"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.sendDeadLetterEmail = exports.startDefaultPartitionEntryCheckService = exports.startDeadLetterQueueCheckService = void 0;
const luxon_1 = require("luxon");
const app_logs_1 = require("../../../app.logs");
const messageBroker_amqp_to_monitor_sub_1 = require("../../../messageBroker/messageBroker.amqp-to-monitor.sub");
const messageBroker_cli_1 = require("../../../messageBroker/messageBroker.cli");
const messagebroker_models_1 = require("../../../messageBroker/messagebroker.models");
const common_utils_1 = require("../../common.utils");
const models_1 = require("../models");
const monitor_config_1 = __importDefault(require("../monitor.config"));
const incidentNotification_1 = require("./incidentNotification");
const app_config_1 = require("../../../app.config");
const cli_queries_1 = require("../../../dal/access/psql/cli-queries");
const pgmaintenance_db_utils_1 = require("../../pgmaintenance/db/pgmaintenance.db.utils");
const appTag = "[app.monitor.deadLetterQueueCheck]";
async function startDeadLetterQueueCheckService(pool) {
    app_logs_1.logger.info("%sConnecting to dead letter queue", appTag);
    app_logs_1.logger.info("%sStarting dead letter queue status", appTag);
    setDeadLetterInterval(pool);
}
exports.startDeadLetterQueueCheckService = startDeadLetterQueueCheckService;
async function startDefaultPartitionEntryCheckService(pool) {
    setInterval(async () => {
        try {
            const tablesToReport = await (0, cli_queries_1.systemTransaction)(pool, async (trx) => {
                return checkDefaultPartitionEntries(trx);
            });
            if (tablesToReport.length) {
                const emailContent = defaultPartitionEntryEmailContent(tablesToReport);
                await (0, incidentNotification_1.notifyMonitorServiceEmailReceivers)(emailContent);
            }
        }
        catch (error) {
            app_logs_1.logger.error(`%sError while checking default partition entries %s`, appTag);
            app_logs_1.logger.error(error);
        }
    }, 7 * 24 * 60 * 60 * 1000);
}
exports.startDefaultPartitionEntryCheckService = startDefaultPartitionEntryCheckService;
async function checkDefaultPartitionEntries(trx) {
    const tablesToReport = [];
    const tablesWithPartitions = await (0, pgmaintenance_db_utils_1.findTablesWithPartitions)(trx);
    for (const twp of tablesWithPartitions) {
        const defaulPartitionName = twp.table_name + "_default";
        const defaultTableEntriesDbResult = await trx.query(`
			SELECT DATE_TRUNC('month', "${twp.partition_key}") AS month, COUNT(*) AS "count"
			FROM "${twp.schema}"."${defaulPartitionName}"
			GROUP BY DATE_TRUNC('month', "${twp.partition_key}")
			ORDER BY DATE_TRUNC('month', "${twp.partition_key}") ASC
		`);
        if (defaultTableEntriesDbResult.rowCount) {
            tablesToReport.push({
                tableName: twp.table_name,
                schemaName: twp.schema,
                entries: defaultTableEntriesDbResult.rows,
            });
        }
    }
    return tablesToReport;
}
async function checkDeadLetterQueue() {
    const queueDetails = await (0, messageBroker_cli_1.getQueueDetails)({
        hostname: monitor_config_1.default.AMQP_HOSTNAME,
        port: monitor_config_1.default.AMQP_MANAGEMENT_API_PORT,
        queueName: messagebroker_models_1.MessageBrokerNames.deadLetter.deviceQueue,
        authorization: monitor_config_1.default.AMQP_MANAGEMENT_AUTH,
    });
    if (queueDetails.messages > 0) {
        const deadLetterQueueMessages = await getDeadLetterQueueMessages();
        return {
            messages: deadLetterQueueMessages,
            idleSince: luxon_1.DateTime.fromSQL(queueDetails.idle_since, { zone: "UTC" }).toJSDate().toLocaleString(monitor_config_1.default.LOCALE),
        };
    }
    return null;
}
async function getDeadLetterQueueMessages() {
    const getDeadLetterQueueMessagesResponse = await (0, messageBroker_cli_1.getQueueMessages)({
        hostname: monitor_config_1.default.AMQP_HOSTNAME,
        port: monitor_config_1.default.AMQP_MANAGEMENT_API_PORT,
        queueName: messagebroker_models_1.MessageBrokerNames.deadLetter.deviceQueue,
        authorization: monitor_config_1.default.AMQP_MANAGEMENT_AUTH,
    });
    return getDeadLetterQueueMessagesResponse.map((r) => {
        return {
            organizationId: r.properties.headers.o,
            deviceId: r.properties.headers.d,
        };
    });
}
function setDeadLetterInterval(pool) {
    setInterval(async () => {
        app_logs_1.logger.info("%sDeadLetter checking...", appTag);
        try {
            const connection = await (0, common_utils_1.connectAmqpWithProcessEnv)(monitor_config_1.default, "monitor.messageBroker.manager.ts");
            await (0, messageBroker_amqp_to_monitor_sub_1.amqpToMonitorChannelInit)(connection, pool);
            setTimeout(async () => {
                const queueStatus = await checkDeadLetterQueue();
                if (queueStatus?.messages?.length) {
                    app_logs_1.logger.info(`%squeueMessageCount: ${queueStatus.messages.length}, idleSince: ${queueStatus.idleSince}`, appTag);
                    await sendDeadLetterEmail({ type: models_1.MonitorEmailType.IntervalDeadLetterControl, deadLetterQueueStatus: queueStatus });
                }
                else {
                    app_logs_1.logger.info("%sDeadLetter is empty.", appTag);
                }
                await (0, messageBroker_amqp_to_monitor_sub_1.closeDeadLetterChannel)();
            }, monitor_config_1.default.DEAD_LETTER_CHECK_DURATION_IN_SECONDS * 1000);
        }
        catch (error) {
            if (error.code === "ECONNREFUSED") {
                app_logs_1.logger.info("%sFailed to connect Amqp Management API.", appTag);
                await sendDeadLetterEmail({ type: models_1.MonitorEmailType.ManagementApiFailed });
            }
            else {
                app_logs_1.logger.info("%DeadLetter İnterval Encountered an error." + error, appTag);
                await sendDeadLetterEmail({ type: models_1.MonitorEmailType.UnknownError, error });
            }
        }
    }, monitor_config_1.default.DEAD_LETTER_CHECK_INTERVAL_IN_SECONDS * 1000);
}
async function sendDeadLetterEmail(params) {
    try {
        app_logs_1.logger.warn("%sNotification sending...", appTag);
        let content;
        switch (params.type) {
            case models_1.MonitorEmailType.DeadLetterMessage:
                content = deadLetterHasMessageEmailContent(params.deadLetterQueueStatus);
                break;
            case models_1.MonitorEmailType.ManagementApiFailed:
                content = managementApiFailedEmailContent();
                break;
            case models_1.MonitorEmailType.IntervalDeadLetterControl:
                content = intervalCheckEmailContent(params.deadLetterQueueStatus);
                break;
            case models_1.MonitorEmailType.UnknownError:
                content = unknownErrorEmailContent(params.error);
                break;
        }
        await (0, incidentNotification_1.notifyMonitorServiceEmailReceivers)(content);
        app_logs_1.logger.warn("%sNotification sended", appTag);
    }
    catch (error) {
        app_logs_1.logger.error("%sNotification couldn't send: " + error, appTag);
    }
}
exports.sendDeadLetterEmail = sendDeadLetterEmail;
function deadLetterHasMessageEmailContent(status) {
    if (monitor_config_1.default.LOCALE === "en") {
        return {
            subject: `DeadLetter Message Has Message!`,
            text: `
				Dear System Administrator,
				Dead letter Queue caught a message and shut itself down. 
				You should check the server.

				Server name: ${app_config_1.appConfig.serverName},
				OrganizationId in Incoming Message: ${status.messages[0].organizationId},
				DeviceId in Incoming Message: ${status.messages[0].deviceId},
				Number of Messages in Queue : ${status.messages.length},
				Date of Last Message Received: ${status.idleSince},
				`,
        };
    }
    else {
        return {
            subject: `DeadLetter Mesaj Yakaladı!`,
            text: `
			Sayın Sistem Yöneticisi,
			Deadletter Kuyruğu bir mesaj yakaladı ve kendini kapattı.
			Sunucuyu kontrol etmelisiniz.
			
			Sunucu adı: ${app_config_1.appConfig.serverName},
			Gelen Mesajdaki OrganizationId: ${status.messages[0].organizationId},
			Gelen Mesajdaki DeviceId: ${status.messages[0].deviceId},
			DeadLetter' daki Mesaj Sayısı: ${status.messages.length},
			Son Mesajın Alındığı Tarih: ${status.idleSince}, 
				`,
        };
    }
}
function intervalCheckEmailContent(status) {
    if (monitor_config_1.default.LOCALE === "en") {
        return {
            subject: `Monitor Service DeadLetter Check Alert!`,
            text: `
			Dear System Administrator,
			There are messages in DeadLetter Queue on the server whose information is given below.
			
			Server Name: ${app_config_1.appConfig.serverName},
			Affected OrganizationIds: ${[...new Set(status.messages.map((m) => m.organizationId))]},
			Affected DeviceIds: ${[...new Set(status.messages.map((m) => m.deviceId))]},
			Number of Messages in Queue: ${status.messages.length},
			Date of Last Message Received: ${status.idleSince}, 
				`,
        };
    }
    else {
        return {
            subject: `Monitor Servisi DeadLetter Kontrolü Alarmı!`,
            text: `
			Sayın Sistem Yöneticisi,
			Aşağıda bilgileri verilen sunucuda DeadLetter Kuyruğunda mesajlar bulunmaktadır.
			
			Sunucu adı: ${app_config_1.appConfig.serverName},
			Etkilenen OrganizationIdleri: ${[...new Set(status.messages.map((m) => m.organizationId))]},
			Etkilenen DeviceIdleri: ${[...new Set(status.messages.map((m) => m.deviceId))]},
			Sıradaki Mesaj Sayısı: ${status.messages.length},
			Son Mesajın Alındığı Tarih: ${status.idleSince}, 
				`,
        };
    }
}
function unknownErrorEmailContent(error) {
    if (monitor_config_1.default.LOCALE === "en") {
        return {
            subject: `The Monitor Service Encountered an Unknown Error`,
            text: `
			Dear System Administrator,
			Dead Letter Check encountered an unknown error detailed below.
			Server: ${app_config_1.appConfig.serverName}
			Error: ${error} 
				`,
        };
    }
    else {
        return {
            subject: `Monitor Servisi Bilinmeyen Hata Yakaladı!`,
            text: `
				Sayın Sistem Yöneticisi,
				Dead Letter kontrolü aşağıda detayları verilen bilinmeyen bir hatayla karşılaştı.
				Sunucu: ${app_config_1.appConfig.serverName}
				Error: ${error}
				`,
        };
    }
}
function managementApiFailedEmailContent() {
    if (monitor_config_1.default.LOCALE === "en") {
        return {
            subject: `Monitor Service Could not Reach the Management API.`,
            text: `
				Dear System Administrator,
				Monitor Service could not reach RabbitMQ Management API service.
				Server Name: ${app_config_1.appConfig.serverName},
				Date: ${new Date().toLocaleString("en")}
				`,
        };
    }
    else {
        return {
            subject: `Monitor Servisi Management API'ya ulaşamadı.`,
            text: `
				Sayın Sistem Yöneticisi,
				Monitor Servisi, RabbitMQ Management API servisine ulaşamadı.
				Sunucu adi: ${app_config_1.appConfig.serverName},
				Tarih: ${new Date().toLocaleString("tr")}
				`,
        };
    }
}
function defaultPartitionEntryEmailContent(tablesWithPartitions) {
    if (monitor_config_1.default.LOCALE === "en") {
        return {
            subject: `There are entries en default partitions.`,
            text: `
				Dear System Administrator,
				Monitor Service detected default partition entries for tables below.
				Server Name: ${app_config_1.appConfig.serverName},
				Date: ${new Date().toLocaleString("en")}
				Details: ${tablesWithPartitions.map((m) => {
                return `
					Schema: ${m.schemaName}
					Table: ${m.tableName}
					Entry Counts: ${m.entries
                    .map((mm) => `
						Month: ${mm.month.toLocaleDateString("default", { month: "2-digit", year: "numeric" })}, Entry Count: ${mm.count}`)
                    .join()}`;
            }).join(`
					===========================`)}
				`,
        };
    }
    else {
        return {
            subject: `Default partitionda entry olduğu saptandı.`,
            text: `
				Sayın Sistem Yöneticisi,
				Monitor Servisi, Aşağıda bilgileri olan tablolar için default partition girdisi saptadı.
				Sunucu adi: ${app_config_1.appConfig.serverName},
				Tarih: ${new Date().toLocaleString("tr")}
				Detaylar: ${tablesWithPartitions.map((m) => {
                return `
					Şema: ${m.schemaName}
					Tablo: ${m.tableName}
					Satır Sayıları: ${m.entries
                    .map((mm) => `
						Ay: ${mm.month.toLocaleDateString("default", { month: "2-digit", year: "numeric" })}, Satır Sayısı: ${mm.count}`)
                    .join()}`;
            }).join(`
					===========================`)}
				`,
        };
    }
}
