"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.EnerjisaPerakendeUserTask = void 0;
const integration_base_1 = require("../integration.base");
const dal_db_armon_schema_1 = require("../../../../dal/db/armon/dal.db.armon.schema");
const app_enums_1 = require("../../../../app.enums");
const predefined_roles_1 = require("../../../../dal/db/predefined/predefined.roles");
const uuid_1 = __importDefault(require("uuid"));
const luxon_1 = require("luxon");
const app_logs_1 = require("../../../../app.logs");
const integration_config_1 = __importDefault(require("../../integration.config"));
class EnerjisaPerakendeUserTask {
    constructor() {
        this.organizationId = integration_config_1.default.INTEGRATION_ORGANIZATION;
        this.getRandomColor = () => {
            const letters = "0123456789ABCDEF";
            let color = "";
            for (let i = 0; i < 6; i++) {
                color += letters[Math.floor(Math.random() * 16)];
            }
            return color;
        };
    }
    static create() {
        return new EnerjisaPerakendeUserTask();
    }
    async execute(trx, token, taskId, data) {
        app_logs_1.logger.info("Executing task for: " + data.id);
        if (!EnerjisaPerakendeUserTask.unitTypeId || !EnerjisaPerakendeUserTask.officeTypeId) {
            const typeId = await trx.query(`SELECT vals.id, (vals.value)::jsonb->>'caption_tr' as name
                FROM "${this.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.organizationUserDefinedLists}" list
                INNER JOIN "${this.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.organizationUserDefinedListValues}" vals
                    ON vals."userDefinedListId" = list.id
                WHERE list.type = $1 AND ((vals.value)::jsonb->>'caption_tr' = 'Birim' OR (vals.value)::jsonb->>'caption_tr' = 'Ofis')`, [app_enums_1.enums.OrganizationUserDefinedListType.OrganizationUnitType]);
            EnerjisaPerakendeUserTask.unitTypeId = typeId.rows.find((r) => r.name === "Birim").id;
            EnerjisaPerakendeUserTask.officeTypeId = typeId.rows.find((r) => r.name === "Ofis").id;
            EnerjisaPerakendeUserTask.fixedData = {
                "0090": { id: "0090", name: "ENERJİSA ENERJİ", shortCode: "0090", typeId: EnerjisaPerakendeUserTask.unitTypeId },
                "0690": { id: "0690", name: "TOROSLAR PERAKENDE", shortCode: "0690", typeId: EnerjisaPerakendeUserTask.unitTypeId },
                "0670": { id: "0670", name: "AYESAŞ", shortCode: "0670", typeId: EnerjisaPerakendeUserTask.unitTypeId },
                "0220": { id: "0220", name: "BAŞKENT PERAKENDE", shortCode: "0220", typeId: EnerjisaPerakendeUserTask.unitTypeId },
                "0291": { id: "0291", name: "EŞARJ", shortCode: "0291", typeId: EnerjisaPerakendeUserTask.unitTypeId },
                "0290": { id: "0290", name: "MÜŞTERİ ÇÖZÜMLERİ", shortCode: "0290", typeId: EnerjisaPerakendeUserTask.unitTypeId },
                "0292": { id: "0292", name: "FİLO", shortCode: "0292", typeId: EnerjisaPerakendeUserTask.unitTypeId },
                location: { id: "loc-root", name: "Lokasyonlar", shortCode: "Lokasyonlar", typeId: EnerjisaPerakendeUserTask.officeTypeId },
            };
        }
        if (!EnerjisaPerakendeUserTask.unitAdminId || !EnerjisaPerakendeUserTask.standardUserId || !EnerjisaPerakendeUserTask.orgWideStandardUserId) {
            const typeId = await trx.query(`SELECT id, "typeId"
                FROM "${this.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.roles}" roles
                WHERE "typeId" = ANY($1::uuid[]) AND "deletedAt" IS NULL
                `, [[predefined_roles_1.PredefinedRoles.UnitAdministrator.id, predefined_roles_1.PredefinedRoles.StandartUser.id, predefined_roles_1.PredefinedRoles.StandartUserOrganization.id]]);
            EnerjisaPerakendeUserTask.unitAdminId = typeId.rows.find((r) => r.typeId === predefined_roles_1.PredefinedRoles.UnitAdministrator.id).id;
            EnerjisaPerakendeUserTask.standardUserId = typeId.rows.find((r) => r.typeId === predefined_roles_1.PredefinedRoles.StandartUser.id).id;
            EnerjisaPerakendeUserTask.orgWideStandardUserId = typeId.rows.find((r) => r.typeId === predefined_roles_1.PredefinedRoles.StandartUserOrganization.id).id;
        }
        if (!EnerjisaPerakendeUserTask.unitAdminId ||
            !EnerjisaPerakendeUserTask.standardUserId ||
            !EnerjisaPerakendeUserTask.orgWideStandardUserId ||
            !EnerjisaPerakendeUserTask.unitTypeId ||
            !EnerjisaPerakendeUserTask.officeTypeId) {
            app_logs_1.logger.error("Predefined ids could not be determined, aborting task");
            return integration_base_1.OperationStatus.Failure;
        }
        const taskdata = {
            uniqueId: data.id,
            taskid: taskId,
        };
        const receivedOrganizations = [];
        const orgs = (data.organizations ?? [])
            .map((ou) => {
            return {
                id: data.bukrs + ou.id,
                shortCode: ou.id,
                name: EnerjisaPerakendeUserTask.prefixes[data.bukrs] + ou.name,
                typeId: EnerjisaPerakendeUserTask.unitTypeId,
            };
        })
            .reverse();
        orgs.unshift(EnerjisaPerakendeUserTask.fixedData[data.bukrs]);
        let result = await this.syncOrganizationUnitTree(trx, token, taskdata, orgs);
        if (!result) {
            return integration_base_1.OperationStatus.Failure;
        }
        if (data.ismanager) {
            app_logs_1.logger.info("User will be manager for " + result + " when user profile is syncronized");
            receivedOrganizations.push({ id: result, role: EnerjisaPerakendeUserTask.unitAdminId });
        }
        else {
            receivedOrganizations.push({ id: result, role: EnerjisaPerakendeUserTask.standardUserId });
        }
        const locations = [data.location1, data.location2, data.location3]
            .map((loc) => {
            if (loc) {
                return {
                    id: "loc-" + loc,
                    shortCode: loc,
                    name: loc,
                    typeId: EnerjisaPerakendeUserTask.officeTypeId,
                };
            }
            else {
                return undefined;
            }
        })
            .filter((loc) => loc);
        if (locations.length > 0) {
            locations.unshift(EnerjisaPerakendeUserTask.fixedData["location"]);
            let result = await this.syncOrganizationUnitTree(trx, token, taskdata, locations);
            if (!result) {
                return integration_base_1.OperationStatus.Failure;
            }
            receivedOrganizations.push({ id: result, role: EnerjisaPerakendeUserTask.standardUserId });
        }
        const user = (await trx.query(`SELECT uop.*, uo."isDisabled", uo."roleId"
                FROM "${this.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userOrganizationProfiles}" uop
                INNER JOIN "${this.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userOrganizations}" uo
					ON uop."userOrganizationId" = uo."id"
                WHERE "uniqueId" = $1 AND uo."deletedAt" IS NULL AND uop."deletedAt" IS NULL;`, [data.id])).rows[0];
        if (data.photo) {
            data.photo = data.photo.replace(/^data:image\/\w+;base64,/, "");
        }
        let userGroupId = "";
        let existingGroupId = "";
        if (data.employeetype) {
            try {
                let userGroups = (await trx.query(`SELECT "id", "name"
						FROM "${integration_config_1.default.INTEGRATION_ORGANIZATION}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userGroups}" r
						`)).rows;
                if (user?.extensionFields?.org_employeetype && userGroups.length > 0) {
                    existingGroupId = userGroups.find((group) => group.name === user.extensionFields.org_employeetype).id;
                }
                let groupToAddUser;
                if (userGroups.length > 0) {
                    groupToAddUser = userGroups.find((group) => group.name === data.employeetype);
                }
                if (groupToAddUser) {
                    userGroupId = groupToAddUser.id;
                }
                else {
                    let randomColor = this.getRandomColor();
                    let addGroupBody = {
                        name: data.employeetype,
                        colorCode: randomColor,
                    };
                    let addedGroup = await token.httpRequest(taskdata, "POST", `/u/v1/${integration_config_1.default.INTEGRATION_ORGANIZATION}/usergroup/add`, addGroupBody);
                    if (addedGroup?.status === 200) {
                        app_logs_1.logger.info("Group " + data.employeetype + " added successfully");
                    }
                    else {
                        app_logs_1.logger.error("Could not insert group: " + addedGroup?.status + " - " + addedGroup?.data?.message);
                        return integration_base_1.OperationStatus.Failure;
                    }
                    userGroupId = addedGroup.data.id;
                }
            }
            catch (error) {
                app_logs_1.logger.error("Error during user group operation.");
                app_logs_1.logger.error(error);
            }
        }
        else {
            let userGroupNames = (await trx.query(`SELECT ug."id", ug."name"
					FROM "${integration_config_1.default.INTEGRATION_ORGANIZATION}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userGroups}" ug
					INNER JOIN "${this.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userGroupUserOrganizations}" uguo
					ON uguo."userGroupId" = ug."id"
                	WHERE uguo."userOrganizationId" = $1 AND uguo."deletedAt" IS NULL;`, [user.userOrganizationId])).rows;
            let hasMYAKA = userGroupNames.find((groupName) => {
                groupName.name === "MYAKA";
            });
            let hasBYAKA = userGroupNames.find((groupName) => {
                groupName.name === "BYAKA";
            });
            if (hasMYAKA) {
                existingGroupId = hasMYAKA.id;
            }
            else if (hasBYAKA) {
                existingGroupId = hasBYAKA.id;
            }
        }
        if (!user) {
            app_logs_1.logger.info("Will insert new user for " + data.id);
            const requestBody = {
                id: uuid_1.default.v4(),
                username: null,
                publicKey: null,
                organizationProfile: {
                    roleId: EnerjisaPerakendeUserTask.orgWideStandardUserId,
                    isDisabled: !data.active,
                    uniqueId: data.id,
                    name: data.name,
                    surname: data.surname,
                    email: data.email,
                    phoneNumber: data.phone,
                    extensionFields: {
                        org_company_code: data.bukrs,
                        org_position: data.positionlongtext,
                        org_position_id: data.positionid,
                        org_location: locations
                            ? locations
                                .slice(1)
                                .map((l) => l.name)
                                .join(",")
                            : undefined,
                        org_workunit: data.location4,
                        org_employeetype: data.employeetype,
                    },
                    employmentStartUtc: data.hiredate ? luxon_1.DateTime.fromISO(data.hiredate).setZone("UTC").toString() : undefined,
                    employmentEndUtc: data.firedate ? luxon_1.DateTime.fromISO(data.firedate).setZone("UTC").toString() : undefined,
                    previousServiceDuration: 0,
                    pacsEnabledRemainedAnnualPPermission: 0,
                    locale: "tr",
                    isAnonymized: false,
                },
                organizationUnits: receivedOrganizations.map((o) => {
                    return { organizationUnitId: o.id, roleId: o.role };
                }),
                accessRights: [],
                credentials: [],
                regions: [],
                userGroupIds: [userGroupId],
                addDefaultAccessRightsOfUnits: false,
            };
            const insertedUser = await token.httpRequest(taskdata, "POST", `/u/v1/${this.organizationId}/identity/add`, requestBody);
            if (insertedUser?.status === 200) {
                app_logs_1.logger.info("User " + data.id + " inserted with id: " + insertedUser.data.id);
                if (data.photo) {
                    const thumbnail = await token.httpRequest(taskdata, "POST", `/u/v1/${this.organizationId}/identity/${insertedUser.data.id}/thumbnail/upsert`, {
                        data: data.photo,
                    });
                    if (thumbnail?.status === 200) {
                        app_logs_1.logger.info("Profile picture added");
                    }
                    else {
                        app_logs_1.logger.error("Could not update profile picture: " + thumbnail?.status + " - " + thumbnail?.data?.message);
                    }
                }
            }
            else {
                app_logs_1.logger.error("Could not insert user: " + insertedUser?.status + " - " + insertedUser?.data?.message);
                return integration_base_1.OperationStatus.Failure;
            }
        }
        else {
            app_logs_1.logger.info("Will update user for " + data.id + " with id " + user.userId);
            const existingUnits = (await trx.query(`SELECT "organizationUnitId" as id, uoou."roleId" as role
                    FROM "${this.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userOrganizationOrganizationUnits}" uoou
                    INNER JOIN "${this.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.organizationUnits}" ou
                        ON ou.id = uoou."organizationUnitId" AND uoou."deletedAt" IS NULL AND ou."deletedAt" IS NULL
                    WHERE uoou."userOrganizationId" = $1`, [user.userOrganizationId])).rows;
            const requestBody = {
                addedOrganizationUnits: receivedOrganizations
                    .filter((o) => !existingUnits.find((e) => e.id === o.id))
                    .map((o) => {
                    return {
                        organizationUnitId: o.id,
                        roleId: o.role,
                        extraPermissions: [],
                    };
                }),
                updatedOrganizationUnits: receivedOrganizations
                    .filter((o) => existingUnits.find((e) => e.id === o.id && e.role != o.role && [EnerjisaPerakendeUserTask.unitAdminId, EnerjisaPerakendeUserTask.standardUserId].includes(e.role)))
                    .map((o) => {
                    return {
                        organizationUnitId: o.id,
                        roleId: o.role,
                        extraPermissions: [],
                    };
                }),
                removedOrganizationUnitIds: existingUnits.filter((e) => !receivedOrganizations.find((o) => o.id === e.id)).map((e) => e.id),
                addedAccessRights: [],
                updatedAccessRights: [],
                removedAccessControlPointIds: [],
                addedCredentials: [],
                updatedCredentials: [],
                removedCredentialIds: [],
                addedUserGroupIds: existingGroupId === userGroupId ? [] : [userGroupId],
                removedUserGroupIds: existingGroupId === userGroupId ? [] : [existingGroupId],
                removedRegionIds: [],
                addedRegions: [],
                updatedRegions: [],
                id: user.userId,
                organizationProfile: {
                    roleId: user.roleId !== EnerjisaPerakendeUserTask.orgWideStandardUserId ? user.roleId : EnerjisaPerakendeUserTask.orgWideStandardUserId,
                    isDisabled: user.isDisabled ? user.isDisabled : !data.active,
                    uniqueId: data.id,
                    name: data.name,
                    surname: data.surname,
                    email: data.email,
                    phoneNumber: data.phone,
                    extensionFields: {
                        org_company_code: data.bukrs,
                        org_position: data.positionlongtext,
                        org_position_id: data.positionid,
                        org_location: locations
                            ? locations
                                .slice(1)
                                .map((l) => l.name)
                                .join(",")
                            : undefined,
                        org_workunit: data.location4,
                        org_employeetype: data.employeetype,
                    },
                    employmentStartUtc: data.hiredate ? luxon_1.DateTime.fromISO(data.hiredate).setZone("UTC").toString() : undefined,
                    employmentEndUtc: data.firedate ? luxon_1.DateTime.fromISO(data.firedate).setZone("UTC").toString() : undefined,
                    previousServiceDuration: 0,
                    pacsEnabledRemainedAnnualPPermission: 0,
                    locale: "tr",
                    isAnonymized: false,
                },
                addDefaultAccessRightsOfUnits: false,
                removeDefaultAccessRightsOfUnits: false,
            };
            const updatedUser = await token.httpRequest(taskdata, "POST", `/u/v1/${this.organizationId}/identity/update`, requestBody);
            if (updatedUser?.status === 200) {
                app_logs_1.logger.info("User " + data.id + " updated successfully");
                if (data.photo) {
                    const thumbnail = await token.httpRequest(taskdata, "POST", `/u/v1/${this.organizationId}/identity/${user.userId}/thumbnail/upsert`, {
                        data: data.photo,
                    });
                    if (thumbnail?.status === 200) {
                        app_logs_1.logger.info("Profile picture updated");
                    }
                    else {
                        app_logs_1.logger.error("Could not update profile picture: " + thumbnail?.status + " - " + thumbnail?.data?.message);
                    }
                }
            }
            else {
                app_logs_1.logger.error("Could not insert user: " + updatedUser?.status + " - " + updatedUser?.data?.message);
                return integration_base_1.OperationStatus.Failure;
            }
        }
        return integration_base_1.OperationStatus.Success;
    }
    async syncOrganizationUnitTree(trx, token, taskdata, units) {
        const internalids = (await trx.query(`SELECT ids."armonId", ids."extId", units.name, units."shortCode", units."parentId"
                FROM "${this.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.integration_idmap}" ids
                INNER JOIN "${this.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.organizationUnits}" units
                    ON ids.type = $1 AND units.id = ids."armonId" AND units."deletedAt" IS NULL
                WHERE "extId" = ANY($2)`, [integration_base_1.IntegrationEntityType.OrganizationUnit, units.map((o) => o.id)])).rows;
        let parentId = null;
        for (const org of units) {
            const counterpart = internalids.find((i) => org.id === i.extId);
            if (!counterpart) {
                const id = await this.upsertOrganizationUnit(taskdata, token, {
                    name: org.name,
                    shortCode: org.shortCode,
                    parentId: parentId,
                    typeId: org.typeId,
                });
                if (!id) {
                    return null;
                }
                else {
                    parentId = id;
                    await trx.query(`INSERT INTO "${this.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.integration_idmap}"
                        ("armonId", "extId", type) VALUES ($1, $2, $3);`, [id, org.id, integration_base_1.IntegrationEntityType.OrganizationUnit]);
                }
            }
            else {
                if (counterpart.name !== org.name || counterpart.shortCode !== org.shortCode || counterpart.parentId != parentId) {
                    const id = await this.upsertOrganizationUnit(taskdata, token, {
                        id: counterpart.armonId,
                        name: org.name,
                        shortCode: org.shortCode,
                        parentId: parentId,
                        typeId: org.typeId,
                    });
                    if (!id) {
                        return null;
                    }
                    parentId = id;
                }
                else {
                    parentId = counterpart.armonId;
                }
            }
        }
        return parentId;
    }
    async upsertOrganizationUnit(taskdata, token, unit) {
        if (unit.parentId === null) {
            delete unit.parentId;
        }
        const response = await token.httpRequest(taskdata, "POST", `/u/v1/${this.organizationId}/organizationunit`, unit);
        if (response?.status === 200) {
            app_logs_1.logger.info(`Unit ${unit.name} successfully upserted with id: ${response.data.id}`);
            return response.data.id;
        }
        else {
            app_logs_1.logger.error(`Unit ${unit.name} could not be upserted`);
            return null;
        }
    }
}
exports.EnerjisaPerakendeUserTask = EnerjisaPerakendeUserTask;
EnerjisaPerakendeUserTask.id = "03dc5927-07b8-4f65-ac71-a6cb9b3043da";
EnerjisaPerakendeUserTask.prefixes = {
    "0090": "(EE) ",
    "0690": "(T) ",
    "0670": "(A) ",
    "0220": "(B) ",
    "0291": "(EŞ) ",
    "0290": "(M) ",
    "0292": "(F) ",
};
