"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
const integration_base_1 = require("./integration.base");
const integration_config_1 = __importDefault(require("../integration.config"));
const dal_db_armon_schema_1 = require("../../../dal/db/armon/dal.db.armon.schema");
const app_logs_1 = require("../../../app.logs");
const api_validatorhelper_1 = require("../../../api/api.validatorhelper");
const cli_queries_1 = require("../../../dal/access/psql/cli-queries");
const app_enums_1 = require("../../../app.enums");
const userUpsert_1 = require("./enerjisa_perakende/userUpsert");
class EnerjisaPerakendeOrg {
    constructor(_pool, apiToken, taskFactory) {
        this.pool = _pool;
        this.apiToken = apiToken;
        this.taskFactory = taskFactory;
    }
    async initialize(router) {
        app_logs_1.logger.info(`Registering custom handler for /integration/${EnerjisaPerakendeOrg.orgId}/user/upsert`);
        router.post(`/integration/${EnerjisaPerakendeOrg.orgId}/user/upsert`, this.upsertUser.bind(this));
        app_logs_1.logger.info(`Registering custom handler for /integration/${EnerjisaPerakendeOrg.orgId}/user/upsert/queued`);
        router.post(`/integration/${EnerjisaPerakendeOrg.orgId}/user/upsert/queued`, this.upsertUserQueued.bind(this));
        this.taskFactory.registerTaskType(userUpsert_1.EnerjisaPerakendeUserTask.id, userUpsert_1.EnerjisaPerakendeUserTask.create);
    }
    async upsertUser(req, res) {
        const body = req.body;
        try {
            const validator = new api_validatorhelper_1.ValidatorHelper();
            this.validateUserRequest(body, validator);
            validator.finalize();
        }
        catch (error) {
            return res.status(400).send(error);
        }
        try {
            let response = await (0, cli_queries_1.systemTransaction)(this.pool, async (trx) => {
                return await userUpsert_1.EnerjisaPerakendeUserTask.create().execute(trx, this.apiToken, "instant", body);
            });
            if (response === integration_base_1.OperationStatus.Success) {
                res.status(200).send();
            }
            else {
                res.status(app_enums_1.enums.HttpStatusCode.INTERNAL_ERROR).send();
            }
        }
        catch (error) {
            app_logs_1.logger.error("Error while executing task:");
            app_logs_1.logger.error(error);
            res.status(app_enums_1.enums.HttpStatusCode.INTERNAL_ERROR).send();
        }
    }
    async upsertUserQueued(req, res) {
        const body = req.body;
        app_logs_1.logger.info("Operation received");
        try {
            const validator = new api_validatorhelper_1.ValidatorHelper();
            validator.validateObjectArray("items", false, body.items);
            if (body.items && Array.isArray(body.items)) {
                for (const item of body.items ?? []) {
                    this.validateUserRequest(item, validator);
                }
            }
            validator.finalize();
        }
        catch (error) {
            app_logs_1.logger.error("Validation errors: " + JSON.stringify(error));
            return res.status(400).send(error);
        }
        try {
            const response = await (0, cli_queries_1.systemTransaction)(this.pool, async (trx) => {
                const opid = await trx.query(`INSERT INTO "${EnerjisaPerakendeOrg.orgId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.integration_ops}"
                    (id, status, "taskCount", "createdAt") VALUES (gen_random_uuid(), $1, $2, now())
                    RETURNING id;`, [integration_base_1.OperationStatus.Pending, body.items.length]);
                const taskids = await trx.query(`INSERT INTO "${EnerjisaPerakendeOrg.orgId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.integration_tasks}"
                    (id, "operationId", index, "typeId", data, status, "createdAt")
                    SELECT gen_random_uuid(), $1, elem.index, $2, elem.item, $3, now()
                    FROM jsonb_array_elements($4::jsonb) WITH ORDINALITY elem(item, index)
                    RETURNING id;`, [opid.rows[0].id, userUpsert_1.EnerjisaPerakendeUserTask.id, integration_base_1.OperationStatus.Pending, JSON.stringify(body.items)]);
                const tasksResponse = {
                    op: opid.rows[0].id,
                    taskids: taskids.rows.map((r) => r.id),
                };
                app_logs_1.logger.info(JSON.stringify(tasksResponse));
                return tasksResponse;
            });
            res.json(response);
        }
        catch (error) {
            app_logs_1.logger.error("Error while receiving task:");
            app_logs_1.logger.error(error);
            res.status(app_enums_1.enums.HttpStatusCode.INTERNAL_ERROR).send();
        }
    }
    validateUserRequest(raw, v) {
        v.validateString({ field: "id", optional: false, input: raw.id, maxLength: 255 });
        v.validateString({ field: raw.id + " name", optional: false, input: raw.name, maxLength: 255 });
        v.validateString({ field: raw.id + " surname", optional: false, input: raw.surname, maxLength: 255 });
        v.validateIsBoolean(raw.id + " active", false, raw.active);
        v.validateDateWithFormat(raw.id + " hiredate", false, "YYYY-MM-DD", raw.hiredate);
        v.validateDateWithFormat(raw.id + " firedate", true, "YYYY-MM-DD", raw.firedate);
        v.validateObjectArray(raw.id + " organizations", true, raw.organizations);
        if (raw.organizations && Array.isArray(raw.organizations)) {
            for (let index = 0; index < raw.organizations.length; ++index) {
                v.validateString({ field: `${raw.id} organizations[${index}].id`, optional: false, input: raw.organizations[index].id, maxLength: 255 });
                v.validateString({ field: `${raw.id} organizations[${index}].name`, optional: false, input: raw.organizations[index].name, maxLength: 255 });
            }
        }
        v.validateIsBoolean(raw.id + " ismanager", true, raw.ismanager);
        v.validateString({ field: raw.id + " positionid", optional: true, input: raw.positionid, maxLength: 255 });
        v.validateString({ field: raw.id + " positionlongtext", optional: true, input: raw.positionlongtext, maxLength: 255 });
        v.validateString({ field: raw.id + " email", optional: true, input: raw.email, maxLength: 255 });
        v.validateString({ field: raw.id + " phone", optional: true, input: raw.phone, maxLength: 255 });
        v.validateString({ field: raw.id + " location1", optional: true, input: raw.location1, maxLength: 255 });
        v.validateString({ field: raw.id + " location2", optional: true, input: raw.location2, maxLength: 255 });
        v.validateString({ field: raw.id + " location3", optional: true, input: raw.location3, maxLength: 255 });
        v.validateString({ field: raw.id + " location4", optional: true, input: raw.location4, maxLength: 255 });
        v.validateEnumString(raw.id + " bukrs", false, userUpsert_1.EnerjisaPerakendeUserTask.prefixes, raw.bukrs);
        v.validateString({ field: raw.id + " employeetype", optional: true, input: raw.employeetype, maxLength: 255 });
        v.validateString({ field: raw.id + " photo", optional: true, input: raw.photo });
    }
}
EnerjisaPerakendeOrg.orgId = integration_config_1.default.INTEGRATION_ORGANIZATION;
exports.default = EnerjisaPerakendeOrg;
