"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
const api_validatorhelper_1 = require("../../../api/api.validatorhelper");
const app_enums_1 = require("../../../app.enums");
const app_logs_1 = require("../../../app.logs");
const cli_queries_1 = require("../../../dal/access/psql/cli-queries");
const dal_db_armon_schema_1 = require("../../../dal/db/armon/dal.db.armon.schema");
const integration_config_1 = __importDefault(require("../integration.config"));
class FenerbahceOrg {
    constructor(_pool, apiToken, taskFactory) {
        this.pool = _pool;
        this.apiToken = apiToken;
        this.taskFactory = taskFactory;
    }
    async initialize(router) {
        app_logs_1.logger.info(`Registering custom handler for /integration/${FenerbahceOrg.orgId}/working-summary`);
        router.post(`/integration/${FenerbahceOrg.orgId}/working-summary`, this.getWorkingSummary.bind(this));
    }
    async getWorkingSummary(req, res) {
        const body = req.body;
        try {
            const validator = new api_validatorhelper_1.ValidatorHelper();
            this.validateSummaryRequest(body, validator);
            validator.finalize();
        }
        catch (error) {
            return res.status(400).send(error);
        }
        try {
            let userIds = [];
            let unitIds = [];
            await (0, cli_queries_1.systemTransaction)(this.pool, async (trx) => {
                const users = (await trx.query(`SELECT uop."userId", uop."uniqueId"
					FROM "${integration_config_1.default.INTEGRATION_ORGANIZATION}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userOrganizationProfiles}" uop
					WHERE uop."uniqueId" =ANY($1::varchar[])`, [body.userIds ?? []])).rows;
                userIds = users.map((user) => user.userId);
                const units = (await trx.query(`SELECT ou."id", ou."shortCode"
					FROM "${integration_config_1.default.INTEGRATION_ORGANIZATION}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.organizationUnits}" ou
					WHERE ou."shortCode" =ANY($1::varchar[])`, [body.unitIds ?? []])).rows;
                unitIds = units.map((unit) => unit.id);
            });
            let reqBody = {
                userFilter: {
                    userIds: userIds,
                    organizationUnitIds: unitIds,
                    applyOrganizationUnitFilterHierarchically: true,
                },
                dateRange: {
                    startDateTime: body.fromDate,
                    endDateTime: body.toDate,
                },
                sortOrder: "ASC",
                sortType: 1,
                paginationRequest: {
                    skip: body.skip,
                    take: body.take,
                },
                customSearch: body.customSearch,
            };
            const taskdata = {
                uniqueId: FenerbahceOrg.orgId,
                taskid: "instant",
            };
            const response = await this.apiToken.httpRequest(taskdata, "POST", `/p/v2/${integration_config_1.default.INTEGRATION_ORGANIZATION}/workingdetails/daterange/multi-employee`, reqBody);
            if (response.status === 200) {
                let resData = response.data;
                res.status(app_enums_1.enums.HttpStatusCode.SUCCESS).send(resData);
            }
            else {
                app_logs_1.logger.error(response);
                res.status(app_enums_1.enums.HttpStatusCode.INTERNAL_ERROR).send();
            }
        }
        catch (error) {
            app_logs_1.logger.error("Error while executing task:");
            app_logs_1.logger.error(error);
            res.status(app_enums_1.enums.HttpStatusCode.INTERNAL_ERROR).send();
        }
    }
    validateSummaryRequest(raw, v) {
        for (let i = 0; i < raw.userIds?.length; i++) {
            v.validateString({ field: `userId[${i}]`, optional: true, input: raw.userIds[i], maxLength: 255 });
        }
        for (let i = 0; i < raw.unitIds?.length; i++) {
            v.validateString({ field: `unitId[${i}]`, optional: true, input: raw.unitIds[i], maxLength: 255 });
        }
        v.validateDateWithFormat(raw.fromDate + " fromDate", false, "YYYY-MM-DD", raw.fromDate);
        v.validateDateWithFormat(raw.toDate + " toDate", false, "YYYY-MM-DD", raw.toDate);
        v.validateDateRange(["fromDate", "toDate"], {
            start: raw.fromDate,
            end: raw.toDate,
            startOptional: false,
            endOptional: false,
        });
        v.validateInteger({ field: raw.skip + " skip", optional: false, input: raw.skip });
        v.validateInteger({ field: raw.take + " take", optional: false, input: raw.take });
        v.validateDateRange(["fromDate", "toDate"], { startOptional: false, start: raw.startDate, endOptional: false, end: raw.endDate });
        v.validateObjectSingleKey("customSearch", true, raw.customSearch);
    }
}
FenerbahceOrg.orgId = integration_config_1.default.INTEGRATION_ORGANIZATION;
exports.default = FenerbahceOrg;
