"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.DefaultWorkPlanTask = void 0;
const luxon_1 = require("luxon");
const app_enums_1 = require("../../../../app.enums");
const app_logs_1 = require("../../../../app.logs");
const dal_db_armon_schema_1 = require("../../../../dal/db/armon/dal.db.armon.schema");
const integration_config_1 = __importDefault(require("../../integration.config"));
const integration_base_1 = require("../../integrations/integration.base");
class DefaultWorkPlanTask {
    static create() {
        return new DefaultWorkPlanTask();
    }
    async execute(trx, token, taskId, workPlanData) {
        app_logs_1.logger.info(`Executing task for user: ${workPlanData.userId}`);
        if (!workPlanData.range.endUtc) {
            workPlanData.range.endUtc = workPlanData.range.startUtc;
        }
        const taskdata = {
            uniqueId: workPlanData.userId,
            taskid: taskId,
        };
        const user = (await trx.query(`SELECT uop."userId", uop."uniqueId"
				FROM "${integration_config_1.default.INTEGRATION_ORGANIZATION}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userOrganizationProfiles}" uop
				WHERE uop."uniqueId" = $1`, [workPlanData.userId])).rows[0];
        if (!user)
            throw {
                status: app_enums_1.enums.HttpStatusCode.BAD_REQUEST,
                message: "Kullanıcı bulunamadı",
            };
        const workplan = (await trx.query(`SELECT wp.offset
            FROM "${integration_config_1.default.INTEGRATION_ORGANIZATION}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.workPlans}" wp
            WHERE wp.id = $1`, [workPlanData.workPlanId])).rows[0];
        if (!workplan)
            throw {
                status: app_enums_1.enums.HttpStatusCode.BAD_REQUEST,
                message: "Çalışma planı bulunamadı",
            };
        const workplanPeriods = (await trx.query(`SELECT wpp."offset", wpp."periodStartDateTime"
            FROM "${integration_config_1.default.INTEGRATION_ORGANIZATION}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.workPlanPeriods}" wpp
            WHERE wpp."workPlanId" = $1 ORDER BY wpp."periodStartDateTime" ASC`, [workPlanData.workPlanId])).rows;
        let startPeriod;
        let endPeriod;
        workplanPeriods.forEach((period) => {
            if (luxon_1.DateTime.fromJSDate(period.periodStartDateTime) < luxon_1.DateTime.fromISO(workPlanData.range.startUtc)) {
                startPeriod = period;
            }
            if (luxon_1.DateTime.fromJSDate(period.periodStartDateTime) < luxon_1.DateTime.fromISO(workPlanData.range.endUtc)) {
                endPeriod = period;
            }
        });
        if (this.isDateString(workPlanData.range.startUtc) && this.isDateString(workPlanData.range.endUtc)) {
            let startDate = luxon_1.DateTime.fromISO(workPlanData.range.startUtc);
            let endDate = luxon_1.DateTime.fromISO(workPlanData.range.endUtc);
            startDate = startDate.plus({ hours: startPeriod?.offset ?? workplan.offset });
            endDate = endDate.plus({ hours: endPeriod?.offset ?? workplan.offset });
            workPlanData.range.startUtc = startDate.toString();
            workPlanData.range.endUtc = endDate.toString();
        }
        const reqBody = {
            items: [
                {
                    workplanId: workPlanData.workPlanId,
                    range: {
                        startDateTime: workPlanData.range.startUtc,
                        endDateTime: workPlanData.range.endUtc,
                    },
                },
            ],
            range: {
                startDateTime: workPlanData.range.startUtc,
                endDateTime: workPlanData.range.endUtc,
            },
            userId: user.userId,
        };
        let response;
        try {
            response = await token.httpRequest(taskdata, "POST", `/p/v1/${integration_config_1.default.INTEGRATION_ORGANIZATION}/workplanmembership/updaterange`, reqBody);
        }
        catch (error) {
            app_logs_1.logger.error("Could not apply workplan: " + error.status + " - " + error.message);
            throw error;
        }
        if (response?.status === 200) {
            app_logs_1.logger.info(`Workplan successfully applied to user with id: ${workPlanData.userId}`);
        }
        return integration_base_1.OperationStatus.Success;
    }
    isDateString(dateString) {
        const dateTime = luxon_1.DateTime.fromISO(dateString);
        return dateTime.isValid && dateTime.hour === 0 && dateTime.minute === 0 && dateTime.second === 0;
    }
}
exports.DefaultWorkPlanTask = DefaultWorkPlanTask;
DefaultWorkPlanTask.id = "1a00768c-0743-4208-a792-755f69e09aec";
