"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.WorkPlanAPI = void 0;
const api_validatorhelper_1 = require("../../../../api/api.validatorhelper");
const app_logs_1 = require("../../../../app.logs");
const cli_queries_1 = require("../../../../dal/access/psql/cli-queries");
const dal_db_armon_schema_1 = require("../../../../dal/db/armon/dal.db.armon.schema");
const integration_config_1 = __importDefault(require("../../integration.config"));
const integration_base_1 = require("../../integrations/integration.base");
const task_workplan_1 = require("./task.workplan");
const luxon_1 = require("luxon");
const api_error_1 = require("../../../../api/api.error");
const app_enums_1 = require("../../../../app.enums");
const app_constants_1 = require("../../../../app.constants");
class WorkPlanAPI {
    constructor(_pool, apiToken, taskFactory) {
        this.pool = _pool;
        this.apiToken = apiToken;
        this.taskFactory = taskFactory;
        this.taskFactory.registerTaskType(task_workplan_1.DefaultWorkPlanTask.id, task_workplan_1.DefaultWorkPlanTask.create);
    }
    async insertWorkPlan(req, res) {
        try {
            const workPlanData = req.body;
            try {
                const validator = new api_validatorhelper_1.ValidatorHelper();
                this.validateWorkPlanRequest(workPlanData, validator);
                validator.finalize();
            }
            catch (error) {
                return res.status(app_enums_1.enums.HttpStatusCode.BAD_REQUEST).send(error);
            }
            try {
                let response = await (0, cli_queries_1.systemTransaction)(this.pool, async (trx) => {
                    app_logs_1.logger.info("Execution starting for client: " + req.header(app_constants_1.ArmonHeaders.X_Armon_ClientId));
                    return await task_workplan_1.DefaultWorkPlanTask.create().execute(trx, this.apiToken, "instant", workPlanData);
                });
                if (response === integration_base_1.OperationStatus.Success) {
                    res.status(app_enums_1.enums.HttpStatusCode.SUCCESS).send();
                }
                else {
                    res.status(app_enums_1.enums.HttpStatusCode.INTERNAL_ERROR).send();
                }
            }
            catch (error) {
                app_logs_1.logger.error("Error while executing task:");
                app_logs_1.logger.error(error);
                if (error.status && error.message) {
                    res.status(error.status).send(error.message);
                }
                else {
                    res.status(app_enums_1.enums.HttpStatusCode.INTERNAL_ERROR).send();
                }
            }
        }
        catch (error) {
            app_logs_1.logger.error("Error while receiving task:");
            app_logs_1.logger.error(error);
            res.status(app_enums_1.enums.HttpStatusCode.INTERNAL_ERROR).send();
        }
    }
    async insertWorkPlanQueued(req, res) {
        const workPlanData = req.body;
        app_logs_1.logger.info("Operation received");
        try {
            const validator = new api_validatorhelper_1.ValidatorHelper();
            validator.validateObjectArray("items", false, workPlanData.items);
            if (workPlanData.items && Array.isArray(workPlanData.items)) {
                for (const item of workPlanData.items ?? []) {
                    this.validateWorkPlanRequest(item, validator);
                }
            }
            validator.finalize();
        }
        catch (error) {
            app_logs_1.logger.error("Validation errors: " + JSON.stringify(error));
            return res.status(app_enums_1.enums.HttpStatusCode.BAD_REQUEST).send(error);
        }
        try {
            const response = await (0, cli_queries_1.systemTransaction)(this.pool, async (trx) => {
                const opid = await trx.query(`INSERT INTO "${integration_config_1.default.INTEGRATION_ORGANIZATION}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.integration_ops}"
                    (id, status, "taskCount", "createdAt", "clientId") VALUES (gen_random_uuid(), $1, $2, now(), $3)
                    RETURNING id;`, [integration_base_1.OperationStatus.Pending, workPlanData.items.length, req.header(app_constants_1.ArmonHeaders.X_Armon_ClientId)]);
                const taskids = await trx.query(`INSERT INTO "${integration_config_1.default.INTEGRATION_ORGANIZATION}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.integration_tasks}"
                    (id, "operationId", index, "typeId", data, status, "createdAt")
                    SELECT gen_random_uuid(), $1, elem.index, $2, elem.item, $3, now()
                    FROM jsonb_array_elements($4::jsonb) WITH ORDINALITY elem(item, index)
                    RETURNING id;`, [opid.rows[0].id, task_workplan_1.DefaultWorkPlanTask.id, integration_base_1.OperationStatus.Pending, JSON.stringify(workPlanData.items)]);
                const tasksResponse = {
                    op: opid.rows[0].id,
                    taskids: taskids.rows.map((r) => r.id),
                };
                app_logs_1.logger.info(JSON.stringify(tasksResponse));
                return tasksResponse;
            });
            res.json(response);
        }
        catch (error) {
            app_logs_1.logger.error("Error while receiving task:");
            app_logs_1.logger.error(error);
            res.status(app_enums_1.enums.HttpStatusCode.INTERNAL_ERROR).send();
        }
    }
    async listWorkPlans(req, res) {
        const listReq = req.body;
        try {
            const validator = new api_validatorhelper_1.ValidatorHelper();
            this.validateListRequest(listReq, validator);
            validator.finalize();
        }
        catch (error) {
            return res.status(app_enums_1.enums.HttpStatusCode.BAD_REQUEST).send(error);
        }
        try {
            const taskdata = {
                uniqueId: integration_config_1.default.INTEGRATION_ORGANIZATION,
                taskid: "instant",
            };
            const response = await this.apiToken.httpRequest(taskdata, "POST", `/p/v1/${integration_config_1.default.INTEGRATION_ORGANIZATION}/workplan/list`, listReq);
            if (response.status === 200) {
                let resData = response.data.items.map((item) => {
                    return {
                        id: item.id,
                        name: item.name,
                    };
                });
                res.status(app_enums_1.enums.HttpStatusCode.SUCCESS).send(resData);
            }
            else {
                res.status(app_enums_1.enums.HttpStatusCode.INTERNAL_ERROR).send();
            }
        }
        catch (error) {
            app_logs_1.logger.error("Error while fetching workplans:");
            app_logs_1.logger.error(error);
            res.status(app_enums_1.enums.HttpStatusCode.INTERNAL_ERROR).send();
        }
        return null;
    }
    detectDateOrDatetime(dateString) {
        let dateTime = luxon_1.DateTime.fromFormat(dateString, "yyyy-MM-dd");
        if (dateTime.isValid) {
            return "date";
        }
        else if (!dateTime.isValid) {
            dateTime = luxon_1.DateTime.fromISO(dateString);
            if (dateTime.isValid) {
                return "datetime";
            }
            else {
                return "invalid";
            }
        }
    }
    isStartDateBeforeEndDate(startDateString, endDateString) {
        const startDate = luxon_1.DateTime.fromISO(startDateString);
        const endDate = luxon_1.DateTime.fromISO(endDateString);
        return startDate < endDate;
    }
    validateWorkPlanRequest(raw, v) {
        let isOnlyStart = false;
        let checkStartDate = this.detectDateOrDatetime(raw.range.startUtc);
        if (!raw.range.endUtc) {
            isOnlyStart = true;
        }
        if (!isOnlyStart) {
            let checkEndDate = this.detectDateOrDatetime(raw.range.endUtc);
            if (checkStartDate === "date") {
                if (checkEndDate !== "date") {
                    throw (0, api_error_1.generateBadRequestApiError)({
                        message: "Invalid date formats",
                    });
                }
            }
            else if (checkStartDate === "datetime") {
                if (checkEndDate !== "datetime") {
                    throw (0, api_error_1.generateBadRequestApiError)({
                        message: "Invalid date formats",
                    });
                }
            }
            else {
                throw (0, api_error_1.generateBadRequestApiError)({
                    message: "Invalid date formats",
                });
            }
            if (!this.isStartDateBeforeEndDate(raw.range.startUtc, raw.range.endUtc)) {
                throw (0, api_error_1.generateBadRequestApiError)({
                    message: "Start date should be earlier than end date",
                });
            }
        }
        else {
            if (checkStartDate !== "date") {
                throw (0, api_error_1.generateBadRequestApiError)({
                    message: "Invalid date formats",
                });
            }
        }
        v.validateString({ field: raw.userId + " userId", optional: false, input: raw.userId });
        v.validateUUID(raw.workPlanId + " workPlanId", false, raw.workPlanId);
        v.finalize();
    }
    validateListRequest(raw, v) {
        v.validateInteger({ field: "type", optional: false, input: raw.type });
    }
}
exports.WorkPlanAPI = WorkPlanAPI;
exports.default = WorkPlanAPI;
