"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.OrganizationUnitAPI = void 0;
const api_validatorhelper_1 = require("../../../../api/api.validatorhelper");
const app_logs_1 = require("../../../../app.logs");
const cli_queries_1 = require("../../../../dal/access/psql/cli-queries");
const dal_db_armon_schema_1 = require("../../../../dal/db/armon/dal.db.armon.schema");
const integration_config_1 = __importDefault(require("../../integration.config"));
const integration_base_1 = require("../../integrations/integration.base");
const task_organizationUnit_1 = require("./task.organizationUnit");
const app_enums_1 = require("../../../../app.enums");
const app_constants_1 = require("../../../../app.constants");
class OrganizationUnitAPI {
    constructor(_pool, apiToken, taskFactory) {
        this.pool = _pool;
        this.apiToken = apiToken;
        this.taskFactory = taskFactory;
        this.taskFactory.registerTaskType(task_organizationUnit_1.DefaultOrganizationUnitTask.id, task_organizationUnit_1.DefaultOrganizationUnitTask.create);
    }
    async upsertUnit(req, res) {
        const unitData = req.body;
        try {
            const validator = new api_validatorhelper_1.ValidatorHelper();
            this.validateUnitRequest(unitData, validator);
            validator.finalize();
        }
        catch (error) {
            return res.status(app_enums_1.enums.HttpStatusCode.BAD_REQUEST).send(error);
        }
        try {
            let response = await (0, cli_queries_1.systemTransaction)(this.pool, async (trx) => {
                app_logs_1.logger.info("Execution starting for client: " + req.header(app_constants_1.ArmonHeaders.X_Armon_ClientId));
                return await task_organizationUnit_1.DefaultOrganizationUnitTask.create().execute(trx, this.apiToken, "instant", unitData);
            });
            if (response === integration_base_1.OperationStatus.Success) {
                res.status(app_enums_1.enums.HttpStatusCode.SUCCESS).send();
            }
            else {
                res.status(app_enums_1.enums.HttpStatusCode.INTERNAL_ERROR).send();
            }
        }
        catch (error) {
            app_logs_1.logger.error("Error while executing task:");
            app_logs_1.logger.error(error);
            if (error.status && error.message) {
                res.status(error.status).send(error.message);
            }
            else {
                res.status(app_enums_1.enums.HttpStatusCode.INTERNAL_ERROR).send();
            }
        }
        return null;
    }
    async upsertUnitQueued(req, res) {
        const unitData = req.body;
        app_logs_1.logger.info("Operation received");
        try {
            const validator = new api_validatorhelper_1.ValidatorHelper();
            validator.validateObjectArray("items", false, unitData.items);
            if (unitData.items && Array.isArray(unitData.items)) {
                for (const item of unitData.items ?? []) {
                    this.validateUnitRequest(item, validator);
                }
            }
            validator.finalize();
        }
        catch (error) {
            app_logs_1.logger.error("Validation errors: " + JSON.stringify(error));
            return res.status(app_enums_1.enums.HttpStatusCode.BAD_REQUEST).send(error);
        }
        try {
            const response = await (0, cli_queries_1.systemTransaction)(this.pool, async (trx) => {
                const opid = await trx.query(`INSERT INTO "${integration_config_1.default.INTEGRATION_ORGANIZATION}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.integration_ops}"
                    (id, status, "taskCount", "createdAt", "clientId") VALUES (gen_random_uuid(), $1, $2, now(), $3)
                    RETURNING id;`, [integration_base_1.OperationStatus.Pending, unitData.items.length, req.header(app_constants_1.ArmonHeaders.X_Armon_ClientId)]);
                const taskids = await trx.query(`INSERT INTO "${integration_config_1.default.INTEGRATION_ORGANIZATION}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.integration_tasks}"
                    (id, "operationId", index, "typeId", data, status, "createdAt")
                    SELECT gen_random_uuid(), $1, elem.index, $2, elem.item, $3, now()
                    FROM jsonb_array_elements($4::jsonb) WITH ORDINALITY elem(item, index)
                    RETURNING id;`, [opid.rows[0].id, task_organizationUnit_1.DefaultOrganizationUnitTask.id, integration_base_1.OperationStatus.Pending, JSON.stringify(unitData.items)]);
                const tasksResponse = {
                    op: opid.rows[0].id,
                    taskids: taskids.rows.map((r) => r.id),
                };
                app_logs_1.logger.info(JSON.stringify(tasksResponse));
                return tasksResponse;
            });
            res.json(response);
        }
        catch (error) {
            app_logs_1.logger.error("Error while receiving task:");
            app_logs_1.logger.error(error);
            res.status(app_enums_1.enums.HttpStatusCode.INTERNAL_ERROR).send();
        }
    }
    validateUnitRequest(raw, v) {
        v.validateString({ field: "id", optional: false, input: raw.id, maxLength: 255 });
        v.validateString({ field: raw.id + " name", optional: false, input: raw.name, maxLength: 255 });
        v.validateString({ field: raw.id + " parentId", optional: true, input: raw.parentId, maxLength: 255 });
        v.validateString({ field: raw.id + " managerId", optional: true, input: raw.managerId, maxLength: 255 });
    }
}
exports.OrganizationUnitAPI = OrganizationUnitAPI;
exports.default = OrganizationUnitAPI;
