"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.LeaveAPI = void 0;
const api_validatorhelper_1 = require("../../../../api/api.validatorhelper");
const app_logs_1 = require("../../../../app.logs");
const cli_queries_1 = require("../../../../dal/access/psql/cli-queries");
const dal_db_armon_schema_1 = require("../../../../dal/db/armon/dal.db.armon.schema");
const integration_config_1 = __importDefault(require("../../integration.config"));
const integration_base_1 = require("../../integrations/integration.base");
const task_leave_1 = require("./task.leave");
const luxon_1 = require("luxon");
const api_error_1 = require("../../../../api/api.error");
const app_enums_1 = require("../../../../app.enums");
const app_constants_1 = require("../../../../app.constants");
class LeaveAPI {
    constructor(_pool, apiToken, taskFactory) {
        this.pool = _pool;
        this.apiToken = apiToken;
        this.taskFactory = taskFactory;
        this.taskFactory.registerTaskType(task_leave_1.DefaultLeaveTask.id, task_leave_1.DefaultLeaveTask.create);
    }
    async listLeaves(req, res) {
        const listReq = req.body;
        try {
            const validator = new api_validatorhelper_1.ValidatorHelper();
            this.validateListRequest(listReq, validator);
            validator.finalize();
        }
        catch (error) {
            return res.status(app_enums_1.enums.HttpStatusCode.BAD_REQUEST).send(error);
        }
        try {
            const taskdata = {
                uniqueId: integration_config_1.default.INTEGRATION_ORGANIZATION,
                taskid: "instant",
            };
            const response = await this.apiToken.httpRequest(taskdata, "POST", `/p/v1/${integration_config_1.default.INTEGRATION_ORGANIZATION}/ppermission/type/list`, { pagination: listReq });
            if (response.status === 200) {
                let resData = [];
                if (response?.data?.items && response.data.items.length > 0) {
                    resData = response.data.items.map((item) => {
                        return {
                            id: item.id,
                            name: item.name,
                        };
                    });
                }
                res.status(app_enums_1.enums.HttpStatusCode.SUCCESS).send(resData);
            }
            else {
                res.status(app_enums_1.enums.HttpStatusCode.INTERNAL_ERROR).send();
            }
        }
        catch (error) {
            app_logs_1.logger.error("Error while fetching leave types:");
            app_logs_1.logger.error(error);
            res.status(app_enums_1.enums.HttpStatusCode.INTERNAL_ERROR).send();
        }
        return null;
    }
    isMidnight(isoString) {
        const dateTime = luxon_1.DateTime.fromISO(isoString);
        return dateTime.hour === 0 && dateTime.minute === 0 && dateTime.second === 0;
    }
    isMidDay(isoString) {
        const dateTime = luxon_1.DateTime.fromISO(isoString);
        return dateTime.hour === 12 && dateTime.minute === 0 && dateTime.second === 0;
    }
    async insertLeave(req, res) {
        const leaveData = req.body;
        try {
            const validator = new api_validatorhelper_1.ValidatorHelper();
            await this.validateLeaveRequest(leaveData, validator);
            validator.finalize();
        }
        catch (error) {
            app_logs_1.logger.error("Validation errors: " + JSON.stringify(error));
            return res.status(app_enums_1.enums.HttpStatusCode.BAD_REQUEST).send(error);
        }
        try {
            let response = await (0, cli_queries_1.systemTransaction)(this.pool, async (trx) => {
                app_logs_1.logger.info("Execution starting for client: " + req.header(app_constants_1.ArmonHeaders.X_Armon_ClientId));
                return await task_leave_1.DefaultLeaveTask.create().execute(trx, this.apiToken, "instant", leaveData);
            });
            if (response === integration_base_1.OperationStatus.Success) {
                res.status(app_enums_1.enums.HttpStatusCode.SUCCESS).send();
            }
            else {
                res.status(app_enums_1.enums.HttpStatusCode.INTERNAL_ERROR).send();
            }
        }
        catch (error) {
            app_logs_1.logger.error("Error while executing task:");
            app_logs_1.logger.error(error);
            if (error.status && error.message) {
                res.status(error.status).send(error.message);
            }
            else {
                res.status(app_enums_1.enums.HttpStatusCode.INTERNAL_ERROR).send();
            }
        }
    }
    async insertLeaveQueued(req, res) {
        const leaveData = req.body;
        app_logs_1.logger.info("Operation received");
        try {
            const validator = new api_validatorhelper_1.ValidatorHelper();
            validator.validateObjectArray("items", false, leaveData.items);
            if (leaveData.items && Array.isArray(leaveData.items)) {
                for (const item of leaveData.items ?? []) {
                    this.validateLeaveRequest(item, validator);
                }
            }
            validator.finalize();
        }
        catch (error) {
            app_logs_1.logger.error("Validation errors: " + JSON.stringify(error));
            return res.status(app_enums_1.enums.HttpStatusCode.BAD_REQUEST).send(error);
        }
        try {
            const response = await (0, cli_queries_1.systemTransaction)(this.pool, async (trx) => {
                const opid = await trx.query(`INSERT INTO "${integration_config_1.default.INTEGRATION_ORGANIZATION}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.integration_ops}"
                    (id, status, "taskCount", "createdAt", "clientId") VALUES (gen_random_uuid(), $1, $2, now(), $3)
                    RETURNING id;`, [integration_base_1.OperationStatus.Pending, leaveData.items.length, req.header(app_constants_1.ArmonHeaders.X_Armon_ClientId)]);
                const taskids = await trx.query(`INSERT INTO "${integration_config_1.default.INTEGRATION_ORGANIZATION}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.integration_tasks}"
                    (id, "operationId", index, "typeId", data, status, "createdAt")
                    SELECT gen_random_uuid(), $1, elem.index, $2, elem.item, $3, now()
                    FROM jsonb_array_elements($4::jsonb) WITH ORDINALITY elem(item, index)
                    RETURNING id;`, [opid.rows[0].id, task_leave_1.DefaultLeaveTask.id, integration_base_1.OperationStatus.Pending, JSON.stringify(leaveData.items)]);
                const tasksResponse = {
                    op: opid.rows[0].id,
                    taskids: taskids.rows.map((r) => r.id),
                };
                app_logs_1.logger.info(JSON.stringify(tasksResponse));
                return tasksResponse;
            });
            res.json(response);
        }
        catch (error) {
            app_logs_1.logger.error("Error while receiving task:");
            app_logs_1.logger.error(error);
            res.status(app_enums_1.enums.HttpStatusCode.INTERNAL_ERROR).send();
        }
    }
    async validateLeaveRequest(raw, v) {
        v.validateIsArrayOfString("ids", false, raw.ids);
        v.validateUUID("leaveId", false, raw.leaveId);
        v.validateDate("startUtc", false, raw.startUtc);
        v.validateDate("endUtc", false, raw.endUtc);
        let ppermissionType;
        await (0, cli_queries_1.systemTransaction)(this.pool, async (trx) => {
            ppermissionType = (await trx.query(`SELECT pt."minValue", pt."isDailyScheduled"
					FROM "${integration_config_1.default.INTEGRATION_ORGANIZATION}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.ppermissionTypes}" pt
					WHERE pt."id" = $1`, [raw.leaveId])).rows[0];
        });
        if (ppermissionType.isDailyScheduled) {
            if (ppermissionType.minValue !== 0) {
                let properDateTimes = this.isMidnight(raw.startUtc) && this.isMidnight(raw.endUtc);
                if (!properDateTimes) {
                    app_logs_1.logger.error(`Times should be exactly midnight for this type of leave.`);
                    throw (0, api_error_1.generateBadRequestApiError)({
                        message: "Times should be exactly midnight for this type of leave.",
                    });
                }
            }
            else {
                let properDateTimes = (this.isMidnight(raw.startUtc) || this.isMidDay(raw.startUtc)) && (this.isMidnight(raw.endUtc) || this.isMidDay(raw.endUtc));
                if (!properDateTimes) {
                    app_logs_1.logger.error(`Times should be exactly midnight or mid day for this type of leave.`);
                    throw (0, api_error_1.generateBadRequestApiError)({
                        message: "Times should be exactly midnight or mid day for this type of leave.",
                    });
                }
            }
        }
    }
    validateListRequest(raw, v) {
        v.validateInteger({ field: "skip", optional: false, input: raw.skip });
        v.validateInteger({ field: "take", optional: false, input: raw.take });
    }
}
exports.LeaveAPI = LeaveAPI;
exports.default = LeaveAPI;
