"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.IdentityAPI = void 0;
const api_validatorhelper_1 = require("../../../../api/api.validatorhelper");
const app_constants_1 = require("../../../../app.constants");
const app_logs_1 = require("../../../../app.logs");
const cli_queries_1 = require("../../../../dal/access/psql/cli-queries");
const dal_db_armon_schema_1 = require("../../../../dal/db/armon/dal.db.armon.schema");
const integration_config_1 = __importDefault(require("../../integration.config"));
const integration_base_1 = require("../../integrations/integration.base");
const task_identity_1 = require("./task.identity");
const app_enums_1 = require("../../../../app.enums");
class IdentityAPI {
    constructor(_pool, apiToken, taskFactory) {
        this.pool = _pool;
        this.apiToken = apiToken;
        this.taskFactory = taskFactory;
        this.taskFactory.registerTaskType(task_identity_1.DefaultUserTask.id, task_identity_1.DefaultUserTask.create);
    }
    async upsertIdentity(req, res) {
        const identityData = req.body;
        try {
            const validator = new api_validatorhelper_1.ValidatorHelper();
            this.validateUserRequest(identityData, validator);
            validator.finalize();
        }
        catch (error) {
            return res.status(app_enums_1.enums.HttpStatusCode.BAD_REQUEST).send(error);
        }
        try {
            let response = await (0, cli_queries_1.systemTransaction)(this.pool, async (trx) => {
                app_logs_1.logger.info("Execution starting for client: " + req.header(app_constants_1.ArmonHeaders.X_Armon_ClientId));
                return await task_identity_1.DefaultUserTask.create().execute(trx, this.apiToken, "instant", identityData);
            });
            if (response === integration_base_1.OperationStatus.Success) {
                res.status(app_enums_1.enums.HttpStatusCode.SUCCESS).send();
            }
            else {
                res.status(app_enums_1.enums.HttpStatusCode.INTERNAL_ERROR).send();
            }
        }
        catch (error) {
            app_logs_1.logger.error("Error while executing task:");
            app_logs_1.logger.error(error);
            if (error.status && error.message) {
                res.status(error.status).send(error.message);
            }
            else {
                res.status(app_enums_1.enums.HttpStatusCode.INTERNAL_ERROR).send();
            }
        }
    }
    async upsertUserQueued(req, res) {
        const body = req.body;
        app_logs_1.logger.info("Operation received");
        try {
            const validator = new api_validatorhelper_1.ValidatorHelper();
            validator.validateObjectArray("items", false, body.items);
            if (body.items && Array.isArray(body.items)) {
                for (const item of body.items ?? []) {
                    this.validateUserRequest(item, validator);
                }
            }
            validator.finalize();
        }
        catch (error) {
            app_logs_1.logger.error("Validation errors: " + JSON.stringify(error));
            return res.status(app_enums_1.enums.HttpStatusCode.BAD_REQUEST).send(error);
        }
        try {
            const response = await (0, cli_queries_1.systemTransaction)(this.pool, async (trx) => {
                const opid = await trx.query(`INSERT INTO "${integration_config_1.default.INTEGRATION_ORGANIZATION}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.integration_ops}"
                    (id, status, "taskCount", "createdAt", "clientId") VALUES (gen_random_uuid(), $1, $2, now(), $3)
                    RETURNING id;`, [integration_base_1.OperationStatus.Pending, body.items.length, req.header(app_constants_1.ArmonHeaders.X_Armon_ClientId)]);
                const taskids = await trx.query(`INSERT INTO "${integration_config_1.default.INTEGRATION_ORGANIZATION}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.integration_tasks}"
                    (id, "operationId", index, "typeId", data, status, "createdAt")
                    SELECT gen_random_uuid(), $1, elem.index, $2, elem.item, $3, now()
                    FROM jsonb_array_elements($4::jsonb) WITH ORDINALITY elem(item, index)
                    RETURNING id;`, [opid.rows[0].id, task_identity_1.DefaultUserTask.id, integration_base_1.OperationStatus.Pending, JSON.stringify(body.items)]);
                const tasksResponse = {
                    op: opid.rows[0].id,
                    taskids: taskids.rows.map((r) => r.id),
                };
                app_logs_1.logger.info(JSON.stringify(tasksResponse));
                return tasksResponse;
            });
            res.json(response);
        }
        catch (error) {
            app_logs_1.logger.error("Error while receiving task:");
            app_logs_1.logger.error(error);
            res.status(app_enums_1.enums.HttpStatusCode.INTERNAL_ERROR).send();
        }
    }
    validateUserRequest(raw, v) {
        v.validateString({ field: "id", optional: false, input: raw.id, maxLength: 255 });
        v.validateString({ field: raw.id + " name", optional: false, input: raw.name, maxLength: 255 });
        v.validateString({ field: raw.id + " surname", optional: false, input: raw.surname, maxLength: 255 });
        v.validateIsBoolean(raw.id + " active", false, raw.active);
        v.validateString({ field: raw.id + " roleId", optional: true, input: raw.roleId });
        v.validateObject(raw.id + " fields", true, raw.fields, v);
        v.validateObjectArray(raw.id + " units", true, raw.units);
        raw.units?.forEach((unit) => {
            v.validateString({ field: "orgUnitId", optional: false, input: unit.orgUnitId, maxLength: 255 });
            v.validateIsBoolean("isManager", false, unit.isManager);
        });
        v.validateObjectArray(raw.id + "  credentials", true, raw.credentials);
        raw.credentials?.forEach((credential) => {
            v.validateString({ field: "data", optional: false, input: credential.data, maxLength: 255 });
            v.validateInteger({ field: "type", optional: false, input: credential.type });
            v.validateDate("expirationUtc", true, credential.expirationUtc);
            v.validateObject(raw.id + " extensionFields", true, raw.extensionFields, v);
        });
        v.validateString({ field: "username", optional: true, input: raw.account?.username, maxLength: 255 });
        v.validateString({ field: "password", optional: true, input: raw.account?.password });
    }
}
exports.IdentityAPI = IdentityAPI;
exports.default = IdentityAPI;
