"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.CredentialAPI = void 0;
const api_validatorhelper_1 = require("../../../../api/api.validatorhelper");
const app_enums_1 = require("../../../../app.enums");
const app_logs_1 = require("../../../../app.logs");
const cli_queries_1 = require("../../../../dal/access/psql/cli-queries");
const dal_constants_1 = require("../../../../dal/dal.constants");
const dal_db_armon_schema_1 = require("../../../../dal/db/armon/dal.db.armon.schema");
const integration_config_1 = __importDefault(require("../../integration.config"));
const models_credential_1 = require("./models.credential");
class CredentialAPI {
    constructor(_pool, apiToken, taskFactory) {
        this.pool = _pool;
        this.apiToken = apiToken;
    }
    async searchCredential(req, res) {
        const searchReq = req.body;
        try {
            const validator = new api_validatorhelper_1.ValidatorHelper();
            this.validateSearchRequest(searchReq, validator);
            validator.finalize();
        }
        catch (error) {
            return res.status(app_enums_1.enums.HttpStatusCode.BAD_REQUEST).send(error);
        }
        let responseData = {
            existance: models_credential_1.ExistanceType.NOTEXIST,
        };
        try {
            const taskdata = {
                uniqueId: integration_config_1.default.INTEGRATION_ORGANIZATION,
                taskid: "instant",
            };
            const findCredReqBody = {
                credentialData: searchReq.data.replace(/\s+/g, ""),
            };
            const credResponse = await this.apiToken.httpRequest(taskdata, "POST", `/u/v1/${integration_config_1.default.INTEGRATION_ORGANIZATION}/member/findbycredential`, findCredReqBody);
            if (credResponse.status === 200) {
                const credData = credResponse.data;
                return await (0, cli_queries_1.systemTransaction)(this.pool, async (trx) => {
                    const userCredential = (await trx.query(`SELECT id, "expiresOn", "userId"
							FROM "${integration_config_1.default.INTEGRATION_ORGANIZATION}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userOrganizationCredentials}"
							WHERE data ilike $1 AND "deletedAt" IS NULL;`, [findCredReqBody.credentialData])).rows[0];
                    if (credData.existance == dal_constants_1.DalConstants.CredentialExistance.NotExits || credData.existance === dal_constants_1.DalConstants.CredentialExistance.Visitor) {
                        const orgVisitorProfiles = (await trx.query(`SELECT id, "userOrganizationId", name || ' ' || surname AS fullname, "uniqueId"
								FROM "${integration_config_1.default.INTEGRATION_ORGANIZATION}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.organizationVisitorProfiles}"
								WHERE ("extensionFields" ->> 'licencePlate') IS NOT NULL
  								AND "extensionFields" ->> 'licencePlate' ilike $1 AND "deletedAt" IS NULL;`, [findCredReqBody.credentialData])).rows[0];
                        if (orgVisitorProfiles) {
                            responseData.existance = models_credential_1.ExistanceType.VISITOR;
                            responseData = await this.getVisitorData(trx, orgVisitorProfiles.id, responseData);
                            return res.status(app_enums_1.enums.HttpStatusCode.SUCCESS).send(responseData);
                        }
                        else if (userCredential) {
                            const orgVisitorProfile = (await trx.query(`SELECT ovp.id as id, ovp."userOrganizationId" as "userOrganizationId"
								FROM "${integration_config_1.default.INTEGRATION_ORGANIZATION}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.organizationVisitorProfiles}" ovp
								INNER JOIN "${integration_config_1.default.INTEGRATION_ORGANIZATION}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userOrganizations}" uo ON uo."id" = ovp."userOrganizationId"
								WHERE uo."userId" = $1;`, [userCredential.userId])).rows[0];
                            responseData.existance = models_credential_1.ExistanceType.VISITOR;
                            if (userCredential?.expiresOn && userCredential?.expiresOn < new Date()) {
                                responseData.state = models_credential_1.State.EXPIRED;
                                return res.status(app_enums_1.enums.HttpStatusCode.SUCCESS).send(responseData);
                            }
                            responseData = await this.getVisitorData(trx, orgVisitorProfile.id, responseData);
                            return res.status(app_enums_1.enums.HttpStatusCode.SUCCESS).send(responseData);
                        }
                        else {
                            return res.status(app_enums_1.enums.HttpStatusCode.NOT_FOUND).send(responseData);
                        }
                    }
                    else if (credData.existance === dal_constants_1.DalConstants.CredentialExistance.Member) {
                        responseData.existance = models_credential_1.ExistanceType.MEMBER;
                        responseData.memberInfo = {
                            uniqueId: credData.member.uniqueId,
                            fullname: credData.member.fullname,
                            groups: credData.member.userGroups,
                        };
                        if (userCredential?.expiresOn && userCredential?.expiresOn < new Date()) {
                            responseData.state = models_credential_1.State.EXPIRED;
                            return res.status(app_enums_1.enums.HttpStatusCode.SUCCESS).send(responseData);
                        }
                        const memberForbiddances = (await trx.query(`SELECT id, "regionId"
								FROM "${integration_config_1.default.INTEGRATION_ORGANIZATION}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userOrganizationForbiddances}"
								WHERE "userId" = $1 AND "deletedAt" IS NULL
  								AND now() >= "startUtc" AND now() <= "endUtc" 
								AND ("credentialIds" IS NULL
    									OR "credentialIds"::jsonb @> to_jsonb(ARRAY[$2])::jsonb)`, [credData.member.id, userCredential.id])).rows;
                        responseData.state = models_credential_1.State.ACTIVE;
                        if (memberForbiddances.length > 0) {
                            responseData.state = models_credential_1.State.FORBIDDEN;
                            let forbiddedRegionIds = [];
                            for (let forbiddance of memberForbiddances) {
                                if (forbiddance.regionId) {
                                    forbiddedRegionIds.push(forbiddance.regionId);
                                }
                                else {
                                    forbiddedRegionIds = [];
                                    break;
                                }
                            }
                            if (forbiddedRegionIds.length > 0) {
                                const forbiddedRegions = (await trx.query(`SELECT id, name
										FROM "${integration_config_1.default.INTEGRATION_ORGANIZATION}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.regions}"
										WHERE "id" = ANY($1::uuid[])`, [forbiddedRegionIds])).rows;
                                responseData.forbiddenAreas = forbiddedRegions;
                            }
                            else {
                                responseData.forbiddenAreas = [];
                            }
                        }
                        return res.status(app_enums_1.enums.HttpStatusCode.SUCCESS).send(responseData);
                    }
                });
            }
            else {
                return res.status(app_enums_1.enums.HttpStatusCode.INTERNAL_ERROR).send(credResponse);
            }
        }
        catch (error) {
            app_logs_1.logger.error("Error while finding credential:");
            app_logs_1.logger.error(error);
            return res.status(app_enums_1.enums.HttpStatusCode.INTERNAL_ERROR).send();
        }
    }
    async getVisitorData(trx, visiorProfileId, responseData) {
        const visitsOfVisitor = (await trx.query(`SELECT id, "expectedStartUtc", "expectedEndUtc", "state"
				FROM "${integration_config_1.default.INTEGRATION_ORGANIZATION}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.organizationActiveVisits}"
				WHERE "organizationVisitorProfileId" = $1
				AND ("expectedEndUtc" IS NULL OR now() <= "expectedEndUtc")
				AND ("state" = ${dal_constants_1.DalConstants.OrganizationVisitorStates.Active} OR "state" = ${dal_constants_1.DalConstants.OrganizationVisitorStates.Expected})`, [visiorProfileId])).rows;
        if (visitsOfVisitor.length === 0) {
            responseData.existance = models_credential_1.ExistanceType.NOTEXIST;
        }
        else {
            responseData.state = models_credential_1.State.ACTIVE;
            const visitorForbiddances = (await trx.query(`SELECT id, "startUtc", "endUtc", "state"
					FROM "${integration_config_1.default.INTEGRATION_ORGANIZATION}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.organizationVisitorStates}"
					WHERE "organizationVisitorProfileId" = $1 AND "deletedAt" IS NULL
					AND (("startUtc" IS NULL OR now() >= "startUtc") AND ("endUtc" IS NULL OR now() <= "endUtc"))
					AND "state" = ${dal_constants_1.DalConstants.OrganizationVisitorStates.Forbidden}`, [visiorProfileId])).rows;
            if (visitorForbiddances.length > 0) {
                responseData.state = models_credential_1.State.FORBIDDEN;
            }
        }
        return responseData;
    }
    validateSearchRequest(raw, v) {
        v.validateString({ field: "data", optional: false, input: raw.data });
    }
}
exports.CredentialAPI = CredentialAPI;
exports.default = CredentialAPI;
