"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.AppwebApi = void 0;
const dal_db_armon_schema_1 = require("../../dal/db/armon/dal.db.armon.schema");
const integration_config_1 = __importDefault(require("./integration.config"));
const dal_constants_1 = require("../../dal/dal.constants");
const axios_1 = __importDefault(require("axios"));
const app_config_1 = require("../../app.config");
const luxon_1 = require("luxon");
const app_logs_1 = require("../../app.logs");
const cli_queries_1 = require("../../dal/access/psql/cli-queries");
const dal_access_psql_organization_1 = require("../../dal/access/psql/dal.access.psql.organization");
const taskqueue_1 = require("../../utils/taskqueue");
const app_enums_1 = require("../../app.enums");
class AppwebApi {
    constructor() {
        this.rediskey = "integration_" + integration_config_1.default.INTEGRATION_ORGANIZATION + "_tokendata";
        this.taskQueue = new taskqueue_1.TaskQueue(30000, "EnerjiSA SAP");
    }
    async init(pool, redis) {
        const method = (await pool.query(`SELECT id FROM public."${dal_db_armon_schema_1.ArmonSchema.tableNames.organizationAuthenticationMethods}"
                WHERE "organizationId" = $1 AND "deletedAt" IS NULL AND "authenticationMethod" = $2 AND "grantType" = $3;`, [integration_config_1.default.INTEGRATION_ORGANIZATION, dal_constants_1.DalConstants.AuthenticationMethod.Native, dal_constants_1.DalConstants.AuthGrantType.UsernamePassword])).rows[0]?.id;
        if (method) {
            this.loginMethodId = method;
        }
        else {
            app_logs_1.logger.error("Authentication method id could not be found.");
        }
        const loginBody = await (0, cli_queries_1.systemTransaction)(pool, async (trx) => {
            return (0, dal_access_psql_organization_1.getOrganizationSecretById)(integration_config_1.default.INTEGRATION_ORGANIZATION, integration_config_1.default.APPWEB_API_SECRET_ID, trx);
        });
        if (loginBody) {
            this.loginBody = loginBody;
        }
        this.redis = redis;
        const orgSettings = await pool.query(`SELECT settings FROM "${integration_config_1.default.INTEGRATION_ORGANIZATION}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.organizations}"
            WHERE id = $1`, [integration_config_1.default.INTEGRATION_ORGANIZATION]);
        this.smtpSettings = orgSettings.rows[0]?.settings?.notification?.smtpSettings;
        if (integration_config_1.default.ERROR_EMAILS_SECRET_ID) {
            this.emailRecipients = (await (0, dal_access_psql_organization_1.getOrganizationSecretById)(integration_config_1.default.INTEGRATION_ORGANIZATION, integration_config_1.default.ERROR_EMAILS_SECRET_ID, pool))?.recipients || [];
        }
    }
    async getToken() {
        try {
            const redisdata = await this.redis.getValue(this.rediskey);
            if (redisdata) {
                const tokendata = JSON.parse(redisdata);
                if (luxon_1.DateTime.now() < luxon_1.DateTime.fromISO(tokendata.expiration)) {
                    return tokendata.token;
                }
                else {
                    app_logs_1.logger.info("Cached token expired, will refresh");
                    const params = new URLSearchParams({ refreshToken: tokendata.refreshToken, organizationId: integration_config_1.default.INTEGRATION_ORGANIZATION });
                    const response = await axios_1.default.post("/auth/refresh/", params.toString(), {
                        baseURL: app_config_1.appConfig.httpServer.apiBasePath,
                        headers: {
                            Authorization: tokendata.token,
                        },
                    });
                    if (response.status === 200) {
                        return await this.handleNewToken(response.data);
                    }
                    else {
                        app_logs_1.logger.error("Could not refresh token: " + response.status + " - " + response?.data?.message);
                    }
                }
            }
            else {
                app_logs_1.logger.info("Token not found in cache, will perform new login");
                const params = new URLSearchParams(this.loginBody);
                const response = await axios_1.default.post("/auth/usernamepass/" + this.loginMethodId, params.toString(), { baseURL: app_config_1.appConfig.httpServer.apiBasePath });
                if (response.status == 200) {
                    return await this.handleNewToken(response.data);
                }
                else {
                    app_logs_1.logger.error("Could not login: " + response.status + " - " + response?.data?.message);
                }
            }
        }
        catch (error) {
            app_logs_1.logger.error("Error renewing token:");
            if (error.response.status === app_enums_1.enums.HttpStatusCode.UNAUTHORIZED_ACCESS && error.response.data.error_description === "Refresh token timed out, login again") {
                this.redis.delValue(this.rediskey);
                return await this.getToken();
            }
            app_logs_1.logger.error(error);
        }
        this.redis.delValue(this.rediskey);
        return null;
    }
    async handleNewToken(response) {
        app_logs_1.logger.info("Token renewed");
        await this.redis.setValue(this.rediskey, JSON.stringify({
            token: "Bearer " + response.token,
            refreshToken: response.refreshToken,
            expiration: luxon_1.DateTime.now()
                .plus({ seconds: response.expiresIn - 60 })
                .toISO(),
        }));
        return "Bearer " + response.token;
    }
    async httpRequest(taskdata, method, url, data, config) {
        let retry = 3;
        while (retry > 0) {
            try {
                const response = await axios_1.default.request(Object.assign({}, config, {
                    url: url,
                    baseURL: app_config_1.appConfig.httpServer.apiBasePath,
                    data: data,
                    method: method,
                    headers: Object.assign({
                        Authorization: await this.getToken(),
                    }, config?.headers),
                }));
                return response;
            }
            catch (error) {
                const axioserror = error;
                if (axioserror.code === "401" && axioserror.response?.data && !axioserror.response.data.missingPermissions) {
                    app_logs_1.logger.error("Token expired or revoked while sending request, will try again");
                    this.redis.delValue(this.rediskey);
                    retry--;
                }
                else {
                    app_logs_1.logger.error("Error while api request:");
                    this.sendErrorMail("User: " +
                        taskdata.uniqueId +
                        "\nTask: " +
                        taskdata.taskid +
                        "\nDate: " +
                        new Date().toLocaleString() +
                        "\nStatus: " +
                        (axioserror.response?.status ?? "unknown") +
                        "\n" +
                        (axioserror.response?.data || error)?.message);
                    throw {
                        status: axioserror.response?.status ?? "unknown",
                        message: (axioserror.response?.data || error)?.message,
                    };
                }
            }
        }
        if (retry === 0) {
            this.sendErrorMail("User: " + taskdata.uniqueId + "\nTask: " + taskdata.taskid + "\nDate: " + new Date().toLocaleString() + "\nCould not refresh integration token");
        }
    }
    sendErrorMail(text) {
        if (this.smtpSettings?.enabled && this.emailRecipients?.length > 0) {
            this.emailRecipients.map((r) => {
                this.taskQueue.push(new taskqueue_1.SendEmailTask(this.smtpSettings, {
                    from: this.smtpSettings.defaultFromAddress,
                    to: r,
                    subject: "Armon SAP Integration Error",
                    text: text,
                }));
            });
        }
    }
}
exports.AppwebApi = AppwebApi;
