"use strict";
var __createBinding = (this && this.__createBinding) || (Object.create ? (function(o, m, k, k2) {
    if (k2 === undefined) k2 = k;
    var desc = Object.getOwnPropertyDescriptor(m, k);
    if (!desc || ("get" in desc ? !m.__esModule : desc.writable || desc.configurable)) {
      desc = { enumerable: true, get: function() { return m[k]; } };
    }
    Object.defineProperty(o, k2, desc);
}) : (function(o, m, k, k2) {
    if (k2 === undefined) k2 = k;
    o[k2] = m[k];
}));
var __setModuleDefault = (this && this.__setModuleDefault) || (Object.create ? (function(o, v) {
    Object.defineProperty(o, "default", { enumerable: true, value: v });
}) : function(o, v) {
    o["default"] = v;
});
var __importStar = (this && this.__importStar) || function (mod) {
    if (mod && mod.__esModule) return mod;
    var result = {};
    if (mod != null) for (var k in mod) if (k !== "default" && Object.prototype.hasOwnProperty.call(mod, k)) __createBinding(result, mod, k);
    __setModuleDefault(result, mod);
    return result;
};
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
const elastic_apm_node_1 = __importDefault(require("elastic-apm-node"));
if (process.env.ELASTIC_APM_SERVER_URL &&
    process.env.ELASTIC_APM_SECRET_TOKEN &&
    process.env.ELASTIC_APM_PREFIX &&
    ((process.env.ELASTIC_APM_VERIFY_SERVER_CERT === "true" && process.env.ELASTIC_APM_SERVER_CA_CERT_FILE) || process.env.ELASTIC_APM_VERIFY_SERVER_CERT === "false")) {
    elastic_apm_node_1.default.start({
        serviceName: `${process.env.ELASTIC_APM_PREFIX}-integration`,
        serverUrl: process.env.ELASTIC_APM_SERVER_URL,
        secretToken: process.env.ELASTIC_APM_SECRET_TOKEN,
        environment: "production",
        serverCaCertFile: process.env.ELASTIC_APM_SERVER_CA_CERT_FILE ?? null,
        verifyServerCert: process.env.ELASTIC_APM_VERIFY_SERVER_CERT === "true",
    });
}
else {
    console.warn(`Elastic APM is not enabled. Please set ELASTIC_APM_SERVER_URL, ELASTIC_APM_SECRET_TOKEN, ELASTIC_APM_PREFIX, ELASTIC_APM_VERIFY_SERVER_CERT, and ELASTIC_APM_SERVER_CA_CERT_FILE environment variables.`);
}
const express_1 = __importDefault(require("express"));
const fs = __importStar(require("fs"));
const http_1 = __importDefault(require("http"));
const https_1 = __importDefault(require("https"));
const path_1 = __importDefault(require("path"));
const pg_1 = require("pg");
const swagger_ui_express_1 = __importDefault(require("swagger-ui-express"));
const app_config_1 = require("../../app.config");
const app_logs_1 = require("../../app.logs");
const integration_config_1 = __importDefault(require("./integration.config"));
const integration_base_1 = require("./integrations/integration.base");
const task_factory_1 = require("./tasks/task.factory");
const express_body_parser_error_handler_1 = __importDefault(require("express-body-parser-error-handler"));
const yaml = __importStar(require("js-yaml"));
const app_constants_1 = require("../../app.constants");
const cli_queries_1 = require("../../dal/access/psql/cli-queries");
const dal_access_psql_organization_1 = require("../../dal/access/psql/dal.access.psql.organization");
const dal_access_cache_client_1 = __importDefault(require("../../dal/access/redis/dal.access.cache.client"));
const dal_access_cache_redis_1 = require("../../dal/access/redis/dal.access.cache.redis");
const appwebApi_1 = require("./appwebApi");
const joi_1 = __importDefault(require("joi"));
const luxon_1 = require("luxon");
const crypto_1 = __importDefault(require("crypto"));
const app_enums_1 = require("../../app.enums");
class IntegrationApp {
    constructor() { }
    async start() {
        try {
            await app_config_1.appConfig.init(app_constants_1.ServiceNames.Integration);
            app_logs_1.appLogger.init(integration_config_1.default.LOG_DIRECTORY);
            this.pool = new pg_1.Pool({
                host: integration_config_1.default.PGHOST,
                port: parseInt(integration_config_1.default.PGPORT),
                database: integration_config_1.default.PGDATABASE,
                user: integration_config_1.default.PGUSER,
                application_name: "integration_" + process.env.HOSTNAME,
                max: 100,
            });
            this.redisCache = new dal_access_cache_redis_1.RedisCache(await dal_access_cache_client_1.default.connect({
                url: `redis://${app_config_1.appConfig.db.redis.host}:${app_config_1.appConfig.db.redis.port}`,
            }));
            this.loginData = await (0, cli_queries_1.systemTransaction)(this.pool, async (trx) => {
                return (0, dal_access_psql_organization_1.getOrganizationSecretById)(integration_config_1.default.INTEGRATION_ORGANIZATION, integration_config_1.default.BASIC_AUTH_SECRET_ID, trx);
            });
            if (this.loginData) {
                this.clientToken = Buffer.from(this.loginData.username + ":" + this.loginData.password).toString("base64");
            }
            this.apiToken = new appwebApi_1.AppwebApi();
            await this.apiToken.init(this.pool, this.redisCache);
            this.expressApp = (0, express_1.default)();
            this.expressApp.use(express_1.default.json({ limit: "100mb" }));
            this.expressApp.use((0, express_body_parser_error_handler_1.default)({
                errorMessage(err) {
                    return `Invalid JSON value at request, please check request body`;
                },
            }));
            this.expressApp.use(async (req, res, next) => {
                if (req.path.includes(`/integration/${integration_config_1.default.INTEGRATION_ORGANIZATION}/api-docs`)) {
                    return next();
                }
                const auth = req.headers.authorization?.split(" ");
                if (auth && auth[0] && auth[0] === "Basic" && auth[1] && auth[1] === this.clientToken) {
                    return next();
                }
                try {
                    const validation = joi_1.default.object({
                        [app_constants_1.ArmonHeaders.X_Armon_ClientId]: joi_1.default.string().uuid().required(),
                        [app_constants_1.ArmonHeaders.X_Armon_Timestamp]: joi_1.default.string().length(14).required(),
                        [app_constants_1.ArmonHeaders.X_Armon_Signature]: joi_1.default.string().alphanum().required(),
                    }).validate(req.headers, {
                        allowUnknown: true,
                    });
                    if (validation.error) {
                        return res.status(401).send(validation.error.message);
                    }
                    const clientId = req.header(app_constants_1.ArmonHeaders.X_Armon_ClientId);
                    const timestamp = req.header(app_constants_1.ArmonHeaders.X_Armon_Timestamp);
                    const signature = req.header(app_constants_1.ArmonHeaders.X_Armon_Signature);
                    const clientCreds = await (0, cli_queries_1.systemTransaction)(this.pool, async (trx) => {
                        return (0, dal_access_psql_organization_1.getOrganizationIntegrationCredsById)(integration_config_1.default.INTEGRATION_ORGANIZATION, clientId.toString(), trx);
                    });
                    if (!clientCreds) {
                        return res.status(401).send("not found");
                    }
                    if (clientCreds.validUntil) {
                        const expireDate = luxon_1.DateTime.fromISO(clientCreds.validUntil);
                        if (expireDate < luxon_1.DateTime.now()) {
                            return res.status(401).send("expired");
                        }
                    }
                    const format = "ddMMyyyyHHmmss";
                    const requestTimestamp = luxon_1.DateTime.fromFormat(timestamp.toString(), format);
                    const currentTimestamp = luxon_1.DateTime.now();
                    if (Math.abs(currentTimestamp.diff(requestTimestamp, "milliseconds").milliseconds) > 50000) {
                        return res.status(401).send("timestamp difference");
                    }
                    const clientSecret = clientCreds.clientSecret;
                    const data = clientId.toString() + timestamp.toString();
                    const expectedSignature = crypto_1.default.createHmac("sha256", clientSecret).update(data).digest("hex");
                    if (signature !== expectedSignature) {
                        return res.status(401).send("signatures does not match");
                    }
                    req.body.integrationClientId = clientId;
                    return next();
                }
                catch (error) {
                    app_logs_1.logger.error("Error during authentication:", error);
                    return res.status(app_enums_1.enums.HttpStatusCode.INTERNAL_ERROR).send();
                }
            });
            this.taskFactory = new task_factory_1.TaskFactory();
            try {
                const organizationFile = path_1.default.join(__dirname, "integrations", integration_config_1.default.INTEGRATION_ORGANIZATION + ".js");
                if (fs.existsSync(organizationFile)) {
                    this.override = new (require(organizationFile).default)(this.pool, this.apiToken, this.taskFactory);
                    await this.override.initialize(this.expressApp);
                }
                else {
                    app_logs_1.logger.info("Configured organization does not have an integration file");
                }
            }
            catch (error) {
                app_logs_1.logger.error("Error while loading organization integration file");
                app_logs_1.logger.error(error);
            }
            this.baseApi = new integration_base_1.IntegrationBasic(this.pool, this.apiToken, this.taskFactory);
            await this.baseApi.initialize(this.expressApp);
            const swaggerDocument = yaml.load(fs.readFileSync(path_1.default.join(__dirname, "swagger.yaml"), "utf8"));
            swaggerDocument.servers[0].url = `${process.env.HTTP_PROTOCOL}://${process.env.DOMAIN_NAME}:${process.env.APIPORT}`;
            this.expressApp.use(`/integration/${integration_config_1.default.INTEGRATION_ORGANIZATION}/api-docs`, swagger_ui_express_1.default.serve, swagger_ui_express_1.default.setup(swaggerDocument));
            let server = undefined;
            if (integration_config_1.default.HTTP_PROTOCOL === "https" && app_config_1.appConfig.httpServer.ssl?.keyFileName && app_config_1.appConfig.httpServer.ssl?.certFileName) {
                server = https_1.default.createServer({
                    key: fs.readFileSync(app_config_1.appConfig.httpServer.ssl?.keyFileName, "utf8"),
                    cert: fs.readFileSync(app_config_1.appConfig.httpServer.ssl?.certFileName, "utf8"),
                }, this.expressApp);
            }
            else {
                server = http_1.default.createServer(this.expressApp);
            }
            server.listen(integration_config_1.default.INTEGRATION_PORT);
            setTimeout(this.workloop.bind(this), 30000);
            app_logs_1.logger.info("integration server started");
        }
        catch (error) {
            app_logs_1.logger.error(error || error.message);
            process.exit(1);
        }
    }
    async workloop() {
        try {
            let hasTask = true;
            while (hasTask) {
                await (0, cli_queries_1.systemTransaction)(this.pool, async (trx) => {
                    const nexttask = (await trx.query(`WITH current_op AS (
								SELECT id, status, "taskCount"
								FROM "${integration_config_1.default.INTEGRATION_ORGANIZATION}"."integration_ops" ops
								WHERE status = ANY(ARRAY[$1, $2]::integer[]) AND "taskCount" > 0
								ORDER BY "createdAt" ASC
								LIMIT 1
							)
							SELECT
								tasks.id, tasks."operationId", tasks."typeId", tasks.data, tasks.index, current_op."taskCount", current_op.status as "opStatus"
							FROM "${integration_config_1.default.INTEGRATION_ORGANIZATION}"."integration_tasks" tasks
							INNER JOIN current_op
								ON current_op.id = tasks."operationId" AND tasks.status = $1
							ORDER BY tasks."createdAt" ASC, tasks.index ASC
							LIMIT 1
							FOR UPDATE;`, [integration_base_1.OperationStatus.Pending, integration_base_1.OperationStatus.InProgress])).rows[0];
                    if (nexttask) {
                        if (nexttask.opStatus === integration_base_1.OperationStatus.Pending) {
                            await trx.query(`UPDATE "${integration_config_1.default.INTEGRATION_ORGANIZATION}"."integration_ops"
								SET status = $1, "startedAt" = clock_timestamp()
								WHERE id = $2;`, [integration_base_1.OperationStatus.InProgress, nexttask.operationId]);
                        }
                        await trx.query(`UPDATE "${integration_config_1.default.INTEGRATION_ORGANIZATION}"."integration_tasks"
							SET status = $1, "startedAt" = clock_timestamp()
							WHERE id = $2;`, [integration_base_1.OperationStatus.InProgress, nexttask.id]);
                        await trx.query(`SAVEPOINT sp1`);
                        app_logs_1.logger.info("Starting task: " + nexttask.operationId + " - " + nexttask.id);
                        let result = integration_base_1.OperationStatus.Failure;
                        try {
                            result = await this.taskFactory.createTask(nexttask.typeId).execute(trx, this.apiToken, nexttask.id, nexttask.data);
                        }
                        catch (error) {
                            app_logs_1.logger.error("Error while executing task:");
                            app_logs_1.logger.error(error);
                            await trx.query(`ROLLBACK TO SAVEPOINT sp1`);
                        }
                        await trx.query(`UPDATE "${integration_config_1.default.INTEGRATION_ORGANIZATION}"."integration_tasks"
							SET status = $1, "finishedAt" = clock_timestamp()
							WHERE id = $2;`, [result, nexttask.id]);
                        if (nexttask.index === nexttask.taskCount) {
                            await trx.query(`UPDATE "${integration_config_1.default.INTEGRATION_ORGANIZATION}"."integration_ops"
								SET "finishedAt" = clock_timestamp(),
								status = (
									SELECT CASE
										WHEN $1 = ANY(results) AND $2 = ANY(results) THEN $3
										WHEN $1 = ANY(results) THEN $1
										ELSE $2 END as result
									FROM (
										SELECT ARRAY_AGG(DISTINCT status) as results
										FROM "${integration_config_1.default.INTEGRATION_ORGANIZATION}".integration_tasks
										WHERE "operationId" = $4
									) sq
								)
								WHERE id = $4;`, [integration_base_1.OperationStatus.Success, integration_base_1.OperationStatus.Failure, integration_base_1.OperationStatus.Partial, nexttask.operationId]);
                            const delay = (ms) => new Promise((resolve) => setTimeout(resolve, ms));
                            await delay(1);
                        }
                    }
                    else {
                        hasTask = false;
                    }
                });
            }
        }
        catch (error) {
            app_logs_1.logger.error("Error while performing tasks:");
            app_logs_1.logger.error(error);
        }
        setTimeout(this.workloop.bind(this), 60 * 1000);
    }
}
const app = new IntegrationApp();
app.start();
