"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.getMigrationFiles = exports.checkFileSystemAgainstMigrationsTable = void 0;
const fs_1 = __importDefault(require("fs"));
const path_1 = __importDefault(require("path"));
const app_logs_1 = require("../../../../app.logs");
const dal_db_armon_schema_1 = require("../../../../dal/db/armon/dal.db.armon.schema");
async function checkFileSystemAgainstMigrationsTable(client, targetMigrationName) {
    const migrationFSFiles = getMigrationFiles();
    let migrationList = [];
    let result = {
        isDbCompatible: true,
        migrationNeeded: false,
        migrationList,
    };
    if (targetMigrationName) {
        if (migrationFSFiles.length <= 0) {
            throw Error("Migration Files not found in File System");
        }
        if (!migrationFSFiles.includes(targetMigrationName)) {
            throw Error("There is no file as " + targetMigrationName + " to migrate");
        }
    }
    await client
        .query(`
                SELECT name FROM public.${dal_db_armon_schema_1.ArmonSchema.tableNames.migration}
				WHERE "name" >= $1                      
                ORDER BY "name";
			`, [migrationFSFiles[0]])
        .then((dbResult) => {
        migrationList = dbResult.rowCount > 0 ? dbResult.rows.map((r) => r.name) : [];
        result.migrationList = migrationList;
    });
    if (targetMigrationName && targetMigrationName != "" && migrationList.includes(targetMigrationName)) {
        const targetMigrationIndex = migrationList.indexOf(targetMigrationName);
        for (let i = targetMigrationIndex; i < migrationList.length; i++) {
            if (migrationList[i] != migrationFSFiles[i]) {
                throw Error("There are mismatched migrations between fs and db, please get in touch with development team!");
            }
        }
        result.migrationNeeded = true;
        result.nextMigrationIndex = -1;
    }
    else if (migrationFSFiles.length === migrationList.length) {
        app_logs_1.logger.info("DB and FS migrations are the same, no migration needed.");
        result = { isDbCompatible: true, migrationNeeded: false, migrationList };
    }
    else if (migrationList.length < 0) {
        app_logs_1.logger.info("Empty migration table");
        app_logs_1.logger.info("There are " + migrationFSFiles.length + " migration(s) waiting to be applied!");
        result = { isDbCompatible: true, migrationNeeded: true, nextMigrationIndex: 0, migrationList };
    }
    else {
        for (let index = 0; index < migrationFSFiles.length; index++) {
            if (index >= migrationList.length) {
                result.nextMigrationIndex = index;
                break;
            }
            if (migrationFSFiles[index] !== migrationList[index]) {
                throw Error(migrationFSFiles[index] + " from file system and " + migrationList[index] + " from database does not match!");
            }
            result.nextMigrationIndex = index + 1;
        }
        result.migrationNeeded = true;
    }
    return result;
}
exports.checkFileSystemAgainstMigrationsTable = checkFileSystemAgainstMigrationsTable;
function getMigrationFiles() {
    try {
        return fs_1.default
            .readdirSync(path_1.default.resolve(__dirname, "..", "..", "..", "..", "dal", "db", "migrations", "postgresql"))
            .filter((f) => f.endsWith(".js"))
            .sort();
    }
    catch (err) {
        app_logs_1.logger.error(err);
        return [];
    }
}
exports.getMigrationFiles = getMigrationFiles;
