"use strict";
var __createBinding = (this && this.__createBinding) || (Object.create ? (function(o, m, k, k2) {
    if (k2 === undefined) k2 = k;
    var desc = Object.getOwnPropertyDescriptor(m, k);
    if (!desc || ("get" in desc ? !m.__esModule : desc.writable || desc.configurable)) {
      desc = { enumerable: true, get: function() { return m[k]; } };
    }
    Object.defineProperty(o, k2, desc);
}) : (function(o, m, k, k2) {
    if (k2 === undefined) k2 = k;
    o[k2] = m[k];
}));
var __setModuleDefault = (this && this.__setModuleDefault) || (Object.create ? (function(o, v) {
    Object.defineProperty(o, "default", { enumerable: true, value: v });
}) : function(o, v) {
    o["default"] = v;
});
var __importStar = (this && this.__importStar) || function (mod) {
    if (mod && mod.__esModule) return mod;
    var result = {};
    if (mod != null) for (var k in mod) if (k !== "default" && Object.prototype.hasOwnProperty.call(mod, k)) __createBinding(result, mod, k);
    __setModuleDefault(result, mod);
    return result;
};
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.updateUsernameCmd = void 0;
const child_process_1 = __importDefault(require("child_process"));
const fs = __importStar(require("fs"));
const inquirer_1 = __importDefault(require("inquirer"));
const joi_1 = __importDefault(require("joi"));
const os_1 = __importDefault(require("os"));
const path_1 = __importDefault(require("path"));
const pg_1 = require("pg");
const app_constants_1 = require("../../app.constants");
const app_enums_1 = require("../../app.enums");
const app_logs_1 = require("../../app.logs");
const cli_queries_1 = require("../../dal/access/psql/cli-queries");
const dal_access_cache_client_1 = __importDefault(require("../../dal/access/redis/dal.access.cache.client"));
const dal_access_cache_redis_1 = require("../../dal/access/redis/dal.access.cache.redis");
const dal_constants_1 = require("../../dal/dal.constants");
const restapi_1 = require("../../lib/es/models/restapi");
const cli_config_1 = __importDefault(require("./cli.config"));
const commander_menu_1 = require("./commander-menu");
const db_backup_and_restore_1 = require("./services/db-management/db-backup-and-restore");
const db_scaffold_1 = require("./services/db-management/db-scaffold");
const delete_one_time_used_user_filters_1 = require("./services/db-management/delete-one-time-used-user-filters");
const migration_manager_1 = require("./services/db-management/migration-manager");
const client_1 = __importDefault(require("./services/rabbit-management/client"));
const uuid_1 = __importDefault(require("uuid"));
const partition_manager_1 = require("../pgmaintenance/db/partition-manager");
const update_user_profiles_1 = require("./services/db-management/update-user-profiles");
let pool = null;
const mainMenuChoices = [
    {
        name: "Database management",
        value: 1,
    },
    {
        name: "User management",
        value: 2,
    },
    {
        name: "Create New Organization",
        value: 3,
    },
    {
        name: "Restart Armon Services ( webapp / pacs / report / notification / pgredis / pgrabbit / pgmaintenance)",
        value: 4,
    },
    {
        name: "Fetch Dead Letter Messages",
        value: 5,
    },
    {
        name: "Exit Application",
        value: 99,
    },
];
const dbManagementMenuChoices = [
    {
        name: "Database migration",
        value: 1,
    },
    {
        name: "Partition Recovery",
        value: 2,
    },
    {
        name: "Scaffold Database (for new servers)",
        value: 3,
    },
    {
        name: "Backup Database and Archive Expired Data (for onPremiseServers)",
        value: 4,
    },
    {
        name: "Delete One Time Used User Filters",
        value: 6,
    },
    {
        name: "Back to Main Menu",
        value: 99,
    },
];
const userManagementMenuChoices = [
    {
        name: "Update Username",
        value: 2,
    },
    {
        name: "Reset Password of User",
        value: 3,
    },
    {
        name: "Import Workplan Memberships",
        value: 4,
    },
    {
        name: "Update User Profiles of Org From Latest Data",
        value: 5,
    },
    {
        name: "Back to Main Menu",
        value: 99,
    },
];
if (process.env.NODE_ENV === "development") {
    dbManagementMenuChoices.push({
        name: "Truncate and Scaffold Db Again (Without any data and schema)",
        value: 98,
    });
    dbManagementMenuChoices.push({
        name: "Anonymize Organization",
        value: 5,
    });
}
async function mainMenu() {
    let isExitPressed = false;
    let initialTimeout = await setTimeout(() => {
        app_logs_1.logger.info("Exiting due to inactivity for 30 seconds");
        process.exit(0);
    }, 30000);
    while (!isExitPressed) {
        const takenAction = await inquirer_1.default.prompt({
            type: "list",
            name: "action",
            message: "Select the action that you want to take?",
            choices: mainMenuChoices,
        });
        if (initialTimeout) {
            clearTimeout(initialTimeout);
        }
        if (!takenAction.action) {
            app_logs_1.logger.info(`Quitting cli application`);
            process.exit(0);
        }
        switch (takenAction.action) {
            case 3:
                await createOrganizationCmd();
                break;
            case 1:
                await dbManagementMenu();
                break;
            case 2:
                await userManagementMenu();
                break;
            case 4:
                await restartServicesMenu();
                break;
            case 5:
                await fetchDeadLetterMessagesCmd();
                break;
            case 99:
                app_logs_1.logger.info("Exiting manager app");
                isExitPressed = true;
                process.exit(0);
            default:
                app_logs_1.logger.info(`Unknown action! Quitting management application!`);
                process.exit(0);
        }
    }
    return;
}
async function restartServicesMenu() {
    const serviceRestartChoices = [
        {
            name: "All",
            value: "all",
        },
        {
            name: "Web Application",
            value: app_constants_1.ServiceNames.WebApplication,
        },
        {
            name: "Report Service",
            value: app_constants_1.ServiceNames.ReportService,
        },
        {
            name: "Notification Service",
            value: app_constants_1.ServiceNames.NotificationService,
        },
        {
            name: "PACS Service",
            value: app_constants_1.ServiceNames.PACSService,
        },
        {
            name: "PG-Rabbit Service",
            value: app_constants_1.ServiceNames.PGRabbitService,
        },
        {
            name: "PG-Redis Service",
            value: app_constants_1.ServiceNames.PGRedisService,
        },
        {
            name: "PG-Maintenance Service",
            value: app_constants_1.ServiceNames.PGMaintenanceService,
        },
        {
            name: "Back to Main Menu",
            value: "back",
        },
    ];
    const takenAction = await inquirer_1.default.prompt({
        type: "list",
        name: "action",
        message: "Select the action that you want to take?",
        choices: serviceRestartChoices,
    });
    if (!takenAction.action) {
        app_logs_1.logger.info(`Quitting cli application`);
        process.exit(0);
    }
    let serviceNamesToRestart = [];
    if (takenAction.action === "back") {
        return;
    }
    else if (takenAction.action === "all") {
        serviceNamesToRestart = Object.values(app_constants_1.ServiceNames);
    }
    else {
        serviceNamesToRestart = [takenAction.action];
    }
    const restartServiceOsDependentPrefix = os_1.default.platform() === "win32" ? "nssm" : "systemctl";
    const restartServiceCommandPrefix = restartServiceOsDependentPrefix + " restart";
    for (const serviceName of serviceNamesToRestart) {
        child_process_1.default.execSync(`${restartServiceCommandPrefix} ${serviceName}`);
    }
}
async function dbManagementMenu() {
    let isBackPressed = false;
    while (!isBackPressed) {
        const takenAction = await inquirer_1.default.prompt({
            type: "list",
            name: "action",
            message: "Select the action that you want to take?",
            choices: dbManagementMenuChoices,
        });
        if (!takenAction.action) {
            app_logs_1.logger.info(`Quitting cli application`);
            process.exit(0);
        }
        switch (takenAction.action) {
            case 1:
                await migrateDbCmd();
                break;
            case 2:
                await partitionRecoveryCmd();
                break;
            case 4:
                await runDbBackupServiceCmd();
                break;
            case 98:
                await dropAndScaffoldDbAgainCmd();
                break;
            case 3:
                await scaffoldDbCmd();
                break;
            case 5:
                await anonymizeOrganizationCmd(cli_config_1.default.ANONYMIZATION_FILE_PATH);
                break;
            case 6:
                await deleteOneTimeUsedUserFiltersCmd();
                break;
            case 99:
                isBackPressed = true;
                break;
            default:
                app_logs_1.logger.info(`Unknown action! Quitting management application!`);
                process.exit(0);
        }
    }
    return;
}
async function userManagementMenu() {
    let isBackPressed = false;
    while (!isBackPressed) {
        const takenAction = await inquirer_1.default.prompt({
            type: "list",
            name: "action",
            message: "Select the action that you want to take?",
            choices: userManagementMenuChoices,
        });
        if (!takenAction.action) {
            app_logs_1.logger.info(`Quitting cli application`);
            process.exit(0);
        }
        switch (takenAction.action) {
            case 2:
                await updateUsernameCmd();
                break;
            case 3:
                app_logs_1.logger.info("UserManagementMenuActions.PasswordReset not implemented yet");
                break;
            case 5:
                await updateUserProfilesCmd();
                break;
            case 99:
                isBackPressed = true;
                break;
            default:
                app_logs_1.logger.info(`Unknown action! Quitting management application!`);
                process.exit(0);
        }
    }
    return;
}
async function anonymizeOrganizationCmd(filepath) {
    try {
        let anonymizationData = JSON.parse(fs.readFileSync(path_1.default.resolve(filepath), "utf-8"));
        if (anonymizationData) {
            const validation = joi_1.default.object({
                organizationId: joi_1.default.string().uuid().required(),
                newOrganizationName: joi_1.default.string().required(),
                anonymizeLogins: joi_1.default.boolean(),
                entityNameReplacements: joi_1.default.array()
                    .items({
                    regex: joi_1.default.string().required(),
                    regexOptions: joi_1.default.string().optional(),
                    replacement: joi_1.default.string().required(),
                })
                    .optional(),
                addLogins: joi_1.default.array()
                    .items({
                    name: joi_1.default.string().required(),
                    surname: joi_1.default.string().required(),
                    email: joi_1.default.string().optional(),
                    phoneNumber: joi_1.default.string().optional(),
                    username: joi_1.default.string().required(),
                    password: joi_1.default.string().required(),
                })
                    .optional(),
                users: joi_1.default.array()
                    .items({
                    name: joi_1.default.string().required(),
                    surname: joi_1.default.string().required(),
                })
                    .required(),
            }).validate(anonymizationData, {
                allowUnknown: false,
            });
            if (validation.error) {
                throw validation.error;
            }
            await (0, cli_queries_1.systemTransaction)(pool, async (trx) => {
                await (0, cli_queries_1.anonymizeOrganization)(anonymizationData, trx);
            });
            app_logs_1.logger.info("Anonymization Completed");
        }
        else {
            app_logs_1.logger.error("Anonymization data not found");
            process.exit(1);
        }
    }
    catch (error) {
        app_logs_1.logger.error("Error during anonymization");
        app_logs_1.logger.error(error);
        process.exit(-1);
    }
}
async function updateUsernameCmd() {
    const answers = await inquirer_1.default.prompt([
        {
            type: "input",
            name: "oldUsername",
            message: "Please enter old username you want to change!",
            validate: (value) => {
                if (value !== null && value !== undefined) {
                    return true;
                }
                return "Username cannot be empty!";
            },
        },
        {
            type: "input",
            name: "newUsername",
            message: "Please enter new username !",
            validate: (value) => {
                if (value !== null && value !== undefined) {
                    return true;
                }
                return "Username cannot be empty!";
            },
        },
    ]);
    try {
        await (0, cli_queries_1.systemTransaction)(pool, async (trx) => {
            await (0, cli_queries_1.updateUsername)({ organizationId: null, oldUsername: answers.oldUsername, newUsername: answers.newUsername, trx });
        });
    }
    catch (err) {
        app_logs_1.logger.error("Error while updateing username!");
        app_logs_1.logger.error(err);
        process.exit(-1);
    }
}
exports.updateUsernameCmd = updateUsernameCmd;
async function runDbBackupServiceCmd() {
    let client = null;
    try {
        client = await pool.connect();
        await (0, db_backup_and_restore_1.runDbBackupService)(client);
        client.release();
    }
    catch (err) {
        app_logs_1.logger.error("Error while running backup operations!");
        app_logs_1.logger.error(err);
        client?.release(err);
        process.exit(-1);
    }
}
async function migrateDbCmd() {
    const answers = await inquirer_1.default.prompt([
        {
            type: "input",
            name: "migration",
            message: "Migrate to ?. Leave empty to migrate to the last migration",
        },
    ]);
    try {
        await (0, cli_queries_1.systemTransaction)(pool, async (trx) => {
            await (0, migration_manager_1.migrate)(trx, cli_config_1.default.PGUSER, cli_config_1.default.PGSUPERUSER, answers.migration);
        });
    }
    catch (err) {
        app_logs_1.logger.error("Error while running migration!");
        app_logs_1.logger.error(err);
        process.exit(-1);
    }
}
async function partitionRecoveryCmd() {
    try {
        await (0, cli_queries_1.systemTransaction)(pool, async (trx) => {
            await (0, partition_manager_1.partitionRecovery)(trx);
        });
        app_logs_1.logger.info(`Missing partitions created successfully`);
    }
    catch (error) {
        app_logs_1.logger.error(`Error while recovering missing partitions`);
        app_logs_1.logger.error(error);
    }
}
async function fetchDeadLetterMessagesCmd() {
    app_logs_1.logger.info(`
	**********************************************************
			** Data Fetch Operation Starting... **
	**********************************************************
	`);
    const deadLetterInitClients = async () => {
        try {
            const client = await dal_access_cache_client_1.default.connect({
                url: cli_config_1.default.REDIS_SERVER_URL,
                cert: cli_config_1.default.REDIS_CLIENT_TLS_CERT,
                key: cli_config_1.default.REDIS_CLIENT_TLS_KEY,
                ca: cli_config_1.default.REDIS_CLIENT_TLS_CA,
            });
            const redis = new dal_access_cache_redis_1.RedisCache(client);
            const pool = new pg_1.Pool({
                user: cli_config_1.default.PGUSER,
                application_name: "cli_application_deadLetter_" + process.env.HOSTNAME,
                max: 5,
            });
            app_logs_1.logger.debug("deadLetter clients successfully initiated");
            return { pool, redis };
        }
        catch (err) {
            app_logs_1.logger.error(`There is a error while initiating deadLetter clients`);
        }
    };
    await (0, client_1.default)(deadLetterInitClients);
    const waitUntilFetchAll = () => {
        return new Promise(async (resolve, reject) => {
            if (false) {
                resolve();
            }
        });
    };
    await waitUntilFetchAll();
}
async function updateUserProfilesCmd() {
    app_logs_1.logger.info(`
		**********************************************************
				** User Profile Picture Update Operation Starting... **
		**********************************************************
		`);
    try {
        await (0, cli_queries_1.systemTransaction)(pool, async (trx) => {
            await (0, update_user_profiles_1.updateUserProfiles)(trx);
        });
    }
    catch (err) {
        app_logs_1.logger.error("Error while updating user profile pictures!");
        app_logs_1.logger.error(err);
        process.exit(-1);
    }
}
async function createOrganizationCmd() {
    const userFriendlyCredentialTypes = [
        {
            name: "MiFare Card",
            value: app_enums_1.enums.libEnumsV2.CredentialType.MiFare,
        },
        {
            name: "Proximity Card",
            value: app_enums_1.enums.libEnumsV2.CredentialType.ProximityCard,
        },
        {
            name: "Bluetooth",
            value: app_enums_1.enums.libEnumsV2.CredentialType.BLE,
        },
        {
            name: "NFC",
            value: app_enums_1.enums.libEnumsV2.CredentialType.NFC,
        },
        {
            name: "Fingerprint",
            value: app_enums_1.enums.libEnumsV2.CredentialType.FingerPrintISO19794,
        },
        {
            name: "MRZ (TR Identity Card / Passport)",
            value: app_enums_1.enums.libEnumsV2.CredentialType.MRZ,
        },
        {
            name: "Qr Code",
            value: app_enums_1.enums.libEnumsV2.CredentialType.QrCode,
        },
        {
            name: "UHF Tag (RFID)",
            value: app_enums_1.enums.libEnumsV2.CredentialType.UHFRfid,
        },
        {
            name: "Licence Plate ",
            value: app_enums_1.enums.libEnumsV2.CredentialType.VehiclePlate,
        },
    ];
    const answers = await inquirer_1.default.prompt([
        {
            type: "input",
            name: "organizationName",
            message: "Please enter organization name to create!",
            validate: (value) => {
                if (value === null || value === undefined || value === "") {
                    return "Organization name cannot be empty!";
                }
                if (value.length >= 255) {
                    return "Organization name should be shorter than 255 characters!";
                }
                if (value.length <= 3) {
                    return "Organization name should be longer than 3 characters!";
                }
                return true;
            },
        },
        {
            type: "input",
            name: "id",
            message: "Please enter organization id, leave empty for random id",
            validate: (value) => {
                if (!value || /^[0-9a-f]{8}-[0-9a-f]{4}-[1-5][0-9a-f]{3}-[89ab][0-9a-f]{3}-[0-9a-f]{12}$/i.test(value)) {
                    return true;
                }
                else {
                    return "Invalid UUID";
                }
            },
        },
        {
            type: "confirm",
            name: "canCreateUserAccount",
            message: "Can create users for new organization (Y default)",
            validate: (value) => {
                if (value instanceof Boolean) {
                    return true;
                }
                return "Invalid choice";
            },
            default: true,
        },
        {
            type: "list",
            name: "usernamePolicy",
            message: "Choose username policy from list below",
            choices: getKeyValueMappingsOfEnum(app_enums_1.enums.OrganizationUsernamePolicy),
            validate: (value) => {
                if (Number.isInteger(value) && app_enums_1.enums.OrganizationUsernamePolicy[value]) {
                    return true;
                }
                return "Invalid username policy selection";
            },
            default: app_enums_1.enums.OrganizationUsernamePolicy.FreeAlphanumeric,
        },
        {
            type: "confirm",
            name: "hasEmergencyManagement",
            message: "Is emergency management enabled for organization (Y default)",
            validate: (value) => {
                if (value instanceof Boolean) {
                    return true;
                }
                return "Invalid choice";
            },
            default: true,
        },
        {
            type: "input",
            name: "administratorUsername",
            message: "Please enter username of administration account!",
            validate: (value) => {
                if (value === null || value === undefined || value === "") {
                    return "Administrator username cannot be empty!";
                }
                if (value.length >= 127) {
                    return "Administrator username should be shorter than 127 characters!";
                }
                if (value.length <= 3) {
                    return "Administrator username should be longer than 3 characters!";
                }
                return true;
            },
        },
        {
            type: "password",
            name: "administratorPassword",
            message: "Please enter password of administration account!",
            validate: (value) => {
                if (value === null || value === undefined || value === "") {
                    return "Administrator password cannot be empty!";
                }
                if (value.length > app_constants_1.ACCOUNT_PASSWORD_MAX_LENGTH) {
                    return "Administrator password should be shorter than " + (app_constants_1.ACCOUNT_PASSWORD_MAX_LENGTH + 1) + " characters!";
                }
                if (value.length < app_constants_1.ACCOUNT_PASSWORD_MIN_LENGTH) {
                    return "Administrator password should be longer than " + (app_constants_1.ACCOUNT_PASSWORD_MIN_LENGTH - 1) + " characters!";
                }
                return true;
            },
        },
        {
            type: "password",
            name: "administratorPasswordReEnter",
            message: "Please enter password of administration account again!",
            validate: (value, answers) => {
                if (value === null || value === undefined || value === "") {
                    return "Administrator password cannot be empty!";
                }
                if (value.length > app_constants_1.ACCOUNT_PASSWORD_MAX_LENGTH) {
                    return "Administrator password should be shorter than " + (app_constants_1.ACCOUNT_PASSWORD_MAX_LENGTH + 1) + " characters!";
                }
                if (value.length < app_constants_1.ACCOUNT_PASSWORD_MIN_LENGTH) {
                    return "Administrator password should be longer than " + (app_constants_1.ACCOUNT_PASSWORD_MIN_LENGTH - 1) + " characters!";
                }
                if (answers.administratorPassword !== value) {
                    return "Administrator password does not match with previously entered password! Please enter password again!";
                }
                return true;
            },
        },
        {
            type: "input",
            name: "administratorName",
            message: "Please enter profile name of administrator!",
            validate: (value) => {
                if (value === null || value === undefined || value === "") {
                    return "Profile name cannot be empty!";
                }
                if (value.length >= 127) {
                    return "Profile name should be shorter than 255 characters!";
                }
                if (value.length <= 3) {
                    return "Profile name should be longer than 3 characters!";
                }
                return true;
            },
        },
        {
            type: "input",
            name: "administratorSurname",
            message: "Please enter profile surname of administrator!",
            validate: (value) => {
                if (value === null || value === undefined || value === "") {
                    return "Profile surname cannot be empty!";
                }
                if (value.length >= 127) {
                    return "Profile surname should be shorter than 255 characters!";
                }
                if (value.length <= 3) {
                    return "Profile surname should be longer than 3 characters!";
                }
                return true;
            },
        },
        {
            type: "input",
            name: "administratorUniqueId",
            message: "Please enter unique id of administrator!",
            validate: (value) => {
                if (value === null || value === undefined || value === "") {
                    return "Unique id cannot be empty!";
                }
                if (value.length >= 64) {
                    return "Unique id should be shorter than 64 characters!";
                }
                if (value.length <= 3) {
                    return "Unique id should be longer than 3 characters!";
                }
                return true;
            },
        },
        {
            type: "checkbox",
            name: "credentialTypes",
            message: "Please choose available credentials for new organization!",
            choices: userFriendlyCredentialTypes,
            validate: (value) => {
                if (value === null || value === undefined) {
                    value = [];
                }
                if (value.length === 0) {
                }
                for (let index = 0; index < value.length; index++) {
                    if (!Number.isInteger(value[index]) || !app_enums_1.enums.libEnumsV2.CredentialType[value[index]]) {
                        return "Unavailable choice made!";
                    }
                }
                return true;
            },
        },
        {
            type: "list",
            name: "locale",
            message: "Please select locale for new organization!",
            choices: dal_constants_1.DalConstants.SupportedLocalesList.map((s) => {
                return { name: s, value: s };
            }),
            validate: (value) => {
                if (value === null || value === undefined || value === "") {
                    return "Supported Locale cannot be empty!";
                }
                if (!dal_constants_1.DalConstants.SupportedLocalesList.includes(value)) {
                    return "Supported Locale not found in definition!";
                }
                return true;
            },
        },
        {
            type: "list",
            name: "userExpression",
            message: "Please select user expression (Users / Personnels / Employees etc) for new organization!",
            choices: getKeyValueMappingsOfEnum(restapi_1.UserExpression),
            validate: (value) => {
                if (value === null || value === undefined) {
                    value = [];
                }
                if (value.length === 0) {
                }
                for (let index = 0; index < value.length; index++) {
                    if (!Number.isInteger(value[index]) || !restapi_1.UserExpression[value[index]]) {
                        return "Unavailable choice made!";
                    }
                }
                return true;
            },
        },
        {
            type: "checkbox",
            name: "credentialTypes",
            message: "Please choose available credentials for new organization!",
            choices: userFriendlyCredentialTypes,
            validate: (value) => {
                if (value === null || value === undefined) {
                    value = [];
                }
                if (value.length === 0) {
                }
                for (let index = 0; index < value.length; index++) {
                    if (!Number.isInteger(value[index]) || !app_enums_1.enums.libEnumsV3.CredentialType[value[index]]) {
                        return "Unavailable choice made!";
                    }
                }
                return true;
            },
        },
    ]);
    let IndexedNotificationMediumType;
    (function (IndexedNotificationMediumType) {
        IndexedNotificationMediumType[IndexedNotificationMediumType["web"] = 0] = "web";
        IndexedNotificationMediumType[IndexedNotificationMediumType["email"] = 1] = "email";
        IndexedNotificationMediumType[IndexedNotificationMediumType["pushNotification"] = 2] = "pushNotification";
        IndexedNotificationMediumType[IndexedNotificationMediumType["sms"] = 3] = "sms";
    })(IndexedNotificationMediumType || (IndexedNotificationMediumType = {}));
    const notificationMediumAnswers = await inquirer_1.default.prompt([
        {
            type: "checkbox",
            name: "notificationMediums",
            message: "Please choose available mediums for new organization!",
            choices: getKeyValueMappingsOfEnum(IndexedNotificationMediumType),
            validate: (value) => {
                if (value === null || value === undefined) {
                    value = [];
                }
                if (value.length === 0) {
                    return "At least 1 medium must be chosen!";
                }
                for (let index = 0; index < value.length; index++) {
                    if (!Number.isInteger(value[index]) || !IndexedNotificationMediumType[value[index]]) {
                        return "Unavailable choice made!";
                    }
                }
                return true;
            },
        },
    ]);
    const createOrganizationReq = {
        id: answers.id || uuid_1.default.v4(),
        name: answers.organizationName,
        administratorAccount: {
            name: answers.administratorName,
            surname: answers.administratorSurname,
            username: answers.administratorUsername,
            password: answers.administratorPassword,
            uniqueId: answers.administratorUniqueId,
        },
        canCreateUserAccount: answers.canCreateUserAccount,
        credentialTypes: answers.credentialTypes,
        hasEmergencyManagement: answers.hasEmergencyManagement,
        usernamePolicy: answers.usernamePolicy,
        settings: {
            webRtc: {
                twilio: {
                    enabled: false,
                },
                iceServers: [],
            },
            enableTriggerThrowErrorIfUserRoleChangeCauseTerminalHasNoWriteAdmin: true,
            locale: answers.locale,
            userExpression: answers.userExpression,
            logo: null,
            notification: {
                enableSendingCreatedAccountEmail: true,
                availableMediums: {
                    email: false,
                    pushNotification: false,
                    sms: false,
                    web: true,
                },
                mediumSettings: {
                    general: {
                        email: false,
                        pushNotification: false,
                        sms: false,
                        web: true,
                    },
                },
                enabledTypes: dal_constants_1.DalConstants.organizationBaseModuleNotificationTypes,
                smsSettings: undefined,
                smtpSettings: undefined,
            },
            passiveUserSettings: {
                anonymize: false,
            },
            checkMobileDeviceMismatchOnLogin: false,
            snapshotSettings: {
                quality: 60,
                width: 180,
                discardSnapshot: false,
            },
            integrations: [],
            multiFactorAuthenticationSettings: {
                option: app_enums_1.enums.MultifactorAuthenticationOption.DISABLE,
            },
        },
    };
    for (const iterator of notificationMediumAnswers.notificationMediums) {
        createOrganizationReq.settings.notification.mediumSettings.general[IndexedNotificationMediumType[iterator]] = true;
    }
    try {
        const organizationId = await (0, cli_queries_1.systemTransaction)(pool, async (trx) => {
            const organizationId = await (0, cli_queries_1.createOrganization)({
                request: createOrganizationReq,
                dbuser: cli_config_1.default.PGUSER,
                dbsuperuser: cli_config_1.default.PGSUPERUSER,
                trx,
            }, pool);
            return organizationId;
        });
        app_logs_1.logger.info(`
				**********************************************************
				** ORGANIZATIONID: ${organizationId} **
				**********************************************************
		`);
    }
    catch (err) {
        app_logs_1.logger.error("Error while creating organization!");
        app_logs_1.logger.error(err);
        process.exit(-1);
    }
}
async function dropAndScaffoldDbAgainCmd() {
    try {
        await (0, cli_queries_1.systemTransaction)(pool, async (trx) => {
            await (0, db_scaffold_1.dropAndCreateDbAgain)(trx, cli_config_1.default.PGUSER, cli_config_1.default.PGSUPERUSER);
        });
    }
    catch (err) {
        app_logs_1.logger.error("Error while dropping and re-scaffolding database!");
        app_logs_1.logger.error(err);
        process.exit(-1);
    }
}
async function scaffoldDbCmd() {
    try {
        await (0, cli_queries_1.systemTransaction)(pool, async (trx) => {
            await (0, db_scaffold_1.scaffoldDb)(trx, cli_config_1.default.PGUSER, cli_config_1.default.PGSUPERUSER);
        });
    }
    catch (err) {
        app_logs_1.logger.error("Error while scaffolding database!");
        app_logs_1.logger.error(err);
        process.exit(-1);
    }
}
async function deleteOneTimeUsedUserFiltersCmd() {
    try {
        await (0, cli_queries_1.systemTransaction)(pool, async (trx) => {
            await (0, delete_one_time_used_user_filters_1.deleteOneTimeUsedUserFilters)(trx);
        });
    }
    catch (err) {
        app_logs_1.logger.error("Error while deleting one time used user filters!");
        app_logs_1.logger.error(err);
        process.exit(-1);
    }
}
async function init() {
    app_logs_1.logger.info("Application starting...");
    app_logs_1.appLogger.init(cli_config_1.default.LOG_DIRECTORY);
    app_logs_1.logger.info("Config is loaded");
    try {
        pool = new pg_1.Pool({
            host: cli_config_1.default.PGHOST,
            port: parseInt(cli_config_1.default.PGPORT),
            database: cli_config_1.default.PGDATABASE,
            user: cli_config_1.default.PGSUPERUSER,
            application_name: "cli_application_" + process.env.HOSTNAME,
            max: 100,
        });
        app_logs_1.logger.info("DB connected");
        const cliParameter = (0, commander_menu_1.parseCliArguments)();
        if (cliParameter.rawArgs.length > 2) {
            try {
                if (cliParameter.autoMigrate) {
                    await (0, cli_queries_1.systemTransaction)(pool, async (trx) => {
                        await (0, migration_manager_1.migrate)(trx, cli_config_1.default.PGUSER, cli_config_1.default.PGSUPERUSER);
                    });
                }
                if (cliParameter.down) {
                    await (0, cli_queries_1.systemTransaction)(pool, async (trx) => {
                        await (0, migration_manager_1.migrate)(trx, cli_config_1.default.PGUSER, cli_config_1.default.PGSUPERUSER, cliParameter.down);
                    });
                }
                if (cliParameter.backupAndArchive) {
                    await (0, cli_queries_1.systemTransaction)(pool, async (trx) => {
                        await (0, db_backup_and_restore_1.runDbBackupService)(trx);
                    });
                }
                if (cliParameter.anonymizeOrganization) {
                    anonymizeOrganizationCmd(cliParameter.anonymizeOrganization);
                }
                if (cliParameter.removeOneTimeUserFilter) {
                    await deleteOneTimeUsedUserFiltersCmd();
                }
            }
            catch (err) {
                app_logs_1.logger.error(err);
                app_logs_1.logger.error("Error occured cli functions called with command line arguments");
                app_logs_1.logger.error("cliParameters:" + JSON.stringify(cliParameter.rawArgs, null, 4));
                process.exit(-1);
            }
        }
        else {
            await mainMenu();
        }
    }
    catch (err) {
        app_logs_1.logger.error("Error while starting cli application");
        app_logs_1.logger.error(err);
        process.exit(-1);
    }
}
init();
function getKeyValueMappingsOfEnum(myEnum) {
    return Object.keys(myEnum)
        .filter((e) => Number.isInteger(myEnum[e]))
        .map((k) => {
        return {
            name: k,
            value: myEnum[k],
        };
    });
}
