"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.isCredentialsRequired = exports.isAccessControlPointIdsRequired = exports.registerVisitor = exports.preRegisterVisitor = exports.validateVisitFields = exports.resendVisitorInvitationNotificaton = exports.sendQrCodeToVisitor = exports.disableVisitorModule = exports.enableVisitorModule = exports.BusinessVisitor = exports.hideVisitorUniqueId = exports.hideSensitiveFieldsOfVisitors = exports.hideSensitiveFieldsOfVisitorFormSearch = exports.hideSensitiveFieldsOfVisits = exports.transformVisitorActionItemsLocalizedMessages = exports.armonHeaderLogo = exports.armonLogo = void 0;
const i18n_1 = __importDefault(require("i18n"));
const path_1 = __importDefault(require("path"));
const qrcode_1 = __importDefault(require("qrcode"));
const uuid_1 = __importDefault(require("uuid"));
const api_error_1 = require("../../api/api.error");
const app_config_1 = require("../../app.config");
const app_enums_1 = require("../../app.enums");
const app_httpserver_1 = require("../../app.httpserver");
const app_logs_1 = require("../../app.logs");
const app_util_1 = require("../../app.util");
const dal_access_error_1 = require("../../dal/access/dal.access.error");
const dal_access_models_1 = require("../../dal/access/dal.access.models");
const dal_memcache_1 = require("../../dal/access/dal.memcache");
const dal_visitor_utils_1 = require("../../dal/access/dal.visitor.utils");
const dal_manager_1 = require("../../dal/dal.manager");
const dal_db_armon_schema_1 = require("../../dal/db/armon/dal.db.armon.schema");
const restapi_1 = require("../../lib/es/models/restapi");
const messagebroker_models_1 = require("../../messageBroker/messagebroker.models");
const messageBroker_notification_pub_1 = require("../../messageBroker/messageBroker.notification.pub");
const messageBroker_server_to_tablet_pub_1 = require("../../messageBroker/messageBroker.server-to-tablet.pub");
const business_common_1 = require("../business.common");
const business_hooks_1 = require("../business.hooks");
const business_visitor_excel_1 = require("./business.visitor.excel");
exports.armonLogo = "http://docs.armongate.com/armon_light_logo.png";
exports.armonHeaderLogo = "https://docs.armongate.com/old/header-image.png";
function transformVisitorActionItemsLocalizedMessages(items, locale) {
    let results = [];
    for (const item of items) {
        if (!item.actionUser || !item.actionUser.fullname) {
            continue;
        }
        let message = "";
        let userFullname = item.actionUser.fullname;
        switch (item.action) {
            case app_enums_1.enums.UserVisitorActionType.NewVisit:
                message = (0, app_util_1.stringFormat)(i18n_1.default.__({ phrase: "VISITOR.ACTIONHISTORY.NEWVISIT", locale: locale }), [userFullname]);
                break;
            case app_enums_1.enums.UserVisitorActionType.NewExpectedVisit:
                message = (0, app_util_1.stringFormat)(i18n_1.default.__({ phrase: "VISITOR.ACTIONHISTORY.NEWEXPECTEDVISIT", locale: locale }), [userFullname]);
                break;
            case app_enums_1.enums.UserVisitorActionType.TerminateVisit:
                message = (0, app_util_1.stringFormat)(i18n_1.default.__({ phrase: "VISITOR.ACTIONHISTORY.TERMINATEVISIT", locale: locale }), [userFullname]);
                break;
            case app_enums_1.enums.UserVisitorActionType.SetVisitorAsBlacklisted:
                message = (0, app_util_1.stringFormat)(i18n_1.default.__({ phrase: "VISITOR.ACTIONHISTORY.SETBLACKLIST", locale: locale }), [userFullname]);
                break;
            case app_enums_1.enums.UserVisitorActionType.SetVisitorAsTemporaryPermitted:
                message = (0, app_util_1.stringFormat)(i18n_1.default.__({ phrase: "VISITOR.ACTIONHISTORY.SETTEMPORARYPERMIT", locale: locale }), [userFullname]);
                break;
            case app_enums_1.enums.UserVisitorActionType.UnsetVisitorAsBlacklisted:
                message = (0, app_util_1.stringFormat)(i18n_1.default.__({ phrase: "VISITOR.ACTIONHISTORY.UNSETBLACKLIST", locale: locale }), [userFullname]);
                break;
            case app_enums_1.enums.UserVisitorActionType.UnsetVisitorAsTemporaryPermitted:
                message = (0, app_util_1.stringFormat)(i18n_1.default.__({ phrase: "VISITOR.ACTIONHISTORY.UNSETTEMPORARYPERMIT", locale: locale }), [userFullname]);
                break;
            case app_enums_1.enums.UserVisitorActionType.UpdateVisitorProfile:
                message = (0, app_util_1.stringFormat)(i18n_1.default.__({ phrase: "VISITOR.ACTIONHISTORY.UPDATEVISITORPROFILE", locale: locale }), [userFullname]);
                break;
            case app_enums_1.enums.UserVisitorActionType.InsertVisitorProfile:
                message = (0, app_util_1.stringFormat)(i18n_1.default.__({ phrase: "VISITOR.ACTIONHISTORY.INSERTVISITORPROFILE", locale: locale }), [userFullname]);
                break;
            case app_enums_1.enums.UserVisitorActionType.UpsertVisitorRegistrationPoint:
                message = (0, app_util_1.stringFormat)(i18n_1.default.__({ phrase: "VISITOR.ACTIONHISTORY.UPSERTREGISTRATIONPOINT", locale: locale }), [userFullname]);
                break;
            case app_enums_1.enums.UserVisitorActionType.MergeVisitorProfile:
                message = (0, app_util_1.stringFormat)(i18n_1.default.__({ phrase: "VISITOR.ACTIONHISTORY.MERGEVISITORPROFILE", locale: locale }), [userFullname]);
                break;
            default:
                break;
        }
        results.push({
            action: item.action,
            actionUser: item.actionUser,
            actionUtc: item.actionUtc,
            message: message,
        });
    }
    return results;
}
exports.transformVisitorActionItemsLocalizedMessages = transformVisitorActionItemsLocalizedMessages;
function hideSensitiveFieldsOfVisits(organizationId, jwt, visits, settings) {
    let hiddenVisitFields = settings.visitFormFields
        .filter((v) => v.requiredPermissions && v.requiredPermissions.reduce((a, b) => a && !jwt.hasPermissionForOrganizationOrAnyUnit(organizationId, b), true))
        .map((v) => v.name);
    for (let terminatedVisit of visits.items) {
        for (const hiddenField of hiddenVisitFields) {
            if (terminatedVisit.visitFields[hiddenField] != null || terminatedVisit.visitFields[hiddenField] != undefined)
                terminatedVisit.visitFields[hiddenField] = null;
        }
    }
    let hiddenVisitorFields = settings.visitorProfileFormFields
        .filter((v) => v.requiredPermissions && v.requiredPermissions.reduce((a, b) => a && !jwt.hasPermissionForOrganizationOrAnyUnit(organizationId, b), true))
        .map((v) => v.name);
    for (let terminatedVisit of visits.items) {
        for (const hiddenField of hiddenVisitorFields) {
            if (terminatedVisit.visitorProfileFields[hiddenField] != null || terminatedVisit.visitorProfileFields[hiddenField] != undefined)
                terminatedVisit.visitorProfileFields[hiddenField] = null;
        }
        hideVisitorUniqueId(terminatedVisit.visitorProfileFields, settings);
    }
    return visits;
}
exports.hideSensitiveFieldsOfVisits = hideSensitiveFieldsOfVisits;
const hideSensitiveFieldsOfVisitorFormSearch = (organizationId, jwt, formSearchResultItem, settings) => {
    let hiddenVisitorFields = settings.visitorProfileFormFields
        .filter((v) => v.requiredPermissions && v.requiredPermissions.reduce((a, b) => a && !jwt.hasPermissionForOrganizationOrAnyUnit(organizationId, b), true))
        .map((v) => v.name);
    for (let profile of formSearchResultItem) {
        for (const hiddenField of hiddenVisitorFields) {
            if (profile.searchFields[hiddenField] != null || profile.searchFields[hiddenField] != undefined)
                profile.searchFields[hiddenField] = null;
        }
        let uniqueId = profile.searchFields.find((searchField) => searchField.name === dal_access_models_1.DbAccessModel.PredefinedVisitorProfileFormFields.uniqueId)?.value;
        if (uniqueId && settings?.uniqueIdSettings?.hide) {
            profile.searchFields.find((searchField) => searchField.name === dal_access_models_1.DbAccessModel.PredefinedVisitorProfileFormFields.uniqueId).value =
                uniqueId.substring(0, settings.uniqueIdSettings.showNumberOfInitial) +
                    "***********".substring(0, 11 - settings.uniqueIdSettings.showNumberOfInitial - settings.uniqueIdSettings.showNumberOfLast) +
                    uniqueId.substring(uniqueId.length - settings.uniqueIdSettings.showNumberOfLast);
        }
    }
    return formSearchResultItem;
};
exports.hideSensitiveFieldsOfVisitorFormSearch = hideSensitiveFieldsOfVisitorFormSearch;
function hideSensitiveFieldsOfVisitors(organizationId, jwt, visitors, settings) {
    let hiddenVisitorFields = settings.visitorProfileFormFields
        .filter((v) => v.requiredPermissions && v.requiredPermissions.reduce((a, b) => a && !jwt.hasPermissionForOrganizationOrAnyUnit(organizationId, b), true))
        .map((v) => v.name);
    for (let profile of visitors) {
        for (const hiddenField of hiddenVisitorFields) {
            if (profile.fields[hiddenField] != null || profile.fields[hiddenField] != undefined)
                profile.fields[hiddenField] = null;
        }
        hideVisitorUniqueId(profile.fields, settings);
    }
    return visitors;
}
exports.hideSensitiveFieldsOfVisitors = hideSensitiveFieldsOfVisitors;
function hideVisitorUniqueId(visitor, settings) {
    if (!visitor) {
        return;
    }
    let uniqueId = visitor[dal_access_models_1.DbAccessModel.PredefinedVisitorProfileFormFields.uniqueId];
    if (uniqueId && settings?.uniqueIdSettings?.hide) {
        visitor[dal_access_models_1.DbAccessModel.PredefinedVisitorProfileFormFields.uniqueId] =
            uniqueId.substring(0, settings.uniqueIdSettings.showNumberOfInitial) +
                "***********".substring(0, 11 - settings.uniqueIdSettings.showNumberOfInitial - settings.uniqueIdSettings.showNumberOfLast) +
                uniqueId.substring(uniqueId.length - settings.uniqueIdSettings.showNumberOfLast);
    }
}
exports.hideVisitorUniqueId = hideVisitorUniqueId;
var BusinessVisitor;
(function (BusinessVisitor) {
    async function endVisitByQrCode(organizationId, visitorRegistrationPointId, organizationVisitorModuleSettings) {
        let registrationPoint = await dal_manager_1.dbManager.accessVisitor.getRegistrationPoint(organizationId, visitorRegistrationPointId);
        if (registrationPoint.tabletDeviceId && !registrationPoint.isConnected) {
            throw (0, api_error_1.generateTranslatedError)(app_enums_1.enums.HttpStatusCode.TABLET_DEVICE_NOT_CONNECTED, "ERRORS.VISITOR.TABLETNOTCONNECTED", null, true);
        }
        let response = null;
        let sessionId = uuid_1.default.v4();
        let requestApprovalPromise = new Promise((resolve, reject) => {
            messageBroker_server_to_tablet_pub_1.amqpServerToTabletPub.sendToRpcQueue(registrationPoint.tabletDeviceId, {
                e: messagebroker_models_1.MessageBrokerModels.ServerToTabletMessageTypes.EndVisitByQr,
                s: sessionId,
            }, organizationVisitorModuleSettings.gdprSettings.approvementTimeout * 1000, (err, success) => {
                if (err) {
                    return reject(err);
                }
                response = success;
                resolve(null);
            });
        });
        await requestApprovalPromise.catch((err) => {
            throw (0, api_error_1.generateTranslatedError)(app_enums_1.enums.HttpStatusCode.QR_CODE_READ_TIMEOUT, "ERRORS.VISITOR.QR_CODE_READ_TIMEOUT", null, true);
        });
        if (response.e == messagebroker_models_1.MessageBrokerModels.TabletToServerMessageTypes.QrCodeClosed) {
            throw (0, api_error_1.generateTranslatedError)(app_enums_1.enums.HttpStatusCode.QR_CODE_READ_CANCELLED, "ERRORS.VISITOR.QR_CODE_READ_CANCELLED", null, true);
        }
        if (response.e == messagebroker_models_1.MessageBrokerModels.TabletToServerMessageTypes.QrCodeTimeout) {
            throw (0, api_error_1.generateTranslatedError)(app_enums_1.enums.HttpStatusCode.QR_CODE_READ_TIMEOUT, "ERRORS.VISITOR.QR_CODE_READ_TIMEOUT", null, true);
        }
        if (response.e == messagebroker_models_1.MessageBrokerModels.TabletToServerMessageTypes.QrCodeData) {
            return response.p;
        }
        else {
            throw (0, api_error_1.generateTranslatedError)(app_enums_1.enums.HttpStatusCode.TABLET_SEQUENCE_INVALID_RESPONSE, "ERRORS.VISITOR.INVALID_SEQUENCE_RESPONSE", null, true);
        }
    }
    BusinessVisitor.endVisitByQrCode = endVisitByQrCode;
    async function obtainVisitIdViaQrCode(organizationId, visitorRegistrationPointId, organizationVisitorModuleSettings) {
        let registrationPoint = await dal_manager_1.dbManager.accessVisitor.getRegistrationPoint(organizationId, visitorRegistrationPointId);
        if (registrationPoint.tabletDeviceId && !registrationPoint.isConnected) {
            throw (0, api_error_1.generateTranslatedError)(app_enums_1.enums.HttpStatusCode.TABLET_DEVICE_NOT_CONNECTED, "ERRORS.VISITOR.TABLETNOTCONNECTED", null, true);
        }
        let response = null;
        let sessionId = uuid_1.default.v4();
        let requestApprovalPromise = new Promise((resolve, reject) => {
            messageBroker_server_to_tablet_pub_1.amqpServerToTabletPub.sendToRpcQueue(registrationPoint.tabletDeviceId, {
                e: messagebroker_models_1.MessageBrokerModels.ServerToTabletMessageTypes.QrCodeRequired,
                s: sessionId,
            }, organizationVisitorModuleSettings.gdprSettings.approvementTimeout * 1000, (err, success) => {
                if (err) {
                    return reject(err);
                }
                response = success;
                resolve(null);
            });
        });
        await requestApprovalPromise.catch((err) => {
            throw (0, api_error_1.generateTranslatedError)(app_enums_1.enums.HttpStatusCode.QR_CODE_READ_TIMEOUT, "ERRORS.VISITOR.QR_CODE_READ_TIMEOUT", null, true);
        });
        if (response.e == messagebroker_models_1.MessageBrokerModels.TabletToServerMessageTypes.QrCodeClosed) {
            throw (0, api_error_1.generateTranslatedError)(app_enums_1.enums.HttpStatusCode.QR_CODE_READ_CANCELLED, "ERRORS.VISITOR.QR_CODE_READ_CANCELLED", null, true);
        }
        if (response.e == messagebroker_models_1.MessageBrokerModels.TabletToServerMessageTypes.QrCodeTimeout) {
            throw (0, api_error_1.generateTranslatedError)(app_enums_1.enums.HttpStatusCode.QR_CODE_READ_TIMEOUT, "ERRORS.VISITOR.QR_CODE_READ_TIMEOUT", null, true);
        }
        if (response.e == messagebroker_models_1.MessageBrokerModels.TabletToServerMessageTypes.QrCodeData) {
            return response.p;
        }
        else {
            throw (0, api_error_1.generateTranslatedError)(app_enums_1.enums.HttpStatusCode.TABLET_SEQUENCE_INVALID_RESPONSE, "ERRORS.VISITOR.INVALID_SEQUENCE_RESPONSE", null, true);
        }
    }
    BusinessVisitor.obtainVisitIdViaQrCode = obtainVisitIdViaQrCode;
    async function newVisit(organizationId, userId, request, organizationVisitorModuleSettings, trx, locale) {
        let dbResult = null;
        let thumbnail = null;
        let registrationPoint = await dal_manager_1.dbManager.accessVisitor.getRegistrationPoint(organizationId, request.visitorRegistrationPointId, trx);
        await dal_manager_1.dbManager.accessVisitor.verifyActiveVisit(organizationId, request.visitFields, request.credentials, request.visitorProfileFields, request.visitorProfileId, trx, request.replacedVisitorProfileId);
        if (organizationVisitorModuleSettings.gdprEnabled && registrationPoint.tabletDeviceId) {
            if (!registrationPoint.isConnected) {
                throw (0, api_error_1.generateTranslatedError)(app_enums_1.enums.HttpStatusCode.TABLET_DEVICE_NOT_CONNECTED, "ERRORS.VISITOR.TABLETNOTCONNECTED", null, true);
            }
            let response = null;
            let visitorFullName = "";
            if (request.visitorProfileId) {
                let visitor = await dal_manager_1.dbManager.accessVisitor.getVisitorDetailed(organizationId, request.visitorProfileId, organizationVisitorModuleSettings);
                for (const key of Object.keys(visitor.profile.fields)) {
                    if (key == "name") {
                        visitorFullName = visitor.profile.fields[key];
                        break;
                    }
                }
            }
            else {
                let visitorField = request.visitorProfileFields.find((t) => t.name == "name");
                visitorFullName = visitorField ? visitorField.value : "";
            }
            let sessionId = uuid_1.default.v4();
            let requestApprovalPromise = new Promise((resolve, reject) => {
                messageBroker_server_to_tablet_pub_1.amqpServerToTabletPub.sendToRpcQueue(registrationPoint.tabletDeviceId, {
                    e: messagebroker_models_1.MessageBrokerModels.ServerToTabletMessageTypes.RequestApproval,
                    s: sessionId,
                    p: {
                        fullname: visitorFullName,
                        visitId: request.visitId,
                    },
                }, organizationVisitorModuleSettings.gdprSettings.approvementTimeout * 1000, (err, success) => {
                    if (err) {
                        return reject(err);
                    }
                    response = success;
                    resolve(null);
                });
            });
            await requestApprovalPromise.catch((err) => {
                throw (0, api_error_1.generateTranslatedError)(app_enums_1.enums.HttpStatusCode.VISITOR_APPROVAL_TIMEOUT, "ERRORS.VISITOR.APPROVAL_TIMEOUT", null, true);
            });
            if (response.e == messagebroker_models_1.MessageBrokerModels.TabletToServerMessageTypes.Rejected) {
                throw (0, api_error_1.generateTranslatedError)(app_enums_1.enums.HttpStatusCode.TABLET_DEVICE_NOT_APPROVED, "ERRORS.VISITOR.NOTAPPROVED", null, true);
            }
            if (response.e == messagebroker_models_1.MessageBrokerModels.TabletToServerMessageTypes.ApprovalTimeout) {
                throw (0, api_error_1.generateTranslatedError)(app_enums_1.enums.HttpStatusCode.VISITOR_APPROVAL_TIMEOUT, "ERRORS.VISITOR.APPROVAL_TIMEOUT", null, true);
            }
            if (response.e == messagebroker_models_1.MessageBrokerModels.TabletToServerMessageTypes.Approved) {
                let p = response.p;
                if (p.photo) {
                    thumbnail = p.photo;
                }
            }
            else {
                throw (0, api_error_1.generateTranslatedError)(app_enums_1.enums.HttpStatusCode.TABLET_SEQUENCE_INVALID_RESPONSE, "ERRORS.VISITOR.INVALID_SEQUENCE_RESPONSE", null, true);
            }
        }
        const beforeVisitorSaveHook = business_hooks_1.armonHookManager.getBeforeVisitorSaveHook(organizationId);
        if (beforeVisitorSaveHook) {
            await beforeVisitorSaveHook({
                organizationId,
                visitFields: request.visitFields,
                visitorProfileFields: request.visitorProfileFields,
                organizationSettings: organizationVisitorModuleSettings,
                visitorProfileId: request.visitorProfileId,
                trx,
                force: request.force,
            });
        }
        const beforeVisitorUpsertHook = business_hooks_1.armonHookManager.getBeforeVisitorUpsert(organizationId);
        if (beforeVisitorUpsertHook && request.visitorProfileId && request.visitorProfileFields) {
            request.visitorProfileFields = beforeVisitorUpsertHook({
                id: request.visitorProfileId,
                fields: request.visitorProfileFields,
            }, organizationVisitorModuleSettings).fields;
        }
        dbResult = await dal_manager_1.dbManager.accessVisitor.insertActiveVisit(organizationId, userId, {
            organizationVisitorModuleSettings,
            visitFields: request.visitFields,
            accessControlPointIds: request.accessControlPointIds,
            credentials: request.credentials,
            visitorProfileFields: request.visitorProfileFields,
            visitorProfileId: request.visitorProfileId,
            visitId: request.visitId,
            replacedVisitorProfileId: request.replacedVisitorProfileId,
            visitorRegistrationPointId: request.visitorRegistrationPointId,
            processTime: request.processTime,
            thumbnail,
            force: request.force,
            locale,
        }, trx);
        let afterVisitorSaveHook = business_hooks_1.armonHookManager.getAfterVisitorSaveHook(organizationId);
        if (afterVisitorSaveHook) {
            await afterVisitorSaveHook({
                organizationId: organizationId,
                visitId: dbResult.visitId,
                visitorProfileId: dbResult.visitorProfileId,
                trx: trx,
            });
        }
        if (dbResult) {
            let userInfo = await dal_manager_1.dbManager.accessUser.getBasicVisitorInfoWithRoleIdKnex(organizationId, dbResult.visitorProfileId, trx);
            await dal_manager_1.dbManager.accessRedisCache.setExpireValue(userInfo.id, JSON.stringify(userInfo), business_common_1.secondsInDay);
            const noPreregistrationAndQrCodeGenerated = !!(!request.visitId && dbResult.invitationInfo?.qrCodeData);
            if (noPreregistrationAndQrCodeGenerated) {
                dal_manager_1.dbManager.organizationTransaction(async (trx) => {
                    (0, exports.sendQrCodeToVisitor)(organizationId, dbResult.invitationInfo, dbResult.visitorUserId, organizationVisitorModuleSettings, trx);
                }, userId, organizationId);
            }
        }
        return dbResult;
    }
    BusinessVisitor.newVisit = newVisit;
    async function generateQrCodeFile(data) {
        let filePath = path_1.default.join(app_config_1.appConfig.tmpDirectory, uuid_1.default.v4());
        await qrcode_1.default.toFile(filePath, [
            {
                data: JSON.stringify(data),
                mode: "byte",
            },
        ], {
            errorCorrectionLevel: "M",
            width: 512,
        });
        return filePath;
    }
    BusinessVisitor.generateQrCodeFile = generateQrCodeFile;
    async function downloadVisitorExcel(organizationId, userId, locale, organizationVisitorModuleSettings, preRegistration) {
        let visitorExcel = new business_visitor_excel_1.VisitorsUploadExcel(organizationId, userId, locale, organizationVisitorModuleSettings);
        let visitorExcelResponse = {};
        const timeout = setTimeout(() => {
            visitorExcelResponse.errorMessage = i18n_1.default.__({ phrase: "VISITOR-EXCEL.TIMEOUT.WRITE", locale: locale });
            return visitorExcelResponse;
        }, 10 * 1000);
        visitorExcelResponse = await visitorExcel.generateExcel(preRegistration);
        clearTimeout(timeout);
        return visitorExcelResponse;
    }
    BusinessVisitor.downloadVisitorExcel = downloadVisitorExcel;
    async function uploadVisitorExcel(res, organizationId, userId, locale, organizationVisitorModuleSettings, fileBuffer, preRegistration, autoRegistration, visitFormFields) {
        let visitorExcel = new business_visitor_excel_1.VisitorsUploadExcel(organizationId, userId, locale, organizationVisitorModuleSettings);
        let visitorExcelResponse = {};
        let scheduledJobVisitData = {
            visitCount: 0,
            visitorRegistrationPointId: "",
            visitFormFields: [],
            accessControlPointIds: [],
            eventName: null,
            autoRegistration: false,
            autoTermination: false,
        };
        try {
            scheduledJobVisitData = (0, exports.validateVisitFields)(visitFormFields, preRegistration, autoRegistration, organizationVisitorModuleSettings, locale);
        }
        catch (error) {
            return {
                errorMessage: error.message,
            };
        }
        const loadTimeout = setTimeout(() => {
            visitorExcelResponse.errorMessage = i18n_1.default.__({ phrase: "VISITOR-EXCEL.TIMEOUT.LOAD", locale: locale });
            return visitorExcelResponse;
        }, 10 * 1000);
        await visitorExcel.loadExcel(fileBuffer);
        clearTimeout(loadTimeout);
        const readTimeout = setTimeout(() => {
            visitorExcelResponse.errorMessage = i18n_1.default.__({ phrase: "VISITOR-EXCEL.TIMEOUT.READ", locale: locale });
            return visitorExcelResponse;
        }, 10 * 1000);
        await visitorExcel.read(autoRegistration ? false : preRegistration);
        clearTimeout(readTimeout);
        const validationTimeout = setTimeout(() => {
            visitorExcelResponse.errorMessage = i18n_1.default.__({ phrase: "VISITOR-EXCEL.TIMEOUT.VALIDATION", locale: locale });
            return visitorExcelResponse;
        }, 30 * 1000);
        visitorExcel.validateMeta();
        let validation = await visitorExcel.validateRecords(res);
        scheduledJobVisitData.visitCount = validation.totalVisitorCount;
        const trx = await dal_manager_1.dbManager.transaction();
        if (validation.validationSuccess) {
            validation = await visitorExcel.validateExcelOnDB(scheduledJobVisitData, res, trx);
        }
        clearTimeout(validationTimeout);
        const visitorList = visitorExcel.getVisitorList();
        let failedRows = [];
        if (validation.validationSuccess) {
            let registrationScheduledJobId = null;
            let terminationScheduledJobId = null;
            let expectedDate = scheduledJobVisitData.visitFormFields.find((p) => p.name == dal_access_models_1.DbAccessModel.PredefinedVisitFormFields.expectedDate);
            if (scheduledJobVisitData.autoRegistration) {
                registrationScheduledJobId = await dal_manager_1.dbManager.accessVisitor.createVisitorExcelScheduledJob(organizationId, userId, {
                    type: 33,
                    interval: 10,
                    executionDate: new Date(expectedDate.value.startDate),
                    data: scheduledJobVisitData,
                }, trx);
            }
            if (scheduledJobVisitData.autoTermination) {
                terminationScheduledJobId = await dal_manager_1.dbManager.accessVisitor.createVisitorExcelScheduledJob(organizationId, userId, { type: 34, interval: 10, executionDate: new Date(expectedDate.value.endDate) }, trx);
            }
            if (preRegistration) {
                for (let index = 0; index < visitorList.length; index += 1) {
                    let result = await (0, exports.preRegisterVisitor)(organizationId, userId, organizationVisitorModuleSettings, visitorList[index], scheduledJobVisitData.visitFormFields, locale, trx);
                    if (!result.success) {
                        failedRows.push({
                            row: 2 + index,
                            error: i18n_1.default.__({ phrase: result.reason, locale: locale }),
                        });
                    }
                    else {
                        if (scheduledJobVisitData.autoRegistration) {
                            await dal_manager_1.dbManager.accessVisitor.mapVisitToScheduledJob(organizationId, result.visitId, registrationScheduledJobId, trx);
                        }
                        if (scheduledJobVisitData.autoTermination) {
                            await dal_manager_1.dbManager.accessVisitor.mapVisitToScheduledJob(organizationId, result.visitId, terminationScheduledJobId, trx);
                        }
                    }
                }
            }
            else {
                for (let index = 0; index < visitorList.length; index += 1) {
                    let result = await (0, exports.registerVisitor)(organizationId, userId, organizationVisitorModuleSettings, visitorList[index], scheduledJobVisitData, locale, trx);
                    if (!result.success) {
                        failedRows.push({
                            row: 2 + index,
                            error: i18n_1.default.__({ phrase: result.reason, locale: locale }),
                        });
                    }
                    else {
                        if (scheduledJobVisitData.autoTermination) {
                            await dal_manager_1.dbManager.accessVisitor.mapVisitToScheduledJob(organizationId, result.visitId, terminationScheduledJobId, trx);
                        }
                    }
                }
            }
            if (failedRows && failedRows.length > 0) {
                trx.rollback();
                visitorExcel.indicateFailedRows(failedRows);
                validation = await visitorExcel.downloadResultExcel(res);
                validation.validationSuccess = false;
            }
            else {
                app_httpserver_1.httpServer.sendVisitorListChangedNotification(organizationId);
            }
        }
        trx.commit();
        return validation;
    }
    BusinessVisitor.uploadVisitorExcel = uploadVisitorExcel;
})(BusinessVisitor = exports.BusinessVisitor || (exports.BusinessVisitor = {}));
async function enableVisitorModule(organizationId, trx) {
    await dal_manager_1.dbManager.accessVisitor.enableVisitorModule(organizationId, trx);
}
exports.enableVisitorModule = enableVisitorModule;
async function disableVisitorModule(organizationId, trx) {
    await dal_manager_1.dbManager.accessVisitor.disableVisitorModule(organizationId, trx);
    return Promise.resolve();
}
exports.disableVisitorModule = disableVisitorModule;
const sendQrCodeToVisitor = async (organizationId, invitationInfo, visitorUserId, organizationVisitorModuleSettings, trx) => {
    const qrcodeFilePath = await BusinessVisitor.generateQrCodeFile(invitationInfo.qrCodeData);
    const instanceData = {
        s: organizationVisitorModuleSettings.qrSettings.communicationMedium.type,
        receiverUserIds: [visitorUserId],
        q: qrcodeFilePath,
        i: invitationInfo,
        e: Array.isArray(invitationInfo.visitorProfile.fields)
            ? invitationInfo.visitorProfile.fields.find((field) => field.name === organizationVisitorModuleSettings.qrSettings.communicationMedium.field)?.value
            : invitationInfo.visitorProfile.fields[organizationVisitorModuleSettings.qrSettings.communicationMedium.field],
    };
    const notificationId = await (0, dal_memcache_1.getCacheUniqueNotificationIdOfOrganization)(organizationId, app_enums_1.enums.NotificationType.VisitorInvitation, trx);
    const eventId = await dal_manager_1.dbManager.accessNotifications.addNotificationEvent({
        createdT: new Date(),
        notificationId,
        organizationId,
        instanceData,
        trx,
    });
    (0, messageBroker_notification_pub_1.publishToNotificationService)({
        a: 0,
        i: eventId,
        n: app_enums_1.enums.AmqpMessageCode.Empty,
        o: organizationId,
        v: "1",
        d: instanceData,
    }, 10);
};
exports.sendQrCodeToVisitor = sendQrCodeToVisitor;
const resendVisitorInvitationNotificaton = async (organizationId, visitId, trx) => {
    const result = {
        invitationInfo: {
            organizationContactInfo: null,
        },
    };
    const organizationVisitorModuleSettings = await dal_manager_1.dbManager.accessVisitor.getVisitAndVisitorFormSettings(organizationId);
    const visitorDetailed = await dal_manager_1.dbManager.accessVisitor.getVisitorDetailedByVisitId(organizationId, visitId, organizationVisitorModuleSettings);
    const qrCodeData = dal_visitor_utils_1.DalVisitorUtils.generateIdBasedQrCodeContent(visitId);
    const activeVisitDetailed = await dal_manager_1.dbManager.accessVisitor.getActiveVisitDetailed(organizationId, visitId, organizationVisitorModuleSettings);
    if (!visitorDetailed.profile.fields[organizationVisitorModuleSettings.qrSettings.communicationMedium.field]) {
        if (organizationVisitorModuleSettings.qrSettings.communicationMedium.type === restapi_1.VisitorCommunicationMediumType.Email) {
            (0, dal_access_error_1.throwDbAccessBadRequestErrorTr)("ERROR.VISITOR.NOEMAIL", {}, false);
        }
        else {
            (0, dal_access_error_1.throwDbAccessBadRequestErrorTr)("ERROR.VISITOR.NOPHONENUMBER", {}, false);
        }
    }
    result.invitationInfo = {
        visitId,
        visitorProfile: visitorDetailed.profile,
        organizationContactInfo: null,
        visitedOrganizationUnitName: null,
        visitedUser: null,
        visitorRegistrationPointName: null,
        qrCodeData: qrCodeData,
    };
    const { rows: organizationInfoRows } = await trx.query(`SELECT "name", "contactInfo" FROM "${organizationId}".${dal_db_armon_schema_1.ArmonSchema.tableNames.organizations}
				 WHERE id = $1`, [organizationId]);
    result.invitationInfo.organizationContactInfo = Object.assign(organizationInfoRows[0]?.contactInfo || {}, { name: organizationInfoRows[0].name });
    if (visitorDetailed.lastVisit?.fields["visitedPerson"]?.userId) {
        const { rows: visitedUserRows } = await trx.query(`SELECT uop."userId", uop.name, uop.surname, uop."uniqueId" 
					 FROM "${organizationId}". "${dal_db_armon_schema_1.ArmonSchema.tableNames.users}" u
					 INNER JOIN "${organizationId}". "${dal_db_armon_schema_1.ArmonSchema.tableNames.userOrganizationProfiles}" uop
					 	ON u.id = uop."userId"
					 WHERE u."id" = $1`, [visitorDetailed.lastVisit?.fields["visitedPerson"]?.userId]);
        const visitedUserBadge = await dal_manager_1.dbManager.accessRedisCache.getUserBadgeCache({ organizationId, userId: visitorDetailed.lastVisit?.fields["visitedPerson"]?.userId, trx });
        result.invitationInfo.visitedUser = {
            id: visitedUserRows[0]?.userId,
            captionLines: visitedUserBadge.caption,
            fullname: visitedUserRows[0]?.name + " " + visitedUserRows[0]?.surname,
            uniqueId: visitedUserRows[0]?.uniqueId,
        };
    }
    if (visitorDetailed.lastVisit?.fields["visitedOrganizationUnit"]?.id) {
        const { rows } = await trx.query(`SELECT name FROM "${dal_db_armon_schema_1.ArmonSchema.tableNames.organizationUnits}" 
					 WHERE id = $1`, [visitorDetailed.lastVisit?.fields["visitedOrganizationUnit"]?.id]);
        result.invitationInfo.visitedOrganizationUnitName = rows[0]?.name;
    }
    if (activeVisitDetailed.visitorRegistrationPointId) {
        const { rows } = await trx.query(`SELECT name FROM "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.visitorRegistrationPoints}"
				 WHERE id = $1`, [activeVisitDetailed.visitorRegistrationPointId]);
        result.invitationInfo.visitorRegistrationPointName = rows[0]?.name;
    }
    await (0, exports.sendQrCodeToVisitor)(organizationId, result.invitationInfo, activeVisitDetailed.userId, organizationVisitorModuleSettings, trx);
};
exports.resendVisitorInvitationNotificaton = resendVisitorInvitationNotificaton;
const validateVisitFields = (visitFields, preRegistration, autoRegistration, organizationVisitorModuleSettings, locale) => {
    let visitFieldsForExcel = {
        visitFormFields: [],
        accessControlPointIds: [],
        visitorRegistrationPointId: "",
        eventName: null,
        autoRegistration: false,
        autoTermination: false,
        visitCount: 0,
    };
    let providedParameter;
    try {
        providedParameter = JSON.parse(visitFields);
    }
    catch (error) {
        app_logs_1.logger.error("visitFields parameter is not a valid object.");
        throw Error(i18n_1.default.__({ phrase: "VISITOR-EXCEL.BAD-REQUEST", locale: locale }));
    }
    let badRequest = false;
    try {
        if ((!preRegistration || autoRegistration) && (0, exports.isAccessControlPointIdsRequired)(organizationVisitorModuleSettings)) {
            if (!providedParameter.accessControlPointIds) {
                badRequest = true;
                app_logs_1.logger.error("Access control point ids list is required.");
            }
            else if (!Array.isArray(providedParameter.accessControlPointIds)) {
                badRequest = true;
                app_logs_1.logger.error("accessControlPointIds parameter must be an array.");
            }
            else if (providedParameter.accessControlPointIds.length == 0) {
                badRequest = true;
                app_logs_1.logger.error("Access control point ids list cannot be empty.");
            }
        }
        if (!providedParameter.visitorRegistrationPointId) {
            badRequest = true;
            app_logs_1.logger.error("visitorRegistrationPointId is required.");
        }
        if (!providedParameter.visitFormFields) {
            badRequest = true;
            app_logs_1.logger.error("visitFormFields is required.");
        }
        if (!providedParameter.eventName) {
            badRequest = true;
            app_logs_1.logger.error("eventName is required.");
        }
        if (!preRegistration && !providedParameter.hasOwnProperty("autoTermination")) {
            badRequest = true;
            app_logs_1.logger.error("autoTermination is required for registration.");
        }
        if (preRegistration && autoRegistration == null) {
            badRequest = true;
            app_logs_1.logger.error("autoRegistration is required for preregistration.");
        }
        if (badRequest) {
            throw Error(i18n_1.default.__({ phrase: "VISITOR-EXCEL.BAD-REQUEST", locale: locale }));
        }
        visitFieldsForExcel.eventName = providedParameter.eventName;
        visitFieldsForExcel.accessControlPointIds = providedParameter.accessControlPointIds;
        visitFieldsForExcel.visitorRegistrationPointId = providedParameter.visitorRegistrationPointId;
        visitFieldsForExcel.visitFormFields = providedParameter.visitFormFields;
        visitFieldsForExcel.force = providedParameter.force;
        visitFieldsForExcel.processTime = providedParameter.processTime;
        visitFieldsForExcel.autoTermination = providedParameter.autoTermination;
        visitFieldsForExcel.autoRegistration = autoRegistration;
        let expectedDateRegEx = /\d{4}-\d{2}-\d{2}T\d{2}:\d{2}:\d{2}(.\d{3})?Z/;
        let expectedDate = providedParameter.visitFormFields.find((p) => p.name == dal_access_models_1.DbAccessModel.PredefinedVisitFormFields.expectedDate);
        let expectedDateIndex = visitFieldsForExcel.visitFormFields.indexOf(expectedDate);
        if (!expectedDate) {
            app_logs_1.logger.error("expectedDate visitField is required.");
            throw Error(i18n_1.default.__({ phrase: "VISITOR-EXCEL.BAD-REQUEST", locale: locale }));
        }
        if (expectedDate.value.startDate && expectedDate.value.startDate != "" && expectedDateRegEx.test(expectedDate.value.startDate)) {
            visitFieldsForExcel.visitFormFields[expectedDateIndex].value.startDate = new Date(expectedDate.value.startDate);
            if (visitFieldsForExcel.visitFormFields[expectedDateIndex].value.startDate.toString() == "Invalid Date") {
                app_logs_1.logger.error("startDate of expectedDate is invalid.");
                badRequest = true;
            }
        }
        if (expectedDate.value.endDate && expectedDate.value.endDate != "" && expectedDateRegEx.test(expectedDate.value.endDate)) {
            visitFieldsForExcel.visitFormFields[expectedDateIndex].value.endDate = new Date(expectedDate.value.endDate);
            if (visitFieldsForExcel.visitFormFields[expectedDateIndex].value.endDate.toString() == "Invalid Date") {
                app_logs_1.logger.error("endDate of expectedDate is invalid.");
                badRequest = true;
            }
        }
        for (let field of visitFieldsForExcel.visitFormFields) {
            if (organizationVisitorModuleSettings.visitFormFields.map((pair) => pair.name).includes(field.name)) {
                if (field.value == "") {
                    visitFieldsForExcel.visitFormFields.splice(visitFieldsForExcel.visitFormFields.indexOf(field), 1);
                }
            }
            else {
                app_logs_1.logger.error(`Unexpected visitField: ${field.name} with value ${field.value}`);
                badRequest = true;
                break;
            }
        }
    }
    catch (e) {
        app_logs_1.logger.error(`Unexpected error: ${e.message}`);
        badRequest = true;
    }
    if (badRequest) {
        throw Error(i18n_1.default.__({ phrase: "VISITOR-EXCEL.BAD-REQUEST", locale: locale }));
    }
    return visitFieldsForExcel;
};
exports.validateVisitFields = validateVisitFields;
const preRegisterVisitor = async (organizationId, userId, visitorModuleSettings, visitorFields, visitFields, locale, trx) => {
    let visitId = null;
    try {
        let dbResult = await dal_manager_1.dbManager.accessVisitor.searchVisitorProfileForm(organizationId, visitorModuleSettings, {
            take: 1,
            skip: 0,
            filterFields: [
                {
                    name: dal_access_models_1.DbAccessModel.PredefinedVisitorProfileFormFields.uniqueId,
                    filter: visitorFields.find((e) => e.name == dal_access_models_1.DbAccessModel.PredefinedVisitorProfileFormFields.uniqueId).value,
                    formField: visitorModuleSettings.visitorProfileFormFields.find((e) => e.name == dal_access_models_1.DbAccessModel.PredefinedVisitorProfileFormFields.uniqueId),
                },
            ],
        }, true, userId, trx);
        let result = await dal_manager_1.dbManager.accessVisitor.upsertPreregisteredVisit(organizationId, userId, visitFields.find((p) => p.name == dal_access_models_1.DbAccessModel.PredefinedVisitFormFields.visitedPerson)
            ? visitFields.find((p) => p.name == dal_access_models_1.DbAccessModel.PredefinedVisitFormFields.visitedPerson).value
            : null, visitorModuleSettings, visitFields.filter((p) => p.value != ""), visitorFields, dbResult.total > 0 ? dbResult.items[0].id : null, true, trx, null, locale);
        visitId = result.visitId;
    }
    catch (error) {
        let translationPhrase = error.message && i18n_1.default.__({ phrase: error.message, locale: locale }) !== error.message
            ? i18n_1.default.__({ phrase: error.message, locale: locale })
            : error.phrase
                ? i18n_1.default.__({ phrase: error.message, locale: locale })
                : error.message
                    ? error.message
                    : "VISITOR-EXCEL.VALIDATIONS.FAILS.UNKNOWN_ERROR";
        app_logs_1.logger.warn(`Visit [${visitorFields.map((field) => `${field.name}: ${field.value}`).join(",")}] could not be preregistered: ${error}`);
        return {
            success: false,
            reason: translationPhrase,
        };
    }
    return {
        success: true,
        visitId: visitId,
    };
};
exports.preRegisterVisitor = preRegisterVisitor;
const registerVisitor = async (organizationId, userId, visitorModuleSettings, visitorFields, visitFields, locale, trx) => {
    let registrationResult;
    try {
        let dbResult = await dal_manager_1.dbManager.accessVisitor.searchVisitorProfileForm(organizationId, visitorModuleSettings, {
            take: 1,
            skip: 0,
            filterFields: [
                {
                    name: dal_access_models_1.DbAccessModel.PredefinedVisitorProfileFormFields.uniqueId,
                    filter: visitorFields.find((e) => e.name == dal_access_models_1.DbAccessModel.PredefinedVisitorProfileFormFields.uniqueId).value,
                    formField: visitorModuleSettings.visitorProfileFormFields.find((e) => e.name == dal_access_models_1.DbAccessModel.PredefinedVisitorProfileFormFields.uniqueId),
                },
            ],
        }, true, userId, trx);
        registrationResult = await BusinessVisitor.newVisit(organizationId, userId, {
            visitFields: visitFields.visitFormFields,
            visitorProfileFields: visitorFields,
            visitorProfileId: dbResult.total > 0 ? dbResult.items[0].id : null,
            accessControlPointIds: visitFields.accessControlPointIds,
            visitorRegistrationPointId: visitFields.visitorRegistrationPointId,
            force: visitFields.force,
            processTime: visitFields.processTime,
        }, visitorModuleSettings, trx);
        if (!registrationResult.visitId) {
            throw (0, dal_access_error_1.throwInternalErrorTr)();
        }
    }
    catch (error) {
        app_logs_1.logger.warn(`Visit [${visitorFields.map((field) => `${field.name}: ${field.value}`).join(",")}] could not be preregistered: ${error}`);
        let translationPhrase = error.message && i18n_1.default.__({ phrase: error.message, locale: locale }) !== error.message
            ? i18n_1.default.__({ phrase: error.message, locale: locale })
            : error.phrase
                ? i18n_1.default.__({ phrase: error.message, locale: locale })
                : error.message
                    ? error.message
                    : "VISITOR-EXCEL.VALIDATIONS.FAILS.UNKNOWN_ERROR";
        return {
            success: false,
            reason: translationPhrase,
        };
    }
    return {
        success: true,
        visitId: registrationResult.visitId,
    };
};
exports.registerVisitor = registerVisitor;
const isAccessControlPointIdsRequired = (organizationVisitorModuleSettings) => {
    const visitFormFields = organizationVisitorModuleSettings.visitFormFields.find((field) => field.name == dal_access_models_1.DbAccessModel.PredefinedVisitFormFields.accessControlPoints);
    if (visitFormFields) {
        if (visitFormFields.invisible || !visitFormFields.required) {
            return false;
        }
    }
    else {
        return false;
    }
    return true;
};
exports.isAccessControlPointIdsRequired = isAccessControlPointIdsRequired;
const isCredentialsRequired = (organizationVisitorModuleSettings) => {
    const visitFormFields = organizationVisitorModuleSettings.visitFormFields.find((field) => field.name == dal_access_models_1.DbAccessModel.PredefinedVisitFormFields.credentials);
    if (visitFormFields) {
        if (visitFormFields.required) {
            return true;
        }
    }
    else {
        return false;
    }
    return false;
};
exports.isCredentialsRequired = isCredentialsRequired;
