"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.VisitorsUploadExcel = void 0;
const faker_1 = require("@faker-js/faker");
const exceljs_1 = __importDefault(require("exceljs"));
const fs_1 = __importDefault(require("fs"));
const i18n_1 = __importDefault(require("i18n"));
const lodash_1 = require("lodash");
const moment_1 = __importDefault(require("moment"));
const path_1 = __importDefault(require("path"));
const randexp_1 = __importDefault(require("randexp"));
const uuid_1 = __importDefault(require("uuid"));
const app_config_1 = require("../../app.config");
const app_logs_1 = require("../../app.logs");
const dal_access_error_1 = require("../../dal/access/dal.access.error");
const dal_access_models_1 = require("../../dal/access/dal.access.models");
const dal_constants_1 = require("../../dal/dal.constants");
const dal_manager_1 = require("../../dal/dal.manager");
const dynamicFormCurrent_1 = require("../../lib/es/models/dynamicFormCurrent");
const business_visitor_metadata_1 = require("./business.visitor.metadata");
const business_visitor_models_1 = require("./business.visitor.models");
class VisitorsUploadExcel {
    constructor(organizationId, userId, locale, visitorModuleSettings) {
        this._acceptedCharactersForBoolean = { true: ["y", "yes", "true", "1", "evet"], false: ["n", "no", "false", "0", "hayır"] };
        this._acceptedCharactersForBooleanRegEx = /([01yn]|yes|no|evet|hayır)?/;
        this._fileId = uuid_1.default.v4();
        this._requestedUserId = userId;
        this._organizationId = organizationId;
        this._locale = locale;
        this._filePath = path_1.default.join(app_config_1.appConfig.tmpDirectory, this._fileId + ".xlsx");
        if (fs_1.default.existsSync(this._filePath)) {
            fs_1.default.unlinkSync(this._filePath);
        }
        this._options = visitorModuleSettings;
        this.setStyles();
    }
    async loadExcel(data) {
        this._wbReader = new exceljs_1.default.Workbook();
        await this._wbReader.xlsx.load(data);
    }
    async read(preRegisteration) {
        this._preRegistration = preRegisteration;
        this._organizationName = (await dal_manager_1.dbManager.accessOrganization.getOrganizationBasic(this._organizationId)).name;
        app_logs_1.logger.info("Organization Name : " + this._organizationName);
        let { caption: requestedUserCaption } = await dal_manager_1.dbManager.accessRedisCache.getUserBadgeCache({ organizationId: this._organizationId, userId: this._requestedUserId });
        if (!requestedUserCaption) {
            throw new Error("Not found");
        }
        this._requestedUserCaption = requestedUserCaption[0].text[0];
        app_logs_1.logger.info("Creator Name : " + this._requestedUserCaption);
        this.getHeaders();
        this._wbReader.getWorksheet(1).properties.defaultColWidth = 35;
        this.setColumnKeys(this._wbReader.getWorksheet(1));
    }
    validateMeta() {
        try {
            if (this._wbReader.worksheets.filter((sheet) => sheet.name == "meta").length == 0) {
                throw new Error("Meta sheet could not found.");
            }
            let metaKeysNHeaders = {};
            let inputHeaders = [];
            for (let sheet of this._wbReader.worksheets) {
                if (sheet.name == i18n_1.default.__({ phrase: `VISITOR-EXCEL.SHEET-NAMES.${"META"}`, locale: this._locale })) {
                    for (let header of this._headers) {
                        metaKeysNHeaders[header.key] = header.header;
                    }
                    this._createdAt = sheet.getCell("A4").toString();
                    let metaSheet = new business_visitor_metadata_1.MetaSheet(sheet.id, this._organizationId, this._preRegistration, this._createdAt, metaKeysNHeaders);
                    if (!metaSheet.validateHash(sheet.getCell("A6").toString())) {
                        throw new Error("Hash codes do not match.");
                    }
                }
                else if (sheet.name == i18n_1.default.__({ phrase: `VISITOR-EXCEL.SHEET-NAMES.${"INPUT"}`, locale: this._locale })) {
                    let headerRow = sheet.getRow(1);
                    headerRow.eachCell((cell) => {
                        inputHeaders.push(cell.value);
                    });
                }
            }
            let index = 0;
            for (let key of Object.keys(metaKeysNHeaders)) {
                if (metaKeysNHeaders[key] != inputHeaders[index]) {
                    throw new Error("Headers in input sheet do not match with the ones in meta sheet.");
                }
                index += 1;
            }
        }
        catch (e) {
            app_logs_1.logger.error(`An error encountered during validation of meta sheet. Error: ${e}`);
            throw new Error(i18n_1.default.__({ phrase: "VISITOR-EXCEL.INCORRECT-HASH", locale: this._locale }));
        }
    }
    async validateRecords(res) {
        let totalVisitorCount = 0;
        let validationSuccess = true;
        let dateRangeEndHeader = null;
        let endColumnIndex = null;
        if (this._headers.filter((header) => header.key.endsWith("End")).length > 0) {
            dateRangeEndHeader = this._headers.filter((header) => header.key.endsWith("End"))[0];
            endColumnIndex = this._headers.indexOf(dateRangeEndHeader) + 1;
        }
        this.createWorkbook();
        let sheet = this._wbReader.getWorksheet(1);
        totalVisitorCount = sheet.actualRowCount - 1;
        sheet.eachColumnKey((column) => {
            if (!column.key.includes("empty")) {
                let header = this._headers.filter((header) => header.key == column.key)[0];
                column.eachCell((cell, row) => {
                    if (row != 1) {
                        let validation = {
                            valid: true,
                            failReasons: [],
                        };
                        validation = this.requiredConstraint(header.required, validation, header.type, cell.value);
                        validation = this.uniquenessConstraintWithinForm(header.unique, validation, cell.value, row, column.values);
                        validation = this.regexConstraint(header.regex, validation, cell);
                        validation = this.optionConstraint(header.type, validation, header.option, cell.value, header.key, new Date(this._createdAt));
                        validation = this.dateRangeConstraint(dateRangeEndHeader, sheet.getRow(row), cell, endColumnIndex, validation);
                        if (validation.valid) {
                            cell.style = this._styles.success;
                            if (cell.note) {
                                this.cleanNotes(cell);
                            }
                        }
                        else {
                            cell.style = this._styles.fail;
                            this.addNoteToCell(sheet, cell, "-" + validation.failReasons.join("\n-"), true);
                        }
                        validationSuccess = validation.valid && validationSuccess;
                    }
                });
            }
        });
        if (!validationSuccess) {
            return await this.downloadResultExcel(res);
        }
        return {
            validationSuccess: validationSuccess,
            totalVisitorCount,
        };
    }
    async validateExcelOnDB(visitFields, res, trx) {
        let failedRows = [];
        for (let rowNumber = 2; rowNumber <= this._wbReader.getWorksheet(1).rowCount; rowNumber += 1) {
            const row = this._wbReader.getWorksheet(1).getRow(rowNumber);
            let visitorFields = [];
            for (let col = 0; col < this._headers.length; col += 1) {
                visitorFields.push({
                    name: this._headers[col].key,
                    value: row.getCell(col + 1).value,
                });
            }
            try {
                const activeVisitList = await dal_manager_1.dbManager.accessVisitor.listActiveVisitsOfVisitorByUniqueId(this._organizationId, visitorFields.find((e) => e.name == "uniqueId").value, trx);
                if (activeVisitList.length > 0) {
                    for (let visit of activeVisitList) {
                        let expectedDateOfNewVisit = visitFields.visitFormFields.find((p) => p.name == dal_access_models_1.DbAccessModel.PredefinedVisitFormFields.expectedDate);
                        if (visit.state == dal_constants_1.DalConstants.OrganizationVisitorStates.Expected && visit.expectedEndUtc && visit.expectedStartUtc) {
                            if ((expectedDateOfNewVisit.value.startDate.valueOf() >= visit.expectedStartUtc.valueOf() &&
                                expectedDateOfNewVisit.value.startDate.valueOf() <= visit.expectedEndUtc.valueOf()) ||
                                (expectedDateOfNewVisit.value.startDate.valueOf() < visit.expectedStartUtc.valueOf() &&
                                    expectedDateOfNewVisit.value.startDate.valueOf() >= visit.expectedStartUtc.valueOf())) {
                                throw (0, dal_access_error_1.throwDbAccessBadRequestErrorTr)("VISITOR-EXCEL.VALIDATIONS.FAILS.VISIT-CONFLICT", null, false);
                            }
                        }
                        else if (visit.state == dal_constants_1.DalConstants.OrganizationVisitorStates.Active) {
                            if (visit.expectedEndUtc) {
                                if (expectedDateOfNewVisit.value.startDate.valueOf() <= visit.expectedEndUtc.valueOf()) {
                                    throw (0, dal_access_error_1.throwDbAccessBadRequestErrorTr)("VISITOR-EXCEL.VALIDATIONS.FAILS.VISIT-CONFLICT", null, false);
                                }
                            }
                        }
                    }
                }
            }
            catch (error) {
                let stringifiedRow = "[";
                row.eachCell((cell) => {
                    stringifiedRow += cell.value.toString() + ",";
                });
                stringifiedRow += "]";
                app_logs_1.logger.error(`Excel row validation on Database:\n row: #${rowNumber} - ${stringifiedRow}\n error: ${i18n_1.default.__({ phrase: error.phrase, locale: this._locale })}`);
                failedRows.push({ row: rowNumber, error: i18n_1.default.__({ phrase: error.phrase ?? "ERROR.FAILS.UNKNOWN_ERROR", locale: this._locale }) });
            }
        }
        if (failedRows && failedRows.length > 0) {
            this.indicateFailedRows(failedRows);
            return {
                validationSuccess: false,
                ...(await this.downloadResultExcel(res)),
            };
        }
        return {
            validationSuccess: true,
            filePath: "",
        };
    }
    requiredConstraint(required, validation, type, cell) {
        if (!required)
            return validation;
        let valid = validation.valid;
        let failReasons = validation.failReasons;
        if (cell == null || cell == undefined)
            valid = false;
        else {
            switch (type) {
                case dynamicFormCurrent_1.FormFieldType.Text:
                case dynamicFormCurrent_1.FormFieldType.Date:
                case dynamicFormCurrent_1.FormFieldType.Time:
                case dynamicFormCurrent_1.FormFieldType.DateTime:
                case dynamicFormCurrent_1.FormFieldType.Select:
                case dynamicFormCurrent_1.FormFieldType.Textarea:
                case dynamicFormCurrent_1.FormFieldType.Number:
                case dynamicFormCurrent_1.FormFieldType.DateRange:
                    if (cell.toString().trim().length == 0) {
                        valid = false;
                    }
                    break;
                case dynamicFormCurrent_1.FormFieldType.Checkbox:
                    if (this._acceptedCharactersForBooleanRegEx.test(cell.toString()))
                        break;
                    else {
                        valid = false;
                    }
                    break;
                case dynamicFormCurrent_1.FormFieldType.ImageCapture:
                case dynamicFormCurrent_1.FormFieldType.RadioGroup:
                case dynamicFormCurrent_1.FormFieldType.Custom:
                default:
                    break;
            }
        }
        if (!valid) {
            failReasons.push(i18n_1.default.__({ phrase: "VISITOR-EXCEL.VALIDATIONS.FAILS.REQUIRED", locale: this._locale }));
        }
        return {
            valid: valid && validation.valid,
            failReasons: valid ? validation.failReasons : failReasons,
        };
    }
    uniquenessConstraintWithinForm(unique, validation, cell, row, columnValues) {
        if (!unique)
            return validation;
        let failedReasons = validation.failReasons;
        let valid = validation.valid;
        let otherValues = columnValues.filter((val, index) => val && index != 1 && index != row);
        if (otherValues.indexOf(cell) == -1)
            valid = true;
        else
            valid = false;
        if (!valid) {
            failedReasons.push(i18n_1.default.__({ phrase: "VISITOR-EXCEL.VALIDATIONS.FAILS.UNIQUE", locale: this._locale }));
        }
        return {
            valid: valid && validation.valid,
            failReasons: valid ? validation.failReasons : failedReasons,
        };
    }
    regexConstraint(regex, validation, cell) {
        if (!regex)
            return validation;
        if (cell == null || cell == undefined || cell.toString().trim().length == 0)
            return validation;
        let failedReasons = validation.failReasons;
        let valid = validation.valid;
        if (new RegExp(regex).test(cell.toString()))
            valid = true;
        else
            valid = false;
        if (!valid) {
            failedReasons.push(i18n_1.default.__({ phrase: "VISITOR-EXCEL.VALIDATIONS.FAILS.FORMAT.REGEX", locale: this._locale }));
        }
        return {
            valid: valid && validation.valid,
            failReasons: valid ? validation.failReasons : failedReasons,
        };
    }
    optionConstraint(type, validation, option, cell, key, createdAt) {
        if (cell == null || cell == undefined || cell.toString().trim().length == 0)
            return validation;
        let valid = validation.valid;
        let failedReasons = validation.failReasons;
        switch (type) {
            case dynamicFormCurrent_1.FormFieldType.Text:
                let textOptions = option;
                if (textOptions.minLength) {
                    if (textOptions.minLength <= cell.toString().length)
                        valid = true && valid;
                    else {
                        valid = false;
                        failedReasons.push(i18n_1.default.__({ phrase: "VISITOR-EXCEL.VALIDATIONS.FAILS.MIN-LENGTH", locale: this._locale }, { length: textOptions.minLength.toString() }));
                    }
                }
                if (textOptions.maxLength) {
                    if (textOptions.maxLength >= cell.toString().length)
                        valid = true && valid;
                    else {
                        valid = false;
                        failedReasons.push(i18n_1.default.__({ phrase: "VISITOR-EXCEL.VALIDATIONS.FAILS.MAX-LENGTH", locale: this._locale }, { length: textOptions.maxLength.toString() }));
                    }
                }
                if (textOptions.onlyNumber) {
                    if (/^\d*$/.test(cell.toString()))
                        valid = true && valid;
                    else {
                        valid = false;
                        failedReasons.push(i18n_1.default.__({ phrase: "VISITOR-EXCEL.VALIDATIONS.FAILS.ONLY-NUMBER", locale: this._locale }));
                    }
                }
                break;
            case dynamicFormCurrent_1.FormFieldType.Date:
                let dateOptions = option;
                if (dateOptions.min) {
                    if (dateOptions.min <= new Date(this.convertDateFormat(cell.toString())))
                        valid = true && valid;
                    else {
                        valid = false;
                        failedReasons.push(i18n_1.default.__({ phrase: "VISITOR-EXCEL.VALIDATIONS.FAILS.MAX-DATE", locale: this._locale }, { date: this.dateFormat(dateOptions.min, false) }));
                    }
                }
                if (dateOptions.max) {
                    if (dateOptions.max >= new Date(this.convertDateFormat(cell.toString())))
                        valid = true && valid;
                    else {
                        valid = false;
                        failedReasons.push(i18n_1.default.__({ phrase: "VISITOR-EXCEL.VALIDATIONS.FAILS.MAX-DATE", locale: this._locale }, { date: this.dateFormat(dateOptions.max, false) }));
                    }
                }
                if (this.validDateTimeFormat(cell, false, false))
                    valid = true && valid;
                else {
                    valid = false;
                    failedReasons.push(i18n_1.default.__({ phrase: "VISITOR-EXCEL.VALIDATIONS.FAILS.FORMAT.DATE", locale: this._locale }));
                }
                break;
            case dynamicFormCurrent_1.FormFieldType.Time:
                let timeOptions = option;
                if (timeOptions.minMinutes) {
                    if (timeOptions.minMinutes <= parseInt(cell.toString().split(":")[1]))
                        valid = true && valid;
                    else {
                        valid = false;
                        failedReasons.push(i18n_1.default.__({ phrase: "VISITOR-EXCEL.VALIDATIONS.FAILS.MIN-MINUTE", locale: this._locale }, { minute: timeOptions.minMinutes.toString() }));
                    }
                }
                if (timeOptions.maxMinutes) {
                    if (timeOptions.maxMinutes >= parseInt(cell.toString().split(":")[1]))
                        valid = true && valid;
                    else {
                        valid = false;
                        failedReasons.push(i18n_1.default.__({ phrase: "VISITOR-EXCEL.VALIDATIONS.FAILS.MAX-MINUTE", locale: this._locale }, { minute: timeOptions.maxMinutes.toString() }));
                    }
                }
                if (timeOptions.minHours) {
                    if (timeOptions.minHours <= parseInt(cell.toString().split(":")[0]))
                        valid = true && valid;
                    else {
                        valid = false;
                        failedReasons.push(i18n_1.default.__({ phrase: "VISITOR-EXCEL.VALIDATIONS.FAILS.MIN-HOUR", locale: this._locale }, { hour: timeOptions.minHours.toString() }));
                    }
                }
                if (timeOptions.maxHours) {
                    if (timeOptions.maxHours >= parseInt(cell.toString().split(":")[0]))
                        valid = true && valid;
                    else {
                        valid = false;
                        failedReasons.push(i18n_1.default.__({ phrase: "VISITOR-EXCEL.VALIDATIONS.FAILS.MAX-HOUR", locale: this._locale }, { hour: timeOptions.maxHours.toString() }));
                    }
                }
                if (this.validDateTimeFormat(cell, true, true))
                    valid = true && valid;
                else {
                    valid = false;
                    failedReasons.push(i18n_1.default.__({ phrase: "VISITOR-EXCEL.VALIDATIONS.FAILS.FORMAT.TIME", locale: this._locale }));
                }
                break;
            case dynamicFormCurrent_1.FormFieldType.DateTime:
                let dateTimeOptions = option;
                if (dateTimeOptions.min) {
                    if (dateTimeOptions.min <= new Date(this.convertDateFormat(cell.toString())))
                        valid = true && valid;
                    else {
                        valid = false;
                        failedReasons.push(i18n_1.default.__({ phrase: "VISITOR-EXCEL.VALIDATIONS.FAILS.MIN-DATE", locale: this._locale }, { date: this.dateFormat(dateTimeOptions.min, true) }));
                    }
                }
                if (dateTimeOptions.max) {
                    if (dateTimeOptions.max >= new Date(this.convertDateFormat(cell.toString())))
                        valid = true && valid;
                    else {
                        valid = false;
                        failedReasons.push(i18n_1.default.__({ phrase: "VISITOR-EXCEL.VALIDATIONS.FAILS.MAX-DATE", locale: this._locale }, { date: this.dateFormat(dateTimeOptions.max, true) }));
                    }
                }
                if (this.validDateTimeFormat(cell, true, false))
                    valid = true && valid;
                else {
                    valid = false;
                    failedReasons.push(i18n_1.default.__({ phrase: "VISITOR-EXCEL.VALIDATIONS.FAILS.FORMAT.DATETIME", locale: this._locale }));
                }
                break;
            case dynamicFormCurrent_1.FormFieldType.Checkbox:
                if (this._acceptedCharactersForBooleanRegEx.test(cell.toString())) {
                    valid = true && valid;
                }
                else {
                    valid = false;
                    failedReasons.push(i18n_1.default.__({ phrase: "VISITOR-EXCEL.VALIDATIONS.FAILS.FORMAT.DATETIME", locale: this._locale }));
                }
                break;
            case dynamicFormCurrent_1.FormFieldType.Select:
                let selectOptions = option;
                let selections = cell
                    .toString()
                    .split(",")
                    .map((str) => str.trim());
                let availableValues = selectOptions.options.map((selectOpt) => selectOpt.value.toString()).concat(selectOptions.options.map((selectOpt) => selectOpt.captionLines[0]));
                if (!selectOptions.multiSelect && selections.length > 1) {
                    valid = true && valid;
                }
                for (let select of selections) {
                    if (!availableValues.includes(select)) {
                        valid = false;
                        failedReasons.push(i18n_1.default.__({ phrase: "VISITOR-EXCEL.VALIDATIONS.FAILS.INVALID-SELECT-OPTION", locale: this._locale }));
                    }
                }
                break;
            case dynamicFormCurrent_1.FormFieldType.Textarea:
                let textAreaOptions = option;
                if (textAreaOptions.minLength) {
                    if (textAreaOptions.minLength <= cell.toString().length)
                        valid = true && valid;
                    else {
                        valid = false;
                        failedReasons.push(i18n_1.default.__({ phrase: "VISITOR-EXCEL.VALIDATIONS.FAILS.MIN-LENGTH", locale: this._locale }, { length: textAreaOptions.minLength.toString() }));
                    }
                }
                if (textAreaOptions.maxLength) {
                    if (textAreaOptions.maxLength >= cell.toString().length)
                        valid = true && valid;
                    else {
                        valid = false;
                        failedReasons.push(i18n_1.default.__({ phrase: "VISITOR-EXCEL.VALIDATIONS.FAILS.MAX-LENGTH", locale: this._locale }, { length: textAreaOptions.maxLength.toString() }));
                    }
                }
                break;
            case dynamicFormCurrent_1.FormFieldType.Number:
                let numberOptions = option;
                if (numberOptions.min) {
                    if (numberOptions.min <= parseInt(cell.toString()))
                        valid = true && valid;
                    else {
                        valid = false;
                        failedReasons.push(i18n_1.default.__({ phrase: "VISITOR-EXCEL.VALIDATIONS.FAILS.MIN", locale: this._locale }, { value: numberOptions.min.toString() }));
                    }
                }
                if (numberOptions.max) {
                    if (numberOptions.max >= parseInt(cell.toString()))
                        valid = true && valid;
                    else {
                        valid = false;
                        failedReasons.push(i18n_1.default.__({ phrase: "VISITOR-EXCEL.VALIDATIONS.FAILS.MAX", locale: this._locale }, { value: numberOptions.max.toString() }));
                    }
                }
                if (numberOptions.step) {
                    if (parseInt(cell.toString()) % numberOptions.step == 0)
                        valid = true && valid;
                    else {
                        valid = false;
                        failedReasons.push(i18n_1.default.__({ phrase: "VISITOR-EXCEL.VALIDATIONS.FAILS.STEP", locale: this._locale }, { value: numberOptions.step.toString() }));
                    }
                }
                break;
            case dynamicFormCurrent_1.FormFieldType.DateRange:
                let dateRangeOptions = option;
                let dynamicDateProperties = {
                    minStartDynamicDate: null,
                    maxStartDynamicDate: null,
                };
                if (!key.endsWith("End")) {
                    dynamicDateProperties.minStartDynamicDate = dateRangeOptions.minStartDynamicDate;
                    dynamicDateProperties.maxStartDynamicDate = dateRangeOptions.maxStartDynamicDate;
                }
                for (let i = 0; i < Object.keys(dynamicDateProperties).length; i += 1) {
                    let property = Object.keys(dynamicDateProperties)[i];
                    let dynamicDateValue = dynamicDateProperties[Object.keys(dynamicDateProperties)[i]];
                    if (!dynamicDateValue)
                        continue;
                    let rule_value;
                    rule_value = new Date(this.convertDateFormat(this._createdAt));
                    switch (dynamicDateValue) {
                        case dynamicFormCurrent_1.DateDynamicValue.Now:
                            break;
                        case dynamicFormCurrent_1.DateDynamicValue.StartOfHour:
                            rule_value.setMinutes(0, 0, 0);
                            break;
                        case dynamicFormCurrent_1.DateDynamicValue.StartOfDay:
                            rule_value.setHours(0, 0, 0, 0);
                            break;
                        case dynamicFormCurrent_1.DateDynamicValue.EndOfHour:
                            rule_value.setMinutes(59, 59, 999);
                            break;
                        case dynamicFormCurrent_1.DateDynamicValue.EndOfDayAfterSevenDays:
                            rule_value = this.addDays(rule_value, 7);
                        case dynamicFormCurrent_1.DateDynamicValue.EndOfDay:
                            rule_value.setHours(23, 59, 59, 999);
                            break;
                        case dynamicFormCurrent_1.DateDynamicValue.EndOfDayAfterFiveDays:
                            rule_value = this.addDays(rule_value, 5);
                            rule_value.setHours(23, 59, 59, 999);
                            break;
                        default:
                            continue;
                    }
                    switch (property) {
                        case "minStartDynamicDate":
                            if (rule_value <= new Date(this.convertDateFormat(cell.toString())))
                                valid = true && valid;
                            else {
                                valid = false;
                                failedReasons.push(i18n_1.default.__({ phrase: "VISITOR-EXCEL.VALIDATIONS.FAILS.MIN-DATE", locale: this._locale }, { date: this.dateFormat(rule_value, true) }));
                            }
                            break;
                        case "maxStartDynamicDate":
                            if (rule_value >= new Date(this.convertDateFormat(cell.toString())))
                                valid = true && valid;
                            else {
                                valid = false;
                                failedReasons.push(i18n_1.default.__({ phrase: "VISITOR-EXCEL.VALIDATIONS.FAILS.MAX-DATE", locale: this._locale }, { date: this.dateFormat(rule_value, true) }));
                            }
                            break;
                        default:
                            continue;
                    }
                }
                break;
            case dynamicFormCurrent_1.FormFieldType.ImageCapture:
            case dynamicFormCurrent_1.FormFieldType.RadioGroup:
            case dynamicFormCurrent_1.FormFieldType.Custom:
            default:
                break;
        }
        return {
            valid: valid && validation.valid,
            failReasons: valid ? validation.failReasons : failedReasons,
        };
    }
    dateRangeConstraint(header, row, cell, endColumnIndex, validation) {
        let valid = validation.valid;
        let cellIndex;
        let failedReasons = [];
        row.eachCell((rowCell, col) => {
            if (rowCell == cell) {
                cellIndex = col;
            }
        });
        if (!header || !endColumnIndex || (cellIndex != endColumnIndex && cellIndex != endColumnIndex - 1))
            return validation;
        let dateRangeOption = header.option;
        if (dateRangeOption.maxDayInterval) {
            if (cellIndex == endColumnIndex) {
                let dayDifference = (new Date(this.convertDateFormat(row.getCell(cellIndex).value.toString())).getTime() -
                    new Date(this.convertDateFormat(row.getCell(cellIndex - 1).value.toString())).getTime()) /
                    (24 * 60 * 60 * 1000);
                if (dayDifference <= dateRangeOption.maxDayInterval)
                    valid = true && valid;
                else {
                    valid = false;
                    failedReasons.push(i18n_1.default.__({ phrase: "VISITOR-EXCEL.VALIDATIONS.FAILS.DATE-RANGE-INTERVAL", locale: this._locale }, { interval: dateRangeOption.maxDayInterval.toString() }));
                }
            }
            else if (cellIndex == endColumnIndex - 1) {
                let dayDifference = (new Date(this.convertDateFormat(row.getCell(cellIndex + 1).value.toString())).getTime() -
                    new Date(this.convertDateFormat(row.getCell(cellIndex).value.toString())).getTime()) /
                    (24 * 60 * 60 * 1000);
                if (dayDifference <= dateRangeOption.maxDayInterval)
                    valid = true && valid;
                else {
                    valid = false;
                    failedReasons.push(i18n_1.default.__({ phrase: "VISITOR-EXCEL.VALIDATIONS.FAILS.DATE-RANGE-INTERVAL", locale: this._locale }, { interval: dateRangeOption.maxDayInterval.toString() }));
                }
            }
        }
        return {
            valid: valid && validation.valid,
            failReasons: valid ? validation.failReasons : failedReasons,
        };
    }
    validDateTimeFormat(cell, includeTime, onlyTime) {
        if (!includeTime) {
            let date = (0, moment_1.default)(cell.toLocaleString("tr-TR"), "DD.MM.YYYY");
            return date.isValid();
        }
        else {
            if (onlyTime) {
                let date = (0, moment_1.default)(cell.toLocaleString("tr-TR"), "hh:mm:ss");
                return date.isValid();
            }
            else {
                let date = (0, moment_1.default)(cell.toLocaleString("tr-TR"), "DD.MM.YYYY hh:mm:ss");
                return date.isValid();
            }
        }
    }
    cleanNotes(cell) {
        delete cell.model.comment;
        delete cell._comment;
    }
    async generateExcel(preRegisteration) {
        this._preRegistration = preRegisteration;
        this._organizationName = (await dal_manager_1.dbManager.accessOrganization.getOrganizationBasic(this._organizationId)).name;
        app_logs_1.logger.info("Organization Name : " + this._organizationName);
        let { caption: requestedUserCaption } = await dal_manager_1.dbManager.accessRedisCache.getUserBadgeCache({ organizationId: this._organizationId, userId: this._requestedUserId });
        if (!requestedUserCaption) {
            app_logs_1.logger.error("Requested user cannot be found during generation of excel.");
            return {
                errorMessage: "User not found!",
            };
        }
        this._requestedUserCaption = requestedUserCaption[0].text[0];
        this.createWorkbook();
        app_logs_1.logger.info("Generating visitors excel...");
        app_logs_1.logger.info("Requester User : " + this._requestedUserCaption);
        this._sheetNames = {
            [business_visitor_models_1.VisitorExcelSheetType.Input]: null,
            [business_visitor_models_1.VisitorExcelSheetType.Sample]: null,
            [business_visitor_models_1.VisitorExcelSheetType.Meta]: null,
        };
        this.createSheets();
        for (let sheetName in this._sheetNames) {
            this._wbWriter.getWorksheet(this._sheetNames[sheetName]).commit();
        }
        await this._wbWriter.commit();
        app_logs_1.logger.info("File is ready -> " + this._filePath);
        let localeFormTypeCaption = i18n_1.default.__({ phrase: `VISITOR-EXCEL.${this._preRegistration ? "PREREGISTRATION" : "REGISTRATION"}`, locale: this._locale });
        return {
            filePath: this._filePath,
            filename: this._organizationName
                .toLocaleLowerCase()
                .replace(new RegExp(/[&\/\\#,+()$~%.'":*?<>{}`\r\n\t]/g), "")
                .replace(new RegExp(/[\s]+/g), "_")
                .replace(new RegExp(/ç/g), "c")
                .replace(new RegExp(/ğ/g), "g")
                .replace(new RegExp(/ı/g), "i")
                .replace(new RegExp(/ö/g), "o")
                .replace(new RegExp(/ş/g), "s")
                .replace(new RegExp(/ü/g), "u") +
                "_" +
                i18n_1.default.__({ phrase: "VISITOR-EXCEL.FILENAME", locale: this._locale }, { formType: localeFormTypeCaption }) +
                ".xlsx",
        };
    }
    createSheets() {
        this.getHeaders();
        for (let sheetName in this._sheetNames) {
            this.addSheetToWorkbook(sheetName);
            this._wbWriter.getWorksheet(this._sheetNames[sheetName]).properties.defaultRowHeight = 65;
            this._wbWriter.getWorksheet(this._sheetNames[sheetName]).properties.defaultColWidth = 15 + 2 * (15 - (this._headers.length >= 15 ? 0 : this._headers.length));
            this.setColumnKeys(this._wbWriter.getWorksheet(this._sheetNames[sheetName]));
            if (sheetName === business_visitor_models_1.VisitorExcelSheetType.Meta) {
                this.createMetaSheet(this._wbWriter.getWorksheet(this._sheetNames[sheetName]), "veryHidden");
            }
            else {
                this.addHeaders(this._wbWriter.getWorksheet(this._sheetNames[sheetName]));
                this.addConstraints(this._wbWriter.getWorksheet(this._sheetNames[sheetName]), sheetName === business_visitor_models_1.VisitorExcelSheetType.Input ? true : false);
            }
        }
    }
    addHeaders(ws) {
        for (let element of this._headers) {
            ws.getColumn(element.key).header = element.header;
        }
        for (let index = 1; index <= this._headers.length; index += 1) {
            ws.getCell(1, index).style = this._styles.headers;
        }
    }
    addSheetToWorkbook(key) {
        let translatedName = i18n_1.default.__({ phrase: `VISITOR-EXCEL.SHEET-NAMES.${key.toUpperCase()}`, locale: this._locale });
        let ws = this._wbWriter.addWorksheet(translatedName, {
            pageSetup: { fitToPage: true, fitToWidth: 1, fitToHeight: 0, paperSize: 9, orientation: "landscape" },
        });
        this._sheetNames[key] = ws.id;
    }
    addConstraints(ws, asNote) {
        if (asNote) {
            let rules = this.getValidationRules(ws);
            for (let index = 0; index < ws.columns.length; index += 1) {
                if (ws.columns[index].key.includes("empty"))
                    continue;
                this.addNoteToCell(ws, ws.getCell(1, index + 1), rules[ws.columns[index].key], true);
            }
        }
        else {
            let rules = this.getValidationRules(ws);
            ws.addRow(rules);
            ws.lastRow.eachCell((cell) => {
                cell.style = this._styles.rules;
            });
            this.addSampleRows(ws, 3);
        }
    }
    getValidationRules(ws) {
        let rules = {};
        for (let index = 0; index < ws.columns.length; index += 1) {
            if (ws.columns[index].key.includes("empty"))
                continue;
            let columnRules = "";
            let field = this._headers[index];
            let requiredText = field.required && !(field.type == dynamicFormCurrent_1.FormFieldType.DateRange)
                ? `-${i18n_1.default.__({ phrase: "VISITOR-EXCEL.VALIDATIONS.REQUIRED", locale: this._locale })}\n`
                : `-${i18n_1.default.__({ phrase: "VISITOR-EXCEL.VALIDATIONS.OPTIONAL", locale: this._locale })}\n`;
            columnRules += requiredText;
            if (field.unique) {
                columnRules += `-${i18n_1.default.__({ phrase: "VISITOR-EXCEL.VALIDATIONS.UNIQUE", locale: this._locale })}\n`;
            }
            let options = field.option;
            switch (field.type) {
                case dynamicFormCurrent_1.FormFieldType.Text:
                    let textOptions = options;
                    columnRules += this.getMinMaxLengthLimitRules(textOptions.minLength, textOptions.maxLength);
                    if (textOptions.onlyNumber) {
                        columnRules += `-${i18n_1.default.__({ phrase: "VISITOR-EXCEL.VALIDATIONS.ONLY-NUMBER", locale: this._locale })}\n`;
                    }
                    if (textOptions.defaultValue) {
                        columnRules += `-${i18n_1.default.__({ phrase: "VISITOR-EXCEL.VALIDATIONS.DEFAULT", locale: this._locale }, { default: textOptions.defaultValue })}\n`;
                    }
                    break;
                case dynamicFormCurrent_1.FormFieldType.Number:
                    let numberOptions = options;
                    columnRules += this.getMinMaxValueLimitRules(numberOptions.min, numberOptions.max);
                    if (numberOptions.step) {
                        columnRules += `-${i18n_1.default.__({ phrase: "VISITOR-EXCEL.VALIDATIONS.NUMBER-STEP", locale: this._locale }, { step: numberOptions.step.toString() })}\n`;
                    }
                    if (numberOptions.defaultValue) {
                        columnRules += `-${i18n_1.default.__({ phrase: "VISITOR-EXCEL.VALIDATIONS.DEFAULT", locale: this._locale }, { default: numberOptions.defaultValue.toString() })}\n`;
                    }
                    break;
                case dynamicFormCurrent_1.FormFieldType.Time:
                    let timeOptions = options;
                    columnRules += this.getMinMaxTimeLimitRules(timeOptions.minHours, timeOptions.maxHours, timeOptions.minMinutes, timeOptions.maxMinutes);
                    if (timeOptions.defaultHours || timeOptions.defaultMinutes) {
                        columnRules += `-${i18n_1.default.__({ phrase: "VISITOR-EXCEL.VALIDATIONS.DEFAULT-TIME", locale: this._locale }, {
                            defaultHours: timeOptions.defaultHours ? timeOptions.defaultHours.toString().padStart(2, "0") : "00",
                            defaultMinutes: timeOptions.defaultMinutes ? timeOptions.defaultMinutes.toString().padStart(2, "0") : "00",
                        })}\n`;
                    }
                    break;
                case dynamicFormCurrent_1.FormFieldType.Date:
                    let dateOptions = options;
                    columnRules += this.getMinMaxValueLimitRules(dateOptions.min, dateOptions.max, false);
                    if (dateOptions.defaultValue) {
                        let dateString = this.dateFormat(dateOptions.defaultValue, false);
                        columnRules += `-${i18n_1.default.__({ phrase: "VISITOR-EXCEL.VALIDATIONS.DEFAULT", locale: this._locale }, { default: dateString })}\n`;
                    }
                    let date = this.dateFormat(new Date(Date.now()), false);
                    columnRules += `-${i18n_1.default.__({ phrase: "VISITOR-EXCEL.VALIDATIONS.FORMAT.DATE", locale: this._locale }, { example: date })}\n`;
                    break;
                case dynamicFormCurrent_1.FormFieldType.DateTime:
                    let dateTimeOptions = options;
                    columnRules += this.getMinMaxValueLimitRules(dateTimeOptions.min, dateTimeOptions.max, true);
                    if (dateTimeOptions.defaultValue) {
                        let dateString = this.dateFormat(dateTimeOptions.defaultValue, true);
                        columnRules += `-${i18n_1.default.__({ phrase: "VISITOR-EXCEL.VALIDATIONS.DEFAULT", locale: this._locale }, { default: dateString })}\n`;
                    }
                    columnRules += `-${i18n_1.default.__({ phrase: "VISITOR-EXCEL.VALIDATIONS.FORMAT.DATETIME", locale: this._locale }, { example: this.dateFormat(new Date(Date.now()), true) })}\n`;
                    break;
                case dynamicFormCurrent_1.FormFieldType.DateRange:
                    let dateRangeOptions = options;
                    if (field.key.endsWith("End")) {
                        requiredText = dateRangeOptions.isEndRequired
                            ? `-${i18n_1.default.__({ phrase: "VISITOR-EXCEL.VALIDATIONS.REQUIRED", locale: this._locale })}\n`
                            : `-${i18n_1.default.__({ phrase: "VISITOR-EXCEL.VALIDATIONS.OPTIONAL", locale: this._locale })}\n`;
                        columnRules += requiredText;
                    }
                    else {
                        requiredText = dateRangeOptions.isStartRequired
                            ? `-${i18n_1.default.__({ phrase: "VISITOR-EXCEL.VALIDATIONS.REQUIRED", locale: this._locale })}\n`
                            : `-${i18n_1.default.__({ phrase: "VISITOR-EXCEL.VALIDATIONS.OPTIONAL", locale: this._locale })}\n`;
                        columnRules += requiredText;
                    }
                    if (dateRangeOptions.showTimer) {
                        columnRules += `-${i18n_1.default.__({ phrase: "VISITOR-EXCEL.VALIDATIONS.FORMAT.DATETIME", locale: this._locale }, { example: this.dateFormat(new Date(Date.now() - (field.key.endsWith("End") ? 0 : (1000 * 60 * 60 * 24 * 3) / 2)), true) })}\n`;
                    }
                    else {
                        columnRules += `-${i18n_1.default.__({ phrase: "VISITOR-EXCEL.VALIDATIONS.FORMAT.DATE", locale: this._locale }, { example: this.dateFormat(new Date(Date.now() - (field.key.endsWith("End") ? 0 : (1000 * 60 * 60 * 24 * 3) / 2)), false) })}\n`;
                    }
                    if (field.key.endsWith("End") && dateRangeOptions.isEndVisible && dateRangeOptions.maxDayInterval) {
                        columnRules += `-${i18n_1.default.__({ phrase: "VISITOR-EXCEL.VALIDATIONS.DATE-INTERVAL", locale: this._locale }, { interval: dateRangeOptions.maxDayInterval.toString() })}\n`;
                    }
                    let dynamicDateProperties = {
                        minStartDynamicDate: null,
                        maxStartDynamicDate: null,
                        startDynamicDate: null,
                        endDynamicDate: null,
                    };
                    if (field.key.endsWith("End")) {
                        dynamicDateProperties.endDynamicDate = dateRangeOptions.endDynamicDate;
                    }
                    else {
                        dynamicDateProperties.minStartDynamicDate = dateRangeOptions.minStartDynamicDate;
                        dynamicDateProperties.maxStartDynamicDate = dateRangeOptions.maxStartDynamicDate;
                        dynamicDateProperties.startDynamicDate = dateRangeOptions.startDynamicDate;
                    }
                    for (let i = 0; i < Object.keys(dynamicDateProperties).length; i += 1) {
                        let property = Object.keys(dynamicDateProperties)[i];
                        let dynamicDateValue = dynamicDateProperties[Object.keys(dynamicDateProperties)[i]];
                        if (!dynamicDateValue)
                            continue;
                        let rule_value;
                        let i18nPhrase = "VISITOR-EXCEL.VALIDATIONS.DYNAMIC-DATE.";
                        switch (property) {
                            case "minStartDynamicDate":
                                i18nPhrase += "MIN-START";
                                break;
                            case "maxStartDynamicDate":
                                i18nPhrase += "MAX-START";
                                break;
                            case "startDynamicDate":
                            case "endDynamicDate":
                                i18nPhrase += "DEFAULT";
                                break;
                            default:
                                continue;
                        }
                        rule_value = new Date(Date.now());
                        switch (dynamicDateValue) {
                            case dynamicFormCurrent_1.DateDynamicValue.Now:
                                break;
                            case dynamicFormCurrent_1.DateDynamicValue.StartOfHour:
                                rule_value.setMinutes(0, 0, 0);
                                break;
                            case dynamicFormCurrent_1.DateDynamicValue.StartOfDay:
                                rule_value.setHours(0, 0, 0, 0);
                                break;
                            case dynamicFormCurrent_1.DateDynamicValue.EndOfHour:
                                rule_value.setMinutes(59, 59, 999);
                                break;
                            case dynamicFormCurrent_1.DateDynamicValue.EndOfDayAfterSevenDays:
                                rule_value = this.addDays(rule_value, 7);
                            case dynamicFormCurrent_1.DateDynamicValue.EndOfDay:
                                rule_value.setHours(23, 59, 59, 999);
                                break;
                            case dynamicFormCurrent_1.DateDynamicValue.EndOfDayAfterFiveDays:
                                rule_value = this.addDays(rule_value, 5);
                                rule_value.setHours(23, 59, 59, 999);
                                break;
                            default:
                                continue;
                        }
                        columnRules += `-${i18n_1.default.__({ phrase: i18nPhrase, locale: this._locale }, { date: this.dateFormat(rule_value, true) })}\n`;
                    }
                    break;
                case dynamicFormCurrent_1.FormFieldType.Checkbox:
                    let checkboxOptions = options;
                    if (checkboxOptions.defaultValue) {
                        columnRules += `-${i18n_1.default.__({ phrase: "VISITOR-EXCEL.VALIDATIONS.DEFAULT", locale: this._locale }, { default: checkboxOptions.defaultValue.toString() })}\n`;
                    }
                    columnRules += `-${i18n_1.default.__({ phrase: "VISITOR-EXCEL.VALIDATIONS.CHECKBOX-VALUES", locale: this._locale }, { forTrue: this._acceptedCharactersForBoolean["true"].join(", "), forFalse: this._acceptedCharactersForBoolean["false"].join(", ") })}`;
                    break;
                case dynamicFormCurrent_1.FormFieldType.Textarea:
                    let textAreaOptions = options;
                    columnRules += this.getMinMaxLengthLimitRules(textAreaOptions.minLength, textAreaOptions.maxLength);
                    if (textAreaOptions.defaultValue) {
                        columnRules += `-${i18n_1.default.__({ phrase: "VISITOR-EXCEL.VALIDATIONS.DEFAULT", locale: this._locale }, { default: textAreaOptions.defaultValue })}\n`;
                    }
                    break;
                case dynamicFormCurrent_1.FormFieldType.Select:
                    let selectOptions = options;
                    if (selectOptions.multiSelect) {
                        columnRules += `-${i18n_1.default.__({ phrase: "VISITOR-EXCEL.VALIDATIONS.MULTI-SELECT", locale: this._locale })}`;
                    }
                    if (selectOptions.defaultValue) {
                        if (Array.isArray(selectOptions.defaultValue)) {
                            let valueList = selectOptions.defaultValue.map((item) => item.value.toString()).join(",");
                            let captionList = selectOptions.defaultValue.map((item) => item.captionLines[0]).join(",");
                            columnRules += `-${i18n_1.default.__({ phrase: "VISITOR-EXCEL.VALIDATIONS.DEFAULT-MULTI-SELECT", locale: this._locale }, { valueList: valueList, captionList: captionList })}`;
                        }
                        else {
                            columnRules += `-${i18n_1.default.__({ phrase: "VISITOR-EXCEL.VALIDATIONS.DEFAULT-SELECT", locale: this._locale }, { value: selectOptions.defaultValue.value, caption: selectOptions.defaultValue.captionLines[0] })}`;
                        }
                    }
                    let availableValues = selectOptions.options.map((item) => item.value.toString());
                    let availableCaptions = selectOptions.options.map((item) => item.captionLines[0]);
                    columnRules += `-${i18n_1.default.__({ phrase: "VISITOR-EXCEL.VALIDATIONS.MULTI-VALUES", locale: this._locale }, { values: availableValues.join(","), captions: availableCaptions.join(","), exampleValue: availableValues[0], exampleCaption: availableCaptions[0] })}`;
                    break;
                case dynamicFormCurrent_1.FormFieldType.ImageCapture:
                    break;
                case dynamicFormCurrent_1.FormFieldType.RadioGroup:
                case dynamicFormCurrent_1.FormFieldType.Custom:
                default:
                    throw new Error(i18n_1.default.__({ phrase: "VISITOR-EXCEL.FIELD-TYPE-ERROR", locale: this._locale }));
            }
            rules[field.key] = columnRules + "\n";
        }
        return rules;
    }
    getMinMaxLengthLimitRules(min, max) {
        let rules = "";
        if (min && min > 0 && max && max >= min) {
            if (min == max) {
                rules += `-${i18n_1.default.__({ phrase: "VISITOR-EXCEL.VALIDATIONS.EXACT-LENGTH", locale: this._locale }, { length: min.toString() })}\n`;
            }
            else {
                rules += `-${i18n_1.default.__({ phrase: "VISITOR-EXCEL.VALIDATIONS.BETWEEN-LENGTH", locale: this._locale }, { min: min.toString(), max: max.toString() })}\n`;
            }
        }
        else {
            if (min && min > 0) {
                rules += `-${i18n_1.default.__({ phrase: "VISITOR-EXCEL.VALIDATIONS.MIN-LENGTH", locale: this._locale }, { length: min.toString() })}\n`;
            }
            if (max && max >= min) {
                rules += `-${i18n_1.default.__({ phrase: "VISITOR-EXCEL.VALIDATIONS.MAX-LENGTH", locale: this._locale }, { length: max.toString() })}\n`;
            }
        }
        return rules;
    }
    getMinMaxValueLimitRules(min, max, includeTime) {
        let rules = "";
        if (min && min > 0 && max && max >= min) {
            rules += `-${i18n_1.default.__({ phrase: "VISITOR-EXCEL.VALIDATIONS.BETWEEN", locale: this._locale }, { min: min instanceof Date ? this.dateFormat(min, includeTime) : min.toString(), max: max instanceof Date ? this.dateFormat(max, includeTime) : max.toString() })}\n`;
        }
        else {
            if (min && min > 0) {
                rules += `-${i18n_1.default.__({ phrase: "VISITOR-EXCEL.VALIDATIONS.MIN", locale: this._locale }, { value: min instanceof Date ? this.dateFormat(min, includeTime) : min.toString() })}\n`;
            }
            if (max && max >= min) {
                rules += `-${i18n_1.default.__({ phrase: "VISITOR-EXCEL.VALIDATIONS.MAX", locale: this._locale }, { value: max instanceof Date ? this.dateFormat(max, includeTime) : max.toString() })}\n`;
            }
        }
        return rules;
    }
    getMinMaxTimeLimitRules(minHours, maxHours, minMinutes, maxMinutes) {
        let rules = "";
        if (((minHours && minHours > 0) || (maxHours && maxHours >= minHours)) && ((minMinutes && minMinutes > 0) || (maxMinutes && maxMinutes >= minMinutes))) {
            rules += `-${i18n_1.default.__({ phrase: "VISITOR-EXCEL.VALIDATIONS.BETWEEN-TIME", locale: this._locale }, {
                minHours: minHours ? minHours.toString().padStart(2, "0") : "00",
                maxHours: maxHours ? maxHours.toString().padStart(2, "0") : "00",
                minMinutes: minMinutes ? minMinutes.toString().padStart(2, "0") : "00",
                maxMinutes: maxMinutes ? maxMinutes.toString().padStart(2, "0") : "00",
            })}\n`;
        }
        else {
            if ((minHours && minHours > 0) || (minMinutes && minMinutes > 0)) {
                rules += `-${i18n_1.default.__({ phrase: "VISITOR-EXCEL.VALIDATIONS.MIN-TIME", locale: this._locale }, {
                    minMinutes: minMinutes ? minMinutes.toString().padStart(2, "0") : "00",
                    minHours: minHours ? minHours.toString().padStart(2, "0") : "00",
                })}\n`;
            }
            if ((maxHours && maxHours > 0) || (maxMinutes && maxMinutes > 0)) {
                rules += `-${i18n_1.default.__({ phrase: "VISITOR-EXCEL.VALIDATIONS.MAX-TIME", locale: this._locale }, {
                    maxMinutes: maxMinutes ? maxMinutes.toString().padStart(2, "0") : "00",
                    maxHours: maxHours ? maxHours.toString().padStart(2, "0") : "00",
                })}\n`;
            }
        }
        return rules;
    }
    addSampleRows(ws, sampleCount) {
        faker_1.faker.locale = this._locale;
        for (let count = 0; count < sampleCount; count += 1) {
            let sample = {};
            let lastDateRangeStartSample;
            for (let header in this._headers) {
                let sampleValue = "";
                if (this._headers[header].regex) {
                    if (this.isTextFieldForEmail(this._headers[header])) {
                        sampleValue = faker_1.faker.internet.exampleEmail(faker_1.faker.name.firstName(), faker_1.faker.name.lastName());
                    }
                    else if (this.isTextFieldForPlate(this._headers[header])) {
                        sampleValue = `${faker_1.faker.datatype.number({ min: 0, max: 81 }).toString().padStart(2, "0")}${faker_1.faker.random.alpha({ count: 3, casing: "upper" })}${faker_1.faker.datatype
                            .number({ min: 0, max: 999 })
                            .toString()
                            .padStart(3, "0")}`;
                    }
                    else {
                        sampleValue = new randexp_1.default(this._headers[header].regex).gen();
                    }
                }
                else if (!this._headers[header].extension) {
                    if (this._headers[header].key == "name") {
                        sampleValue = faker_1.faker.name.firstName() + " " + faker_1.faker.name.lastName();
                    }
                    else if (this._headers[header].key == "uniqueId") {
                        if (this._headers[header].type == dynamicFormCurrent_1.FormFieldType.Text) {
                            let opt = this._headers[header].option;
                            if (opt.onlyNumber) {
                                sampleValue = faker_1.faker.datatype
                                    .number({ min: opt.minLength ? Math.pow(10, opt.minLength - 1) : 0, max: opt.maxLength ? Math.pow(10, opt.maxLength) - 1 : 999999 })
                                    .toString();
                            }
                            else {
                                sampleValue = this._organizationName.split(" ")[0] + count.toString().padStart(3, "0");
                            }
                        }
                        else if (this._headers[header].type == dynamicFormCurrent_1.FormFieldType.Number) {
                            let opt = this._headers[header].option;
                            sampleValue = faker_1.faker.datatype.number({ min: opt.min ? opt.min : 0, max: opt.max ? opt.max : 999999 }).toString();
                        }
                        else {
                            app_logs_1.logger.error("Unexpected type for uniqueId field.");
                            continue;
                        }
                    }
                }
                else {
                    switch (this._headers[header].type) {
                        case dynamicFormCurrent_1.FormFieldType.Text:
                            let textOptions = this._headers[header].option;
                            if (textOptions.defaultValue && count == 0) {
                                sampleValue = textOptions.defaultValue;
                            }
                            else if (this.isTextFieldForEmail(this._headers[header])) {
                                sampleValue = faker_1.faker.internet.exampleEmail(faker_1.faker.name.firstName(), faker_1.faker.name.lastName());
                            }
                            else if (this.isTextFieldForPlate(this._headers[header])) {
                                sampleValue = `${faker_1.faker.datatype.number({ min: 0, max: 81 }).toString().padStart(2, "0")}${faker_1.faker.random.alpha({ count: 3, casing: "upper" })}${faker_1.faker.datatype
                                    .number({ min: 0, max: 999 })
                                    .toString()
                                    .padStart(3, "0")}`;
                            }
                            else {
                                let paragraph = faker_1.faker.lorem.paragraph();
                                sampleValue = paragraph.slice(0, textOptions.maxLength ? textOptions.maxLength : 55);
                            }
                            break;
                        case dynamicFormCurrent_1.FormFieldType.Number:
                            let numberOptions = this._headers[header].option;
                            if (numberOptions.defaultValue && count == 0) {
                                sampleValue = numberOptions.defaultValue.toString();
                            }
                            else {
                                let min = 0;
                                let max = 9999999;
                                let step = 1;
                                if (numberOptions.min)
                                    min = numberOptions.min;
                                if (numberOptions.max)
                                    max = numberOptions.max;
                                if (numberOptions.step)
                                    step = numberOptions.step;
                                sampleValue = ((0, lodash_1.random)(min, max, false) * step).toString();
                            }
                            break;
                        case dynamicFormCurrent_1.FormFieldType.Time:
                            let timeOptions = this._headers[header].option;
                            if ((timeOptions.defaultHours || timeOptions.defaultMinutes) && count == 0) {
                                let min = timeOptions.defaultMinutes ? timeOptions.defaultMinutes : (0, lodash_1.random)(0, 60, false);
                                let hour = timeOptions.defaultHours ? timeOptions.defaultHours : (0, lodash_1.random)(0, 24, false);
                                sampleValue = `${hour.toString().padStart(2, "0")}:${min.toString().padStart(2, "0")}`;
                            }
                            else {
                                let minMinutes = timeOptions.minMinutes ? timeOptions.minMinutes : 0;
                                let maxMinutes = timeOptions.maxMinutes ? timeOptions.maxMinutes : 60;
                                let minHours = timeOptions.minHours ? timeOptions.minHours : 0;
                                let maxHours = timeOptions.maxHours ? timeOptions.maxHours : 24;
                                sampleValue = `${(0, lodash_1.random)(minHours, maxHours, false).toString().padEnd(2, "0")}:${(0, lodash_1.random)(minMinutes, maxMinutes, false).toString().padStart(2, "0")}`;
                            }
                            break;
                        case dynamicFormCurrent_1.FormFieldType.Date:
                            let dateOptions = this._headers[header].option;
                            if (dateOptions.defaultValue && count == 0) {
                                sampleValue = this.dateFormat(new Date(dateOptions.defaultValue), false);
                            }
                            else {
                                let now = new Date(Date.now());
                                let minDate = dateOptions.min ? dateOptions.min : now;
                                let maxDate = dateOptions.max ? dateOptions.max : now;
                                if (!dateOptions.max) {
                                    maxDate.setDate(now.getDate() + ((0, lodash_1.random)(0, 10, false) % 28));
                                }
                                let randomDate = now;
                                randomDate.setDate((0, lodash_1.random)(minDate.getDate(), maxDate.getDate(), false));
                                sampleValue = this.dateFormat(randomDate, false);
                            }
                            break;
                        case dynamicFormCurrent_1.FormFieldType.DateTime:
                            let dateTimeOptions = this._headers[header].option;
                            if (dateTimeOptions.defaultValue && count == 0) {
                                sampleValue = this.dateFormat(new Date(dateTimeOptions.defaultValue), true);
                            }
                            else {
                                let now = new Date(Date.now());
                                let minDate = dateTimeOptions.min ? dateTimeOptions.min : now;
                                let maxDate = dateTimeOptions.max ? dateTimeOptions.max : now;
                                if (!dateTimeOptions.max) {
                                    maxDate.setDate(now.getDate() + ((0, lodash_1.random)(0, 10, false) % 28));
                                    maxDate.setHours((0, lodash_1.random)(0, 24, false), (0, lodash_1.random)(0, 60, false), (0, lodash_1.random)(0, 60, false));
                                }
                                let randomDate = now;
                                randomDate.setDate((0, lodash_1.random)(minDate.getDate(), maxDate.getDate(), false));
                                sampleValue = this.dateFormat(randomDate, true);
                            }
                            break;
                        case dynamicFormCurrent_1.FormFieldType.DateRange:
                            let dateRangeOptions = this._headers[header].option;
                            let dynamicDateProperties;
                            if (this._headers[header].key.endsWith("End")) {
                                dynamicDateProperties = {
                                    endDynamicDate: dateRangeOptions.endDynamicDate,
                                };
                            }
                            else {
                                dynamicDateProperties = {
                                    minStartDynamicDate: dateRangeOptions.minStartDynamicDate,
                                    maxStartDynamicDate: dateRangeOptions.maxStartDynamicDate,
                                    startDynamicDate: dateRangeOptions.startDynamicDate,
                                };
                            }
                            let dynamicDatePropertyValues;
                            if (this._headers[header].key.endsWith("End")) {
                                dynamicDatePropertyValues = {
                                    endDynamicDate: null,
                                };
                            }
                            else {
                                dynamicDatePropertyValues = {
                                    minStartDynamicDate: null,
                                    maxStartDynamicDate: null,
                                    startDynamicDate: null,
                                };
                            }
                            for (let i = 0; i < Object.keys(dynamicDateProperties).length; i += 1) {
                                let dynamicDateValue = dynamicDateProperties[Object.keys(dynamicDateProperties)[i]];
                                if (!dynamicDateValue)
                                    continue;
                                let dynamicDate = new Date(Date.now());
                                switch (dynamicDateValue) {
                                    case dynamicFormCurrent_1.DateDynamicValue.Now:
                                        break;
                                    case dynamicFormCurrent_1.DateDynamicValue.StartOfHour:
                                        dynamicDate.setMinutes(0, 0, 0);
                                        break;
                                    case dynamicFormCurrent_1.DateDynamicValue.StartOfDay:
                                        dynamicDate.setHours(0, 0, 0, 0);
                                        break;
                                    case dynamicFormCurrent_1.DateDynamicValue.EndOfHour:
                                        dynamicDate.setMinutes(59, 59, 999);
                                        break;
                                    case dynamicFormCurrent_1.DateDynamicValue.EndOfDayAfterSevenDays:
                                        dynamicDate = this.addDays(dynamicDate, 7);
                                    case dynamicFormCurrent_1.DateDynamicValue.EndOfDay:
                                        dynamicDate.setHours(23, 59, 59, 999);
                                        break;
                                    case dynamicFormCurrent_1.DateDynamicValue.EndOfDayAfterFiveDays:
                                        dynamicDate = this.addDays(dynamicDate, 5);
                                        dynamicDate.setHours(23, 59, 59, 999);
                                        break;
                                    default:
                                        continue;
                                }
                                dynamicDatePropertyValues[Object.keys(dynamicDateProperties)[i]] = dynamicDate;
                            }
                            let time = dateRangeOptions.showTimer;
                            if (this._headers[header].key.endsWith("End")) {
                                if (dateRangeOptions.endDynamicDate && count == 0) {
                                    sampleValue = this.dateFormat(dynamicDatePropertyValues.endDynamicDate, time);
                                }
                                else {
                                    let randomDate = faker_1.faker.date.between(lastDateRangeStartSample, this.addDays(lastDateRangeStartSample, (0, lodash_1.random)(1, (0, lodash_1.min)([dateRangeOptions.maxDayInterval, 28]))));
                                    sampleValue = this.dateFormat(randomDate, time);
                                }
                            }
                            else {
                                if (count == 0) {
                                    sampleValue = this.dateFormat(dynamicDatePropertyValues.startDynamicDate, time);
                                    lastDateRangeStartSample = new Date(sampleValue);
                                }
                                else {
                                    let randomDate = faker_1.faker.date.between(dynamicDatePropertyValues.minStartDynamicDate, dynamicDatePropertyValues.maxStartDynamicDate
                                        ? dynamicDatePropertyValues.maxStartDynamicDate
                                        : this.addDays(dynamicDatePropertyValues.minStartDynamicDate, 6));
                                    sampleValue = this.dateFormat(randomDate, time);
                                    lastDateRangeStartSample = new Date(sampleValue);
                                }
                            }
                            break;
                        case dynamicFormCurrent_1.FormFieldType.Select:
                            let selectOptions = this._headers[header].option;
                            if (selectOptions.defaultValue && count == 0) {
                                if (Array.isArray(selectOptions.defaultValue)) {
                                    sampleValue = selectOptions.defaultValue.map((x) => x.captionLines[0]).join(",");
                                }
                                else {
                                    sampleValue = selectOptions.defaultValue.captionLines[0];
                                }
                            }
                            else {
                                let options = selectOptions.options.map((x) => x.captionLines[0]);
                                if (selectOptions.multiSelect) {
                                    let count = (0, lodash_1.random)(0, options.length - 1, false);
                                    let selections = [];
                                    for (let i = 0; i < count; i += 1) {
                                        selections.push(options.splice((0, lodash_1.random)(0, options.length - 1, false), 1)[0]);
                                    }
                                    sampleValue = selections.join(",");
                                }
                                else {
                                    let index = (0, lodash_1.random)(0, options.length - 1, false);
                                    sampleValue += options[index];
                                }
                            }
                            break;
                        case dynamicFormCurrent_1.FormFieldType.Textarea:
                            let textAreaOptions = this._headers[header].option;
                            let paragraph = faker_1.faker.lorem.paragraph();
                            sampleValue = paragraph.slice(0, textAreaOptions.maxLength ? textAreaOptions.maxLength : 255);
                            break;
                        case dynamicFormCurrent_1.FormFieldType.Checkbox:
                            sampleValue = (0, lodash_1.random)(0, 2, false) == 1 ? "true" : "false";
                            break;
                        case dynamicFormCurrent_1.FormFieldType.Custom:
                        case dynamicFormCurrent_1.FormFieldType.ImageCapture:
                        case dynamicFormCurrent_1.FormFieldType.RadioGroup:
                        default:
                            break;
                    }
                }
                sample[this._headers[header].key] = sampleValue;
            }
            ws.addRow(sample);
            ws.lastRow.eachCell((cell) => {
                cell.style = this._styles.samples;
            });
        }
    }
    createMetaSheet(ws, state) {
        let metaKeysNHeaders = {};
        for (let header of this._headers) {
            metaKeysNHeaders[header.key] = header.header;
        }
        let metaSheet = new business_visitor_metadata_1.MetaSheet(ws.id, this._organizationId, this._preRegistration, this.dateFormat(this._wbWriter.created, true), metaKeysNHeaders);
        metaSheet.generateHash();
        metaSheet.write(ws);
        ws.state = state;
    }
    createWorkbook() {
        const options = {
            filename: this._filePath,
            useSharedStrings: true,
            useStyles: true,
        };
        this._wbWriter = new exceljs_1.default.stream.xlsx.WorkbookWriter(options);
        this._wbWriter.creator = this._requestedUserCaption;
        this._wbWriter.company = this._organizationName;
    }
    isTextFieldForEmail(header) {
        let commonField = false;
        commonField = commonField || header.header.toLocaleLowerCase().includes("mail");
        commonField = commonField || header.key.toLocaleLowerCase().includes("mail");
        return commonField;
    }
    isTextFieldForPlate(header) {
        let commonField = false;
        commonField = commonField || header.header.toLocaleLowerCase().includes("plate");
        commonField = commonField || header.header.toLocaleLowerCase().includes("plaka");
        commonField = commonField || header.key.toLocaleLowerCase().includes("plate");
        return commonField;
    }
    addDays(date, days) {
        let lastDayOfMonth = 28;
        switch (date.getMonth()) {
            case 0:
            case 2:
            case 4:
            case 6:
            case 7:
            case 9:
            case 11:
                lastDayOfMonth = 31;
                break;
            case 3:
            case 5:
            case 8:
            case 10:
                lastDayOfMonth = 30;
                break;
            case 1:
                lastDayOfMonth = date.getFullYear() % 4 == 0 ? 29 : 28;
                break;
            default:
                break;
        }
        if (date.getDate() + days <= lastDayOfMonth) {
            date.setDate(date.getDate() + days);
            return date;
        }
        else {
            if (date.getMonth() == 11) {
                date.setFullYear(date.getFullYear() + 1);
                date.setMonth(0);
                date.setDate(days + date.getDate() - lastDayOfMonth);
                return date;
            }
            date.setMonth(date.getMonth() + 1);
            date.setDate(days + date.getDate() - lastDayOfMonth);
            return date;
        }
    }
    dateFormat(date, includeTime) {
        return (`${date.getDate().toString().padStart(2, "0")}.${(date.getMonth() + 1).toString().padStart(2, "0")}.${date.getFullYear().toString()}` +
            (includeTime ? ` ${date.getHours().toString().padStart(2, "0")}:${date.getMinutes().toString().padStart(2, "0")}:${date.getSeconds().toString().padStart(2, "0")}` : ""));
    }
    addNoteToCell(ws, cell, note, multiline) {
        let lineHeight = 8 / 7;
        let lineWidth = ws.getColumn(cell.col).width * lineHeight;
        let lineCountDueToOverflowedText = 0;
        let rowHeightDepedingOnTextLength;
        let noteLength = note.length;
        cell.note = {
            texts: [
                {
                    text: note,
                },
            ],
            margins: {
                insetmode: "auto",
            },
        };
        if (multiline) {
            let minLineLength = 999;
            let newLineCount = note.split("-").length - 1;
            minLineLength = (0, lodash_1.min)(note.split("-").map((line) => {
                if (line.length < 1)
                    return 999;
                else
                    return line.length;
            }));
            lineCountDueToOverflowedText = (noteLength - minLineLength * newLineCount) / lineHeight;
            if (lineWidth < minLineLength) {
                lineCountDueToOverflowedText += (newLineCount * minLineLength) / lineWidth;
            }
            rowHeightDepedingOnTextLength = (lineCountDueToOverflowedText + newLineCount) * 8;
        }
        else {
            if (noteLength > lineWidth) {
                lineCountDueToOverflowedText = noteLength / lineWidth;
            }
            rowHeightDepedingOnTextLength = (lineCountDueToOverflowedText + 1) * lineHeight;
        }
        if (ws.getRow(parseInt(cell.row)).height < rowHeightDepedingOnTextLength) {
            ws.getRow(parseInt(cell.row)).height = rowHeightDepedingOnTextLength < 25 ? 25 : rowHeightDepedingOnTextLength;
        }
    }
    setColumnKeys(ws) {
        let columns = [];
        for (let element of this._headers) {
            columns.push({
                key: element.key,
                width: ws.properties.defaultColWidth,
            });
        }
        columns.push({
            key: "empty1",
            width: ws.properties.defaultColWidth,
        });
        columns.push({
            key: "empty2",
            width: ws.properties.defaultColWidth,
        });
        ws.columns = columns;
    }
    setStyles() {
        this._styles = {
            headers: {
                alignment: { horizontal: "center", vertical: "middle", wrapText: true },
                fill: {
                    type: "pattern",
                    pattern: "solid",
                    fgColor: { argb: "905FC9F3" },
                },
                border: {
                    top: { style: "medium", color: { argb: "FF000000" } },
                    left: { style: "medium", color: { argb: "FF000000" } },
                    bottom: { style: "thick", color: { argb: "FF000000" } },
                    right: { style: "medium", color: { argb: "FF000000" } },
                },
                font: { bold: true },
            },
            rules: {
                alignment: { horizontal: "left", vertical: "top", wrapText: true, shrinkToFit: true },
            },
            samples: {
                alignment: { horizontal: "left", vertical: "top", wrapText: true },
            },
            fail: {
                font: { color: { argb: "FFFFFFFF" }, bold: true },
                fill: {
                    type: "pattern",
                    pattern: "solid",
                    fgColor: { argb: "90E94D4D" },
                },
                alignment: { horizontal: "left", vertical: "top", wrapText: true },
            },
            success: {
                font: { color: { argb: "FFFFFFFF" } },
                fill: {
                    type: "pattern",
                    pattern: "solid",
                    fgColor: { argb: "9065D537" },
                },
                alignment: { horizontal: "left", vertical: "top", wrapText: true },
            },
        };
    }
    getHeaders() {
        this._headers = [];
        for (let formField of this._options.visitorProfileFormFields) {
            if (formField.invisible || (this._preRegistration && formField.preRegistrationFieldSettings && formField.preRegistrationFieldSettings.invisible))
                continue;
            if (formField.type == dynamicFormCurrent_1.FormFieldType.DateRange) {
                this._headers.push({
                    header: formField.options.startCaption,
                    key: formField.name,
                    type: formField.type,
                    regex: formField.regex,
                    extension: formField.extension,
                    option: formField.options,
                    unique: formField.unique,
                    required: formField.options.isStartRequired,
                });
                if (formField.options.isEndVisible) {
                    this._headers.push({
                        header: formField.options.endCaption,
                        key: formField.name + "End",
                        type: formField.type,
                        regex: formField.regex,
                        extension: formField.extension,
                        option: formField.options,
                        unique: formField.unique,
                        required: formField.options.isEndRequired,
                    });
                }
            }
            else {
                this._headers.push({
                    header: formField.caption,
                    key: formField.name,
                    type: formField.type,
                    regex: formField.regex,
                    extension: formField.extension,
                    option: formField.options,
                    unique: formField.unique,
                    required: formField.required,
                });
            }
        }
        if (this._preRegistration) {
            for (let formField of this._options.visitorProfileFormFields) {
                let preRegisterationField = formField.preRegistrationFieldSettings;
                let headerIndex = 0;
                if (!preRegisterationField || (this._preRegistration && formField.preRegistrationFieldSettings.invisible))
                    continue;
                if (formField.type == dynamicFormCurrent_1.FormFieldType.DateRange) {
                    headerIndex = this._headers.indexOf(this._headers.filter((header) => header.key == formField.name + "End")[0]);
                }
                else {
                    headerIndex = this._headers.indexOf(this._headers.filter((header) => header.key == formField.name)[0]);
                }
                if (preRegisterationField.caption) {
                    this._headers[headerIndex].header = preRegisterationField.caption;
                }
                if (preRegisterationField.type) {
                    this._headers[headerIndex].type = preRegisterationField.type;
                }
                if (preRegisterationField.regex) {
                    this._headers[headerIndex].regex = preRegisterationField.regex;
                }
                if (preRegisterationField.unique) {
                    this._headers[headerIndex].unique = preRegisterationField.unique;
                }
                if (preRegisterationField.options) {
                    switch (formField.type) {
                        case dynamicFormCurrent_1.FormFieldType.Text:
                            let textOptions = preRegisterationField.options;
                            if (textOptions.defaultValue) {
                                this._headers[headerIndex].option.defaultValue = textOptions.defaultValue;
                            }
                            if (textOptions.maxLength) {
                                this._headers[headerIndex].option.maxLength = textOptions.maxLength;
                            }
                            if (textOptions.minLength) {
                                this._headers[headerIndex].option.minLength = textOptions.minLength;
                            }
                            if (textOptions.onlyNumber) {
                                this._headers[headerIndex].option.onlyNumber = textOptions.onlyNumber;
                            }
                            break;
                        case dynamicFormCurrent_1.FormFieldType.Number:
                            let numberOptions = preRegisterationField.options;
                            if (numberOptions.defaultValue) {
                                this._headers[headerIndex].option.defaultValue = numberOptions.defaultValue;
                            }
                            if (numberOptions.min) {
                                this._headers[headerIndex].option.min = numberOptions.min;
                            }
                            if (numberOptions.max) {
                                this._headers[headerIndex].option.max = numberOptions.max;
                            }
                            if (numberOptions.step) {
                                this._headers[headerIndex].option.step = numberOptions.step;
                            }
                            break;
                        case dynamicFormCurrent_1.FormFieldType.Time:
                            let timeOptions = preRegisterationField.options;
                            if (timeOptions.defaultMinutes) {
                                this._headers[headerIndex].option.defaultMinutes = timeOptions.defaultMinutes;
                            }
                            if (timeOptions.defaultHours) {
                                this._headers[headerIndex].option.defaultHours = timeOptions.defaultHours;
                            }
                            if (timeOptions.maxMinutes) {
                                this._headers[headerIndex].option.maxMinutes = timeOptions.maxMinutes;
                            }
                            if (timeOptions.maxHours) {
                                this._headers[headerIndex].option.maxHours = timeOptions.maxHours;
                            }
                            if (timeOptions.minMinutes) {
                                this._headers[headerIndex].option.minMinutes = timeOptions.minMinutes;
                            }
                            if (timeOptions.minHours) {
                                this._headers[headerIndex].option.minHours = timeOptions.minHours;
                            }
                            break;
                        case dynamicFormCurrent_1.FormFieldType.Date:
                            let dateOptions = preRegisterationField.options;
                            if (dateOptions.defaultValue) {
                                this._headers[headerIndex].option.defaultValue = dateOptions.defaultValue;
                            }
                            if (dateOptions.min) {
                                this._headers[headerIndex].option.min = dateOptions.min;
                            }
                            if (dateOptions.max) {
                                this._headers[headerIndex].option.max = dateOptions.max;
                            }
                            break;
                        case dynamicFormCurrent_1.FormFieldType.DateTime:
                            let dateTimeOptions = preRegisterationField.options;
                            if (dateTimeOptions.defaultValue) {
                                this._headers[headerIndex].option.defaultValue = dateTimeOptions.defaultValue;
                            }
                            if (dateTimeOptions.min) {
                                this._headers[headerIndex].option.min = dateTimeOptions.min;
                            }
                            if (dateTimeOptions.max) {
                                this._headers[headerIndex].option.max = dateTimeOptions.max;
                            }
                            break;
                        case dynamicFormCurrent_1.FormFieldType.DateRange:
                            let dateRangeOptions = preRegisterationField.options;
                            if (dateRangeOptions.startCaption) {
                                this._headers[headerIndex].option.startCaption = dateRangeOptions.startCaption;
                            }
                            if (dateRangeOptions.endCaption) {
                                this._headers[headerIndex].option.endCaption = dateRangeOptions.endCaption;
                            }
                            if (dateRangeOptions.minStartDynamicDate) {
                                this._headers[headerIndex].option.minStartDynamicDate = dateRangeOptions.minStartDynamicDate;
                            }
                            if (dateRangeOptions.maxStartDynamicDate) {
                                this._headers[headerIndex].option.maxStartDynamicDate = dateRangeOptions.maxStartDynamicDate;
                            }
                            if (dateRangeOptions.startDynamicDate) {
                                this._headers[headerIndex].option.startDynamicDate = dateRangeOptions.startDynamicDate;
                            }
                            if (dateRangeOptions.isStartRequired) {
                                this._headers[headerIndex].option.isStartRequired = dateRangeOptions.isStartRequired;
                            }
                            if (dateRangeOptions.showTimer) {
                                this._headers[headerIndex].option.showTimer = dateRangeOptions.showTimer;
                            }
                            if (dateRangeOptions.isEndRequired) {
                                this._headers[headerIndex].option.isEndRequired = dateRangeOptions.isEndRequired;
                            }
                            if (dateRangeOptions.isEndVisible) {
                                this._headers[headerIndex].option.isEndVisible = dateRangeOptions.isEndVisible;
                            }
                            if (dateRangeOptions.endDynamicDate) {
                                this._headers[headerIndex].option.endDynamicDate = dateRangeOptions.endDynamicDate;
                            }
                            if (dateRangeOptions.maxDayInterval) {
                                this._headers[headerIndex].option.maxDayInterval = dateRangeOptions.maxDayInterval;
                            }
                            break;
                        case dynamicFormCurrent_1.FormFieldType.Select:
                            let selectOptions = preRegisterationField.options;
                            if (selectOptions.options) {
                                this._headers[headerIndex].option.options = selectOptions.options;
                            }
                            if (selectOptions.multiSelect) {
                                this._headers[headerIndex].option.multiSelect = selectOptions.multiSelect;
                            }
                            if (selectOptions.defaultValue) {
                                this._headers[headerIndex].option.defaultValue = selectOptions.defaultValue;
                            }
                            break;
                        case dynamicFormCurrent_1.FormFieldType.Textarea:
                            let textAreaOptions = preRegisterationField.options;
                            if (textAreaOptions.maxLength) {
                                this._headers[headerIndex].option.maxLength = textAreaOptions.maxLength;
                            }
                            if (textAreaOptions.minLength) {
                                this._headers[headerIndex].option.minLength = textAreaOptions.minLength;
                            }
                            if (textAreaOptions.defaultValue) {
                                this._headers[headerIndex].option.defaultValue = textAreaOptions.defaultValue;
                            }
                            break;
                        case dynamicFormCurrent_1.FormFieldType.Checkbox:
                            let checkboxOptions = preRegisterationField.options;
                            if (checkboxOptions.defaultValue) {
                                this._headers[headerIndex].option.defaultValue = checkboxOptions.defaultValue;
                            }
                            break;
                        case dynamicFormCurrent_1.FormFieldType.Custom:
                        case dynamicFormCurrent_1.FormFieldType.ImageCapture:
                        case dynamicFormCurrent_1.FormFieldType.RadioGroup:
                        default:
                            break;
                    }
                }
            }
        }
    }
    convertDateFormat(date) {
        return `${date.split(" ")[0].split(".").reverse().join("-")}${date.includes(" ") ? "T" + date.split(" ")[1] + "Z" : ""}`;
    }
    getVisitorList() {
        let visitorList = [];
        let sheet = this._wbReader.getWorksheet(1);
        sheet.eachRow((row, rowNumber) => {
            if (rowNumber != 1) {
                let visitor = [];
                row.eachCell((cell, colNumber) => {
                    visitor.push({
                        name: sheet.getColumn(colNumber).key,
                        value: cell.value,
                    });
                });
                visitorList.push(visitor);
            }
        });
        return visitorList;
    }
    indicateFailedRows(failedRows) {
        for (let row of failedRows) {
            for (let col = 1; col <= this._headers.length; col += 1) {
                let cell = this._wbReader.getWorksheet(1).getRow(row.row).getCell(col);
                cell.style = this._styles.fail;
                if (col == 1) {
                    this.addNoteToCell(cell.worksheet, cell, row.error, false);
                }
            }
        }
    }
    async downloadResultExcel(res) {
        await this._wbReader.xlsx.writeFile(this._filePath, {
            stream: res,
        });
        let localeTranslationResult = i18n_1.default.__({ phrase: `VISITOR-EXCEL.${this._preRegistration ? "PREREGISTRATION" : "REGISTRATION"}`, locale: this._locale }) +
            "_" +
            i18n_1.default.__({ phrase: "VISITOR-EXCEL.RESULT", locale: this._locale });
        return {
            filePath: this._filePath,
            filename: this._organizationName
                .toLocaleLowerCase()
                .replace(new RegExp(/[&\/\\#,+()$~%.'":*?<>{}`\r\n\t]/g), "")
                .replace(new RegExp(/[\s]+/g), "_")
                .replace(new RegExp(/ç/g), "c")
                .replace(new RegExp(/ğ/g), "g")
                .replace(new RegExp(/ı/g), "i")
                .replace(new RegExp(/ö/g), "o")
                .replace(new RegExp(/ş/g), "s")
                .replace(new RegExp(/ü/g), "u") +
                "_" +
                i18n_1.default.__({ phrase: "VISITOR-EXCEL.FILENAME", locale: this._locale }, { formType: localeTranslationResult }) +
                ".xlsx",
        };
    }
}
exports.VisitorsUploadExcel = VisitorsUploadExcel;
