"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.addJobEndLog = exports.addJobStartLog = exports.onScheduledJob = void 0;
const app_enums_1 = require("../../app.enums");
const app_logs_1 = require("../../app.logs");
const dal_manager_1 = require("../../dal/dal.manager");
const dal_utils_1 = require("../../dal/dal.utils");
const messageBroker_notification_pub_1 = require("../../messageBroker/messageBroker.notification.pub");
const messageBroker_server_to_report_pub_1 = require("../../messageBroker/messageBroker.server-to-report.pub");
const messageBroker_server_to_server_sub_1 = require("../../messageBroker/messageBroker.server-to-server.sub");
const business_hooks_1 = require("../business.hooks");
const business_scheduledjobs_1 = require("./business.scheduledjobs");
async function onScheduledJob(msg) {
    const notificationEvents = await dal_manager_1.dbManager.pgTransactionMainDb(async (trx) => {
        try {
            const scheduledMessage = JSON.parse(msg.content.toString());
            scheduledMessage.f = new Date(scheduledMessage.f);
            const scheduledJob = await dal_manager_1.dbManager.accessSystem.getScheduledJob({ id: scheduledMessage.i, trx });
            if (!scheduledJob) {
                app_logs_1.logger.info("Scheduled Job Recieved From RabbitMQ but corresponding entry not found in db, discarding!");
                messageBroker_server_to_server_sub_1.amqpServerToServerSub.ack(msg);
                return;
            }
            const { id, organizationId, type } = scheduledJob;
            if (scheduledJob.firstExecutionDate.getTime() !== scheduledMessage.f.getTime() || scheduledJob.interval !== scheduledMessage.a || !scheduledJob.enabled) {
                messageBroker_server_to_server_sub_1.amqpServerToServerSub.ack(msg);
                return;
            }
            if (scheduledJob.interval != 10) {
                const nextExecutionDate = (0, dal_utils_1.calculateNextExecutionDateOfScheduledJob)(scheduledJob.interval, scheduledJob.firstExecutionDate);
                await dal_manager_1.dbManager.accessSystem.setNextExecutionDateOfScheduledJob({ trx, id: scheduledJob.id, nextExecutionDate });
            }
            const startLogId = await addJobStartLog(id, "Started", new Date(), undefined, trx);
            let notificationEvents = [];
            try {
                switch (type) {
                    case 101:
                        await (0, business_scheduledjobs_1.expiredSessionCleaner)(trx);
                        break;
                    case 103:
                        await (0, business_scheduledjobs_1.visitorGdprAnonymization)(trx);
                        break;
                    case 104:
                        await (0, business_scheduledjobs_1.visitTerminator)(organizationId, trx);
                        break;
                    case 105:
                        await (0, business_scheduledjobs_1.dailyExpiredReportRemover)(trx);
                        break;
                    case 102:
                        app_logs_1.logger.warn("Licence Manager Not IMPLEMENTED!!11");
                        break;
                    case 106:
                        await (0, business_scheduledjobs_1.employmentEndDisabler)(trx);
                        break;
                    case 8:
                        notificationEvents = await (0, business_scheduledjobs_1.ppermissionSignatureReminder)(organizationId, trx);
                        break;
                    case 32:
                        notificationEvents = await (0, business_scheduledjobs_1.pacsSelfMissedWorkNotifier)(organizationId, scheduledJob, trx);
                        break;
                    case 31:
                        notificationEvents = await (0, business_scheduledjobs_1.pacsSelfAbsentEmployeeNotifier)(organizationId, scheduledJob, trx);
                        break;
                    case 49:
                        notificationEvents = await (0, business_scheduledjobs_1.pacsSelfOverworkedEmployeeNotifier)(organizationId, scheduledJob, trx);
                        break;
                    case 19:
                        messageBroker_server_to_report_pub_1.amqpServerToReportPub.sendScheduledJobRequest({
                            scheduledJobId: scheduledJob.id,
                            organizationId,
                            notificationId: scheduledJob.notificationId,
                        });
                        break;
                    case 100:
                        const hook = business_hooks_1.armonHookManager.getScheduledJobRoutineHook(organizationId);
                        if (hook) {
                            await hook(organizationId, scheduledJob.id);
                        }
                        break;
                    case 107:
                        await (0, business_scheduledjobs_1.pacsMissingDayEntryFinder)(trx);
                        break;
                    case 109:
                        await (0, business_scheduledjobs_1.webRtcTwilioIceServerRefresher)(trx);
                        break;
                    case 110:
                        await (0, business_scheduledjobs_1.arventoRegionDataSync)(organizationId, trx);
                        break;
                    case 111:
                        await (0, business_scheduledjobs_1.arventoAccessLogsSync)(organizationId, trx);
                        break;
                    case 33:
                        notificationEvents = await (0, business_scheduledjobs_1.registerVisitorsPreregisteredViaExcel)(scheduledJob, trx);
                        break;
                    case 34:
                        await (0, business_scheduledjobs_1.terminateVisitsByActivatedViaExcel)(scheduledJob, trx);
                        break;
                    default:
                        app_logs_1.logger.error("Undefined scheduled job type");
                        break;
                }
                await addJobEndLog(startLogId, id, "Finished", true, trx);
                if (scheduledJob.interval == 10) {
                    await dal_manager_1.dbManager.accessSystem.deleteScheduledJob({ id: scheduledJob.id, trx: trx });
                }
                messageBroker_server_to_server_sub_1.amqpServerToServerSub.ack(msg);
                return notificationEvents;
            }
            catch (error) {
                messageBroker_server_to_server_sub_1.amqpServerToServerSub.nack(msg);
                await addJobEndLog(startLogId, id, error.message ? error.toString() : error, false, trx);
                app_logs_1.logger.error(`Scheduled job[id: ${id}] [type:${type}] crashed error: ${error}`);
            }
        }
        catch (err) {
            app_logs_1.logger.error(`Error in scheduled Job : ${err.message ? err.toString() : err}`);
        }
    });
    if (notificationEvents?.length > 0) {
        for (const event of notificationEvents) {
            (0, messageBroker_notification_pub_1.publishToNotificationService)(event);
        }
    }
}
exports.onScheduledJob = onScheduledJob;
async function addJobStartLog(jobId, name, date, message, trx) {
    let result = await dal_manager_1.dbManager.accessSystem.getExistingScheduledJobLog(jobId, date, trx);
    if (result) {
        return result;
    }
    result = await dal_manager_1.dbManager.accessSystem.upsertScheduledJobHistory({
        name: name,
        jobId: jobId,
        message: message,
        startDate: new Date(),
        executionDate: date,
        success: false,
        tryCount: 0,
    }, trx);
    return result;
}
exports.addJobStartLog = addJobStartLog;
async function addJobEndLog(startLogId, id, message, success, trx) {
    await dal_manager_1.dbManager.accessSystem.updateScheduledJobHistoryFinalStatus(startLogId, new Date(), success, message, trx);
}
exports.addJobEndLog = addJobEndLog;
