"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.convertSecondsToHoursMinutes = exports.hideSensitiveFieldsOfVisitorForReport = exports.hideSensitiveFieldsOfVisitsForReport = exports.getAccessControlPointTypeName = exports.throwDbReportError = exports.getLocalizedTextOfAccessLogReason = exports.parseCaptionV2 = exports.parseCaption = exports.parseCaptionForPdfFilter = exports.parseCaptionForExcel = exports.formatColumn = exports.styleExcelWorksheet = exports.getAccessLogStatusInfo = exports.getDescriptionAccessLogItem = exports.getCredentialDataVisibility = exports.join = exports.rstrip = exports.formatTime = exports.numberToDurationFormatV2 = exports.numberToDurationFormat = exports.formatDateForExcel = exports.formatDateToString = exports.checkHex = exports.hexToDec = exports.getDataText = exports.reverseCredentialData = exports.ReportConstants = void 0;
const enums_1 = require("../../lib/access-model/v2/enums");
const i18n_1 = __importDefault(require("i18n"));
const moment_1 = __importDefault(require("moment"));
const app_enums_1 = require("../../app.enums");
const dal_access_error_1 = require("../../dal/access/dal.access.error");
const dal_access_models_1 = require("../../dal/access/dal.access.models");
const dal_constants_1 = require("../../dal/dal.constants");
var ReportConstants;
(function (ReportConstants) {
    ReportConstants.emptyAvatar = "data:image/jpeg;base64,/9j/4AAQSkZJRgABAQAAAQABAAD/2wBDAAMCAgICAgMCAgIDAwMDBAYEBAQEBAgGBgUGCQgKCgkICQkKDA8MCgsOCwkJDRENDg8QEBEQCgwSExIQEw8QEBD/2wBDAQMDAwQDBAgEBAgQCwkLEBAQEBAQEBAQEBAQEBAQEBAQEBAQEBAQEBAQEBAQEBAQEBAQEBAQEBAQEBAQEBAQEBD/wAARCAAwADADASIAAhEBAxEB/8QAHwAAAQUBAQEBAQEAAAAAAAAAAAECAwQFBgcICQoL/8QAtRAAAgEDAwIEAwUFBAQAAAF9AQIDAAQRBRIhMUEGE1FhByJxFDKBkaEII0KxwRVS0fAkM2JyggkKFhcYGRolJicoKSo0NTY3ODk6Q0RFRkdISUpTVFVWV1hZWmNkZWZnaGlqc3R1dnd4eXqDhIWGh4iJipKTlJWWl5iZmqKjpKWmp6ipqrKztLW2t7i5usLDxMXGx8jJytLT1NXW19jZ2uHi4+Tl5ufo6erx8vP09fb3+Pn6/8QAHwEAAwEBAQEBAQEBAQAAAAAAAAECAwQFBgcICQoL/8QAtREAAgECBAQDBAcFBAQAAQJ3AAECAxEEBSExBhJBUQdhcRMiMoEIFEKRobHBCSMzUvAVYnLRChYkNOEl8RcYGRomJygpKjU2Nzg5OkNERUZHSElKU1RVVldYWVpjZGVmZ2hpanN0dXZ3eHl6goOEhYaHiImKkpOUlZaXmJmaoqOkpaanqKmqsrO0tba3uLm6wsPExcbHyMnK0tPU1dbX2Nna4uPk5ebn6Onq8vP09fb3+Pn6/9oADAMBAAIRAxEAPwD9PaKKiu3aO1ldDhlQkH04oAq3WsQW7mNFMrDg4OAPxotdXguHEboY2bgZOQT9awaKAOtoqK0dpLWJ3OWZASfXipaAChlDKVYZBGCKKzrzWI4SY7dRI46k/dH+NAFK60i4icmBTJGemOo+tFrpFxK4M6mOMdc9T9Kik1S+kJPnlfZQBQmqX0ZB88t7MAaAOjACgKBgDgUVnWesRzER3CiNz0I+6f8ACtGgCG9JFnMQcHYf5VzFdVNGJoniJwHUrmsz+wP+nv8A8h//AF6AMiitf+wP+nv/AMh//Xo/sD/p7/8AIf8A9egDIrp7Ik2cJJydg/lWf/YH/T3/AOQ//r1pwxiGJIgchFC5oA//2Q==";
    ReportConstants.widthProfilePicture = 36;
    ReportConstants.heightProfilePicture = 36;
    ReportConstants.widthOrganizationLogo = 150;
    ReportConstants.heightOrganizationLogo = 72;
    ReportConstants.evenRowColor = "#e6e3e3";
    ReportConstants.alphabet = ["A", "B", "C", "D", "E", "F", "G", "H", "I", "J", "K", "L", "M", "N", "O", "P", "Q", "R", "S", "T", "U", "V", "W", "X", "Y", "Z"];
    let PacsPPermissionStatusFilter;
    (function (PacsPPermissionStatusFilter) {
        PacsPPermissionStatusFilter[PacsPPermissionStatusFilter["WaitingMyApproval"] = 1] = "WaitingMyApproval";
        PacsPPermissionStatusFilter[PacsPPermissionStatusFilter["Waiting"] = 2] = "Waiting";
        PacsPPermissionStatusFilter[PacsPPermissionStatusFilter["RejectedByMe"] = 3] = "RejectedByMe";
        PacsPPermissionStatusFilter[PacsPPermissionStatusFilter["Rejected"] = 4] = "Rejected";
        PacsPPermissionStatusFilter[PacsPPermissionStatusFilter["ApprovedByMe"] = 5] = "ApprovedByMe";
        PacsPPermissionStatusFilter[PacsPPermissionStatusFilter["Approved"] = 6] = "Approved";
    })(PacsPPermissionStatusFilter = ReportConstants.PacsPPermissionStatusFilter || (ReportConstants.PacsPPermissionStatusFilter = {}));
    ReportConstants.PacsUserWorkingStatusDaysList = [
        {
            type: app_enums_1.enums.WorkStatusFilter.Came,
            description: "EXCEL-REPORT.PERSONNEL_CAME_DAYS",
        },
        {
            type: app_enums_1.enums.WorkStatusFilter.NotCame,
            description: "EXCEL-REPORT.PERSONNEL_NOT_CAME_DAYS",
        },
        {
            type: app_enums_1.enums.WorkStatusFilter.LateStart,
            description: "EXCEL-REPORT.PERSONNEL_CAME_LATE_DAYS",
        },
        {
            type: app_enums_1.enums.WorkStatusFilter.EarlyLeave,
            description: "EXCEL-REPORT.PERSONNEL_EARLY_OUT_DAYS",
        },
        {
            type: app_enums_1.enums.WorkStatusFilter.EarlyStart,
            description: "EXCEL-REPORT.PERSONNEL_EARLY_CAME_DAYS",
        },
        {
            type: app_enums_1.enums.WorkStatusFilter.LateLeave,
            description: "EXCEL-REPORT.PERSONNEL_LATE_OUT_DAYS",
        },
    ];
    ReportConstants.PacsPPermissionStatusFilterOptionList = [
        {
            value: app_enums_1.enums.PPermissionFilterStatus.WaitingMyApproval,
            i18nName: "EXCEL-REPORT.PPERMISSION_STATUS_WAITING_ME_OPTION",
        },
        {
            value: app_enums_1.enums.PPermissionFilterStatus.Waiting,
            i18nName: "EXCEL-REPORT.PPERMISSION_STATUS_WAITING_OPTION",
        },
        {
            value: app_enums_1.enums.PPermissionFilterStatus.RejectedByMe,
            i18nName: "EXCEL-REPORT.PPERMISSION_STATUS_REJECTED_ME_OPTION",
        },
        {
            value: app_enums_1.enums.PPermissionFilterStatus.Rejected,
            i18nName: "EXCEL-REPORT.PPERMISSION_STATUS_REJECTED_OPTION",
        },
        {
            value: app_enums_1.enums.PPermissionFilterStatus.ApprovedByMe,
            i18nName: "EXCEL-REPORT.PPERMISSION_STATUS_ACCEPTED_ME_OPTION",
        },
        {
            value: app_enums_1.enums.PPermissionFilterStatus.Approved,
            i18nName: "EXCEL-REPORT.PPERMISSION_STATUS_ACCEPTED_OPTION",
        },
    ];
    ReportConstants.SensorTypes = [
        {
            type: app_enums_1.enums.DryContactTypeV2.StatusSensor,
            description: "PDF-REPORT.CONNECTIONS_TYPE_STATUS_SENSOR",
        },
        {
            type: app_enums_1.enums.DryContactTypeV2.CounterSensor,
            description: "PDF-REPORT.CONNECTIONS_TYPE_COUNTER_SENSOR",
        },
        {
            type: app_enums_1.enums.DryContactTypeV2.TamperSwitch,
            description: "PDF-REPORT.CONNECTIONS_TYPE_TAMPER_SENSOR",
        },
    ];
    ReportConstants.StatusSensorReasons = [
        {
            type: app_enums_1.enums.StatusSensorReason.WiegandReaderTriggeredRelayDrive,
            description: "PDF-REPORT.SENSOR_READER_ACCESS",
        },
        {
            type: app_enums_1.enums.StatusSensorReason.ExitButtonTriggeredRelayDrive,
            description: "PDF-REPORT.SENSOR_EXIT_BUTTON",
        },
        {
            type: app_enums_1.enums.StatusSensorReason.RemoteAccessTriggeredRelayDrive,
            description: "PDF-REPORT.SENSOR_REMOTELY_OPEN",
        },
        {
            type: app_enums_1.enums.StatusSensorReason.UnknownReasonOpenAlthoughRelayIsMapped,
            description: "PDF-REPORT.SENSOR_UNKNOWN_OPEN_RELAY_MAPPED_DESC",
        },
        {
            type: app_enums_1.enums.StatusSensorReason.UnknownReasonOpenRelayIsNotMapped,
            description: "PDF-REPORT.SENSOR_UNKNOWN_OPEN_RELAY_NOT_MAPPED_DESC",
        },
        {
            type: app_enums_1.enums.StatusSensorReason.LeavedOpenned,
            description: "PDF-REPORT.SENSOR_LEAVED_OPENNED",
        },
        {
            type: app_enums_1.enums.StatusSensorReason.Closed,
            description: "PDF-REPORT.SENSOR_CLOSED",
        },
        {
            type: app_enums_1.enums.StatusSensorReason.ClosedAfterLeavedOpennedTimeout,
            description: "PDF-REPORT.SENSOR_CLOSED_TIMEOUT",
        },
    ];
    ReportConstants.CounterSensorReasons = [
        {
            type: app_enums_1.enums.CounterSensorReason.WiegandReaderTriggeredRelayDrive,
            description: "PDF-REPORT.SENSOR_READER_ACCESS",
        },
        {
            type: app_enums_1.enums.CounterSensorReason.ExitButtonTriggeredRelayDrive,
            description: "PDF-REPORT.SENSOR_EXIT_BUTTON",
        },
        {
            type: app_enums_1.enums.CounterSensorReason.RemoteAccessTriggeredRelayDrive,
            description: "PDF-REPORT.SENSOR_REMOTELY_OPEN",
        },
        {
            type: app_enums_1.enums.CounterSensorReason.UnknownReasonCounterAlthoughRelayIsMapped,
            description: "PDF-REPORT.SENSOR_UNKNOWN_OPEN_RELAY_MAPPED_DESC",
        },
        {
            type: app_enums_1.enums.CounterSensorReason.UnknownReasonCounterRelayIsNotMapped,
            description: "PDF-REPORT.SENSOR_UNKNOWN_OPEN_RELAY_NOT_MAPPED_DESC",
        },
    ];
    ReportConstants.StatusSensorStates = [
        {
            type: app_enums_1.enums.StatusSensorState.Closed,
            description: "PDF-REPORT.SENSOR_STATUS_STATE_CLOSED",
        },
        {
            type: app_enums_1.enums.StatusSensorState.Open,
            description: "PDF-REPORT.SENSOR_STATUS_STATE_OPENED",
        },
        {
            type: app_enums_1.enums.StatusSensorState.TimeOut,
            description: "PDF-REPORT.SENSOR_STATUS_STATE_TIMEOUT",
        },
    ];
    ReportConstants.CounterSensorStates = [
        {
            type: app_enums_1.enums.CounterSensorState.ReaderAccessOpenned,
            description: "PDF-REPORT.SENSOR_COUNTER_STATE_READER_ACCESS_OPENED",
        },
        {
            type: app_enums_1.enums.CounterSensorState.ExitButtonOpenned,
            description: "PDF-REPORT.SENSOR_COUNTER_STATE_EXIT_BUTTON_OPENNED",
        },
        {
            type: app_enums_1.enums.CounterSensorState.UnknownOpenned,
            description: "PDF-REPORT.SENSOR_COUNTER_STATE_UNKNOWN_OPENNED",
        },
    ];
    ReportConstants.TamperSwitchStates = [
        {
            type: app_enums_1.enums.TamperSwitchState.Secure,
            description: "PDF-REPORT.SENSOR_TAMPER_SWITCH_SECURE",
        },
        {
            type: app_enums_1.enums.TamperSwitchState.Tampered,
            description: "PDF-REPORT.SENSOR_TAMPER_SWITCH_TAMPERED",
        },
    ];
    ReportConstants.AccessLogReason = [
        {
            type: app_enums_1.enums.AccessLogReason.Success,
            description: "ACCESS_LOG_REASON.SUCCESS",
        },
        {
            type: app_enums_1.enums.AccessLogReason.HasNotRight,
            description: "ACCESS_LOG_REASON.HASNOTRIGHT",
        },
        {
            type: app_enums_1.enums.AccessLogReason.UnknownCredentialOwner,
            description: "ACCESS_LOG_REASON.UNKNOWN_USER",
        },
        {
            type: app_enums_1.enums.AccessLogReason.UnknownAccessPoint,
            description: "ACCESS_LOG_REASON.UNKNOWN_ACCESS_POINT",
        },
        {
            type: app_enums_1.enums.AccessLogReason.UnhandledFailure,
            description: "ACCESS_LOG_REASON.UNHANDLED_FAILURE",
        },
        {
            type: app_enums_1.enums.AccessLogReason.AntiPassbackReject,
            description: "ACCESS_LOG_REASON.ANTIPASSBACK_REJECT",
        },
        {
            type: app_enums_1.enums.AccessLogReason.AntiPassbackTimeOutReject,
            description: "ACCESS_LOG_REASON.ANTIPASSBACK_TIMEOUT_REJECT",
        },
        {
            type: app_enums_1.enums.AccessLogReason.CredentialExpired,
            description: "ACCESS_LOG_REASON.CREDENTIAL_EXPIRED",
        },
        {
            type: app_enums_1.enums.AccessLogReason.InsufficientFund,
            description: "ACCESS_LOG_REASON.INSUFFICIENT_FUND",
        },
        {
            type: app_enums_1.enums.AccessLogReason.RuleReject,
            description: "ACCESS_LOG_REASON.RULE_REJECT",
        },
        {
            type: app_enums_1.enums.AccessLogReason.Manual,
            description: "ACCESS_LOG_REASON.MANUAL",
        },
        {
            type: app_enums_1.enums.AccessLogReason.StateOpened,
            description: "ACCESS_LOG_REASON.STATE_OPENNED",
        },
        {
            type: app_enums_1.enums.AccessLogReason.StateDisabled,
            description: "ACCESS_LOG_REASON.STATE_DISABLED",
        },
        {
            type: app_enums_1.enums.AccessLogReason.IncompleteMultiFactor,
            description: "ACCESS_LOG_REASON.INCOMPLETE_MULTI_FACTOR",
        },
        {
            type: app_enums_1.enums.AccessLogReason.UserDisabled,
            description: "ACCESS_LOG_REASON.USER_DISABLED",
        },
        {
            type: app_enums_1.enums.AccessLogReason.Forbidden,
            description: "ACCESS_LOG_REASON.FORBIDDEN",
        },
        {
            type: enums_1.EnumsV2.AccessLogReason.RelayNotAvailable,
            description: "ACCESS_LOG_REASON.RELAY_NOT_AVAILABLE",
        },
        {
            type: app_enums_1.enums.AccessLogReason.ChallengeFail,
            description: "ACCESS_LOG_REASON.CHALLENGE_FAIL",
        },
        {
            type: app_enums_1.enums.AccessLogReason.InvalidChallenge,
            description: "ACCESS_LOG_REASON.INVALID_CHALLENGE",
        },
        {
            type: app_enums_1.enums.AccessLogReason.MifareFingerPrintNotMatch,
            description: "ACCESS_LOG_REASON.MIFARE_CARD_FINGERPRINT_NOT_MATCH",
        },
        {
            type: app_enums_1.enums.AccessLogReason.MifareFingerPrintTimeout,
            description: "ACCESS_LOG_REASON.MIFARE_CARD_FINGERPRINT_TIMEOUT",
        },
        {
            type: app_enums_1.enums.AccessLogReason.RegionCapacityFull,
            description: "ACCESS_LOG_REASON.AREA_CAPACITY_FULL",
        },
        {
            type: app_enums_1.enums.AccessLogReason.CapacityBasedRuleReject,
            description: "ACCESS_LOG_REASON.CAPACITY_BASED_RULE_REJECT",
        },
        {
            type: app_enums_1.enums.AccessLogReason.HesCodeNoRecord,
            description: "ACCESS_LOG_REASON.HES_CODE_NO_RECORD",
        },
        {
            type: app_enums_1.enums.AccessLogReason.HesCodeRiskly,
            description: "ACCESS_LOG_REASON.HES_CODE_RISKY",
        },
        {
            type: app_enums_1.enums.AccessLogReason.HesCodeError,
            description: "ACCESS_LOG_REASON.HES_CODE_ERROR",
        },
    ];
    ReportConstants.Direction = [
        {
            type: app_enums_1.enums.AccessDirection.All,
            description: "PDF-REPORT.DIRECTION_ENTRANCE_ALL",
        },
        {
            type: app_enums_1.enums.AccessDirection.Entrance,
            description: "PDF-REPORT.DIRECTION_ENTRANCE",
        },
        {
            type: app_enums_1.enums.AccessDirection.Exit,
            description: "PDF-REPORT.DIRECTION_EXIT",
        },
    ];
    ReportConstants.AccessResultTypes = [
        {
            type: 0,
            description: "PDF-REPORT.ACCESS_LOGS_ALL",
        },
        {
            type: 1,
            description: "PDF-REPORT.ACCESS_LOGS_SUCCESS",
        },
        {
            type: 2,
            description: "PDF-REPORT.ACCESS_LOGS_FAILED",
        },
    ];
    ReportConstants.CredentialTypes = [
        {
            type: app_enums_1.enums.CredentialType.ProximityCard,
            description: "CREDENTIALTYPE_PROXIMITYCARD",
        },
        {
            type: app_enums_1.enums.CredentialType.BLE,
            description: "CREDENTIALTYPE_BLE",
        },
        {
            type: app_enums_1.enums.CredentialType.MiFare,
            description: "CREDENTIALTYPE_MIFARE",
        },
        {
            type: app_enums_1.enums.CredentialType.AccessToken,
            description: "CREDENTIALTYPE_ACCESS_TOKEN",
        },
        {
            type: app_enums_1.enums.CredentialType.UHFRfid,
            description: "CREDENTIALTYPE_UHFRFID",
        },
        {
            type: app_enums_1.enums.CredentialType.VehiclePlate,
            description: "CREDENTIALTYPE_VEHICLE_PLATE",
        },
        {
            type: app_enums_1.enums.CredentialType.NFC,
            description: "CREDENTIALTYPE_NFC",
        },
        {
            type: app_enums_1.enums.CredentialType.FingerPrintISO19794,
            description: "CREDENTIALTYPE_FINGER_PRINT",
        },
        {
            type: app_enums_1.enums.CredentialType.QrCode,
            description: "CREDENTIALTYPE_QR_CODE",
        },
        {
            type: app_enums_1.enums.CredentialType.MRZ,
            description: "CREDENTIALTYPE_MRZ",
        },
    ];
    ReportConstants.CredentialTypesByAuthenticationFactor = [
        {
            type: app_enums_1.enums.AuthenticationFactor.ProximityCard,
            description: "CREDENTIALTYPE_PROXIMITYCARD",
        },
        {
            type: app_enums_1.enums.AuthenticationFactor.Bluetooth,
            description: "CREDENTIALTYPE_BLE",
        },
        {
            type: app_enums_1.enums.AuthenticationFactor.MiFare,
            description: "CREDENTIALTYPE_MIFARE",
        },
        {
            type: app_enums_1.enums.AuthenticationFactor.AccessToken,
            description: "CREDENTIALTYPE_ACCESS_TOKEN",
        },
        {
            type: app_enums_1.enums.AuthenticationFactor.UHFRfid,
            description: "CREDENTIALTYPE_UHFRFID",
        },
        {
            type: app_enums_1.enums.AuthenticationFactor.VehiclePlate,
            description: "CREDENTIALTYPE_VEHICLE_PLATE",
        },
        {
            type: app_enums_1.enums.AuthenticationFactor.NFC,
            description: "CREDENTIALTYPE_NFC",
        },
        {
            type: app_enums_1.enums.AuthenticationFactor.FingerPrint,
            description: "CREDENTIALTYPE_FINGER_PRINT",
        },
        {
            type: app_enums_1.enums.AuthenticationFactor.QR_BarCode,
            description: "CREDENTIALTYPE_QR_CODE",
        },
        {
            type: app_enums_1.enums.AuthenticationFactor.MRZ,
            description: "CREDENTIALTYPE_MRZ",
        },
    ];
})(ReportConstants = exports.ReportConstants || (exports.ReportConstants = {}));
const reverseCredentialData = (data, type) => {
    switch (type) {
        case app_enums_1.enums.CredentialType.MiFare:
        case app_enums_1.enums.CredentialType.ProximityCard:
            return data ? (0, exports.hexToDec)(data).toString() : data;
        default:
            return null;
    }
};
exports.reverseCredentialData = reverseCredentialData;
const getDataText = (data, type) => {
    switch (type) {
        case app_enums_1.enums.CredentialType.FingerPrintISO19794:
            switch (data) {
                case "1":
                    return "FINGER_LEFT_LITTLE";
                case "2":
                    return "FINGER_LEFT_RING";
                case "3":
                    return "FINGER_LEFT_MIDDLE";
                case "4":
                    return "FINGER_LEFT_INDEX";
                case "5":
                    return "FINGER_LEFT_THUMB";
                case "6":
                    return "FINGER_RIGHT_THUMB";
                case "7":
                    return "FINGER_RIGHT_INDEX";
                case "8":
                    return "FINGER_RIGHT_MIDDLE";
                case "9":
                    return "FINGER_RIGHT_RING";
                case "10":
                    return "FINGER_RIGHT_LITTLE";
            }
        default:
            return data;
    }
};
exports.getDataText = getDataText;
const hexToDec = (hexData) => {
    if (!(0, exports.checkHex)(hexData)) {
        return "0";
    }
    return parseInt(hexData, 16).toString(10);
};
exports.hexToDec = hexToDec;
const checkHex = (n) => {
    return /^[0-9A-Fa-f]{1,64}$/.test(n);
};
exports.checkHex = checkHex;
function formatDateToString(date) {
    const dateString = date.toTimeString().substr(0, 5);
    return dateString;
}
exports.formatDateToString = formatDateToString;
function formatDateForExcel(date) {
    let utc = (0, moment_1.default)(date);
    let tz = utc.toDate().getTimezoneOffset();
    utc.add(-tz, "minute");
    return utc.startOf("second").toDate();
}
exports.formatDateForExcel = formatDateForExcel;
function numberToDurationFormat(expectedWorkDuration, locale) {
    let hourAbbr = i18n_1.default.__({ phrase: "EXCEL-REPORT.HOUR_ABBREVATION", locale });
    let minAbbr = i18n_1.default.__({ phrase: "EXCEL-REPORT.MINUTE_ABBREVATION", locale });
    return Math.floor(expectedWorkDuration / 60).toString() + hourAbbr + " " + (expectedWorkDuration % 60).toString() + minAbbr;
}
exports.numberToDurationFormat = numberToDurationFormat;
function numberToDurationFormatV2(expectedWorkDuration, locale) {
    let hourAbbr = i18n_1.default.__({ phrase: "PDF-REPORT.HOUR_ABBREVATION", locale });
    let minAbbr = i18n_1.default.__({ phrase: "PDF-REPORT.MINUTE_ABBREVATION", locale });
    return Math.floor(expectedWorkDuration / 60).toString() + hourAbbr + " " + (expectedWorkDuration % 60).toString() + minAbbr;
}
exports.numberToDurationFormatV2 = numberToDurationFormatV2;
function formatTime(value, locale) {
    if (!value) {
        return "-";
    }
    let minusPrefix = value < 0 ? "- " : "";
    value = Math.abs(value);
    let minutes = value % 60;
    let hours = (value - minutes) / 60;
    let hoursString = (0, moment_1.default)().locale(locale).localeData().relativeTime(hours, true, "hh", false);
    let minutesString = (0, moment_1.default)().locale(locale).localeData().relativeTime(minutes, true, "mm", false);
    return minusPrefix + (hours != 0 ? hoursString + " " + minutesString : minutesString);
}
exports.formatTime = formatTime;
function rstrip(s) {
    return (s || "").replace(/^\s+|\s+$/g, "");
}
exports.rstrip = rstrip;
function join(newElement, sourceArray) {
    let result = "";
    sourceArray.forEach((s) => {
        result += newElement + s;
    });
    return result;
}
exports.join = join;
function getCredentialDataVisibility(type) {
    switch (type) {
        case app_enums_1.enums.CredentialType.UHFRfid:
        case app_enums_1.enums.CredentialType.MiFare:
        case app_enums_1.enums.CredentialType.ProximityCard:
        case app_enums_1.enums.CredentialType.VehiclePlate:
        case app_enums_1.enums.CredentialType.MRZ:
            return true;
        default:
            return false;
    }
}
exports.getCredentialDataVisibility = getCredentialDataVisibility;
function getDescriptionAccessLogItem(reason, direction) {
    switch (reason) {
        case app_enums_1.enums.AccessLogReason.Success:
            if (direction === app_enums_1.enums.AccessDirection.Entrance) {
                return "DESCRIPTION_SUCCEED_ENTRANCE";
            }
            else if (direction === app_enums_1.enums.AccessDirection.Exit) {
                return "DESCRIPTION_SUCCEED_EXIT";
            }
            else {
                return "DESCRIPTION_SUCCEED_ALL";
            }
        case app_enums_1.enums.AccessLogReason.HasNotRight:
            if (direction === app_enums_1.enums.AccessDirection.Entrance) {
                return "DESCRIPTION_HAS_NOT_RIGHT_ENTRANCE";
            }
            else if (direction === app_enums_1.enums.AccessDirection.Exit) {
                return "DESCRIPTION_HAS_NOT_RIGHT_EXIT";
            }
            else {
                return "DESCRIPTION_HAS_NOT_RIGHT_ENTRANCE";
            }
        case app_enums_1.enums.AccessLogReason.UnknownCredentialOwner:
            return "DESCRIPTION_UNKNOWN_ATTEMPT";
        case app_enums_1.enums.AccessLogReason.UnknownAccessPoint:
            return "DESCRIPTION_UNKNOWN_ACP";
        case app_enums_1.enums.AccessLogReason.UnhandledFailure:
            return "DESCRIPTION_UNHANDLED_FAILURE";
        case app_enums_1.enums.AccessLogReason.AntiPassbackReject:
            if (direction === app_enums_1.enums.AccessDirection.Entrance) {
                return "DESCRIPTION_FAILED_ANTIPASSBACK_ENTRANCE";
            }
            else if (direction === app_enums_1.enums.AccessDirection.Exit) {
                return "DESCRIPTION_FAILED_ANTIPASSBACK_EXIT";
            }
            else {
                return "DESCRIPTION_FAILED_ANTIPASSBACK_ALL";
            }
        case app_enums_1.enums.AccessLogReason.AntiPassbackTimeOutReject:
            if (direction === app_enums_1.enums.AccessDirection.Entrance) {
                return "DESCRIPTION_FAILED_TIMEOUT_ENTRANCE";
            }
            else if (direction === app_enums_1.enums.AccessDirection.Exit) {
                return "DESCRIPTION_FAILED_TIMEOUT_EXIT";
            }
            else {
                return "DESCRIPTION_FAILED_TIMEOUT_ALL";
            }
        case app_enums_1.enums.AccessLogReason.CredentialExpired:
            return "DESCRIPTION_CREDENTIAL_EXPIRED";
        case app_enums_1.enums.AccessLogReason.InsufficientFund:
            if (direction === app_enums_1.enums.AccessDirection.Entrance) {
                return "DESCRIPTION_INSUFFICENT_FUND_ENTRANCE";
            }
            else if (direction === app_enums_1.enums.AccessDirection.Exit) {
                return "DESCRIPTION_INSUFFICENT_FUND_EXIT";
            }
            else {
                return "DESCRIPTION_INSUFFICENT_FUND_ALL";
            }
        case app_enums_1.enums.AccessLogReason.RuleReject:
            if (direction === app_enums_1.enums.AccessDirection.Entrance) {
                return "DESCRIPTION_RULE_REJECT_ENTRANCE";
            }
            else if (direction === app_enums_1.enums.AccessDirection.Exit) {
                return "DESCRIPTION_RULE_REJECT_EXIT";
            }
            else {
                return "DESCRIPTION_RULE_REJECT_ALL";
            }
        case app_enums_1.enums.AccessLogReason.Manual:
            if (direction === app_enums_1.enums.AccessDirection.Entrance) {
                return "DESCRIPTION_MANUAL_ENTRANCE";
            }
            else if (direction === app_enums_1.enums.AccessDirection.Exit) {
                return "DESCRIPTION_MANUAL_EXIT";
            }
            else {
                return "DESCRIPTION_MANUAL_ALL";
            }
        case app_enums_1.enums.AccessLogReason.StateOpened:
            return "DESCRIPTION_STATE_OPENNED";
        case app_enums_1.enums.AccessLogReason.StateDisabled:
            return "DESCRIPTION_STATE_DISABLED";
        case app_enums_1.enums.AccessLogReason.IncompleteMultiFactor:
            if (direction === app_enums_1.enums.AccessDirection.Entrance) {
                return "DESCRIPTION_MULTI_FACTOR_ENTRANCE";
            }
            else if (direction === app_enums_1.enums.AccessDirection.Exit) {
                return "DESCRIPTION_MULTI_FACTOR_EXIT";
            }
            else {
                return "DESCRIPTION_MULTI_FACTOR_ALL";
            }
        case app_enums_1.enums.AccessLogReason.UserDisabled:
            return "DESCRIPTION_USER_DISABLED";
        case app_enums_1.enums.AccessLogReason.Forbidden:
            return "DESCRIPTION_FORBIDDEN";
        case app_enums_1.enums.AccessLogReason.RelayNotAvailable:
            return "DESCRIPTION_RELAY_NOT_AVAILABLE";
        case app_enums_1.enums.AccessLogReason.ChallengeFail:
            return "DESCRIPTION_CHALLENGE_FAIL";
        case app_enums_1.enums.AccessLogReason.InvalidChallenge:
            return "DESCRIPTION_INVALID_CHALLENGE";
        case app_enums_1.enums.AccessLogReason.MifareFingerPrintNotMatch:
            return "DESCRIPTION_MIFARE_FINGERPRINT_NOT_MATCH";
        case app_enums_1.enums.AccessLogReason.MifareFingerPrintTimeout:
            return "DESCRIPTION_MIFARE_FINGERPRINT_TIMEOUT";
        case app_enums_1.enums.AccessLogReason.RegionCapacityFull:
            return "DESCRIPTION_REGION_CAPACITY_FULL";
        case app_enums_1.enums.AccessLogReason.CapacityBasedRuleReject:
            return "DESCRIPTION_CAPACITY_BASED_RULE_REJECT";
        case app_enums_1.enums.AccessLogReason.HesCodeNoRecord:
            return "DESCRIPTION_HES_CODE_NO_RECORD";
        case app_enums_1.enums.AccessLogReason.HesCodeRiskly:
            return "DESCRIPTION_HES_CODE_RISKLY";
        case app_enums_1.enums.AccessLogReason.HesCodeError:
            return "DESCRIPTION_HES_CODE_ERROR";
        default:
            return "DESCRIPTION_UNKNOWN_ATTEMPT";
    }
}
exports.getDescriptionAccessLogItem = getDescriptionAccessLogItem;
function getAccessLogStatusInfo(direction, locale, isSuccess) {
    switch (direction) {
        case app_enums_1.enums.AccessDirection.Entrance:
            return isSuccess
                ? i18n_1.default.__({ phrase: "PDF-REPORT.ACCESS_LOGS_SUCCESS_ENTRANCE_SUMMARY", locale })
                : i18n_1.default.__({ phrase: "PDF-REPORT.ACCESS_LOGS_FAILED_ENTRANCE_SUMMARY", locale });
        case app_enums_1.enums.AccessDirection.Exit:
            return isSuccess
                ? i18n_1.default.__({ phrase: "PDF-REPORT.ACCESS_LOGS_SUCCESS_EXIT_SUMMARY", locale })
                : i18n_1.default.__({ phrase: "PDF-REPORT.ACCESS_LOGS_FAILED_EXIT_SUMMARY", locale });
        default:
            return isSuccess ? i18n_1.default.__({ phrase: "PDF-REPORT.ACCESS_LOGS_SUCCESS_ALL_SUMMARY", locale }) : i18n_1.default.__({ phrase: "PDF-REPORT.ACCESS_LOGS_FAILED_ALL_SUMMARY", locale });
    }
}
exports.getAccessLogStatusInfo = getAccessLogStatusInfo;
function styleExcelWorksheet(sheet, locale) {
    sheet.eachRow((row, rowNumber) => {
        row.eachCell((cell, colNumber) => {
            if (colNumber <= sheet.columns.length) {
                if (rowNumber == 1 || (rowNumber === sheet._rows.length && sheet.name === i18n_1.default.__({ phrase: "EXCEL-REPORT.GENERAL", locale }))) {
                    row.getCell(colNumber).fill = {
                        type: "pattern",
                        pattern: "solid",
                        fgColor: { argb: "4f81bd" },
                    };
                    row.getCell(colNumber).font = {
                        bold: true,
                        size: 11,
                        color: { argb: "ffffff" },
                    };
                }
                else {
                    row.getCell(colNumber).font = {
                        bold: false,
                        size: 10,
                    };
                    if (rowNumber % 2 == 1) {
                        row.getCell(colNumber).fill = {
                            type: "pattern",
                            pattern: "solid",
                            fgColor: { argb: "dce6f1" },
                        };
                    }
                    else {
                        row.getCell(colNumber).fill = {
                            type: "pattern",
                            pattern: "solid",
                            fgColor: { argb: "b8cce4" },
                        };
                    }
                }
                if (row != 1)
                    cell.alignment = { vertical: "top", horizontal: "left", wrapText: true };
                cell.border = {
                    top: { style: "thin", color: { argb: "ffffff" } },
                    left: { style: "thin", color: { argb: "ffffff" } },
                    bottom: { style: "thin", color: { argb: "ffffff" } },
                    right: { style: "thin", color: { argb: "ffffff" } },
                };
            }
        });
    });
}
exports.styleExcelWorksheet = styleExcelWorksheet;
function formatColumn(sheet, colIndex, format) {
    sheet.getColumn(colIndex).numFmt = format;
}
exports.formatColumn = formatColumn;
function parseCaptionForExcel(userCaptions) {
    let result = "";
    userCaptions.forEach((item) => {
        if (item.text != null && item.text != "__avatar") {
            item.text.forEach((element) => {
                if (element != null)
                    result += " " + element;
            });
            result += "\r\n";
        }
    });
    return result;
}
exports.parseCaptionForExcel = parseCaptionForExcel;
function parseCaptionForPdfFilter(userCaptions) {
    let _res = "";
    userCaptions.forEach((userCaption, index) => {
        _res += "(";
        userCaption.captionLines.forEach((item, _index) => {
            if (item.text != null && item.text != "__avatar") {
                item.text.forEach((element) => {
                    if (element != null)
                        _res += element + (_index < userCaptions.length - 1 ? " " : ")");
                });
            }
        });
        _res += index < userCaptions.length - 1 ? ", " : "";
    });
    return _res;
}
exports.parseCaptionForPdfFilter = parseCaptionForPdfFilter;
function parseCaption(userCaptions) {
    let result = "";
    let _image;
    userCaptions.forEach((item) => {
        if (item.text != null && item.text != "__avatar") {
            item.text.forEach((element) => {
                if (element != null)
                    result += element + " ";
            });
            result += "\n";
        }
        else if (item.text == "__avatar" && item.avatar) {
            let image = {
                image: "",
                width: ReportConstants.widthProfilePicture,
                height: ReportConstants.heightProfilePicture,
                style: "center",
                margin: [0, 15, 0, 15],
            };
            image.image = item.avatar;
            _image = image;
        }
    });
    if (_image == null) {
        _image = {
            image: ReportConstants.emptyAvatar,
            width: ReportConstants.widthProfilePicture,
            height: ReportConstants.heightProfilePicture,
            style: "center",
            margin: [0, 15, 0, 15],
        };
    }
    return {
        columns: [_image, { text: result, fontSize: 10, color: "black", margin: [10, 15, 0, 15] }],
    };
}
exports.parseCaption = parseCaption;
function parseCaptionV2(userCaptions) {
    let result = "";
    let _image;
    userCaptions.forEach((item) => {
        if (item.text != null && item.text != "__avatar") {
            item.text.forEach((element) => {
                if (element != null)
                    result += element + " ";
            });
            result += "\n";
        }
        else if (item.text == "__avatar" && item.avatar) {
            let image = {
                image: "",
                width: ReportConstants.widthProfilePicture,
                height: ReportConstants.heightProfilePicture,
                style: "center",
                margin: [0, 15, 0, 15],
            };
            image.image = item.avatar;
            _image = image;
        }
    });
    if (_image == null) {
        _image = {
            image: ReportConstants.emptyAvatar,
            width: ReportConstants.widthProfilePicture,
            height: ReportConstants.heightProfilePicture,
            style: "center",
            margin: [0, 15, 0, 15],
        };
    }
    return {
        columns: [_image, { text: result, fontSize: 6, color: "black", margin: [10, 15, 0, 15] }],
    };
}
exports.parseCaptionV2 = parseCaptionV2;
function getLocalizedTextOfAccessLogReason(reason, locale) {
    let phrase = "";
    switch (reason) {
        case app_enums_1.enums.libEnumsV2.AccessLogReason.Success:
            phrase = "ACCESS_LOG_REASON.SUCCESS";
            break;
        case app_enums_1.enums.libEnumsV2.AccessLogReason.HasNotRight:
            phrase = "ACCESS_LOG_REASON.HASNOTRIGHT";
            break;
        case app_enums_1.enums.libEnumsV2.AccessLogReason.UnknownCredentialOwner:
            phrase = "ACCESS_LOG_REASON.UNKNOWN_USER";
            break;
        case app_enums_1.enums.libEnumsV2.AccessLogReason.UnknownAccessPoint:
            phrase = "ACCESS_LOG_REASON.UNKNOWN_ACCESS_POINT";
            break;
        case app_enums_1.enums.libEnumsV2.AccessLogReason.UnhandledFailure:
            phrase = "ACCESS_LOG_REASON.UNHANDLED_FAILURE";
            break;
        case app_enums_1.enums.libEnumsV2.AccessLogReason.AntiPassbackReject:
            phrase = "ACCESS_LOG_REASON.ANTIPASSBACK_REJECT";
            break;
        case app_enums_1.enums.libEnumsV2.AccessLogReason.AntiPassbackTimeOutReject:
            phrase = "ACCESS_LOG_REASON.ANTIPASSBACK_TIMEOUT_REJECT";
            break;
        case app_enums_1.enums.libEnumsV2.AccessLogReason.CredentialExpired:
            phrase = "ACCESS_LOG_REASON.CREDENTIAL_EXPIRED";
            break;
        case app_enums_1.enums.libEnumsV2.AccessLogReason.InsufficientFund:
            phrase = "ACCESS_LOG_REASON.INSUFFICIENT_FUND";
            break;
        case app_enums_1.enums.libEnumsV2.AccessLogReason.RuleReject:
            phrase = "ACCESS_LOG_REASON.RULE_REJECT";
            break;
        case app_enums_1.enums.libEnumsV2.AccessLogReason.Manual:
            phrase = "ACCESS_LOG_REASON.MANUAL";
            break;
        case app_enums_1.enums.libEnumsV2.AccessLogReason.StateOpened:
            phrase = "ACCESS_LOG_REASON.STATE_OPENNED";
            break;
        case app_enums_1.enums.libEnumsV2.AccessLogReason.StateDisabled:
            phrase = "ACCESS_LOG_REASON.STATE_DISABLED";
            break;
        case app_enums_1.enums.libEnumsV2.AccessLogReason.IncompleteMultiFactor:
            phrase = "ACCESS_LOG_REASON.INCOMPLETE_MULTI_FACTOR";
            break;
        case app_enums_1.enums.libEnumsV2.AccessLogReason.Forbidden:
            phrase = "ACCESS_LOG_REASON.FORBIDDEN";
            break;
        case app_enums_1.enums.libEnumsV2.AccessLogReason.UserDisabled:
            phrase = "ACCESS_LOG_REASON.USER_DISABLED";
            break;
        case app_enums_1.enums.libEnumsV2.AccessLogReason.RelayNotAvailable:
            phrase = "ACCESS_LOG_REASON.RELAY_NOT_AVAILABLE";
            break;
        case app_enums_1.enums.libEnumsV2.AccessLogReason.ChallengeFail:
            phrase = "ACCESS_LOG_REASON.CHALLENGE_FAIL";
            break;
        case app_enums_1.enums.libEnumsV2.AccessLogReason.InvalidChallenge:
            phrase = "ACCESS_LOG_REASON.INVALID_CHALLENGE";
            break;
        case app_enums_1.enums.libEnumsV2.AccessLogReason.MifareFingerPrintNotMatch:
            phrase = "ACCESS_LOG_REASON.MIFARE_CARD_FINGERPRINT_NOT_MATCH";
            break;
        case app_enums_1.enums.libEnumsV2.AccessLogReason.MifareFingerPrintTimeout:
            phrase = "ACCESS_LOG_REASON.MIFARE_CARD_FINGERPRINT_TIMEOUT";
            break;
        case app_enums_1.enums.libEnumsV2.AccessLogReason.RegionCapacityFull:
            phrase = "ACCESS_LOG_REASON.AREA_CAPACITY_FULL";
            break;
        case app_enums_1.enums.libEnumsV2.AccessLogReason.CapacityBasedRuleReject:
            phrase = "ACCESS_LOG_REASON.CAPACITY_BASED_RULE_REJECT";
            break;
        case app_enums_1.enums.libEnumsV2.AccessLogReason.HesCodeNoRecord:
            phrase = "ACCESS_LOG_REASON.HES_CODE_NO_RECORD";
            break;
        case app_enums_1.enums.libEnumsV2.AccessLogReason.HesCodeRiskly:
            phrase = "ACCESS_LOG_REASON.HES_CODE_RISKY";
            break;
        case app_enums_1.enums.libEnumsV2.AccessLogReason.HesCodeError:
            phrase = "ACCESS_LOG_REASON.HES_CODE_ERROR";
            break;
        default:
            break;
    }
    return i18n_1.default.__({ phrase: phrase, locale });
}
exports.getLocalizedTextOfAccessLogReason = getLocalizedTextOfAccessLogReason;
function throwDbReportError(errorMessage, res) {
    switch (errorMessage) {
        case "Timeout":
            (0, dal_access_error_1.throwReportTimeoutErrorTr)();
            break;
        case "Notification":
            (0, dal_access_error_1.throwReportNotificationErrorTr)();
        default:
            (0, dal_access_error_1.throwInternalErrorTr)();
            break;
    }
}
exports.throwDbReportError = throwDbReportError;
function getAccessControlPointTypeName(type, locale) {
    let typeName;
    switch (type) {
        case dal_constants_1.DalConstants.AccessControlPointType.Barrier:
            typeName = "ACCESSCONTROLPOINTTYPE.BARRIER";
            break;
        case dal_constants_1.DalConstants.AccessControlPointType.Door:
            typeName = "ACCESSCONTROLPOINTTYPE.DOOR";
            break;
        case dal_constants_1.DalConstants.AccessControlPointType.DropArmTripotTurnstile:
            typeName = "ACCESSCONTROLPOINTTYPE.DROPARMTURNSTILE";
            break;
        case dal_constants_1.DalConstants.AccessControlPointType.Elevator:
            typeName = "ACCESSCONTROLPOINTTYPE.ELEVATOR";
            break;
        case dal_constants_1.DalConstants.AccessControlPointType.ParkingGate:
            typeName = "ACCESSCONTROLPOINTTYPE.PARKINGGATE";
            break;
        case dal_constants_1.DalConstants.AccessControlPointType.SecurityGate:
            typeName = "ACCESSCONTROLPOINTTYPE.SECURITYGATE";
            break;
        case dal_constants_1.DalConstants.AccessControlPointType.AttendanceControl:
            typeName = "ACCESSCONTROLPOINTTYPE.ATTENDANCECONTROL";
            break;
        case dal_constants_1.DalConstants.AccessControlPointType.TripotTurnstile:
            typeName = "ACCESSCONTROLPOINTTYPE.TRIPOTTURNSTILE";
            break;
        case dal_constants_1.DalConstants.AccessControlPointType.TurnstileGate:
            typeName = "ACCESSCONTROLPOINTTYPE.TURNSTILEGATE";
            break;
        case dal_constants_1.DalConstants.AccessControlPointType.TicketControl:
            typeName = "ACCESSCONTROLPOINTTYPE.TICKETCONTROL";
            break;
        case dal_constants_1.DalConstants.AccessControlPointType.VirtualMobileCheckin:
            typeName = "ACCESSCONTROLPOINTTYPE.VIRTUALMOBILECHECKIN";
            break;
        default:
            break;
    }
    return i18n_1.default.__({ phrase: typeName, locale });
}
exports.getAccessControlPointTypeName = getAccessControlPointTypeName;
function hideSensitiveFieldsOfVisitsForReport(hiddenVisitFields, hiddenVisitorFields, visits, settings) {
    for (let visit of visits) {
        for (const hiddenField of hiddenVisitFields) {
            if (visit.visitFields[hiddenField] != null || visit.visitFields[hiddenField] != undefined)
                visit.visitFields[hiddenField] = null;
        }
    }
    for (let visitor of visits) {
        for (const hiddenField of hiddenVisitorFields) {
            if (visitor.visitorProfileFields[hiddenField] != null || visitor.visitorProfileFields[hiddenField] != undefined)
                visitor.visitorProfileFields[hiddenField] = null;
        }
        if (visitor?.visitorProfileFields?.[dal_access_models_1.DbAccessModel.PredefinedVisitorProfileFormFields.uniqueId] && settings?.hideUniqueIdForVisitor) {
            visitor.visitorProfileFields[dal_access_models_1.DbAccessModel.PredefinedVisitorProfileFormFields.uniqueId] =
                visitor.visitorProfileFields[dal_access_models_1.DbAccessModel.PredefinedVisitorProfileFormFields.uniqueId].substring(0, settings.showNumberOfInitial) +
                    "***********".substring(0, 11 - settings.showNumberOfInitial - settings.showNumberOfLast) +
                    visitor.visitorProfileFields[dal_access_models_1.DbAccessModel.PredefinedVisitorProfileFormFields.uniqueId].substring(visitor.visitorProfileFields[dal_access_models_1.DbAccessModel.PredefinedVisitorProfileFormFields.uniqueId].length - settings.showNumberOfLast);
        }
    }
    return visits;
}
exports.hideSensitiveFieldsOfVisitsForReport = hideSensitiveFieldsOfVisitsForReport;
function hideSensitiveFieldsOfVisitorForReport(hiddenVisitorFields, visitors, settings) {
    for (let visitor of visitors) {
        for (const hiddenField of hiddenVisitorFields) {
            if (visitor.fields[hiddenField] != null || visitor.fields[hiddenField] != undefined)
                visitor.fields[hiddenField] = null;
        }
        if (visitor.fields[dal_access_models_1.DbAccessModel.PredefinedVisitorProfileFormFields.uniqueId] && settings?.hideUniqueIdForVisitor) {
            visitor.fields[dal_access_models_1.DbAccessModel.PredefinedVisitorProfileFormFields.uniqueId] =
                visitor.fields[dal_access_models_1.DbAccessModel.PredefinedVisitorProfileFormFields.uniqueId].substring(0, settings.showNumberOfInitial) +
                    "***********".substring(0, 11 - settings.showNumberOfInitial - settings.showNumberOfLast) +
                    visitor.fields[dal_access_models_1.DbAccessModel.PredefinedVisitorProfileFormFields.uniqueId].substring(visitor.fields[dal_access_models_1.DbAccessModel.PredefinedVisitorProfileFormFields.uniqueId].length - settings.showNumberOfLast);
        }
    }
    return visitors;
}
exports.hideSensitiveFieldsOfVisitorForReport = hideSensitiveFieldsOfVisitorForReport;
function convertSecondsToHoursMinutes(seconds) {
    let min = Math.round(Math.round(seconds) / 60);
    return {
        hours: Math.floor(min / 60) ?? 0,
        minutes: min % 60,
    };
}
exports.convertSecondsToHoursMinutes = convertSecondsToHoursMinutes;
