"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.PDFReportCreator = void 0;
const fs_1 = __importDefault(require("fs"));
const pdfmake_1 = __importDefault(require("pdfmake/build/pdfmake"));
require("pdfmake/build/vfs_fonts");
const path_1 = __importDefault(require("path"));
const uuid_1 = __importDefault(require("uuid"));
const app_config_1 = require("../../../app.config");
const sharp = require("sharp");
const report_util_1 = require("../report.util");
const i18n_1 = __importDefault(require("i18n"));
const moment = require("moment");
const report_pdf_style_1 = require("./report.pdf-style");
const app_enums_1 = require("../../../app.enums");
class PDFReportCreator {
    constructor(locale) {
        this.locale = locale;
        this.rows = [];
        this.filters = [];
    }
    async createPdf(organization, organizationLogo, title, tableColumnWidths) {
        let _dd = await this.createDocumentDefinitionOfPdf(organization, organizationLogo, title, tableColumnWidths);
        let fileId = uuid_1.default.v4();
        let filePath = path_1.default.join(app_config_1.appConfig.tmpDirectory, fileId + ".pdf");
        let pdf = pdfmake_1.default.createPdf(_dd);
        await new Promise((resolve, reject) => {
            pdf.getBuffer((result) => {
                fs_1.default.writeFileSync(filePath, result);
                resolve(filePath);
            });
        });
        return {
            format: app_enums_1.enums.ReportFormat.Pdf,
            preparedReportId: fileId,
            filePath: filePath,
            notificationType: undefined,
            reportType: title,
            sendFileInAttachments: undefined,
        };
    }
    async createDocumentDefinitionOfPdf(organization, organizationLogo, title, tableColumnWidths) {
        let _dd = {
            footer: (currentPage, pageCount) => {
                return this.createFooterInformation(currentPage, pageCount);
            },
            content: [await this.createHeader(organization, organizationLogo), this.createTitle(title), this.createFilterTable(), this.createTable(tableColumnWidths)],
            styles: report_pdf_style_1.ReportPdfStyles,
        };
        return _dd;
    }
    addRow(cells) {
        if (cells == null || cells.length == 0)
            return;
        if (cells.length > this.columns.length)
            return;
        let cellObj = [];
        cells.forEach((cell) => {
            if (!cell.isCaption) {
                cellObj.push({ text: cell.text, fontSize: cell.fontSize, color: cell.color, margin: cell.margin });
            }
            else {
                cellObj.push(cell.text);
            }
        });
        this.rows.push(cellObj);
    }
    addBar(cells) {
        if (cells == null || cells.length == 0 || !this.columns || this.columns.length == 0)
            return;
        let cellObj = [];
        let widths = [];
        let bars = [];
        cells.forEach((cell) => {
            if (cell.width > 0) {
                widths.push(cell.width);
                bars.push({
                    text: cell.text,
                    fillColor: cell.barColor,
                    alignment: cell.text,
                });
            }
        });
        cellObj.push({
            text: "",
            colSpan: this.columns.length,
            alignment: "center",
            table: {
                widths: widths,
                body: [bars],
            },
        });
        this.rows.push(cellObj);
    }
    addColumns(columns) {
        if (columns == null || columns.length == 0)
            return;
        let cellObj = [];
        columns.forEach((column) => {
            cellObj.push({ text: column, style: "tableColumn" });
        });
        this.columns = columns;
        this.rows.push(columns);
    }
    addFilters(filters) {
        let result = [];
        if (filters == null || filters.length == 0) {
            result = [
                { text: " ", style: "filterTitle" },
                { text: " ", style: "filterValue" },
            ];
            this.filters.push(result);
        }
        else {
            filters.forEach((filter) => {
                result = [
                    { text: filter.title + " : ", style: "filterTitle" },
                    { text: filter.value, style: "filterValue" },
                ];
                this.filters.push(result);
            });
        }
    }
    async createHeader(organization, organizationLogo) {
        let header;
        if (organizationLogo == null) {
            header = { text: organization == null ? "" : organization, style: "organizationName" };
        }
        else {
            let logoImage = await sharp(organizationLogo, { failOnError: false }).resize(240, 60).toFormat("png").toBuffer();
            let logoNew = (await sharp({
                create: {
                    width: 240,
                    height: 60,
                    channels: 4,
                    background: { r: 1, g: 50, b: 91, alpha: 0.9 },
                },
                failOnError: false,
            })
                .png()
                .composite([{ input: logoImage }])
                .toBuffer()).toString("base64");
            header = {
                image: "data:image/png;base64," + logoNew,
                width: report_util_1.ReportConstants.widthOrganizationLogo,
                alignment: "left",
            };
        }
        return header;
    }
    createFooterInformation(currentPage, pageCount) {
        let locale = this.locale;
        return {
            columns: [
                {
                    text: i18n_1.default.__({ phrase: "PDF-REPORT.report_create_date", locale }) + ": " + moment().locale(locale).format("DD/MM/YYYY HH:mm") + " ",
                    style: "footerReportCreationDate",
                },
                { text: currentPage.toString() + " / " + pageCount, style: "footerPageNumber" },
            ],
        };
    }
    createTitle(title) {
        return {
            text: title + "\n",
            style: "title",
        };
    }
    createFilterTable() {
        if (this.filters.length == 0) {
            this.filters.push([
                { text: " ", style: "filterTitle" },
                { text: " ", style: "filterValue" },
            ]);
        }
        let body = this.filters;
        return {
            table: {
                heights: 20,
                body,
            },
            layout: "noBorders",
        };
    }
    createTable(tableColumnWidths) {
        return {
            style: "table",
            layout: {
                fillColor: function (rowIndex, node, columnIndex) {
                    return rowIndex % 2 === 0 ? report_util_1.ReportConstants.evenRowColor : null;
                },
            },
            table: {
                headerRows: 1,
                dontBreakRows: true,
                keepWithHeaderRows: 1,
                widths: tableColumnWidths,
                body: this.rows,
            },
        };
    }
}
exports.PDFReportCreator = PDFReportCreator;
