"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.PdfReportWorkSummaryV2 = void 0;
const sharp_1 = __importDefault(require("sharp"));
const fs_1 = __importDefault(require("fs"));
const path_1 = __importDefault(require("path"));
const uuid_1 = __importDefault(require("uuid"));
const pdfmake_1 = __importDefault(require("pdfmake/build/pdfmake"));
const app_config_1 = require("../../../app.config");
const moment_1 = __importDefault(require("moment"));
const report_util_1 = require("../report.util");
const i18n_1 = __importDefault(require("i18n"));
const dal_manager_1 = require("../../../dal/dal.manager");
const dal_constants_1 = require("../../../dal/dal.constants");
const app_enums_1 = require("../../../app.enums");
const worker_threads_1 = require("worker_threads");
const app_logs_1 = require("../../../app.logs");
const generator_1 = require("../generator");
var Constants;
(function (Constants) {
    Constants.ColorCodes = {
        ON_TIME: "#7C9923",
        TOLERATED: "#D56626",
        NOT_TOLERATED: "#A11913",
        PERMISSION: "4C2AA8",
    };
})(Constants || (Constants = {}));
class PdfReportWorkSummaryV2 extends generator_1.ReportGenerator {
    constructor(request, locale) {
        super(request, locale);
        this.minuteWidth = 300 / (24 * 60);
        this._options = request.filter;
    }
    async generateReport() {
        this.organization = await dal_manager_1.dbManager.accessOrganization.getOrganizationLogoAndName(this._request.organizationId);
        const option = Object.assign(this._options, {
            paginationRequest: {
                take: app_config_1.appConfig.reportExportRowLimit,
                skip: undefined,
            },
        });
        this._data = await dal_manager_1.dbManager.accessPacs2.workSummaryOfFilteredEmployeesAtDate(this._request.requesterUserId, this._request.organizationId, option);
        let rows = [];
        let locale = this._locale;
        let employee = i18n_1.default.__({ phrase: "PDF-REPORT.employee", locale });
        let status = i18n_1.default.__({ phrase: "PDF-REPORT.status", locale });
        let dummyAccess = false;
        this._data.items.forEach((result) => {
            let rowObj = [];
            let statusText;
            let statusColor = this.findColorCodeByPerformance(result);
            if (result.physicallyInRegionDuration && result.physicallyInRegionDuration > 0) {
                statusText = i18n_1.default.__({ phrase: "PDF-REPORT.arrived", locale: locale });
                statusText += "\n";
            }
            else if (result.expectedWorkDuration == 0) {
                statusText = "";
            }
            else {
                statusText = i18n_1.default.__({ phrase: "PDF-REPORT.not_arrived", locale: locale });
                statusText += "\n";
            }
            rowObj.push((0, report_util_1.parseCaptionV2)(result.userCaption));
            let arrSecondCol = [];
            let cellObj = [];
            let widths = [];
            let bars = [];
            if (result.missingWorkDuration) {
                widths.push(parseFloat((this.minuteWidth * result.missingWorkDuration).toFixed(1)));
                bars.push({
                    text: null,
                    fillColor: this.getColor(false, dal_constants_1.DalConstants.EmployeeStatus.AtWorkingHours),
                    alignment: null,
                    outerWidth: 0,
                });
            }
            if (result.normalWorkDuration) {
                widths.push(parseFloat((this.minuteWidth * result.normalWorkDuration).toFixed(1)));
                bars.push({
                    text: null,
                    fillColor: this.getColor(true, dal_constants_1.DalConstants.EmployeeStatus.AtWorkingHours),
                    alignment: null,
                    outerWidth: 0,
                });
            }
            if (result.accountedOffTimePermissionDuration) {
                widths.push(parseFloat((this.minuteWidth * result.accountedOffTimePermissionDuration).toFixed(1)));
                bars.push({
                    text: null,
                    fillColor: this.getColor(false, dal_constants_1.DalConstants.EmployeeStatus.AtOffTimePermission),
                    alignment: null,
                    outerWidth: 0,
                });
            }
            if (result.extraWorkDuration) {
                widths.push(parseFloat((this.minuteWidth * result.extraWorkDuration).toFixed(1)));
                bars.push({
                    text: null,
                    fillColor: this.getColor(true, dal_constants_1.DalConstants.EmployeeStatus.AtOutOfWorkingHours),
                    alignment: null,
                    outerWidth: 0,
                });
            }
            widths.push("*");
            bars.push({
                text: null,
                fillColor: this.getColor(false, dal_constants_1.DalConstants.EmployeeStatus.AtOutOfWorkingHours),
                alignment: null,
                outerWidth: 0,
            });
            let innerColumns = [];
            let totalWorkDuration = 0;
            if (result.missingWorkDuration) {
                innerColumns.push([
                    {
                        text: "\n",
                    },
                    {
                        text: {
                            text: (0, report_util_1.numberToDurationFormatV2)(result.missingWorkDuration, locale),
                            fontSize: 10,
                            color: "#cccccc",
                            margin: [12, 12, 12, 12],
                            width: "auto",
                            alignment: "center",
                        },
                    },
                    {
                        text: {
                            text: i18n_1.default.__({ phrase: "PDF-REPORT.less_working_with_permission", locale }),
                            fontSize: 5,
                            color: this.getColor(false, dal_constants_1.DalConstants.EmployeeStatus.AtWorkingHours),
                            margin: [12, 0, 12, 12],
                            width: "auto",
                            alignment: "center",
                        },
                    },
                ]);
            }
            if (result.expectedWorkDuration) {
                innerColumns.push([
                    {
                        text: "\n",
                    },
                    {
                        text: {
                            text: (0, report_util_1.numberToDurationFormatV2)(result.expectedWorkDuration, locale),
                            fontSize: 10,
                            color: "#cccccc",
                            margin: [12, 12, 12, 12],
                            width: "auto",
                            alignment: "center",
                        },
                    },
                    {
                        text: {
                            text: i18n_1.default.__({ phrase: "PDF-REPORT.expected_workplan", locale }),
                            fontSize: 5,
                            color: "#cccccc",
                            margin: [12, 0, 12, 12],
                            width: "auto",
                            alignment: "center",
                        },
                    },
                ]);
            }
            if (result.normalWorkDuration) {
                totalWorkDuration += result.normalWorkDuration;
                innerColumns.push([
                    {
                        text: "\n",
                    },
                    {
                        text: {
                            text: (0, report_util_1.numberToDurationFormatV2)(result.normalWorkDuration, locale),
                            fontSize: 10,
                            color: "#cccccc",
                            margin: [12, 0, 12, 12],
                            width: "auto",
                            alignment: "center",
                        },
                    },
                    {
                        text: {
                            text: i18n_1.default.__({ phrase: "PDF-REPORT.actual_working", locale }),
                            fontSize: 5,
                            color: this.getColor(true, dal_constants_1.DalConstants.EmployeeStatus.AtWorkingHours),
                            margin: [12, 0, 12, 12],
                            width: "auto",
                            alignment: "center",
                        },
                    },
                ]);
            }
            if (result.accountedOffTimePermissionDuration) {
                innerColumns.push([
                    {
                        text: "\n",
                    },
                    {
                        text: {
                            text: (0, report_util_1.numberToDurationFormatV2)(result.accountedOffTimePermissionDuration, locale),
                            fontSize: 10,
                            color: "#cccccc",
                            margin: [12, 12, 12, 12],
                            width: "auto",
                            alignment: "center",
                        },
                    },
                    {
                        text: {
                            text: i18n_1.default.__({ phrase: "PDF-REPORT.day_off", locale }),
                            fontSize: 5,
                            color: this.getColor(false, dal_constants_1.DalConstants.EmployeeStatus.AtOffTimePermission),
                            margin: [12, 0, 12, 12],
                            width: "auto",
                            alignment: "center",
                        },
                    },
                ]);
            }
            if (result.extraWorkDuration) {
                totalWorkDuration += result.extraWorkDuration;
                innerColumns.push([
                    {
                        text: "\n",
                    },
                    {
                        text: {
                            text: (0, report_util_1.numberToDurationFormatV2)(result.extraWorkDuration, locale),
                            fontSize: 10,
                            color: "#cccccc",
                            margin: [12, 12, 12, 12],
                            width: "auto",
                            alignment: "center",
                        },
                    },
                    {
                        text: {
                            text: i18n_1.default.__({ phrase: "PDF-REPORT.extra_working", locale }),
                            fontSize: 5,
                            color: this.getColor(true, dal_constants_1.DalConstants.EmployeeStatus.AtOutOfWorkingHours),
                            margin: [12, 0, 12, 12],
                            width: "auto",
                            alignment: "center",
                        },
                    },
                ]);
            }
            if (totalWorkDuration > 0) {
                innerColumns.push([
                    {
                        text: "\n",
                    },
                    {
                        text: { text: (0, report_util_1.numberToDurationFormatV2)(totalWorkDuration, locale), fontSize: 10, color: "#cccccc", margin: [12, 12, 12, 12], width: "auto", alignment: "center" },
                    },
                    {
                        text: {
                            text: i18n_1.default.__({ phrase: "PDF-REPORT.total_work_duration", locale }),
                            fontSize: 5,
                            color: "#cccccc",
                            margin: [12, 0, 12, 12],
                            width: "auto",
                            alignment: "center",
                        },
                    },
                ]);
            }
            if (result.firstStart) {
                let startTime = (0, moment_1.default)(result.firstStart).format("HH:mm");
                let endTime;
                if (result.lastLeave) {
                    endTime = (0, moment_1.default)(result.lastLeave).format("HH:mm");
                }
                else {
                    dummyAccess = true;
                    endTime = "__:__";
                }
                innerColumns.push([
                    {
                        text: statusText,
                        fontSize: 8,
                        color: "white",
                        margin: [12, 12, 0, 12],
                        width: 80,
                        background: statusColor,
                        alignment: "right",
                    },
                    result.lastLeave
                        ? {
                            text: startTime + "-" + endTime,
                            fontSize: 8,
                            color: "#4a4a4a",
                            margin: [0, 0, 0, 0],
                            width: 80,
                            alignment: "right",
                        }
                        : {
                            text: startTime + "-" + endTime,
                            fontSize: 8,
                            color: "#000000",
                            margin: [0, 0, 0, 0],
                            width: 80,
                            background: "#FFFF00",
                            alignment: "right",
                        },
                ]);
            }
            else {
                innerColumns.push([
                    {
                        text: statusText,
                        fontSize: 8,
                        color: "white",
                        margin: [12, 12, 0, 12],
                        width: 80,
                        background: statusColor,
                        alignment: "right",
                    },
                ]);
            }
            cellObj.push([
                {
                    table: {
                        width: "auto",
                        widths: widths,
                        body: [bars],
                    },
                },
                {
                    columns: innerColumns,
                },
            ]);
            arrSecondCol.push(cellObj);
            rowObj.push(arrSecondCol);
            rows.push(rowObj);
        });
        let header;
        if (this.organization == null) {
            header = {
                table: {
                    widths: ["*"],
                    headerRows: 1,
                    body: [
                        [
                            { text: i18n_1.default.__({ phrase: "PDF-REPORT.daily_work_summary_report", locale }) + "\n", style: "header", alignment: "center" },
                            { text: (0, moment_1.default)(this._options.date).locale(this._locale).format("LL"), style: "date", alignment: "right" },
                        ],
                    ],
                },
                layout: "noBorders",
            };
        }
        if (this.organization.logo == null) {
            header = {
                table: {
                    widths: ["auto", "*"],
                    headerRows: 1,
                    body: [
                        [
                            { text: this.organization.name, style: "organizationName", alignment: "left" },
                            [
                                { text: i18n_1.default.__({ phrase: "PDF-REPORT.daily_work_summary_report", locale }) + "\n", style: "header", alignment: "center" },
                                { text: (0, moment_1.default)(this._options.date).locale(this._locale).format("LL"), style: "date", alignment: "right" },
                            ],
                        ],
                    ],
                },
                layout: "noBorders",
            };
        }
        else {
            let logoImage = await (0, sharp_1.default)(this.organization.logo, { failOnError: false }).resize(240, 60).toFormat("png").toBuffer();
            let logoNew = (await (0, sharp_1.default)({
                create: {
                    width: 240,
                    height: 60,
                    channels: 4,
                    background: { r: 1, g: 50, b: 91, alpha: 0.9 },
                },
                failOnError: false,
            })
                .png()
                .composite([{ input: logoImage }])
                .toBuffer()).toString("base64");
            header = {
                table: {
                    widths: ["auto", "*"],
                    headerRows: 1,
                    body: [
                        [
                            {
                                image: "data:image/png;base64," + logoNew,
                                width: report_util_1.ReportConstants.widthOrganizationLogo,
                                alignment: "left",
                            },
                            [
                                { text: i18n_1.default.__({ phrase: "PDF-REPORT.daily_work_summary_report", locale }) + "\n", style: "header", alignment: "center" },
                                { text: (0, moment_1.default)(this._options.date).locale(this._locale).format("LL"), style: "date", alignment: "right" },
                            ],
                        ],
                    ],
                },
                layout: "noBorders",
            };
        }
        this._dd = {
            footer: ((currentPage, pageCount) => {
                return [
                    {
                        text: dummyAccess ? i18n_1.default.__({ phrase: "PDF-REPORT.WARNING_FOR_ACCESS_LOGS", locale }) : "",
                        alignment: "left",
                        fontSize: "8",
                        margin: [40, 0, 0, 0],
                    },
                    {
                        columns: [
                            {
                                text: i18n_1.default.__({ phrase: "PDF-REPORT.report_create_date", locale }) +
                                    ": " +
                                    (0, moment_1.default)().locale(this._locale).format("LL") +
                                    " " +
                                    (0, moment_1.default)().locale(this._locale).format("HH:mm"),
                                alignment: "left",
                                fontSize: "8",
                                margin: [40, 0, 0, 0],
                            },
                            { text: currentPage.toString() + " / " + pageCount, alignment: "right", fontSize: "8", margin: [40, 0] },
                        ],
                    },
                ];
            }),
            content: [
                header,
                {
                    style: "tableExample",
                    table: {
                        headerRows: 0,
                        dontBreakRows: true,
                        keepWithHeaderRows: 0,
                        widths: ["auto", "*"],
                        body: rows,
                    },
                },
            ],
            styles: {
                header: {
                    fontSize: 18,
                    bold: true,
                    margin: [0, 0, 0, 10],
                    alignment: "left",
                },
                subheader: {
                    fontSize: 16,
                    bold: true,
                    margin: [0, 10, 0, 5],
                },
                tableExample: {
                    margin: [0, 5, 0, 15],
                },
                tableHeader: {
                    bold: true,
                    fontSize: 8,
                    color: "black",
                },
                status: {
                    margin: [0, 15],
                    fontSize: 8,
                    color: "#cccccc",
                },
                center: {
                    alignment: "center",
                },
                date: {
                    fontSize: 11,
                    color: "black",
                    alignment: "right",
                },
                organizationName: {
                    fontSize: 11,
                    bold: true,
                    alignment: "left",
                },
            },
            defaultStyle: {},
        };
        let fileId = uuid_1.default.v4();
        let filePath = path_1.default.join(app_config_1.appConfig.tmpDirectory, fileId + ".pdf");
        let pdf = pdfmake_1.default.createPdf(this._dd);
        await new Promise((resolve, reject) => {
            pdf.getBuffer((result) => {
                fs_1.default.writeFileSync(filePath, result);
                resolve(filePath);
            });
        });
        if (!worker_threads_1.isMainThread) {
            app_logs_1.logger.debug("[Worker Thread] Work summary pdf report is ready -> %s", filePath);
        }
        return {
            format: app_enums_1.enums.ReportFormat.Pdf,
            preparedReportId: fileId,
            filePath: filePath,
            notificationType: undefined,
            reportType: app_enums_1.enums.ReportCode.PacsDailyWorkingHours,
            sendFileInAttachments: undefined,
            reportContainsEmptyData: this._request.reportTemplateId ? this._data.items.length === 0 : undefined,
        };
    }
    findColorCodeByPerformance(employeeData) {
        let colorCode = null;
        if (employeeData.expectedWorkDuration == 0) {
            colorCode = null;
        }
        else if (!employeeData.firstStart || employeeData.lateStart > 0) {
            colorCode = Constants.ColorCodes.NOT_TOLERATED;
        }
        else {
            if (employeeData.toleratedLateStart && employeeData.toleratedLateStart > 0) {
                colorCode = Constants.ColorCodes.TOLERATED;
            }
            else {
                colorCode = Constants.ColorCodes.ON_TIME;
            }
        }
        if (employeeData.expectedOffTimePermissionDuration > 0 || employeeData.expectedHolidayDuration > 0) {
            colorCode = Constants.ColorCodes.PERMISSION;
        }
        return colorCode;
    }
    getColor(working, status, inRegion = null, tolerated = false, isIgnoredPermission = false, isDeclaredPermission = false) {
        if (isIgnoredPermission == true)
            return "#e12860";
        if (isDeclaredPermission == true)
            return "#BCD3C1";
        if (working == true) {
            switch (status) {
                case dal_constants_1.DalConstants.EmployeeStatus.AtBreak:
                    if (inRegion == null || inRegion == undefined) {
                        return "#926d88";
                    }
                    else {
                        return inRegion ? "#f29f29" : "#f29f29";
                    }
                case dal_constants_1.DalConstants.EmployeeStatus.AtHoliday:
                    return "#1CCBCB";
                case dal_constants_1.DalConstants.EmployeeStatus.AtOffTimePermission:
                    return "#D91FE6";
                case dal_constants_1.DalConstants.EmployeeStatus.AtOutOfWorkingHours:
                    return "#03820b";
                case dal_constants_1.DalConstants.EmployeeStatus.AtWorkingHours:
                    if (tolerated == true)
                        return "#f29f29";
                    if (inRegion == null || inRegion == undefined) {
                        return "#A2EEA7";
                    }
                    else {
                        return inRegion ? "#A2EEA7" : "#A2EEA7";
                    }
            }
        }
        else {
            switch (status) {
                case dal_constants_1.DalConstants.EmployeeStatus.AtBreak:
                    return "#be9fbf";
                case dal_constants_1.DalConstants.EmployeeStatus.AtHoliday:
                    return "#80b0b0";
                case dal_constants_1.DalConstants.EmployeeStatus.AtOffTimePermission:
                    if (inRegion == null || inRegion == undefined) {
                        return "#ef9df5";
                    }
                    else {
                        return inRegion ? "#ef9df5" : "#ef9df5";
                    }
                case dal_constants_1.DalConstants.EmployeeStatus.AtOutOfWorkingHours:
                    return "#CCCCCC";
                case dal_constants_1.DalConstants.EmployeeStatus.AtWorkingHours:
                    return "#F86F6F";
            }
        }
    }
}
exports.PdfReportWorkSummaryV2 = PdfReportWorkSummaryV2;
