"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.PdfReportRegionState = void 0;
const i18n_1 = __importDefault(require("i18n"));
const report_util_1 = require("../report.util");
const moment_1 = __importDefault(require("moment"));
const app_enums_1 = require("../../../app.enums");
const dal_manager_1 = require("../../../dal/dal.manager");
const report_pdf_report_creator_1 = require("../report-helpers/report.pdf-report-creator");
const report_color_enum_1 = require("../report-helpers/report.color.enum");
const report_pdf_cell_1 = require("../report-helpers/report.pdf-cell");
const report_filter_1 = require("../report-helpers/report.filter");
const business_report_export_1 = require("../business.report.export");
const uuid_1 = __importDefault(require("uuid"));
const generator_1 = require("../generator");
const report_pdf_cell_bar_1 = require("../report-helpers/report.pdf-cell-bar");
class PdfReportRegionState extends generator_1.ReportGenerator {
    constructor(request, locale) {
        super(request, locale);
        this._options = this._request.filter;
        this.locale = this._locale;
    }
    async generateReport() {
        this._data = await (0, business_report_export_1.generateRegionStateData)({
            organizationId: this._request.organizationId,
            requesterUserId: this._request.requesterUserId,
            regionId: this._options.regionId,
            filter: this._options,
        });
        let locale = this.locale;
        this.pdfReportCreator = new report_pdf_report_creator_1.PDFReportCreator(this._locale);
        this.organization = await dal_manager_1.dbManager.accessOrganization.getOrganizationLogoAndName(this._request.organizationId);
        this.createColumns();
        this.createRows();
        await this.createFilters();
        this._fileId = uuid_1.default.v4();
        let result = await this.pdfReportCreator.createPdf(this.organization.name, this.organization.logo, app_enums_1.enums.ReportCode.RegionState, ["auto", "auto", "auto", 80]);
        result.reportContainsEmptyData = this._request.reportTemplateId ? this._data.items.length === 0 : undefined;
        return result;
    }
    createColumns() {
        let locale = this.locale;
        let columns = [];
        columns.push(i18n_1.default.__({ phrase: "PDF-REPORT.name_surname", locale }));
        columns.push(i18n_1.default.__({ phrase: "PDF-REPORT.REGION_STATE_LAST_ACTION", locale }));
        columns.push(i18n_1.default.__({ phrase: "PDF-REPORT.access_point", locale }));
        columns.push(i18n_1.default.__({ phrase: "PDF-REPORT.status", locale }));
        this.pdfReportCreator.addColumns(columns);
    }
    createRows() {
        this._data.items.forEach((record) => {
            let cells = [];
            let cellForIdentity = this.createCellForIdentity(record);
            let cellForLastAction = this.createCellForLastAction(record);
            let cellForAccessPoint = this.createCellForAccessPoint(record);
            let cellForStatus = this.createCellForStatus(record);
            if (cellForIdentity != null) {
                cells.push(this.createPdfReportCell(cellForIdentity, true, null, null));
            }
            else {
                cells.push(this.createPdfReportCell("", false, null, null));
            }
            cells.push(this.createPdfReportCell(cellForAccessPoint, false, null, null));
            cells.push(this.createPdfReportCell(cellForLastAction, false, null, null));
            cells.push(this.createPdfReportCell(cellForStatus.description, false, null, cellForStatus.color));
            this.pdfReportCreator.addRow(cells);
        });
    }
    createPdfReportCell(text, isCaption, fontSize, color) {
        let cell = new report_pdf_cell_1.PdfReportCell();
        cell.text = text;
        if (fontSize != null)
            cell.fontSize = fontSize;
        if (color != null)
            cell.color = color;
        cell.isCaption = isCaption;
        return cell;
    }
    createPdfReportCellBar(text, alignment, barColor, width) {
        let cell = new report_pdf_cell_bar_1.PdfReportCellBar();
        if (text)
            cell.text = text;
        if (alignment)
            cell.alignment = alignment;
        if (barColor)
            cell.barColor = barColor;
        cell.width = width;
        return cell;
    }
    createFilterObject(title, value) {
        let filter = new report_filter_1.PdfReportFilter();
        filter.title = title;
        filter.value = value;
        return filter;
    }
    async createFilters() {
        let locale = this.locale;
        let filterResult = await dal_manager_1.dbManager.accessReport.collectRegionStateFilterInfo(this._options.regionId, this._options.userIds, this._request.organizationId);
        let filters = [];
        if (filterResult.region != null && filterResult.region.length > 0) {
            filters.push(this.createFilterObject(i18n_1.default.__({ phrase: "PDF-REPORT.REGION_STATE_REGION", locale }), filterResult.region[0].name == null ? "" : filterResult.region[0].name));
        }
        if (this._options.state != null) {
            filters.push(this.createFilterObject(i18n_1.default.__({ phrase: "PDF-REPORT.STATUS", locale }), this.getStateDescription(this._options.state)));
        }
        if (filterResult.userCaptions != null && filterResult.userCaptions.length > 0) {
            filters.push(this.createFilterObject(i18n_1.default.__({ phrase: "PDF-REPORT.IDENTITES", locale }), (0, report_util_1.parseCaptionForPdfFilter)(filterResult.userCaptions)));
        }
        if (this._options.status != null) {
            filters.push(this.createFilterObject(i18n_1.default.__({ phrase: "PDF-REPORT.REGION_STATE_USER_STATUS", locale }), this.getUserStatusDescription(this._options.status)));
        }
        this.pdfReportCreator.addFilters(filters);
    }
    createCellForIdentity(record) {
        if (record.userCaptions != null && record.userCaptions.length > 0)
            return (0, report_util_1.parseCaption)(record.userCaptions);
        else
            return null;
    }
    createCellForAccessPoint(record) {
        return record.accessControlPoint ? record.accessControlPoint.name : "";
    }
    createCellForLastAction(record) {
        if (record.lastActionUtc != null)
            return (0, moment_1.default)(record.lastActionUtc).locale(this.locale).format("DD/MM/YYYY HH:mm:ss");
        else
            return "";
    }
    createCellForStatus(record) {
        return {
            description: this.getStateDescription(record.state),
            color: this.getStateColor(record.state),
        };
    }
    getStateDescription(state) {
        let locale = this.locale;
        switch (state) {
            case app_enums_1.enums.AntiPassbackState.Unknown:
                return i18n_1.default.__({ phrase: "PDF-REPORT.REGION_STATE_UNSET", locale });
            case app_enums_1.enums.AntiPassbackState.In:
                return i18n_1.default.__({ phrase: "PDF-REPORT.REGION_STATE_IN", locale });
            case app_enums_1.enums.AntiPassbackState.Out:
                return i18n_1.default.__({ phrase: "PDF-REPORT.REGION_STATE_OUT", locale });
            default:
                return "";
        }
    }
    getUserStatusDescription(status) {
        let locale = this.locale;
        switch (status) {
            case app_enums_1.enums.IdentityStatusType.All:
                return i18n_1.default.__({ phrase: "PDF-REPORT.REGION_STATE_ALL", locale });
            case app_enums_1.enums.IdentityStatusType.Enabled:
                return i18n_1.default.__({ phrase: "PDF-REPORT.REGION_STATE_ACTIVE", locale });
            case app_enums_1.enums.IdentityStatusType.Disabled:
                return i18n_1.default.__({ phrase: "PDF-REPORT.REGION_STATE_PASSIVE", locale });
            default:
                return "";
        }
    }
    getStateColor(state) {
        switch (state) {
            case app_enums_1.enums.AntiPassbackState.Unknown:
                return report_color_enum_1.ReportColors.Inverse;
            case app_enums_1.enums.AntiPassbackState.In:
                return report_color_enum_1.ReportColors.Success;
            case app_enums_1.enums.AntiPassbackState.Out:
                return report_color_enum_1.ReportColors.Danger;
            default:
                return report_color_enum_1.ReportColors.Inverse;
        }
    }
}
exports.PdfReportRegionState = PdfReportRegionState;
