"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.PdfIdentityList = void 0;
const sharp_1 = __importDefault(require("sharp"));
const fs_1 = __importDefault(require("fs"));
const path_1 = __importDefault(require("path"));
const uuid_1 = __importDefault(require("uuid"));
const pdfmake_1 = __importDefault(require("pdfmake/build/pdfmake"));
require("pdfmake/build/vfs_fonts");
const app_config_1 = require("../../../app.config");
const moment_1 = __importDefault(require("moment"));
const report_util_1 = require("../report.util");
const i18n_1 = __importDefault(require("i18n"));
const dal_manager_1 = require("../../../dal/dal.manager");
const app_enums_1 = require("../../../app.enums");
const app_logs_1 = require("../../../app.logs");
const worker_threads_1 = require("worker_threads");
const generator_1 = require("../generator");
class PdfIdentityList extends generator_1.ReportGenerator {
    constructor(request, locale) {
        super(request, locale);
        this.minuteWidth = 300 / (24 * 60);
        this._options = request.filter;
        this._pool = dal_manager_1.dbManager.poolMain;
    }
    async getData(query) {
        const { rows } = await dal_manager_1.dbManager.organizationTransaction(async (trx) => {
            return await trx.query(query.query, query.queryParams);
        }, this._request.requesterUserId, this._request.organizationId);
        return rows;
    }
    checkNull(value) {
        if (value) {
            return value;
        }
        else {
            return "";
        }
    }
    async generateReport() {
        if (!worker_threads_1.isMainThread) {
            app_logs_1.logger.debug("[Worker Thread] Generating user list pdf report");
        }
        this.organization = await dal_manager_1.dbManager.accessOrganization.getOrganizationLogoAndName(this._request.organizationId);
        let organizationName = (await dal_manager_1.dbManager.accessOrganization.getOrganizationBasic(this._request.organizationId)).name;
        if (!worker_threads_1.isMainThread) {
            app_logs_1.logger.debug("[Worker Thread] Organization Name : " + organizationName);
        }
        let requestedUserCaption = await dal_manager_1.dbManager.accessUser.getUserOrganizationCaptionLines(this._request.organizationId, [this._request.requesterUserId]);
        if (requestedUserCaption.length < 1) {
            throw new Error("Not found");
        }
        if (!worker_threads_1.isMainThread) {
            app_logs_1.logger.debug("[Worker Thread] Requester User : " + requestedUserCaption[0].captionLines[0].text[0]);
        }
        let query = dal_manager_1.dbManager.accessIdentity.identityListQueryBuilder(this._options, this._request.organizationId, false);
        this._data = await this.getData(query);
        let rows = [];
        if (!worker_threads_1.isMainThread) {
            app_logs_1.logger.debug("[Worker Thread] Fetching data of %d rows", this._data.length);
        }
        let headerNames = [
            {
                text: i18n_1.default.__({ phrase: "EXCEL-REPORT.NAME", locale: this._locale }),
                style: "tableHeader",
            },
            {
                text: i18n_1.default.__({ phrase: "EXCEL-REPORT.SURNAME", locale: this._locale }),
                style: "tableHeader",
            },
            {
                text: i18n_1.default.__({ phrase: "EXCEL-REPORT.EMAIL", locale: this._locale }),
                style: "tableHeader",
            },
            {
                text: i18n_1.default.__({ phrase: "EXCEL-REPORT.PHONE_NUMBER", locale: this._locale }),
                style: "tableHeader",
            },
            {
                text: i18n_1.default.__({ phrase: "EXCEL-REPORT.ADDRESS", locale: this._locale }),
                style: "tableHeader",
            },
            {
                text: i18n_1.default.__({ phrase: "EXCEL-REPORT.ROLE", locale: this._locale }),
                style: "tableHeader",
            },
            {
                text: i18n_1.default.__({ phrase: "EXCEL-REPORT.USERNAME", locale: this._locale }),
                style: "tableHeader",
            },
            {
                text: i18n_1.default.__({ phrase: "EXCEL-REPORT.UNIQUE_ID", locale: this._locale }),
                style: "tableHeader",
            },
            {
                text: i18n_1.default.__({ phrase: "EXCEL-REPORT.ORGANIZATION_UNITS", locale: this._locale }),
                style: "tableHeader",
            },
            {
                text: i18n_1.default.__({ phrase: "EXCEL-REPORT.USER_GROUPS", locale: this._locale }),
                style: "tableHeader",
            },
        ];
        rows.push(headerNames);
        this._data.forEach((result) => {
            let rowObj = [];
            rowObj.push(result.organizationProfile.name?.toUpperCase() ?? "");
            rowObj.push(result.organizationProfile.surname?.toUpperCase() ?? "");
            rowObj.push(result.organizationProfile.email ?? "");
            rowObj.push(result.organizationProfile.phoneNumber ?? "");
            rowObj.push(result.organizationProfile.address ?? "");
            rowObj.push(result.organizationProfile.roleName ? i18n_1.default.__({ phrase: result.organizationProfile.roleName.toUpperCase(), locale: this._locale }) : "");
            rowObj.push(result.organizationProfile.username ?? "");
            rowObj.push(result.organizationProfile.uniqueId ?? "");
            rowObj.push((result.organizationUnits || [])
                .map((ou) => ou.name)
                .join(", ")
                .toUpperCase() ?? "");
            rowObj.push((result.userGroups || [])
                .map((ug) => ug.name)
                .join(", ")
                .toUpperCase() ?? "");
            rows.push(rowObj);
        });
        let header;
        if (this.organization == null) {
            header = {
                table: {
                    widths: ["*"],
                    headerRows: 1,
                    body: [
                        [
                            { text: i18n_1.default.__({ phrase: "EXCEL-REPORT.USER_LIST", locale: this._locale }) + "\n", style: "header", alignment: "center" },
                            { text: (0, moment_1.default)().locale(this._locale).format("LL"), style: "date", alignment: "right" },
                        ],
                    ],
                },
                layout: "noBorders",
            };
        }
        if (this.organization.logo == null) {
            header = {
                table: {
                    widths: ["auto", "*"],
                    headerRows: 1,
                    body: [
                        [
                            { text: this.organization.name, style: "organizationName", alignment: "left" },
                            [
                                { text: i18n_1.default.__({ phrase: "EXCEL-REPORT.USER_LIST", locale: this._locale }) + "\n", style: "header", alignment: "center" },
                                { text: (0, moment_1.default)().locale(this._locale).format("LL"), style: "date", alignment: "right" },
                            ],
                        ],
                    ],
                },
                layout: "noBorders",
            };
        }
        else {
            let logoImage = await (0, sharp_1.default)(this.organization.logo, { failOnError: false }).resize(240, 60).toFormat("png").toBuffer();
            let logoNew = (await (0, sharp_1.default)({
                create: {
                    width: 240,
                    height: 60,
                    channels: 4,
                    background: { r: 1, g: 50, b: 91, alpha: 0.9 },
                },
                failOnError: false,
            })
                .png()
                .composite([{ input: logoImage }])
                .toBuffer()).toString("base64");
            header = {
                table: {
                    widths: ["auto", "*"],
                    headerRows: 1,
                    body: [
                        [
                            {
                                image: "data:image/png;base64," + logoNew,
                                width: report_util_1.ReportConstants.widthOrganizationLogo,
                                alignment: "left",
                            },
                            [
                                { text: i18n_1.default.__({ phrase: "EXCEL-REPORT.USER_LIST", locale: this._locale }) + "\n", style: "header", alignment: "center" },
                                { text: (0, moment_1.default)().locale(this._locale).format("LL"), style: "date", alignment: "right" },
                            ],
                        ],
                    ],
                },
                layout: "noBorders",
            };
        }
        this._dd = {
            pageOrientation: "landscape",
            footer: ((currentPage, pageCount) => {
                return {
                    columns: [
                        {
                            text: i18n_1.default.__({ phrase: "PDF-REPORT.report_create_date", locale: this._locale }) +
                                ": " +
                                (0, moment_1.default)().locale(this._locale).format("LL") +
                                " " +
                                (0, moment_1.default)().locale(this._locale).format("HH:mm"),
                            alignment: "left",
                            fontSize: "8",
                            margin: [40, 0],
                        },
                        { text: currentPage.toString() + " / " + pageCount, alignment: "right", fontSize: "10", margin: [40, 0] },
                    ],
                };
            }),
            content: [
                header,
                {
                    style: "tableExample",
                    table: {
                        headerRows: 1,
                        dontBreakRows: true,
                        keepWithHeaderRows: 1,
                        widths: ["10%", "10%", "10%", "10%", "10%", "10%", "10%", "10%", "10%", "10%", "10%"],
                        body: rows,
                    },
                },
            ],
            styles: {
                header: {
                    fontSize: 8,
                    bold: true,
                    margin: [0, 0, 0, 10],
                    alignment: "left",
                },
                subheader: {
                    fontSize: 8,
                    bold: true,
                    margin: [0, 10, 0, 5],
                },
                tableExample: {
                    margin: [0, 5, 0, 15],
                },
                tableHeader: {
                    bold: true,
                    fontSize: 8,
                    color: "black",
                },
                status: {
                    margin: [0, 15],
                    fontSize: 8,
                    color: "#cccccc",
                },
                center: {
                    alignment: "center",
                },
                date: {
                    fontSize: 8,
                    color: "black",
                    alignment: "right",
                },
                organizationName: {
                    fontSize: 8,
                    bold: true,
                    alignment: "left",
                },
            },
            defaultStyle: {
                fontSize: 8,
            },
        };
        let fileId = uuid_1.default.v4();
        let filePath = path_1.default.join(app_config_1.appConfig.tmpDirectory, fileId + ".pdf");
        let pdf = pdfmake_1.default.createPdf(this._dd);
        await new Promise((resolve, reject) => {
            pdf.getBuffer((result) => {
                fs_1.default.writeFileSync(filePath, result);
                resolve(filePath);
            });
        });
        if (!worker_threads_1.isMainThread) {
            app_logs_1.logger.debug("[Worker Thread] User list pdf report is ready -> %s", filePath);
        }
        return {
            format: app_enums_1.enums.ReportFormat.Pdf,
            preparedReportId: fileId,
            filePath: filePath,
            notificationType: undefined,
            reportType: app_enums_1.enums.ReportCode.Identity,
            sendFileInAttachments: undefined,
        };
    }
}
exports.PdfIdentityList = PdfIdentityList;
