"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.PdfReportAccessLogs = void 0;
const fs_1 = __importDefault(require("fs"));
const path_1 = __importDefault(require("path"));
const uuid_1 = __importDefault(require("uuid"));
const pdfmake_1 = __importDefault(require("pdfmake/build/pdfmake"));
require("pdfmake/build/vfs_fonts");
const app_config_1 = require("../../../app.config");
const moment_1 = __importDefault(require("moment"));
const report_util_1 = require("../report.util");
const i18n_1 = __importDefault(require("i18n"));
const app_enums_1 = require("../../../app.enums");
const sharp_1 = __importDefault(require("sharp"));
const dal_manager_1 = require("../../../dal/dal.manager");
const app_logs_1 = require("../../../app.logs");
const api_error_1 = require("../../../api/api.error");
const lodash_1 = __importDefault(require("lodash"));
const worker_threads_1 = require("worker_threads");
const generator_1 = require("../generator");
class PdfReportAccessLogs extends generator_1.ReportGenerator {
    constructor(request, locale) {
        super(request, locale);
        this._options = request.filter;
    }
    async generateReport() {
        let excelRows = [];
        excelRows.push(this.createColumns(this._locale));
        this._organizationLogoAndName = await dal_manager_1.dbManager.accessOrganization.getOrganizationLogoAndName(this._request.organizationId);
        let fetched = 0;
        if (this._options.authorizeForFilter) {
            let authResultForFilter = await dal_manager_1.dbManager.accessFunctions.dbFuncAuthorizeUserFor({
                organizationId: this._request.organizationId,
                userId: this._request.requesterUserId,
                organizationUnitIds: this._options.organizationUnitIds,
                accessControlPointIds: this._options.accessControlPointIds,
                userIds: this._options.userIds,
                userGroupIds: this._options.userGroupIds,
            });
            if (!authResultForFilter.result) {
                throw (0, api_error_1.generateForbiddenError)({ message: "Unauthorized filter items" });
            }
            if (lodash_1.default.isEmpty(this._options.organizationUnitIds) && lodash_1.default.isEmpty(this._options.userGroupIds) && lodash_1.default.isEmpty(this._options.userIds)) {
                if (this._options.permittedUnitIds) {
                    this._options.organizationUnitIds = this._options.permittedUnitIds;
                }
                else {
                    const userOrganizationUnits = await dal_manager_1.dbManager.armondb
                        .withSchema(this._request.organizationId)
                        .from("users as u")
                        .innerJoin("userOrganizations as uo", "uo.userId", "u.id")
                        .innerJoin("userOrganizationOrganizationUnits as uoou", "uoou.userOrganizationId", "uo.id")
                        .innerJoin("organizationUnits as ou", (join) => {
                        join.on(dal_manager_1.dbManager.armondb.raw(`ou."ancestorIds" like '%' || uoou."organizationUnitId" || '%'`)).orOn("ou.id", "uoou.organizationUnitId");
                    })
                        .whereNull("u.deletedAt")
                        .whereNull("uoou.deletedAt")
                        .whereNull("uo.deletedAt")
                        .where("uo.organizationId", this._request.organizationId)
                        .where("u.id", this._request.requesterUserId)
                        .select("ou.id");
                    this._options.organizationUnitIds = userOrganizationUnits.map((u) => u.id);
                }
            }
        }
        let filterUsers = await dal_manager_1.dbManager.accessFunctions.dbFuncCollectUsersForAccessReportFilter({
            organizationId: this._request.organizationId,
            userIds: this._options.userIds,
            organizationUnitIds: this._options.organizationUnitIds,
            userGroupIds: this._options.userGroupIds,
            filterOrganizationUnitMembersHierarchically: this._options.filterOrganizationUnitMembersHierarchically,
        });
        let filterUserIds = (filterUsers && filterUsers?.length) > 0 ? filterUsers.map((f) => f.userId) : [];
        this._options.userIds = filterUserIds;
        const filter = {
            ...this._options,
            ...{
                take: app_config_1.appConfig.reportExportRowLimit,
                skip: undefined,
            },
        };
        await dal_manager_1.dbManager.accessAccessLog.getAccessLogsReportNew(this._request.organizationId, filter, async (rows) => {
            if (!worker_threads_1.isMainThread) {
                app_logs_1.logger.debug("[Worker Thread] Fetch data %s rows", rows.items?.length);
            }
            fetched += rows.items?.length;
            if (fetched < app_config_1.appConfig.reportExportRowLimit + 1) {
                this.createRows(rows, excelRows);
            }
        });
        await this.createReportTemplate(excelRows);
        if (!worker_threads_1.isMainThread) {
            app_logs_1.logger.debug("[Worker Thread] Pdf report is ready -> " + this._filePath);
        }
        return {
            preparedReportId: this._fileId,
            filePath: this._filePath,
            format: app_enums_1.enums.ReportFormat.Pdf,
            notificationType: undefined,
            reportType: app_enums_1.enums.ReportCode.AccessLogs,
            sendFileInAttachments: undefined,
            reportContainsEmptyData: this._request.reportTemplateId ? fetched === 0 : undefined,
        };
    }
    async createReportTemplate(rows) {
        await this.createDocumentDefinitionOfPdf(this._locale, rows);
        return this.createPdf();
    }
    createPdf() {
        this._fileId = uuid_1.default.v4();
        this._filePath = path_1.default.join(app_config_1.appConfig.tmpDirectory, this._fileId + ".pdf");
        let pdf = pdfmake_1.default.createPdf(this._dd);
        return new Promise((resolve, reject) => {
            pdf.getBuffer((result) => {
                fs_1.default.writeFileSync(this._filePath, result);
                resolve(this._filePath);
            });
        });
    }
    createColumns(locale) {
        let name = i18n_1.default.__({ phrase: "PDF-REPORT.name_surname", locale });
        let accessTime = i18n_1.default.__({ phrase: "PDF-REPORT.access_time", locale });
        let accessPoint = i18n_1.default.__({ phrase: "PDF-REPORT.access_point", locale });
        let status = i18n_1.default.__({ phrase: "PDF-REPORT.status", locale });
        return [
            { text: name, style: "tableHeader" },
            { text: accessTime, style: "tableHeader" },
            { text: accessPoint, style: "tableHeader" },
            { text: status, style: "tableHeader" },
        ];
    }
    createRows(rows, excelRows) {
        rows.items.forEach((record) => {
            let rowObj = [];
            let cellForIdentity = this.createCellForIdentity(record, this._locale);
            let cellForAccessTime = this.createCellForAccessTime(record, this._locale);
            let cellForAccessPoint = this.createCellForAccessPoint(record, this._locale);
            let cellForStatus = this.createCellForStatus(record, this._locale);
            let accessTime = [];
            let accessPoint = [];
            let status = [];
            accessTime.push({ text: cellForAccessTime.mainText + "\n", fontSize: 10, color: "black", margin: [0, 15] });
            accessTime.push({ text: cellForAccessTime.subText, fontSize: 8, color: "gray", margin: [0, 15] });
            accessPoint.push({ text: cellForAccessPoint.an + "\n", fontSize: 10, color: "black", margin: [0, 15] });
            status.push({ text: cellForStatus.description + "\n", fontSize: 10, color: "gray", margin: [0, 15] });
            status.push({ text: cellForStatus.reason + "\n", fontSize: 10, color: "gray", margin: [0, 15] });
            status.push({ text: cellForStatus.log + "\n", fontSize: 10, color: cellForStatus.statusColor, margin: [0, 15] });
            status.push({ text: cellForStatus.statusInfo + "\n", fontSize: 10, color: cellForStatus.statusColor, margin: [0, 15] });
            status.push({ text: cellForStatus.credentialDescription, fontSize: 10, color: "#103652", margin: [0, 15] });
            if (cellForIdentity.caption != null) {
                rowObj.push(cellForIdentity.caption);
            }
            else {
                rowObj.push({
                    text: cellForIdentity.result,
                    fontSize: 10,
                    color: "black",
                    margin: [10, 15, 0, 15],
                });
            }
            rowObj.push({
                text: accessTime,
            });
            rowObj.push({
                text: accessPoint,
            });
            rowObj.push({
                text: status,
            });
            excelRows.push(rowObj);
        });
    }
    createCellForIdentity(record, locale) {
        let result = "";
        let caption;
        if (record.userCaptions != null && record.userCaptions.length > 0)
            caption = (0, report_util_1.parseCaption)(record.userCaptions);
        else {
            if (record.identity && !record.userCaptions) {
                result += record.identity.organizationUnits;
            }
            else {
                result += "";
            }
            if (!record.identity && !record.isExitButton) {
                result += i18n_1.default.__({ phrase: "PDF-REPORT.unknown_identity", locale });
            }
            if (!record.identity && record.isExitButton) {
                result += i18n_1.default.__({ phrase: "PDF-REPORT.identity_exit_button", locale });
            }
            if (!record.identity && !record.isExitButton && record.unknownCredentialData) {
                result += "-" + record.unknownCredentialData;
            }
        }
        return {
            result: result,
            caption: caption,
        };
    }
    createCellForAccessTime(record, locale) {
        return {
            mainText: (0, moment_1.default)(record.utc).locale(locale).format("HH:mm:ss"),
            subText: (0, moment_1.default)(record.utc).locale(locale).format("LL"),
        };
    }
    createCellForAccessPoint(record, locale) {
        let visual = "";
        let counterSensorData = "";
        let statusSensorData = "";
        return {
            an: record.accessControlPoint != null && record.accessControlPoint.name != null ? record.accessControlPoint.name : "",
            visual: visual,
            counterSensorData: counterSensorData,
            statusSensorData: statusSensorData,
        };
    }
    createCellForStatus(record, locale) {
        let log = "";
        if (record.isVisitor) {
            log += i18n_1.default.__({ phrase: "PDF-REPORT.log_visitor", locale }) + "\r\n";
        }
        if (record.isRemoteAccess) {
            log += i18n_1.default.__({ phrase: "PDF-REPORT.log_remote_access", locale }) + "\r\n";
        }
        if (record.isManuallyLogged) {
            log += i18n_1.default.__({ phrase: "PDF-REPORT.log_manual", locale }, { manualCreator: record.manualRecordCreatorCaptions[0].text[0] }) + "\r\n";
        }
        if (record.isExitButton) {
            log += i18n_1.default.__({ phrase: "PDF-REPORT.log_exit_button", locale }) + "\r\n";
        }
        let translationKey = this.getStatusDescription(record.result, record.direction);
        let translatedReason = this.getStatusReason(record.result);
        let isSuccess = this.isSuccess(record);
        let credentialDescription = "";
        if (record.credentials && record.credentials.length > 0) {
            let visibleCredentialData = [];
            record.credentials.forEach((credential) => {
                visibleCredentialData.push((0, report_util_1.getCredentialDataVisibility)(credential.type));
            });
            record.credentials.forEach((credential, index) => {
                let credentialType = report_util_1.ReportConstants.CredentialTypes.find((item) => item.type === credential.type);
                credentialDescription += "\n" + i18n_1.default.__({ phrase: "PDF-REPORT." + credentialType.description, locale }) + " ";
                if (credential.data && visibleCredentialData[index]) {
                    credentialDescription += "(" + credential.data + ")";
                }
            });
        }
        return {
            statusColor: this.getStatusColor(isSuccess),
            description: record.accessControlPoint != null && record.accessControlPoint.name != null
                ? i18n_1.default.__({ phrase: "PDF-REPORT." + translationKey, locale }, {
                    an: record.accessControlPoint.name,
                })
                : i18n_1.default.__({ phrase: "PDF-REPORT." + translationKey, locale }),
            reason: translatedReason,
            log: log,
            statusInfo: (0, report_util_1.getAccessLogStatusInfo)(record.direction, this._locale, isSuccess),
            credentialDescription: credentialDescription,
        };
    }
    isSuccess(record) {
        return record.result == app_enums_1.enums.AccessLogReason.Success || (record.isManuallyLogged && record.result == app_enums_1.enums.AccessLogReason.Manual);
    }
    getStatusColor(isSuccess) {
        return isSuccess ? "#62A420" : "#E62700";
    }
    getStatusDescription(reason, direction) {
        return (0, report_util_1.getDescriptionAccessLogItem)(reason, direction);
    }
    getStatusReason(reason) {
        return (0, report_util_1.getLocalizedTextOfAccessLogReason)(reason, this._locale);
    }
    async createDocumentDefinitionOfPdf(locale, rows) {
        let organization;
        if (this._organizationLogoAndName == null) {
            organization = "";
        }
        if (this._organizationLogoAndName.logo == null) {
            organization = { text: this._organizationLogoAndName.name, style: "organizationName" };
        }
        else {
            let logoImage = await (0, sharp_1.default)(this._organizationLogoAndName.logo, { failOnError: false }).resize(240, 60).toFormat("png").toBuffer();
            let logoNew = (await (0, sharp_1.default)({
                create: {
                    width: 240,
                    height: 60,
                    channels: 4,
                    background: { r: 1, g: 50, b: 91, alpha: 0.9 },
                },
                failOnError: false,
            })
                .png()
                .composite([{ input: logoImage }])
                .toBuffer()).toString("base64");
            organization = {
                image: "data:image/png;base64," + logoNew,
                width: report_util_1.ReportConstants.widthOrganizationLogo,
                alignment: "left",
            };
        }
        this._dd = {
            footer: ((currentPage, pageCount) => {
                return {
                    columns: [
                        {
                            text: i18n_1.default.__({ phrase: "PDF-REPORT.report_create_date", locale }) + ": " + (0, moment_1.default)().locale(locale).format("DD/MM/YYYY HH:mm") + " ",
                            alignment: "left",
                            fontSize: "10",
                            margin: [40, 0],
                        },
                        { text: currentPage.toString() + " / " + pageCount, alignment: "right", fontSize: "10", margin: [40, 0] },
                    ],
                };
            }),
            content: [
                organization,
                { text: i18n_1.default.__({ phrase: "PDF-REPORT.access_logs", locale }) + "\n", style: "header" },
                await this.createFilters(locale),
                {
                    style: "tableStyle",
                    layout: {
                        fillColor: function (rowIndex, node, columnIndex) {
                            return rowIndex % 2 === 0 ? report_util_1.ReportConstants.evenRowColor : null;
                        },
                    },
                    table: {
                        headerRows: 1,
                        dontBreakRows: true,
                        keepWithHeaderRows: 1,
                        widths: ["auto", "auto", "auto", "auto"],
                        body: rows,
                    },
                },
            ],
            styles: {
                header: {
                    fontSize: 18,
                    bold: true,
                    margin: [0, 0, 0, 10],
                    alignment: "center",
                },
                tableStyle: {
                    margin: [0, 5, 0, 15],
                },
                tableHeader: {
                    bold: true,
                    fontSize: 10,
                    color: "black",
                },
                status: {
                    margin: [0, 15],
                    fontSize: 10,
                    color: "gray",
                },
                date: {
                    fontSize: 11,
                    color: "black",
                    alignment: "right",
                },
                organizationName: {
                    fontSize: 11,
                    bold: true,
                    alignment: "left",
                },
                filter: {
                    fontSize: 10,
                    color: "black",
                    alignment: "left",
                },
                filterBold: {
                    fontSize: 10,
                    color: "black",
                    bold: true,
                    alignment: "left",
                },
            },
            defaultStyle: {},
        };
    }
    async createFilters(locale) {
        let body = [];
        if (this._options.startUtc != null && this._options.endUtc) {
            let _result = [];
            _result.push({ text: i18n_1.default.__({ phrase: "PDF-REPORT.ACCESS_LOGS_DATE", locale }) + " : ", style: "filterBold" });
            _result.push({
                text: (0, moment_1.default)(this._options.startUtc).locale(locale).format("DD/MM/YYYY HH:mm") + " - " + (0, moment_1.default)(this._options.endUtc).locale(locale).format("DD/MM/YYYY HH:mm"),
                style: "filter",
            });
            body.push(_result);
        }
        else if (this._options.startUtc != null) {
            let _result = [];
            _result.push({ text: i18n_1.default.__({ phrase: "PDF-REPORT.ACCESS_LOGS_DATE", locale }) + " : ", style: "filterBold" });
            _result.push({
                text: (0, moment_1.default)(this._options.startUtc).locale(locale).format("DD/MM/YYYY HH:mm"),
                style: "filter",
            });
            body.push(_result);
        }
        if (this._options.accessResult) {
            let _result = [];
            _result.push({ text: i18n_1.default.__({ phrase: "PDF-REPORT.STATUS", locale }) + " : ", style: "filterBold" });
            let key = report_util_1.ReportConstants.AccessResultTypes.filter((item) => this._options.accessResult === item.type);
            _result.push({
                text: i18n_1.default.__({ phrase: key.description, locale }),
                style: "filter",
            });
            body.push(_result);
        }
        if (this._options.direction) {
            let _result = [];
            _result.push({ text: i18n_1.default.__({ phrase: "PDF-REPORT.DIRECTION", locale }) + " : ", style: "filterBold" });
            let key = report_util_1.ReportConstants.Direction.filter((item) => this._options.direction === item.type);
            _result.push({
                text: i18n_1.default.__({ phrase: key.description, locale }),
                style: "filter",
            });
            body.push(_result);
        }
        let filterResult = await dal_manager_1.dbManager.accessReport.collectAccessLogReportFilterInfo(this._options.userIds, this._options.userGroupIds, this._options.organizationUnitIds, this._options.accessControlPointIds, this._options.regionIds, this._request.organizationId);
        if (filterResult.userCaptions != null && filterResult.userCaptions.length > 0) {
            let _result = [];
            let _res = "";
            filterResult.userCaptions.forEach((userCaption, index) => {
                _res += "(";
                userCaption.captionLines.forEach((item, _index) => {
                    if (item.text != null && item.text != "__avatar") {
                        item.text.forEach((element) => {
                            if (element != null)
                                _res += element + (_index < filterResult.userCaptions.length - 1 ? " " : ")");
                        });
                    }
                });
                _res += index < filterResult.userCaptions.length - 1 ? ", " : "";
            });
            _result.push({ text: i18n_1.default.__({ phrase: "PDF-REPORT.IDENTITES", locale }) + " : ", style: "filterBold" });
            _result.push({
                text: _res,
                style: "filter",
            });
            body.push(_result);
        }
        if (filterResult.userGroups != null && filterResult.userGroups.length > 0) {
            let _result = [];
            let res = "";
            filterResult.userGroups.forEach((userGroup, index) => {
                res += userGroup.name + (index < filterResult.userGroups.length - 1 ? ", " : " ");
            });
            _result.push({ text: i18n_1.default.__({ phrase: "PDF-REPORT.USER_GROUPS", locale }) + " : ", style: "filterBold" });
            _result.push({
                text: res,
                style: "filter",
            });
            body.push(_result);
        }
        if (filterResult.organizationUnits != null && filterResult.organizationUnits.length > 0) {
            let _result = [];
            let res = "";
            filterResult.organizationUnits.forEach((org, index) => {
                res += org.name + (index < filterResult.organizationUnits.length - 1 ? ", " : " ");
            });
            _result.push({ text: i18n_1.default.__({ phrase: "PDF-REPORT.ORGANIZATION_UNITS", locale }) + " : ", style: "filterBold" });
            _result.push({
                text: res,
                style: "filter",
            });
            body.push(_result);
        }
        if (filterResult.accessControlPoints != null && filterResult.accessControlPoints.length > 0) {
            let _result = [];
            let res = "";
            filterResult.accessControlPoints.forEach((acp, index) => {
                res += acp.name + (index < filterResult.accessControlPoints.length - 1 ? ", " : " ");
            });
            _result.push({ text: i18n_1.default.__({ phrase: "PDF-REPORT.ACCESS_CONTROL_POINTS", locale }) + " : ", style: "filterBold" });
            _result.push({
                text: res,
                style: "filter",
            });
            body.push(_result);
        }
        if (filterResult.regions != null && filterResult.regions.length > 0) {
            let _result = [];
            let res = "";
            filterResult.regions.forEach((acp, index) => {
                res += acp.name + (index < filterResult.regions.length - 1 ? ", " : " ");
            });
            _result.push({ text: i18n_1.default.__({ phrase: "EXCEL-REPORT.REGIONS", locale }) + " : ", style: "filterBold" });
            _result.push({
                text: res,
                style: "filter",
            });
            body.push(_result);
        }
        if (this._options.reasons != null && this._options.reasons.length > 0) {
            let _result = [];
            let res = "";
            this._options.reasons.forEach((reason, index) => {
                let key = report_util_1.ReportConstants.AccessLogReason.filter((item) => reason === item.type);
                res += i18n_1.default.__({ phrase: key.description, locale }) + (index < this._options.reasons.length - 1 ? ", " : " ");
            });
            _result.push({ text: i18n_1.default.__({ phrase: "PDF-REPORT.ACCESS_REASONS", locale }) + " : ", style: "filterBold" });
            _result.push({
                text: res,
                style: "filter",
            });
            body.push(_result);
        }
        if (this._options.credentialData != null && this._options.credentialData != "") {
            let _result = [];
            _result.push({ text: i18n_1.default.__({ phrase: "PDF-REPORT.TITLE_CREDENTIAL", locale }) + " : ", style: "filterBold" });
            _result.push({
                text: this._options.credentialData,
                style: "filter",
            });
            body.push(_result);
        }
        if (this._options.showOnlyVisitors) {
            let _result = [];
            _result.push({
                text: i18n_1.default.__({ phrase: "PDF-REPORT.ACCESS_LOGS_ONLY_VISITORS", locale }),
                style: "filterBold",
            });
            _result.push({
                text: "",
                style: "filter",
            });
            body.push(_result);
        }
        if (this._options.showOnlyExitButtons) {
            let _result = [];
            _result.push({
                text: i18n_1.default.__({ phrase: "PDF-REPORT.ACCESS_LOGS_ONLY_EXIT_BUTTONS", locale }),
                style: "filterBold",
            });
            _result.push({
                text: "",
                style: "filter",
            });
            body.push(_result);
        }
        if (this._options.showOnlyRemoteAccess) {
            let _result = [];
            _result.push({
                text: i18n_1.default.__({ phrase: "PDF-REPORT.ACCESS_LOGS_ONLY_REMOTE_ACCESS", locale }),
                style: "filterBold",
            });
            _result.push({
                text: "",
                style: "filter",
            });
            body.push(_result);
        }
        if (this._options.showOnlyManuallyInserted) {
            let _result = [];
            _result.push({
                text: i18n_1.default.__({ phrase: "PDF-REPORT.ACCESS_LOGS_ONLY_MANUALLY_INSERTED", locale }),
                style: "filterBold",
            });
            _result.push({
                text: "",
                style: "filter",
            });
            body.push(_result);
        }
        if (this._options.manualRecordCreatorUserId) {
            let manualCreator = "";
            const { caption: userCaptions } = await dal_manager_1.dbManager.accessRedisCache.getUserBadgeCache({
                organizationId: this._request.organizationId,
                userId: this._options.manualRecordCreatorUserId,
            });
            if (userCaptions.length > 0) {
                manualCreator = userCaptions[0].text[0];
            }
            let _result = [];
            _result.push({
                text: i18n_1.default.__({ phrase: "REPORT.ACCESS_LOGS_MANUAL_LOG_CREATOR", locale }) + " : ",
                style: "filterBold",
            });
            _result.push({
                text: manualCreator,
                style: "filter",
            });
            body.push(_result);
        }
        return {
            table: {
                heights: 20,
                body,
            },
            layout: "noBorders",
        };
    }
}
exports.PdfReportAccessLogs = PdfReportAccessLogs;
