"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.prepareReportArgumentsForScheduledJob = exports.CustomReportGenerator = exports.ReportGenerator = void 0;
const fs_1 = __importDefault(require("fs"));
const moment_1 = __importDefault(require("moment"));
const path_1 = __importDefault(require("path"));
const i18n_1 = __importDefault(require("i18n"));
const app_config_1 = require("../../app.config");
const app_enums_1 = require("../../app.enums");
const app_logs_1 = require("../../app.logs");
const dal_constants_1 = require("../../dal/dal.constants");
const dal_manager_1 = require("../../dal/dal.manager");
const excel_report_constants_1 = require("./excel.report.constants");
const controller_report_1 = require("../../api/pacs/v2/controllers/controller.report");
class ReportGenerator {
    constructor(request, locale) {
        this._request = request;
        this._locale = locale;
    }
    getRequest() {
        return this._request;
    }
    setDataCell11(params) {
        let c = params.ws.getCell(params.rowIndex, params.colIndex);
        c.value = params.value;
        c.alignment = {
            horizontal: params.alignment ? params.alignment : "left",
            vertical: "middle",
            wrapText: true,
        };
        if (params.border) {
            c.border = params.border;
        }
        if (params.numFmt)
            c.numFmt = params.numFmt;
        if (params.backgroundColorCode)
            c.fill = {
                type: "pattern",
                pattern: "solid",
                fgColor: { argb: params.backgroundColorCode },
            };
        if (params.isTextWhite)
            c.font = {
                color: { argb: "FFFFFF" },
                bold: true,
            };
        if (params.newLineCount > 1)
            params.ws.getRow(params.rowIndex).height = params.newLineCount * 15;
        if (params.textColorCode)
            c.font = {
                color: { argb: params.textColorCode },
            };
        if (params.newLineCount > 1)
            params.ws.getRow(params.rowIndex).height = params.newLineCount * 15;
    }
    addFilterCell11(ws, row, col, value, border, font, numFmt, newLineCount) {
        let c = ws.getCell(row, col);
        c.value = value;
        c.alignment = {
            horizontal: "left",
            vertical: "middle",
            wrapText: true,
        };
        c.border = border;
        c.font = font;
        c.numFmt = numFmt;
        if (newLineCount > 1)
            ws.getRow(row).height = newLineCount * 15;
    }
    addInitCell11(ws, row, col, value, border, height) {
        let c = ws.getCell(row, col);
        c.value = value;
        c.alignment = {
            horizontal: "center",
            vertical: "middle",
            wrapText: true,
        };
        c.font = {
            bold: true,
        };
        c.border = border;
        const rowCell = ws.getRow(row);
        rowCell.height = height;
    }
    addCellsToRow(values, colIndex, rowIndex, ws, border) {
        for (let i = 0; i < values.length; i++) {
            this.setDataCell11({
                rowIndex: rowIndex,
                colIndex: colIndex++,
                value: values[i],
                ws: ws,
                border: border ?? ReportGenerator.Constants.Styling.Dotted,
            });
        }
        return colIndex;
    }
    addCellsToRowWithOptions(values, colIndex, rowIndex, ws) {
        for (let i = 0; i < values.length; i++) {
            this.setDataCell11({
                rowIndex: rowIndex,
                colIndex: colIndex++,
                value: values[i].value,
                ws: ws,
                border: values[i].border ?? ReportGenerator.Constants.Styling.Dotted,
                numFmt: values[i].numFmt ?? undefined,
                backgroundColorCode: values[i].backgroundColorCode ?? undefined,
                alignment: values[i].alignment ?? undefined,
                isTextWhite: values[i].isTextWhite ?? undefined,
                textColorCode: values[i].textColorCode ?? undefined,
            });
        }
        return colIndex;
    }
    addInitCellsToRow(values, columns, startColIndex, startRowIndex, ws, border, height) {
        for (let i = 0; i < values.length; i++) {
            this.addInitCell11(ws, startRowIndex, startColIndex++, values[i].value || i18n_1.default.__({ phrase: values[i].phrase, locale: this._locale }), border ?? ReportGenerator.Constants.Styling.AllThin, height);
            columns.push({ width: values[i].cellWidth || 30 });
        }
        return startColIndex;
    }
    async generateReportAndGetResponse(onTakesLong) {
        let isHugeReport = false;
        const timeout = setTimeout(() => {
            isHugeReport = true;
            onTakesLong();
        }, 10 * 1000);
        this._response = await this.generateReport().then((res) => {
            clearTimeout(timeout);
            res.notificationType = this._request.reportTemplateId ? app_enums_1.enums.NotificationType.ScheduledReport : isHugeReport ? app_enums_1.enums.NotificationType.HugeReport : null;
            return res;
        });
        this._response.sendFileInAttachments = fs_1.default.statSync(this._response.filePath)["size"] / 1000000.0 <= 1.5;
        return this._response;
    }
}
exports.ReportGenerator = ReportGenerator;
ReportGenerator.Constants = excel_report_constants_1.ExcelReportConstants;
class CustomReportGenerator extends ReportGenerator {
    constructor(request, locale) {
        super(request, locale);
    }
    generateReport() {
        const filter = this._request.filter;
        try {
            const report = require(path_1.default.resolve(app_config_1.appConfig.extensionDirectory, "customreports", filter.reportId + ".js"));
            return report.generateReport(this._request, this._locale);
        }
        catch (error) {
            app_logs_1.logger.error("Error while custom report for user request ", error);
            throw error;
        }
    }
}
exports.CustomReportGenerator = CustomReportGenerator;
async function prepareReportArgumentsForScheduledJob(params) {
    const { organizationId } = params;
    const reportTemplate = await dal_manager_1.dbManager.accessReport.getReportTemplateForScheduledReport(params);
    if (!reportTemplate) {
        return;
    }
    const result = {
        createdByUserId: reportTemplate.createdByUserId,
        receiverFilterId: reportTemplate.receiverFilterId,
        request: null,
    };
    switch (reportTemplate.type) {
        case dal_constants_1.DalConstants.ReportTemplateType.AccessLogs:
            result.request = await prepareReportArgumentsAccessLogs(organizationId, reportTemplate);
            break;
        case dal_constants_1.DalConstants.ReportTemplateType.RegionState:
            result.request = await prepareReportArgumentsRegionState(organizationId, reportTemplate);
            break;
        case dal_constants_1.DalConstants.ReportTemplateType.DurationInRegionReportOverview:
            result.request = await prepareReportArgumentsDurationInRegionOverview(organizationId, reportTemplate);
            break;
        case dal_constants_1.DalConstants.ReportTemplateType.EntryByUnit:
            result.request = await prepareReportArgumentsEntryByUnit(organizationId, reportTemplate);
            break;
        case dal_constants_1.DalConstants.ReportTemplateType.FirstAndLastAccessLogs:
            result.request = await firstLastAccesLogsReport(organizationId, reportTemplate);
            break;
        case dal_constants_1.DalConstants.ReportTemplateType.FirstAndLastAccessLogsForSingleUser:
            result.request = await firstLastAccesLogsForSingleUserReport(organizationId, reportTemplate);
            break;
        case dal_constants_1.DalConstants.ReportTemplateType.PACSWorkingDetailsForSingleEmployeeDateRange:
            result.request = await pacsFirstLastAccesLogsForSingleUserReport(organizationId, reportTemplate);
            break;
        case dal_constants_1.DalConstants.ReportTemplateType.PACSDailyWorkingDetailsForEmployees:
            result.request = await pacsDailyWorkingHoursReport(organizationId, reportTemplate);
            break;
        case dal_constants_1.DalConstants.ReportTemplateType.InstantAttendanceStatus:
            result.request = await distributedEmployees(organizationId, reportTemplate);
            break;
        case dal_constants_1.DalConstants.ReportTemplateType.PACSWorkingDetailsForEmployeesInDateRange:
            result.request = await workingDetailsDateRangeReport(organizationId, reportTemplate);
            break;
        case dal_constants_1.DalConstants.ReportTemplateType.PACSWorkingDetailsForSingleEmployeeDateRange:
            result.request = await workingDetailsSingleEmployeeDateRangeReport(organizationId, reportTemplate);
            break;
        case dal_constants_1.DalConstants.ReportTemplateType.CustomReport:
            result.request = await customReport(organizationId, reportTemplate);
            break;
        default:
            throw new Error("Unsupported Scheduled Report type");
    }
    return result;
}
exports.prepareReportArgumentsForScheduledJob = prepareReportArgumentsForScheduledJob;
async function prepareReportArgumentsAccessLogs(organizationId, template) {
    const filter = template.filter;
    switch (template.interval) {
        case 1:
        case 6:
        case 7:
        case 2:
            filter.startUtc = (0, moment_1.default)(new Date()).add(-1, "days").utc().toDate();
            break;
        case 3:
            filter.startUtc = (0, moment_1.default)(new Date()).add(-1, "week").utc().toDate();
            break;
        case 4:
            filter.startUtc = (0, moment_1.default)(new Date()).add(-1, "month").utc().toDate();
            break;
        case 5:
            filter.startUtc = (0, moment_1.default)(new Date()).add(-1, "year").utc().toDate();
            break;
        default:
            break;
    }
    filter.endUtc = new Date();
    return {
        organizationId,
        code: app_enums_1.enums.ReportCode.AccessLogs,
        format: template.format,
        filter,
        reportTemplateId: template.reportTemplateId,
        requesterUserId: template.createdByUserId,
        sendWhenEmpty: template.sendWhenEmpty,
    };
}
async function prepareReportArgumentsRegionState(organizationId, template) {
    return {
        code: template.type,
        filter: template.filter,
        format: template.format,
        organizationId,
        reportTemplateId: template.reportTemplateId,
        requesterUserId: template.userId,
        sendWhenEmpty: template.sendWhenEmpty,
    };
}
async function prepareReportArgumentsDurationInRegionOverview(organizationId, template) {
    const filter = template.filter;
    let range = { start: null, end: null };
    switch (template.interval) {
        case 1:
        case 6:
        case 7:
        case 2:
            range.start = (0, moment_1.default)(new Date()).add(-1, "days").utc().toDate();
            break;
        case 3:
            range.start = (0, moment_1.default)(new Date()).add(-1, "week").utc().toDate();
            break;
        case 4:
            range.start = (0, moment_1.default)(new Date()).add(-1, "month").utc().toDate();
            break;
        case 5:
            range.start = (0, moment_1.default)(new Date()).add(-1, "year").utc().toDate();
            break;
        default:
            break;
    }
    range.end = new Date();
    filter.range = range;
    return {
        code: template.type,
        filter: filter,
        format: template.format,
        organizationId,
        reportTemplateId: template.reportTemplateId,
        requesterUserId: template.userId,
        sendWhenEmpty: template.sendWhenEmpty,
    };
}
async function prepareReportArgumentsEntryByUnit(organizationId, template) {
    const filter = template.filter;
    filter.organizationId = organizationId;
    filter.dateRange = { startDateTime: new Date(), endDateTime: new Date() };
    switch (template.interval) {
        case 1:
        case 6:
        case 7:
        case 2:
            filter.dateRange.startDateTime = (0, moment_1.default)(new Date()).startOf("day").utc().toDate();
            break;
        case 3:
            filter.dateRange.startDateTime = (0, moment_1.default)(new Date()).add(-1, "week").endOf("day").utc().toDate();
            break;
        case 4:
            filter.dateRange.startDateTime = (0, moment_1.default)(new Date()).add(-1, "month").endOf("day").utc().toDate();
            break;
        case 5:
            filter.dateRange.startDateTime = (0, moment_1.default)(new Date()).add(-1, "year").endOf("day").utc().toDate();
            break;
        default:
            break;
    }
    return {
        code: template.type,
        filter: filter,
        format: template.format,
        organizationId,
        reportTemplateId: template.reportTemplateId,
        requesterUserId: template.userId,
        sendWhenEmpty: template.sendWhenEmpty,
    };
}
async function firstLastAccesLogsReport(organizationId, template) {
    const filter = template.filter;
    const options = {
        accessControlPointIds: filter.accessControlPointIds,
        dateRange: null,
        hasOrganizationWide: filter.hasOrganizationWide,
        organizationUnitIds: filter.organizationUnitIds,
        regionIds: filter.regionIds,
        status: filter.status,
        userIds: filter.userIds,
        filterOrganizationUnitMembersHierarchically: filter.filterOrganizationUnitMembersHierarchically,
        userGroupIds: filter.userGroupIds,
    };
    options.dateRange = { startDateTime: new Date(), endDateTime: new Date() };
    switch (template.interval) {
        case 1:
        case 6:
        case 7:
        case 2:
            options.dateRange.startDateTime = (0, moment_1.default)(new Date()).startOf("day").utc().toDate();
            break;
        case 3:
            options.dateRange.startDateTime = (0, moment_1.default)(new Date()).add(-1, "week").endOf("day").utc().toDate();
            break;
        case 4:
            options.dateRange.startDateTime = (0, moment_1.default)(new Date()).add(-1, "month").endOf("day").utc().toDate();
            break;
        case 5:
            options.dateRange.startDateTime = (0, moment_1.default)(new Date()).add(-1, "year").endOf("day").utc().toDate();
            break;
        default:
            break;
    }
    return {
        code: app_enums_1.enums.ReportCode.FirstAndLastAccessLogs,
        format: template.format,
        filter: options,
        reportTemplateId: template.reportTemplateId,
        organizationId: organizationId,
        requesterUserId: template.userId,
        sendWhenEmpty: template.sendWhenEmpty,
    };
}
async function firstLastAccesLogsForSingleUserReport(organizationId, template) {
    const filter = template.filter;
    filter.dateRange = { startDateTime: new Date(), endDateTime: new Date() };
    switch (template.interval) {
        case 1:
        case 6:
        case 7:
        case 2:
            filter.dateRange.startDateTime = (0, moment_1.default)(new Date()).startOf("day").utc().toDate();
            break;
        case 3:
            filter.dateRange.startDateTime = (0, moment_1.default)(new Date()).add(-1, "week").endOf("day").utc().toDate();
            break;
        case 4:
            filter.dateRange.startDateTime = (0, moment_1.default)(new Date()).add(-1, "month").endOf("day").utc().toDate();
            break;
        case 5:
            filter.dateRange.startDateTime = (0, moment_1.default)(new Date()).add(-1, "year").endOf("day").utc().toDate();
            break;
        default:
            break;
    }
    const options = {
        userId: filter.userId,
        accessControlPointIds: filter.accessControlPointIds,
        dateRange: filter.dateRange,
        regionIds: filter.regionIds,
    };
    return {
        code: app_enums_1.enums.ReportCode.FirstAndLastAccessLogsForSingleUser,
        format: template.format,
        filter: options,
        reportTemplateId: template.reportTemplateId,
        organizationId: organizationId,
        requesterUserId: template.userId,
        sendWhenEmpty: template.sendWhenEmpty,
    };
}
async function pacsFirstLastAccesLogsForSingleUserReport(organizationId, template) {
    const filter = template.filter;
    const options = {
        userId: filter.userId,
        dateRange: null,
        onlyWorkingDays: filter.onlyWorkingDays,
        workStatusFilter: filter.workStatusFilter,
        sortType: filter.sortType,
        sortOrder: filter.sortOrder,
        userOrganizationStatus: filter.userOrganizationStatus,
    };
    options.dateRange = { startDateTime: new Date(), endDateTime: new Date() };
    switch (template.interval) {
        case 1:
        case 6:
        case 7:
        case 2:
            options.dateRange.startDateTime = (0, moment_1.default)(new Date()).startOf("day").utc().toDate();
            break;
        case 3:
            options.dateRange.startDateTime = (0, moment_1.default)(new Date()).add(-1, "week").endOf("day").utc().toDate();
            break;
        case 4:
            options.dateRange.startDateTime = (0, moment_1.default)(new Date()).add(-1, "month").endOf("day").utc().toDate();
            break;
        case 5:
            options.dateRange.startDateTime = (0, moment_1.default)(new Date()).add(-1, "year").endOf("day").utc().toDate();
            break;
        default:
            break;
    }
    await (0, controller_report_1.refreshUsersSummaryForToday)({
        requesterUserId: template.createdByUserId,
        organizationId: organizationId,
        userIds: [filter.userId],
    });
    return {
        code: app_enums_1.enums.ReportCode.PacsEmployeeWorkingHoursSummary,
        format: template.format,
        filter: options,
        reportTemplateId: template.reportTemplateId,
        requesterUserId: template.userId,
        organizationId: organizationId,
        sendWhenEmpty: template.sendWhenEmpty,
    };
}
async function pacsDailyWorkingHoursReport(organizationId, template) {
    const filter = template.filter;
    const options = {
        userFilter: filter.userFilter,
        date: (0, moment_1.default)(new Date()).startOf("day").utc().toDate(),
        sortType: filter.sortType,
        sortOrder: filter.sortOrder,
    };
    await (0, controller_report_1.refreshUsersSummaryForToday)({
        requesterUserId: template.createdByUserId,
        applyOrganizationUnitFilterHierarchically: filter.userFilter.applyOrganizationUnitFilterHierarchically,
        userGroupIds: filter.userFilter.userGroupIds,
        organizationId: organizationId,
        organizationUnitIds: filter.userFilter.organizationUnitIds,
        userIds: filter.userFilter.userIds,
        requiredOrganizationWidePermissions: ["j:l", "e:u", "u:b", "g:r"],
        requiredOrganizationUnitWidePermissions: ["j:l", "e:u", "u:b", "g:r"],
        userOrganizationStatus: filter.userFilter.userOrganizationStatus,
    });
    return {
        code: app_enums_1.enums.ReportCode.PacsDailyWorkingHours,
        format: template.format,
        filter: options,
        reportTemplateId: template.reportTemplateId,
        requesterUserId: template.userId,
        organizationId: organizationId,
        sendWhenEmpty: template.sendWhenEmpty,
    };
}
async function customReport(organizationId, template) {
    template.filter = template.filter;
    const options = {
        filters: template.filter.filters,
        reportId: template.customReportId,
    };
    for (const filter of options.filters) {
        switch (template.interval) {
            case 1:
                if (filter.type === dal_constants_1.DalConstants.CustomOrganizationReportFilterType.Date ||
                    filter.type === dal_constants_1.DalConstants.CustomOrganizationReportFilterType.DateTime ||
                    filter.type === dal_constants_1.DalConstants.CustomOrganizationReportFilterType.DateMonth ||
                    filter.type === dal_constants_1.DalConstants.CustomOrganizationReportFilterType.DateDay) {
                    filter.value = (0, moment_1.default)(new Date()).utc().toDate();
                }
                else if (filter.type === dal_constants_1.DalConstants.CustomOrganizationReportFilterType.DateRange) {
                    filter.value.startDay = (0, moment_1.default)(new Date()).add(-1, "hours").utc().toDate();
                    filter.value.endDay = new Date();
                }
                else if (filter.type === dal_constants_1.DalConstants.CustomOrganizationReportFilterType.DateTimeRange) {
                    filter.value.start = (0, moment_1.default)(new Date()).add(-1, "hours").utc().toDate();
                    filter.value.end = new Date();
                }
                break;
            case 6:
            case 7:
            case 2:
                if (filter.type === dal_constants_1.DalConstants.CustomOrganizationReportFilterType.Date ||
                    filter.type === dal_constants_1.DalConstants.CustomOrganizationReportFilterType.DateTime ||
                    filter.type === dal_constants_1.DalConstants.CustomOrganizationReportFilterType.DateMonth ||
                    filter.type === dal_constants_1.DalConstants.CustomOrganizationReportFilterType.DateDay) {
                    filter.value = (0, moment_1.default)(new Date()).utc().toDate();
                }
                else if (filter.type === dal_constants_1.DalConstants.CustomOrganizationReportFilterType.DateRange) {
                    filter.value.startDay = (0, moment_1.default)(new Date()).add(-1, "days").utc().toDate();
                    filter.value.endDay = new Date();
                }
                else if (filter.type === dal_constants_1.DalConstants.CustomOrganizationReportFilterType.DateTimeRange) {
                    filter.value.start = (0, moment_1.default)(new Date()).add(-1, "days").utc().toDate();
                    filter.value.end = new Date();
                }
                break;
            case 3:
                if (filter.type === dal_constants_1.DalConstants.CustomOrganizationReportFilterType.Date ||
                    filter.type === dal_constants_1.DalConstants.CustomOrganizationReportFilterType.DateTime ||
                    filter.type === dal_constants_1.DalConstants.CustomOrganizationReportFilterType.DateMonth ||
                    filter.type === dal_constants_1.DalConstants.CustomOrganizationReportFilterType.DateDay) {
                    filter.value = (0, moment_1.default)(new Date()).utc().toDate();
                }
                else if (filter.type === dal_constants_1.DalConstants.CustomOrganizationReportFilterType.DateRange) {
                    filter.value.startDay = (0, moment_1.default)(new Date()).add(-1, "week").utc().toDate();
                    filter.value.endDay = new Date();
                }
                else if (filter.type === dal_constants_1.DalConstants.CustomOrganizationReportFilterType.DateTimeRange) {
                    filter.value.start = (0, moment_1.default)(new Date()).add(-1, "week").utc().toDate();
                    filter.value.end = new Date();
                }
                break;
            case 4:
                if (filter.type === dal_constants_1.DalConstants.CustomOrganizationReportFilterType.Date ||
                    filter.type === dal_constants_1.DalConstants.CustomOrganizationReportFilterType.DateTime ||
                    filter.type === dal_constants_1.DalConstants.CustomOrganizationReportFilterType.DateMonth ||
                    filter.type === dal_constants_1.DalConstants.CustomOrganizationReportFilterType.DateDay) {
                    filter.value = (0, moment_1.default)(new Date()).utc().toDate();
                }
                else if (filter.type === dal_constants_1.DalConstants.CustomOrganizationReportFilterType.DateRange) {
                    filter.value.startDay = (0, moment_1.default)(new Date()).add(-1, "month").utc().toDate();
                    filter.value.endDay = new Date();
                }
                else if (filter.type === dal_constants_1.DalConstants.CustomOrganizationReportFilterType.DateTimeRange) {
                    filter.value.start = (0, moment_1.default)(new Date()).add(-1, "month").utc().toDate();
                    filter.value.end = new Date();
                }
                break;
            case 5:
                if (filter.type === dal_constants_1.DalConstants.CustomOrganizationReportFilterType.Date ||
                    filter.type === dal_constants_1.DalConstants.CustomOrganizationReportFilterType.DateTime ||
                    filter.type === dal_constants_1.DalConstants.CustomOrganizationReportFilterType.DateMonth ||
                    filter.type === dal_constants_1.DalConstants.CustomOrganizationReportFilterType.DateDay) {
                    filter.value = (0, moment_1.default)(new Date()).utc().toDate();
                }
                else if (filter.type === dal_constants_1.DalConstants.CustomOrganizationReportFilterType.DateRange) {
                    filter.value.startDay = (0, moment_1.default)(new Date()).add(-1, "year").utc().toDate();
                    filter.value.endDay = new Date();
                }
                else if (filter.type === dal_constants_1.DalConstants.CustomOrganizationReportFilterType.DateTimeRange) {
                    filter.value.start = (0, moment_1.default)(new Date()).add(-1, "year").utc().toDate();
                    filter.value.end = new Date();
                }
                break;
            default:
                break;
        }
    }
    return {
        code: app_enums_1.enums.ReportCode.CustomReport,
        format: template.format,
        filter: options,
        reportTemplateId: template.reportTemplateId,
        organizationId: organizationId,
        requesterUserId: template.userId,
        sendWhenEmpty: template.sendWhenEmpty,
    };
}
async function distributedEmployees(organizationId, template) {
    const filter = template.filter;
    const options = {
        employeeStatusAtSegment: filter.employeeStatusAtSegment,
        idBasedUserFilter: filter.idBasedUserFilter,
        dateTime: (0, moment_1.default)(new Date()).startOf("day").utc().toDate(),
        workPlanIds: filter.workPlanIds,
        sortOrder: filter.sortOrder,
        sortType: filter.sortType,
    };
    await (0, controller_report_1.refreshUsersSummaryForToday)({
        requesterUserId: template.createdByUserId,
        applyOrganizationUnitFilterHierarchically: filter.idBasedUserFilter.applyOrganizationUnitFilterHierarchically,
        userGroupIds: filter.idBasedUserFilter.userGroupIds,
        organizationId: organizationId,
        organizationUnitIds: filter.idBasedUserFilter.organizationUnitIds,
        userIds: filter.idBasedUserFilter.userIds,
        requiredOrganizationWidePermissions: ["j:l", "e:u", "u:b", "g:r"],
        requiredOrganizationUnitWidePermissions: ["j:l", "e:u", "u:b", "g:r"],
        userOrganizationStatus: filter.idBasedUserFilter.userOrganizationStatus,
    });
    return {
        code: app_enums_1.enums.ReportCode.InstantAttendanceStatus,
        format: template.format,
        filter: options,
        reportTemplateId: template.reportTemplateId,
        organizationId: organizationId,
        requesterUserId: template.userId,
        sendWhenEmpty: template.sendWhenEmpty,
    };
}
async function workingDetailsDateRangeReport(organizationId, template) {
    const filter = template.filter;
    const options = {
        sortType: filter.sortType,
        sortOrder: filter.sortOrder,
        userFilter: filter.userFilter,
        dateRange: null,
    };
    options.dateRange = { startDateTime: new Date(), endDateTime: new Date() };
    switch (template.interval) {
        case 1:
        case 6:
        case 7:
        case 2:
            options.dateRange.startDateTime = (0, moment_1.default)(new Date()).startOf("day").utc().toDate();
            break;
        case 3:
            options.dateRange.startDateTime = (0, moment_1.default)(new Date()).add(-1, "week").endOf("day").utc().toDate();
            break;
        case 4:
            options.dateRange.startDateTime = (0, moment_1.default)(new Date()).add(-1, "month").endOf("day").utc().toDate();
            break;
        case 5:
            options.dateRange.startDateTime = (0, moment_1.default)(new Date()).add(-1, "year").endOf("day").utc().toDate();
            break;
        default:
            break;
    }
    await (0, controller_report_1.refreshUsersSummaryForToday)({
        requesterUserId: template.createdByUserId,
        applyOrganizationUnitFilterHierarchically: filter.userFilter.applyOrganizationUnitFilterHierarchically,
        userGroupIds: filter.userFilter.userGroupIds,
        organizationId: organizationId,
        organizationUnitIds: filter.userFilter.organizationUnitIds,
        userIds: filter.userFilter.userIds,
        requiredOrganizationWidePermissions: ["j:l", "e:u", "u:b", "g:r"],
        requiredOrganizationUnitWidePermissions: ["j:l", "e:u", "u:b", "g:r"],
    });
    return {
        code: app_enums_1.enums.ReportCode.PacsFirstInLastOutSummaryMultipleUsers,
        format: template.format,
        filter: options,
        reportTemplateId: template.reportTemplateId,
        requesterUserId: template.userId,
        organizationId: organizationId,
        sendWhenEmpty: template.sendWhenEmpty,
    };
}
async function workingDetailsSingleEmployeeDateRangeReport(organizationId, template) {
    const filter = template.filter;
    const options = {
        sortType: filter.sortType,
        sortOrder: filter.sortOrder,
        dateRange: { startDateTime: new Date(), endDateTime: new Date() },
        userId: filter.userId,
        userOrganizationStatus: filter.userOrganizationStatus,
        workStatusFilter: filter.workStatusFilter,
    };
    switch (template.interval) {
        case 1:
        case 6:
        case 7:
        case 2:
            options.dateRange.startDateTime = (0, moment_1.default)(new Date()).startOf("day").utc().toDate();
            break;
        case 3:
            options.dateRange.startDateTime = (0, moment_1.default)(new Date()).add(-1, "week").endOf("day").utc().toDate();
            break;
        case 4:
            options.dateRange.startDateTime = (0, moment_1.default)(new Date()).add(-1, "month").endOf("day").utc().toDate();
            break;
        case 5:
            options.dateRange.startDateTime = (0, moment_1.default)(new Date()).add(-1, "year").endOf("day").utc().toDate();
            break;
        default:
            break;
    }
    await (0, controller_report_1.refreshUsersSummaryForToday)({
        requesterUserId: template.createdByUserId,
        organizationId: organizationId,
        userIds: [filter.userId],
    });
    return {
        code: app_enums_1.enums.ReportCode.PacsEmployeeWorkingHoursSummary,
        format: template.format,
        filter: options,
        reportTemplateId: template.reportTemplateId,
        requesterUserId: template.userId,
        organizationId: organizationId,
        sendWhenEmpty: template.sendWhenEmpty,
    };
}
