"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.RegionTicketUserUnitsExcelReport = void 0;
const path_1 = __importDefault(require("path"));
const exceljs_1 = __importDefault(require("exceljs"));
const app_config_1 = require("../../../app.config");
const dal_manager_1 = require("../../../dal/dal.manager");
const uuid_1 = __importDefault(require("uuid"));
const i18n_1 = __importDefault(require("i18n"));
const app_logs_1 = require("../../../app.logs");
const app_enums_1 = require("../../../app.enums");
const fs_1 = __importDefault(require("fs"));
const moment_1 = __importDefault(require("moment"));
const generator_1 = require("../generator");
const business_report_export_1 = require("../business.report.export");
class RegionTicketUserUnitsExcelReport extends generator_1.ReportGenerator {
    constructor(request, locale) {
        super(request, locale);
        this._fileId = uuid_1.default.v4();
        this._filePath = path_1.default.join(app_config_1.appConfig.tmpDirectory, this._fileId + ".xlsx");
        if (fs_1.default.existsSync(this._filePath)) {
            fs_1.default.unlinkSync(this._filePath);
        }
        this._wb = new exceljs_1.default.stream.xlsx.WorkbookWriter({
            filename: this._filePath,
            useStyles: true,
            useSharedStrings: true,
        });
        this._options = request.filter;
    }
    async generateReport() {
        this._organizationName = (await dal_manager_1.dbManager.accessOrganization.getOrganizationBasic(this._request.organizationId)).name;
        app_logs_1.logger.debug("Organization Name : " + this._organizationName);
        let { caption: requestedUserCaption } = await dal_manager_1.dbManager.accessRedisCache.getUserBadgeCache({ organizationId: this._request.organizationId, userId: this._request.requesterUserId });
        if (!requestedUserCaption) {
            throw new Error("Not found");
        }
        this._requestedUserCaption = requestedUserCaption[0].text[0];
        this._wb.creator = this._requestedUserCaption;
        app_logs_1.logger.debug("Generating region ticket user units excel report...");
        app_logs_1.logger.debug("Requester User : " + this._requestedUserCaption);
        this._wsGeneral = this._wb.addWorksheet(i18n_1.default.__({ phrase: "EXCEL-REPORT.GENERAL", locale: this._locale }), {
            pageSetup: { fitToPage: true, fitToWidth: 1, fitToHeight: 0, paperSize: 9, orientation: "landscape" },
        });
        this._wsGeneral.properties.defaultRowHeight = 15;
        this._generalRowIndex = 10;
        this.initGeneralSheet(this._wsGeneral, 9, 1);
        const filter = await this.createFilters();
        const filters = [];
        if (filter.organizationUnits) {
            filters.push(i18n_1.default.__({ phrase: "EXCEL-REPORT.ORGANIZATION_UNITS", locale: this._locale }) + " : " + filter.organizationUnits);
        }
        if (filter.userGroups) {
            filters.push(i18n_1.default.__({ phrase: "EXCEL-REPORT.USER_GROUPS", locale: this._locale }) + " : " + filter.userGroups);
        }
        if (filter.users) {
            filters.push(i18n_1.default.__({ phrase: "PDF-REPORT.IDENTITES", locale: this._locale }) + " : " + filter.users);
        }
        app_logs_1.logger.debug("\n\nApplied Filters:\n");
        app_logs_1.logger.debug(filters.join("\r\n"));
        app_logs_1.logger.debug("\n\n");
        this.addFilterToSheet(this._wsGeneral, 1, 1, filters.join("\r\n"));
        this._data = await (0, business_report_export_1.generateUserTicketInfoReportData)(this._request.organizationId, this._request.requesterUserId, this._options);
        for (const item of this._data.items) {
            this.addGeneralRow(item);
        }
        this._wsGeneral.pageSetup.printArea = "A1:" + this._wsGeneral.getColumn(this._wsGeneral.columns.length).letter + this._generalRowIndex;
        this._wsGeneral.pageSetup.printTitlesRow = "5:5";
        this._wsGeneral.commit();
        await this._wb.commit();
        app_logs_1.logger.debug("File is ready -> " + this._filePath);
        return {
            format: app_enums_1.enums.ReportFormat.Excel,
            preparedReportId: this._fileId,
            filePath: this._filePath,
            notificationType: undefined,
            reportType: app_enums_1.enums.ReportCode.RegionTicketUserUnits,
            sendFileInAttachments: undefined,
        };
    }
    async createFilters() {
        let filterResult = await dal_manager_1.dbManager.accessReport.collectRegionTickerUserUnitFilterInfo(this._request.organizationId, this._options.organizationUnitIds, this._options.userGroupIds, this._options.userIds);
        let users = "";
        if (filterResult.user != null && filterResult.user.length > 0) {
            let _res = "";
            filterResult.user.forEach((userCaption, index) => {
                _res += "";
                userCaption.captionLines.forEach((item, _index) => {
                    if (item.text != null && item.text != "__avatar") {
                        item.text.forEach((element) => {
                            if (element != null)
                                _res += " " + element + (_index < filterResult.user.length - 1 ? " " : "");
                        });
                    }
                });
                _res += index < filterResult.user.length - 1 ? "\r\n" : "";
            });
            users = _res;
        }
        let organizationUnits = "";
        if (filterResult.organizationUnits != null && filterResult.organizationUnits.length > 0) {
            let res = "";
            filterResult.organizationUnits.forEach((org, index) => {
                res += org.name + (index < filterResult.organizationUnits.length - 1 ? ", " : " ");
            });
            organizationUnits = res;
        }
        let userGroups = "";
        if (filterResult.userGroups != null && filterResult.userGroups.length > 0) {
            let res = "";
            filterResult.userGroups.forEach((userGroup, index) => {
                res += userGroup.name + (index < filterResult.userGroups.length - 1 ? ", " : " ");
            });
            userGroups = res;
        }
        return {
            users: users,
            organizationUnits: organizationUnits,
            userGroups: userGroups,
        };
    }
    initGeneralSheet(wsRaw, startRowIndex, startColIndex) {
        let columns = [];
        let data = [
            {
                phrase: "EXCEL-REPORT.UNIQUE_ID",
                cellWidth: 30
            },
            {
                phrase: "EXCEL-REPORT.USER",
                cellWidth: 45
            },
            {
                phrase: "EXCEL-REPORT.CREDIT_BALANCE",
                cellWidth: 20
            },
            {
                phrase: "EXCEL-REPORT.TOTAL_CREDITS_SPENT",
                cellWidth: 20
            }
        ];
        startColIndex = this.addInitCellsToRow(data, columns, startColIndex, startRowIndex, wsRaw);
        wsRaw.autoFilter = {
            from: "A" + startRowIndex,
            to: wsRaw.getColumn(columns.length).letter + startRowIndex,
        };
        wsRaw.columns = columns;
    }
    addGeneralRow(item) {
        let colIndex = 1;
        const rowData = [
            item.user.uniqueId,
            item.user.fullname.toUpperCase(),
            item.remainingUnits ?? "",
            item.totalSpentUnits ?? ""
        ];
        colIndex = this.addCellsToRow(rowData, colIndex, this._generalRowIndex, this._wsGeneral);
        this._wsGeneral.getRow(this._generalRowIndex).commit();
        this._generalRowIndex++;
    }
    addFilterToSheet(ws, startRowIndex, startColIndex, filter) {
        this.addFilterCell11(ws, startRowIndex, startColIndex, this._organizationName + " " + i18n_1.default.__({ phrase: "EXCEL-REPORT.CREDIT_MANAGEMENT", locale: this._locale }), generator_1.ReportGenerator.Constants.Styling.AllThin, {
            bold: true,
        });
        ws.mergeCells(startRowIndex, startColIndex, startRowIndex, startColIndex + 3);
        this.addFilterCell11(ws, startRowIndex + 1, startColIndex, i18n_1.default.__({ phrase: "EXCEL-REPORT.FILE_CREATION_DATE", locale: this._locale }), generator_1.ReportGenerator.Constants.Styling.AllThin);
        this.addFilterCell11(ws, startRowIndex + 1, startColIndex + 1, (0, moment_1.default)().locale(this._locale).format("LL") + " " + (0, moment_1.default)().locale(this._locale).format("HH:mm:ss"), generator_1.ReportGenerator.Constants.Styling.AllThin, { italic: false }, generator_1.ReportGenerator.Constants.DateFormat);
        this.addFilterCell11(ws, startRowIndex + 2, startColIndex, i18n_1.default.__({ phrase: "EXCEL-REPORT.REQUESTED_USER", locale: this._locale }), generator_1.ReportGenerator.Constants.Styling.AllThin);
        this.addFilterCell11(ws, startRowIndex + 2, startColIndex + 1, this._requestedUserCaption, generator_1.ReportGenerator.Constants.Styling.AllThin);
        ws.mergeCells(startRowIndex + 1, startColIndex + 1, startRowIndex + 1, startColIndex + 3);
        ws.mergeCells(startRowIndex + 2, startColIndex + 1, startRowIndex + 2, startColIndex + 3);
        ws.mergeCells(startRowIndex + 3, startColIndex + 1, startRowIndex + 3, startColIndex + 3);
        this.addFilterCell11(ws, startRowIndex + 3, startColIndex, i18n_1.default.__({ phrase: "EXCEL-REPORT.APPLIED_FILTERS", locale: this._locale }), generator_1.ReportGenerator.Constants.Styling.AllThin);
        this.addFilterCell11(ws, startRowIndex + 3, startColIndex + 1, filter, generator_1.ReportGenerator.Constants.Styling.AllThin);
        let filterlength = filter.split("\r\n").length;
        let row = ws.getRow(startRowIndex + 3);
        row.height = filterlength * 25;
    }
}
exports.RegionTicketUserUnitsExcelReport = RegionTicketUserUnitsExcelReport;
