"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.PermissionManagementExcelReport = void 0;
const path_1 = __importDefault(require("path"));
const exceljs_1 = __importDefault(require("exceljs"));
const app_config_1 = require("../../../app.config");
const dal_manager_1 = require("../../../dal/dal.manager");
const uuid_1 = __importDefault(require("uuid"));
const i18n_1 = __importDefault(require("i18n"));
const app_logs_1 = require("../../../app.logs");
const app_enums_1 = require("../../../app.enums");
const fs_1 = __importDefault(require("fs"));
const moment_1 = __importDefault(require("moment"));
const generator_1 = require("../generator");
const business_report_export_1 = require("../business.report.export");
const report_util_1 = require("../report.util");
const business_pacs_ppermission_1 = require("../../pacs/business.pacs.ppermission");
class PermissionManagementExcelReport extends generator_1.ReportGenerator {
    constructor(request, locale) {
        super(request, locale);
        this._fileId = uuid_1.default.v4();
        this._filePath = path_1.default.join(app_config_1.appConfig.tmpDirectory, this._fileId + ".xlsx");
        if (fs_1.default.existsSync(this._filePath)) {
            fs_1.default.unlinkSync(this._filePath);
        }
        this._wb = new exceljs_1.default.stream.xlsx.WorkbookWriter({
            filename: this._filePath,
            useStyles: true,
            useSharedStrings: true,
        });
        this._options = request.filter;
    }
    async generateReport() {
        this._organizationName = (await dal_manager_1.dbManager.accessOrganization.getOrganizationBasic(this._request.organizationId)).name;
        app_logs_1.logger.debug("Organization Name : " + this._organizationName);
        let { caption: requestedUserCaption } = await dal_manager_1.dbManager.accessRedisCache.getUserBadgeCache({ organizationId: this._request.organizationId, userId: this._request.requesterUserId });
        if (!requestedUserCaption) {
            throw new Error("Not found");
        }
        this._requestedUserCaption = requestedUserCaption[0].text[0];
        this._wb.creator = this._requestedUserCaption;
        app_logs_1.logger.debug("Generating permission management excel report...");
        app_logs_1.logger.debug("Requester User : " + this._requestedUserCaption);
        this._wsGeneral = this._wb.addWorksheet(i18n_1.default.__({ phrase: "EXCEL-REPORT.GENERAL", locale: this._locale }), {
            pageSetup: { fitToPage: true, fitToWidth: 1, fitToHeight: 0, paperSize: 9, orientation: "landscape" },
        });
        this._wsGeneral.properties.defaultRowHeight = 15;
        this._generalRowIndex = 10;
        this.initGeneralSheet(this._wsGeneral, 9, 1);
        const filter = await this.createFilters();
        const filters = [];
        if (filter.users) {
            filters.push(i18n_1.default.__({ phrase: "EXCEL-REPORT.EMPLOYEES", locale: this._locale }) + " : " + filter.users);
        }
        if (filter.permissionType) {
            filters.push(i18n_1.default.__({ phrase: "EXCEL-REPORT.PERMISSION_TYPE", locale: this._locale }) + " : " + filter.permissionType);
        }
        if (filter.permissionDate) {
            filters.push(i18n_1.default.__({ phrase: "EXCEL-REPORT.PERMISSION_START_DATE", locale: this._locale }) + " : " + filter.permissionDate);
        }
        if (filter.acceptanceStatus) {
            filters.push(i18n_1.default.__({ phrase: "EXCEL-REPORT.ACCEPTANCE_STATUS", locale: this._locale }) + " : " + filter.acceptanceStatus);
        }
        app_logs_1.logger.debug("\n\nApplied Filters:\n");
        app_logs_1.logger.debug(filters.join("\r\n"));
        app_logs_1.logger.debug("\n\n");
        this.addFilterToSheet(this._wsGeneral, 1, 1, filters.join("\r\n"));
        this._data = await (0, business_report_export_1.generateListPPermissionsData)(this._request.organizationId, this._request.requesterUserId, this._options);
        this._vacations = await dal_manager_1.dbManager.accessPacs.listAllVacations(this._request.organizationId);
        const allPermissions = [];
        this._data.ppermissions.forEach((elem) => {
            elem.userPPermissionInfo.forEach((elem2) => allPermissions.push({ ...elem, userPPermissionInfo: [elem2] }));
        });
        allPermissions.sort((perm1, perm2) => {
            let fa = perm1.userPPermissionInfo[0].userAndDeputyInfo.user.fullname.toLowerCase(), fb = perm2.userPPermissionInfo[0].userAndDeputyInfo.user.fullname.toLowerCase();
            return fa.localeCompare(fb);
        });
        const ppermissionTypeIds = allPermissions.map((item) => item.ppermissionType.id);
        const ppermissionTypes = await dal_manager_1.dbManager.accessPacs.getPPermissionTypes(this._request.organizationId, ppermissionTypeIds);
        for (const item of allPermissions) {
            const permissionType = ppermissionTypes.find((pt) => pt.id === item.ppermissionType.id);
            let permissionUsageCount = "";
            if (permissionType.isDailyScheduled) {
                permissionUsageCount =
                    (await (0, business_pacs_ppermission_1.calculateDailyVacationUsage)(this._request.organizationId, permissionType.id, item.dateRange.startDateTime, item.dateRange.endDateTime, this._vacations, item.userPPermissionInfo.length > 0 ? item.userPPermissionInfo[0].userAndDeputyInfo.user.id : undefined, permissionType)) +
                        " " +
                        i18n_1.default.__({ phrase: "REPORT.PACS_UNIT_DAY", locale: this._locale });
            }
            else {
                const permissionMinutes = await (0, business_pacs_ppermission_1.estimateHourlyVacationUsage)(this._request.organizationId, permissionType.id, item.dateRange.startDateTime, item.dateRange.endDateTime, this._vacations, permissionType);
                const hours = Math.floor(permissionMinutes / 60);
                const minutes = permissionMinutes % 60;
                permissionUsageCount =
                    (hours ? hours + " " + i18n_1.default.__({ phrase: "REPORT.PACS_UNIT_HOUR", locale: this._locale }) : "") +
                        (minutes ? ", " + minutes + " " + i18n_1.default.__({ phrase: "REPORT.PACS_UNIT_MINUTE", locale: this._locale }) : "");
            }
            this.addGeneralRow(item, permissionUsageCount);
        }
        this._wsGeneral.pageSetup.printArea = "A1:" + this._wsGeneral.getColumn(this._wsGeneral.columns.length).letter + this._generalRowIndex;
        this._wsGeneral.pageSetup.printTitlesRow = "5:5";
        this._wsGeneral.commit();
        await this._wb.commit();
        app_logs_1.logger.debug("File is ready -> " + this._filePath);
        return {
            format: app_enums_1.enums.ReportFormat.Excel,
            preparedReportId: this._fileId,
            filePath: this._filePath,
            notificationType: undefined,
            reportType: app_enums_1.enums.ReportCode.PermissionManagement,
            sendFileInAttachments: undefined,
        };
    }
    async createFilters() {
        let filterResult = await dal_manager_1.dbManager.accessReport.collectPermissionManagementFilterInfo(this._options.ppermissionTypeId, this._options.userIds, this._request.organizationId);
        let locale = this._locale;
        let users = "";
        if (filterResult.userCaptions != null && filterResult.userCaptions.length > 0) {
            let _res = "";
            filterResult.userCaptions.forEach((userCaption, index) => {
                _res += "";
                userCaption.captionLines.forEach((item, _index) => {
                    if (item.text != null && item.text != "__avatar") {
                        item.text.forEach((element) => {
                            if (element != null)
                                _res += element + (_index < filterResult.userCaptions.length - 1 ? " " : "");
                        });
                    }
                });
                _res += index < filterResult.userCaptions.length - 1 ? "\r\n" : "";
            });
            users = _res;
        }
        let acceptanceStatus = "";
        if (this._options.status) {
            let key = report_util_1.ReportConstants.PacsPPermissionStatusFilterOptionList.filter((item) => this._options.status === item.value);
            acceptanceStatus = i18n_1.default.__({ phrase: key[0].i18nName, locale });
        }
        let permissionDate = "";
        if (this._options.startDateTime)
            permissionDate += (0, moment_1.default)(this._options.startDateTime).locale(locale).format("DD/MM/YYYY HH:mm");
        if (this._options.startDateTime && this._options.endDateTime) {
            permissionDate += " - " + (0, moment_1.default)(this._options.endDateTime).locale(locale).format("DD/MM/YYYY HH:mm");
        }
        if (!this._options.startDateTime && this._options.endDateTime) {
            permissionDate += (0, moment_1.default)(this._options.endDateTime).locale(locale).format("DD/MM/YYYY HH:mm");
        }
        return {
            permissionType: filterResult.permissionType == null ? "" : filterResult.permissionType.name,
            acceptanceStatus: acceptanceStatus,
            permissionDate: permissionDate,
            users: users,
        };
    }
    initGeneralSheet(wsRaw, startRowIndex, startColIndex) {
        let columns = [];
        let data = [
            {
                phrase: "EXCEL-REPORT.PERMISSION_SCOPE",
                cellWidth: 40,
            },
            {
                phrase: "EXCEL-REPORT.UNIQUE_ID",
                cellWidth: 25,
            },
            {
                phrase: "EXCEL-REPORT.EMPLOYEE",
                cellWidth: 40,
            },
            {
                phrase: "EXCEL-REPORT.PERMISSION_START_DATE",
                cellWidth: 25,
            },
            {
                phrase: "EXCEL-REPORT.PERMISSION_END_DATE",
                cellWidth: 25,
            },
            {
                phrase: "EXCEL-REPORT.PERMISSION_TYPE",
                cellWidth: 25,
            },
            {
                phrase: "EXCEL-REPORT.ACCEPTANCE_STATUS",
                cellWidth: 20,
            },
            {
                phrase: "EXCEL-REPORT.NOTE",
                cellWidth: 40,
            },
            {
                phrase: "EXCEL-REPORT.PERMISSON_APPROVED_OR_PENDING_DAY_COUNT",
                cellWidth: 40,
            },
        ];
        startColIndex = this.addInitCellsToRow(data, columns, startColIndex, startRowIndex, wsRaw);
        wsRaw.autoFilter = {
            from: "A" + startRowIndex,
            to: wsRaw.getColumn(columns.length).letter + startRowIndex,
        };
        wsRaw.columns = columns;
    }
    addGeneralRow(record, permissionTotalDayCount) {
        let colIndex = 1;
        let approvalInfo = "";
        const userInfo = record.userPPermissionInfo?.[0];
        let isRequiredApprovement = false;
        if (userInfo.approvementInfo && userInfo.approvementInfo.length > 0) {
            isRequiredApprovement = true;
        }
        if (isRequiredApprovement) {
            switch (userInfo.status) {
                case 0:
                    approvalInfo = i18n_1.default.__({ phrase: "EXCEL-REPORT.PPERMISSION_STATUS_WAITING", locale: this._locale });
                    break;
                case 1:
                    approvalInfo = i18n_1.default.__({ phrase: "EXCEL-REPORT.PPERMISSION_STATUS_ACCEPTED", locale: this._locale });
                    break;
                case 2:
                    approvalInfo = i18n_1.default.__({ phrase: "EXCEL-REPORT.PPERMISSION_STATUS_REJECTED", locale: this._locale });
                    break;
                default:
                    break;
            }
        }
        else {
            approvalInfo = i18n_1.default.__({ phrase: "EXCEL-REPORT.PPERMISSION_STATUS_NO_APPROVEMENT", locale: this._locale });
        }
        const rowData = [
            record.userCount > 1 ? i18n_1.default.__({ phrase: "EXCEL-REPORT.PPERMISSION_MULTIUSER", locale: this._locale }) : "",
            userInfo?.userAndDeputyInfo?.user?.uniqueId ?? "",
            userInfo?.userAndDeputyInfo?.user?.fullname ?? "",
            (0, report_util_1.formatDateForExcel)(record.dateRange.startDateTime) ?? "",
            (0, report_util_1.formatDateForExcel)(record.dateRange.endDateTime) ?? "",
            record.ppermissionType.name ?? "",
            approvalInfo ?? "",
            record.note ?? "",
            permissionTotalDayCount,
        ];
        colIndex = this.addCellsToRow(rowData, colIndex, this._generalRowIndex, this._wsGeneral, generator_1.ReportGenerator.Constants.Styling.AllThin);
        this._wsGeneral.getRow(this._generalRowIndex).commit();
        this._generalRowIndex++;
    }
    addFilterToSheet(ws, startRowIndex, startColIndex, filter) {
        this.addFilterCell11(ws, startRowIndex, startColIndex, this._organizationName + " " + i18n_1.default.__({ phrase: "EXCEL-REPORT.PERMISSION_MANAGEMENT", locale: this._locale }), generator_1.ReportGenerator.Constants.Styling.AllThin, {
            bold: true,
        });
        ws.mergeCells(startRowIndex, startColIndex, startRowIndex, startColIndex + 3);
        this.addFilterCell11(ws, startRowIndex + 1, startColIndex, i18n_1.default.__({ phrase: "EXCEL-REPORT.FILE_CREATION_DATE", locale: this._locale }), generator_1.ReportGenerator.Constants.Styling.AllThin);
        this.addFilterCell11(ws, startRowIndex + 1, startColIndex + 1, (0, moment_1.default)().locale(this._locale).format("LL") + " " + (0, moment_1.default)().locale(this._locale).format("HH:mm:ss"), generator_1.ReportGenerator.Constants.Styling.AllThin, { italic: false }, generator_1.ReportGenerator.Constants.DateFormat);
        this.addFilterCell11(ws, startRowIndex + 2, startColIndex, i18n_1.default.__({ phrase: "EXCEL-REPORT.REQUESTED_USER", locale: this._locale }), generator_1.ReportGenerator.Constants.Styling.AllThin);
        this.addFilterCell11(ws, startRowIndex + 2, startColIndex + 1, this._requestedUserCaption, generator_1.ReportGenerator.Constants.Styling.AllThin);
        ws.mergeCells(startRowIndex + 1, startColIndex + 1, startRowIndex + 1, startColIndex + 3);
        ws.mergeCells(startRowIndex + 2, startColIndex + 1, startRowIndex + 2, startColIndex + 3);
        ws.mergeCells(startRowIndex + 3, startColIndex + 1, startRowIndex + 3, startColIndex + 3);
        this.addFilterCell11(ws, startRowIndex + 3, startColIndex, i18n_1.default.__({ phrase: "EXCEL-REPORT.APPLIED_FILTERS", locale: this._locale }), generator_1.ReportGenerator.Constants.Styling.AllThin);
        this.addFilterCell11(ws, startRowIndex + 3, startColIndex + 1, filter, generator_1.ReportGenerator.Constants.Styling.AllThin);
        let filterlength = filter.split("\r\n").length;
        let row = ws.getRow(startRowIndex + 3);
        row.height = filterlength * 25;
    }
}
exports.PermissionManagementExcelReport = PermissionManagementExcelReport;
