"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.PacsPlanningCalendarExcelReport = void 0;
const path_1 = __importDefault(require("path"));
const exceljs_1 = __importDefault(require("exceljs"));
const moment_1 = __importDefault(require("moment"));
const app_config_1 = require("../../../app.config");
const dal_manager_1 = require("../../../dal/dal.manager");
const uuid_1 = __importDefault(require("uuid"));
const i18n_1 = __importDefault(require("i18n"));
const app_logs_1 = require("../../../app.logs");
const app_enums_1 = require("../../../app.enums");
const fs_1 = __importDefault(require("fs"));
const generator_1 = require("../generator");
const moment_range_1 = require("moment-range");
const lodash_1 = require("lodash");
class PacsPlanningCalendarExcelReport extends generator_1.ReportGenerator {
    constructor(request, locale) {
        super(request, locale);
        this._fileId = uuid_1.default.v4();
        this._filePath = path_1.default.join(app_config_1.appConfig.tmpDirectory, this._fileId + ".xlsx");
        if (fs_1.default.existsSync(this._filePath)) {
            fs_1.default.unlinkSync(this._filePath);
        }
        this._wb = new exceljs_1.default.stream.xlsx.WorkbookWriter({
            filename: this._filePath,
            useStyles: true,
            useSharedStrings: true,
        });
        this._options = request.filter;
        this._workplans = [];
        this._totals = [];
    }
    async generateReport() {
        this._organizationName = (await dal_manager_1.dbManager.accessOrganization.getOrganizationBasic(this._request.organizationId)).name;
        app_logs_1.logger.debug("Organization Name : " + this._organizationName);
        let { caption: requestedUserCaption } = await dal_manager_1.dbManager.accessRedisCache.getUserBadgeCache({ organizationId: this._request.organizationId, userId: this._request.requesterUserId });
        if (!requestedUserCaption) {
            throw new Error("Not found");
        }
        this._requestedUserCaption = requestedUserCaption[0].text[0];
        this._wb.creator = this._requestedUserCaption;
        app_logs_1.logger.debug("Generating audit identity excel report...");
        app_logs_1.logger.debug("Requester User : " + this._requestedUserCaption);
        this._wsWorkingSchedule = this._wb.addWorksheet(i18n_1.default.__({ phrase: "EXCEL-REPORT.WORKING_SCHEDULE", locale: this._locale }), {
            pageSetup: { fitToPage: true, fitToWidth: 1, fitToHeight: 0, paperSize: 9, orientation: "landscape" },
        });
        this._wsWorkingSchedule.properties.defaultRowHeight = 15;
        this._wsWorkingSchedule.properties.defaultColWidth = 20;
        this._workingScheduleRowIndex = 7;
        this._wsMemberships = this._wb.addWorksheet(i18n_1.default.__({ phrase: "EXCEL-REPORT.WORKPLAN_MEMBERSHIPS", locale: this._locale }), {
            pageSetup: { fitToPage: true, fitToWidth: 1, fitToHeight: 0, paperSize: 9, orientation: "landscape" },
        });
        this._wsMemberships.properties.defaultRowHeight = 15;
        this._wsMemberships.properties.defaultColWidth = 20;
        this._membershipsRowIndex = 7;
        this._data = await dal_manager_1.dbManager.accessPacs2.getPlanningCalendar(dal_manager_1.dbManager.poolMain, this._request.organizationId, this._request.requesterUserId, {
            pagination: { skip: 0, take: 1000 },
            range: {
                startDateTime: new Date(this._options.range.startDateTime),
                endDateTime: new Date(this._options.range.endDateTime),
            },
            applyOrganizationUnitFilterHierarchically: this._options.applyOrganizationUnitFilterHierarchically,
            organizationUnitIds: this._options.organizationUnitIds,
            userGroupIds: this._options.userGroupIds,
            userIds: this._options.userIds,
            workPlanIds: this._options.workPlanIds,
        });
        this.initWorkingScheduleSheet(this._wsWorkingSchedule, 5, 1);
        this.initMembershipsSheet(this._wsMemberships, 5, 1);
        let filters = await this.getFilters();
        this.addFilterToSheet(this._wsWorkingSchedule, 1, 1, filters.join("\r\n"), filters.length);
        this.addFilterToSheet(this._wsMemberships, 1, 1, filters.join("\r\n"), filters.length);
        for (const user of this._data.items) {
            this.addWorkingScheduleRow(user);
            this.addMembershipsRow(user);
        }
        this.addWorkingScheduleTotalRow();
        this._wsWorkingSchedule.pageSetup.printArea = "A1:" + this._wsWorkingSchedule.getColumn(this._wsWorkingSchedule.columns.length).letter + this._workingScheduleRowIndex;
        this._wsMemberships.pageSetup.printArea = "A1:" + this._wsMemberships.getColumn(this._wsMemberships.columns.length).letter + this._membershipsRowIndex;
        this._wsWorkingSchedule.commit();
        this._wsMemberships.commit();
        await this._wb.commit();
        app_logs_1.logger.debug("File is ready -> " + this._filePath);
        return {
            format: app_enums_1.enums.ReportFormat.Excel,
            preparedReportId: this._fileId,
            filePath: this._filePath,
            notificationType: undefined,
            reportType: app_enums_1.enums.ReportCode.AuditLogsIdentity,
            sendFileInAttachments: undefined,
        };
    }
    async getFilters() {
        let filters = [];
        if (this._options.organizationUnitIds && this._options.organizationUnitIds.length > 0) {
            let organizationUnitNames = await dal_manager_1.dbManager.accessOrganizationUnit.getIdNamePairsOfOrganizationUnits(this._request.organizationId, this._options.organizationUnitIds);
            filters.push(i18n_1.default.__({ phrase: "EXCEL-REPORT.ORGANIZATION_UNITS", locale: this._locale }) + " : " + organizationUnitNames.map((oun) => oun.name).join(","));
            if (this._options.applyOrganizationUnitFilterHierarchically) {
                filters.push(i18n_1.default.__({ phrase: "EXCEL-REPORT.INCLUDE_DESCENDANTS", locale: this._locale }) + " : " + i18n_1.default.__({ phrase: "EXCEL-REPORT.YES", locale: this._locale }));
            }
        }
        if (this._options.userGroupIds && this._options.userGroupIds.length > 0) {
            let userGroupNames = await dal_manager_1.dbManager.accessUserGroup.listUserGroupsByIds(this._request.organizationId, this._options.userGroupIds);
            filters.push(i18n_1.default.__({ phrase: "EXCEL-REPORT.USER_GROUPS", locale: this._locale }) + " : " + userGroupNames.map((ugn) => ugn.name).join(","));
        }
        if (this._options.workPlanIds && this._options.workPlanIds.length > 0) {
            let workplanNames = this._workplans.filter((wp) => this._options.workPlanIds.includes(wp.id)).map((wp) => wp.name);
            filters.push(i18n_1.default.__({ phrase: "REPORT.WORKPLANS", locale: this._locale }) + " : " + workplanNames.join(","));
        }
        if (this._options.userIds && this._options.userIds.length > 0) {
            let userNames = this._data.items.filter((i) => this._options.userIds.includes(i.userId)).map((i) => i.userCaptions[0].text);
            filters.push(i18n_1.default.__({ phrase: "PDF-REPORT.IDENTITES", locale: this._locale }) + " : " + userNames.join(","));
        }
        app_logs_1.logger.debug("Applied Filters:");
        app_logs_1.logger.debug(filters.join("\r\n"));
        return filters;
    }
    initWorkingScheduleSheet(wsRaw, startRowIndex, startColIndex) {
        let columns = [];
        startColIndex++;
        columns.push({ width: 40 });
        moment_1.default.locale(this._locale);
        const start = (0, moment_1.default)(this._options.range.startDateTime).locale(this._locale);
        const end = (0, moment_1.default)(this._options.range.endDateTime).locale(this._locale);
        while (start.isBefore(end)) {
            wsRaw.mergeCells(startRowIndex, startColIndex, startRowIndex, startColIndex + 2);
            this.addInitCell11(wsRaw, startRowIndex, startColIndex, start.format("DD.MM.YYYY") + "\r\n" + moment_1.default.weekdays(start.isoWeekday()), generator_1.ReportGenerator.Constants.Styling.AllThin, 40);
            this.addInitCell11(wsRaw, startRowIndex + 1, startColIndex++, i18n_1.default.__({ phrase: "EXCEL-REPORT.WORKING_HOURS", locale: this._locale }), generator_1.ReportGenerator.Constants.Styling.AllThin, 40);
            this.addInitCell11(wsRaw, startRowIndex + 1, startColIndex++, i18n_1.default.__({ phrase: "REPORT.PACS_COL_EXPECTED_WORK", locale: this._locale }), generator_1.ReportGenerator.Constants.Styling.AllThin, 40);
            this.addInitCell11(wsRaw, startRowIndex + 1, startColIndex++, i18n_1.default.__({ phrase: "REPORT.PACS_COL_WORKPLAN_WORK", locale: this._locale }), generator_1.ReportGenerator.Constants.Styling.AllThin, 40);
            columns.push({ width: 20 }, { width: 10 }, { width: 25 });
            start.add(1, "day");
        }
        wsRaw.columns = columns;
    }
    initMembershipsSheet(wsRaw, startRowIndex, startColIndex) {
        let columns = [];
        this.addInitCell11(wsRaw, startRowIndex, startColIndex++, i18n_1.default.__({ phrase: "EXCEL-REPORT.EMPLOYEE", locale: this._locale }), generator_1.ReportGenerator.Constants.Styling.AllThin, 40);
        columns.push({ width: 40 });
        moment_1.default.locale(this._locale);
        const start = (0, moment_1.default)(this._options.range.startDateTime).locale(this._locale);
        const end = (0, moment_1.default)(this._options.range.endDateTime).locale(this._locale);
        while (start.isBefore(end)) {
            wsRaw.mergeCells(startRowIndex, startColIndex, startRowIndex, startColIndex + 2);
            this.addInitCell11(wsRaw, startRowIndex, startColIndex, start.format("DD.MM.YYYY") + "\r\n" + moment_1.default.weekdays(start.isoWeekday()), generator_1.ReportGenerator.Constants.Styling.AllThin, 40);
            this.addInitCell11(wsRaw, startRowIndex + 1, startColIndex++, i18n_1.default.__({ phrase: "EXCEL-REPORT.WORKING_HOURS", locale: this._locale }), generator_1.ReportGenerator.Constants.Styling.AllThin, 40);
            this.addInitCell11(wsRaw, startRowIndex + 1, startColIndex++, i18n_1.default.__({ phrase: "REPORT.PACS_COL_EXPECTED_WORK", locale: this._locale }), generator_1.ReportGenerator.Constants.Styling.AllThin, 40);
            this.addInitCell11(wsRaw, startRowIndex + 1, startColIndex++, i18n_1.default.__({ phrase: "REPORT.PACS_COL_WORKPLAN_WORK", locale: this._locale }), generator_1.ReportGenerator.Constants.Styling.AllThin, 40);
            columns.push({ width: 20 }, { width: 10 }, { width: 25 });
            start.add(1, "day");
        }
        wsRaw.columns = columns;
    }
    addMembershipsRow(row) {
        let colIndex = 1;
        let maxRowCount = 1;
        if (row.ppermissions?.length > 0 || row.holidays?.length > 0) {
            maxRowCount++;
        }
        this.setDataCell11({
            rowIndex: this._membershipsRowIndex,
            colIndex: colIndex,
            value: row.userCaptions[0].text.join(" ").toUpperCase(),
            ws: this._wsMemberships,
            border: generator_1.ReportGenerator.Constants.Styling.Dotted,
        });
        this._wsMemberships.mergeCells(this._membershipsRowIndex, colIndex, this._membershipsRowIndex + maxRowCount - 1, colIndex++);
        const start = (0, moment_1.default)(this._options.range.startDateTime).locale(this._locale);
        const end = (0, moment_1.default)(this._options.range.endDateTime).locale(this._locale);
        while (start.isBefore(end)) {
            const ranges = [];
            for (const membership of row.memberships) {
                for (const range of membership.ranges) {
                    if (new moment_range_1.DateRange(range.intersectingRange.startDateTime, range.intersectingRange.endDateTime).intersect(new moment_range_1.DateRange(start.clone().startOf("day"), start.clone().add(1, "day").startOf("day")))) {
                        const intersection = new moment_range_1.DateRange(range.intersectingRange.startDateTime, range.intersectingRange.endDateTime).intersect(new moment_range_1.DateRange(start.clone().startOf("day"), start.clone().add(1, "day").startOf("day")));
                        ranges.push(intersection);
                    }
                }
            }
            const ppermissions = row.ppermissions.filter((d) => (0, moment_1.default)(d.intersectingRange.startDateTime).isSame(start, "day"));
            const holidays = row.holidays.filter((i) => (0, moment_1.default)(i.originalRange.startDateTime).isSame(start, "day"));
            let rowIndex = this._membershipsRowIndex;
            let totalHours = 0;
            let text = "";
            if (ranges?.length > 0) {
                for (const rangeWorkplan of ranges) {
                    const relatedPPermissions = ppermissions.map((p) => new moment_range_1.DateRange(p.intersectingRange.startDateTime, p.intersectingRange.endDateTime).intersect(rangeWorkplan));
                    const relatedHolidays = holidays
                        .filter((h) => new moment_range_1.DateRange(h.originalRange.startDateTime, h.originalRange.endDateTime).intersect(rangeWorkplan))
                        .map((p) => new moment_range_1.DateRange(p.originalRange.startDateTime, p.originalRange.endDateTime).intersect(rangeWorkplan));
                    let ranges = [rangeWorkplan];
                    if (relatedPPermissions?.length > 0) {
                        let intersectRanges = [];
                        for (const ppermissionRange of relatedPPermissions) {
                            intersectRanges.push(rangeWorkplan.intersect(ppermissionRange));
                        }
                        if (intersectRanges.length > 0) {
                            ranges = this.subtractRanges(ranges, intersectRanges);
                        }
                    }
                    if (relatedHolidays?.length > 0) {
                        let intersectRanges = [];
                        for (const holidayRange of relatedHolidays) {
                            intersectRanges.push(rangeWorkplan.intersect(holidayRange));
                        }
                        if (intersectRanges.length > 0) {
                            ranges = this.subtractRanges(ranges, intersectRanges);
                        }
                    }
                    if (ranges?.length > 0) {
                        for (const range of ranges) {
                            text = text && text + "\n\n";
                            text += `${(0, moment_1.default)(range.start).format(generator_1.ReportGenerator.Constants.TimeFormat)} - ${(0, moment_1.default)(range.end).format(generator_1.ReportGenerator.Constants.TimeFormat)}`;
                            totalHours += range.duration("hours");
                        }
                    }
                }
            }
            this.setDataCell11({
                rowIndex: rowIndex++,
                colIndex: colIndex,
                value: text,
                ws: this._wsMemberships,
                border: generator_1.ReportGenerator.Constants.Styling.Dotted,
            });
            let ppermissionsHolidays = "";
            if (ppermissions?.length > 0) {
                ppermissionsHolidays += ppermissions
                    .map((p) => p.ppermissionTypeName + " :\r\n" + (0, moment_1.default)(p.intersectingRange.startDateTime).format("HH:mm") + "-" + (0, moment_1.default)(p.intersectingRange.endDateTime).format("HH:mm"))
                    .join("\r\n");
            }
            if (holidays?.length > 0) {
                ppermissionsHolidays += holidays
                    .map((p) => p.name +
                    " :\r\n" +
                    p.workPlanIntersections.map((w) => w.intersectingRange).map((i) => (0, moment_1.default)(i.startDateTime).format("HH:mm") + "-" + (0, moment_1.default)(i.startDateTime).format("HH:mm")))
                    .join("\r\n");
            }
            if (ppermissionsHolidays) {
                this.setDataCell11({
                    rowIndex: rowIndex++,
                    colIndex: colIndex,
                    value: ppermissionsHolidays,
                    ws: this._wsMemberships,
                    border: generator_1.ReportGenerator.Constants.Styling.Dotted,
                });
            }
            this.setDataCell11({
                rowIndex: this._membershipsRowIndex,
                colIndex: colIndex + 1,
                value: totalHours !== 0 ? totalHours + " " + i18n_1.default.__({ phrase: "REPORT.PACS_UNIT_HOUR", locale: this._locale }) : totalHours,
                ws: this._wsMemberships,
                border: generator_1.ReportGenerator.Constants.Styling.Dotted,
            });
            let membershipPlanNames = [];
            row.memberships.forEach(({ ranges, workPlanName }) => {
                ranges.forEach(({ originalRange: { startDateTime, endDateTime } }) => {
                    const rangeStart = startDateTime ? (0, moment_1.default)(startDateTime) : null;
                    const rangeEnd = endDateTime ? (0, moment_1.default)(endDateTime) : null;
                    const isWithinRange = start.isBetween(rangeStart || start, rangeEnd || start, "day", "[]");
                    if (isWithinRange) {
                        membershipPlanNames.push(workPlanName);
                    }
                });
            });
            const uniqueMembershipPlanNames = [...new Set(membershipPlanNames)].join("\n\n");
            this.setDataCell11({
                rowIndex: this._membershipsRowIndex,
                colIndex: colIndex + 2,
                value: uniqueMembershipPlanNames,
                ws: this._wsMemberships,
                border: generator_1.ReportGenerator.Constants.Styling.Dotted,
            });
            this._wsMemberships.mergeCells(this._membershipsRowIndex, colIndex + 1, this._membershipsRowIndex + maxRowCount - 1, colIndex + 1);
            this._wsMemberships.mergeCells(this._membershipsRowIndex, colIndex + 2, this._membershipsRowIndex + maxRowCount - 1, colIndex + 2);
            colIndex += 3;
            start.add(1, "day");
        }
        this._wsMemberships.getRow(this._membershipsRowIndex).commit();
        this._membershipsRowIndex += maxRowCount;
    }
    subtractRanges(source, others) {
        if (!Array.isArray(source)) {
            source = [source];
        }
        return (0, lodash_1.flatten)(source.map((s) => {
            let remaining = [s];
            (0, lodash_1.flatten)(others).forEach((o) => {
                remaining = (0, lodash_1.flatten)(remaining.map((r) => r.subtract(o)));
            });
            return remaining;
        }));
    }
    addWorkingScheduleRow(row) {
        let colIndex = 1;
        let maxRowCount = 1;
        if (row.ppermissions?.length > 0 || row.holidays?.length > 0) {
            maxRowCount++;
        }
        this.setDataCell11({
            rowIndex: this._workingScheduleRowIndex,
            colIndex: colIndex,
            value: row.userCaptions[0].text.join(" ").toUpperCase(),
            ws: this._wsWorkingSchedule,
            border: generator_1.ReportGenerator.Constants.Styling.Dotted,
        });
        this._wsWorkingSchedule.mergeCells(this._workingScheduleRowIndex, colIndex, this._workingScheduleRowIndex + maxRowCount - 1, colIndex++);
        const start = (0, moment_1.default)(this._options.range.startDateTime).locale(this._locale);
        const end = (0, moment_1.default)(this._options.range.endDateTime).locale(this._locale);
        let index = 0;
        while (start.isBefore(end)) {
            const dayData = row.workingSchedule.find((d) => (0, moment_1.default)(d.date).isSame(start, "day"));
            const ppermissions = row.ppermissions.filter((d) => (0, moment_1.default)(d.intersectingRange.startDateTime).isSame(start, "day"));
            const holidays = row.holidays.filter((i) => (0, moment_1.default)(i.originalRange.startDateTime).isSame(start, "day"));
            let rowIndex = this._workingScheduleRowIndex;
            let totalHours = 0;
            let text = "";
            if (dayData?.workingHours?.length > 0) {
                dayData.workingHours.reverse();
                for (const workingHour of dayData.workingHours) {
                    const relatedPPermissions = ppermissions.map((p) => new moment_range_1.DateRange(p.intersectingRange.startDateTime, p.intersectingRange.endDateTime).intersect(new moment_range_1.DateRange(workingHour.range.startDateTime, workingHour.range.endDateTime)));
                    const relatedHolidays = holidays
                        .filter((h) => new moment_range_1.DateRange(h.originalRange.startDateTime, h.originalRange.endDateTime).intersect(new moment_range_1.DateRange(workingHour.range.startDateTime, workingHour.range.endDateTime)))
                        .map((p) => new moment_range_1.DateRange(p.originalRange.startDateTime, p.originalRange.endDateTime).intersect(new moment_range_1.DateRange(workingHour.range.startDateTime, workingHour.range.endDateTime)));
                    const range = new moment_range_1.DateRange(workingHour.range.startDateTime, workingHour.range.endDateTime);
                    let ranges = [new moment_range_1.DateRange(workingHour.range.startDateTime, workingHour.range.endDateTime)];
                    if (relatedPPermissions?.length > 0) {
                        let intersectRanges = [];
                        for (const ppermissionRange of relatedPPermissions) {
                            intersectRanges.push(range.intersect(ppermissionRange));
                        }
                        if (intersectRanges.length > 0) {
                            ranges = this.subtractRanges(ranges, intersectRanges);
                        }
                    }
                    if (relatedHolidays?.length > 0) {
                        let intersectRanges = [];
                        for (const holidayRange of relatedHolidays) {
                            intersectRanges.push(range.intersect(holidayRange));
                        }
                        if (intersectRanges.length > 0) {
                            ranges = this.subtractRanges(ranges, intersectRanges);
                        }
                    }
                    if (ranges?.length > 0) {
                        for (const range of ranges) {
                            text = text && text + "\n\n";
                            text += `${(0, moment_1.default)(range.start).format(generator_1.ReportGenerator.Constants.TimeFormat)} - ${(0, moment_1.default)(range.end).format(generator_1.ReportGenerator.Constants.TimeFormat)}`;
                            totalHours += range.duration("hours");
                        }
                    }
                }
            }
            this.setDataCell11({
                rowIndex: rowIndex++,
                colIndex: colIndex,
                value: text,
                ws: this._wsWorkingSchedule,
                border: generator_1.ReportGenerator.Constants.Styling.Dotted,
            });
            let ppermissionsHolidays = "";
            if (ppermissions?.length > 0) {
                ppermissionsHolidays += ppermissions
                    .map((p) => p.ppermissionTypeName + " :\r\n" + (0, moment_1.default)(p.intersectingRange.startDateTime).format("HH:mm") + "-" + (0, moment_1.default)(p.intersectingRange.endDateTime).format("HH:mm"))
                    .join("\r\n");
            }
            if (holidays?.length > 0) {
                ppermissionsHolidays += holidays
                    .map((p) => p.name +
                    " :\r\n" +
                    p.workPlanIntersections.map((w) => w.intersectingRange).map((i) => (0, moment_1.default)(i.startDateTime).format("HH:mm") + "-" + (0, moment_1.default)(i.startDateTime).format("HH:mm")))
                    .join("\r\n");
            }
            this.setDataCell11({
                rowIndex: rowIndex++,
                colIndex: colIndex,
                value: ppermissionsHolidays,
                ws: this._wsWorkingSchedule,
                border: generator_1.ReportGenerator.Constants.Styling.Dotted,
            });
            this._totals[index] ? (this._totals[index++] += totalHours) : (this._totals[index++] = totalHours);
            this.setDataCell11({
                rowIndex: this._workingScheduleRowIndex,
                colIndex: colIndex + 1,
                value: totalHours !== 0 ? totalHours + " " + i18n_1.default.__({ phrase: "REPORT.PACS_UNIT_HOUR", locale: this._locale }) : totalHours,
                ws: this._wsWorkingSchedule,
                border: generator_1.ReportGenerator.Constants.Styling.Dotted,
            });
            let workPlanNames = [];
            row.workingSchedule.forEach(({ workingHours }) => {
                workingHours.forEach(({ range: { startDateTime, endDateTime }, workPlanName }) => {
                    const rangeStart = startDateTime ? (0, moment_1.default)(startDateTime) : null;
                    const rangeEnd = endDateTime ? (0, moment_1.default)(endDateTime) : null;
                    const isWithinSchedule = start.isBetween(rangeStart || start, rangeEnd || start, "day", "[]");
                    if (isWithinSchedule) {
                        workPlanNames.push(workPlanName);
                    }
                });
            });
            const uniqueWorkPlanNames = [...new Set(workPlanNames)].join("\n\n");
            this.setDataCell11({
                rowIndex: this._workingScheduleRowIndex,
                colIndex: colIndex + 2,
                value: uniqueWorkPlanNames,
                ws: this._wsWorkingSchedule,
                border: generator_1.ReportGenerator.Constants.Styling.Dotted,
            });
            this._wsWorkingSchedule.mergeCells(this._workingScheduleRowIndex, colIndex + 1, this._workingScheduleRowIndex + maxRowCount - 1, colIndex + 1);
            this._wsWorkingSchedule.mergeCells(this._workingScheduleRowIndex, colIndex + 2, this._workingScheduleRowIndex + maxRowCount - 1, colIndex + 2);
            colIndex += 3;
            start.add(1, "day");
        }
        this._wsWorkingSchedule.getRow(this._workingScheduleRowIndex).commit();
        this._workingScheduleRowIndex += maxRowCount;
    }
    addWorkingScheduleTotalRow() {
        let colIndex = 1;
        this.setDataCell11({
            rowIndex: this._workingScheduleRowIndex,
            colIndex: colIndex++,
            value: i18n_1.default.__({ phrase: "EXCEL-REPORT.SUM", locale: this._locale }),
            ws: this._wsWorkingSchedule,
            border: generator_1.ReportGenerator.Constants.Styling.Dotted,
            backgroundColorCode: "bddd97",
        });
        const start = (0, moment_1.default)(this._options.range.startDateTime).locale(this._locale);
        const end = (0, moment_1.default)(this._options.range.endDateTime).locale(this._locale);
        let index = 0;
        while (start.isBefore(end)) {
            this.setDataCell11({
                rowIndex: this._workingScheduleRowIndex,
                colIndex: colIndex,
                value: this._totals[index] !== 0 ? this._totals[index++] + " " + i18n_1.default.__({ phrase: "REPORT.PACS_UNIT_HOUR", locale: this._locale }) : this._totals[index++],
                ws: this._wsWorkingSchedule,
                border: generator_1.ReportGenerator.Constants.Styling.Dotted,
                backgroundColorCode: "bddd97",
            });
            this._wsWorkingSchedule.mergeCells(this._workingScheduleRowIndex, colIndex, this._workingScheduleRowIndex, colIndex + 2);
            colIndex += 3;
            start.add(1, "day");
        }
        this._wsWorkingSchedule.getRow(this._workingScheduleRowIndex).commit();
        this._workingScheduleRowIndex++;
    }
    addFilterToSheet(ws, startRowIndex, startColIndex, filter, filterlength) {
        this.addFilterCell11(ws, startRowIndex, startColIndex, this._organizationName + " " + i18n_1.default.__({ phrase: "EXCEL-REPORT.PLANNING_CALENDAR_REPORT", locale: this._locale }), generator_1.ReportGenerator.Constants.Styling.AllThin, {
            bold: true,
        });
        this.addFilterCell11(ws, startRowIndex + 1, startColIndex, i18n_1.default.__({ phrase: "EXCEL-REPORT.DATE", locale: this._locale }), generator_1.ReportGenerator.Constants.Styling.AllThin);
        this.addFilterCell11(ws, startRowIndex + 1, startColIndex + 1, (0, moment_1.default)(this._options.range.startDateTime).locale(this._locale).format("LL") + " - " + (0, moment_1.default)(this._options.range.endDateTime).locale(this._locale).format("LL"), generator_1.ReportGenerator.Constants.Styling.AllThin, { italic: true }, generator_1.ReportGenerator.Constants.DateFormat);
        ws.mergeCells(startRowIndex, startColIndex, startRowIndex, startColIndex + 4);
        ws.mergeCells(startRowIndex + 1, startColIndex + 1, startRowIndex + 1, startColIndex + 4);
        ws.mergeCells(startRowIndex + 2, startColIndex + 1, startRowIndex + 2, startColIndex + 4);
        this.addFilterCell11(ws, startRowIndex + 2, startColIndex, i18n_1.default.__({ phrase: "EXCEL-REPORT.APPLIED_FILTERS", locale: this._locale }), generator_1.ReportGenerator.Constants.Styling.AllThin);
        this.addFilterCell11(ws, startRowIndex + 2, startColIndex + 1, filter, generator_1.ReportGenerator.Constants.Styling.AllThin);
        if (filterlength) {
            let row = ws.getRow(startRowIndex + 2);
            row.height = filterlength * 20;
        }
    }
}
exports.PacsPlanningCalendarExcelReport = PacsPlanningCalendarExcelReport;
