"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.UserList = void 0;
const exceljs_1 = __importDefault(require("exceljs"));
const fs_1 = __importDefault(require("fs"));
const i18n_1 = __importDefault(require("i18n"));
const moment_1 = __importDefault(require("moment"));
const path_1 = __importDefault(require("path"));
const uuid_1 = __importDefault(require("uuid"));
const app_config_1 = require("../../../app.config");
const app_enums_1 = require("../../../app.enums");
const enums_1 = require("../../../lib/access-model/v2/enums");
const app_logs_1 = require("../../../app.logs");
const dal_constants_1 = require("../../../dal/dal.constants");
const dal_manager_1 = require("../../../dal/dal.manager");
const generator_1 = require("../generator");
const dynamicFormCurrent_1 = require("../../../lib/es/models/dynamicFormCurrent");
const luxon_1 = require("luxon");
const orgProfilePhraseFieldNames = [
    {
        phrase: "EXCEL-REPORT.NAME",
        fieldName: "name",
    },
    {
        phrase: "EXCEL-REPORT.SURNAME",
        fieldName: "surname",
    },
    {
        phrase: "EXCEL-REPORT.EMAIL",
        fieldName: "email",
    },
    {
        phrase: "EXCEL-REPORT.PHONE_NUMBER",
        fieldName: "phoneNumber",
    },
    {
        phrase: "EXCEL-REPORT.ADDRESS",
        fieldName: "address",
    },
    {
        phrase: "EXCEL-REPORT.USERNAME",
        fieldName: "username",
    },
    {
        phrase: "EXCEL-REPORT.UNIQUE_ID",
        fieldName: "uniqueId",
    },
    {
        phrase: "EXCEL-REPORT.EMPLOYMENT_START_DATE",
        fieldName: "employmentStartUtc",
    },
    {
        phrase: "EXCEL-REPORT.EMPLOYMENT_END_DATE",
        fieldName: "employmentEndUtc",
    },
    {
        phrase: "EXCEL-REPORT.BIRTH_DATE",
        fieldName: "birthDateUtc",
    },
];
class UserList extends generator_1.ReportGenerator {
    constructor(request, locale) {
        super(request, locale);
        this._fileId = uuid_1.default.v4();
        this._filePath = path_1.default.join(app_config_1.appConfig.tmpDirectory, this._fileId + ".xlsx");
        if (fs_1.default.existsSync(this._filePath)) {
            fs_1.default.unlinkSync(this._filePath);
        }
        this._wb = new exceljs_1.default.stream.xlsx.WorkbookWriter({
            filename: this._filePath,
            useStyles: true,
            useSharedStrings: true,
        });
        this._currentDate = (0, moment_1.default)();
        this._options = request.filter;
        this._overriddenFields = [];
    }
    async getExtensionFields() {
        const { rows } = await dal_manager_1.dbManager.organizationTransaction(async (trx) => {
            return await trx.query(` 
			SELECT fields FROM "${this._request.organizationId}"."organizationForms" as orgForms
            WHERE orgForms."organizationId" = $1;`, [this._request.organizationId]);
        }, this._request.requesterUserId, this._request.organizationId);
        if (rows[0]) {
            app_logs_1.logger.debug("Connected. There are %s extension fields!", rows[0].fields.length);
            this._extensionFields = rows[0].fields;
            this.setOverriddenFields();
        }
        else {
            app_logs_1.logger.debug("No extension fields...");
            this._extensionFields = null;
        }
    }
    setOverriddenFields() {
        const overriddenFields = [];
        const orgProfileFieldNames = new Set(orgProfilePhraseFieldNames.map((item) => item.fieldName));
        for (const field of this._extensionFields) {
            if (orgProfileFieldNames.has(field.name)) {
                overriddenFields.push(field);
            }
        }
        this._extensionFields = this._extensionFields.filter((field) => !orgProfileFieldNames.has(field.name));
        this._overriddenFields = overriddenFields;
    }
    addGeneralRow(row) {
        let colIndex = 1;
        const rowData = [
            row.organizationProfile.name?.toUpperCase() ?? "",
            row.organizationProfile.surname?.toUpperCase() ?? "",
            row.organizationProfile.email ?? "",
            row.organizationProfile.phoneNumber ?? "",
            row.organizationProfile.address ?? "",
            row.organizationProfile.roleName ? i18n_1.default.__({ phrase: row.organizationProfile.roleName.toUpperCase(), locale: this._locale }) : "",
            row.organizationProfile.username ?? "",
            row.organizationProfile.uniqueId ?? "",
            row.organizationProfile.birthDateUtc ? luxon_1.DateTime.fromISO(row.organizationProfile.birthDateUtc).toFormat("dd-MM-yyyy") : "",
            row.organizationProfile.employmentStartUtc ? luxon_1.DateTime.fromISO(row.organizationProfile.employmentStartUtc).toFormat("dd-MM-yyyy") : "",
            row.organizationProfile.employmentEndUtc ? luxon_1.DateTime.fromISO(row.organizationProfile.employmentEndUtc).toFormat("dd-MM-yyyy") : "",
            (row.organizationUnits || [])
                .map((ou) => ou.name)
                .join(", ")
                .toUpperCase() ?? "",
            (row.userGroups || [])
                .map((ug) => ug.name)
                .join(", ")
                .toUpperCase() ?? "",
            (row.credentials || [])
                .map((c) => c.data)
                .filter((a) => a)
                .join(", ") ?? "",
        ];
        colIndex = this.addCellsToRow(rowData, colIndex, this._generalRowIndex, this._wsGeneral);
        const extensionFieldsData = row.organizationProfile.extensionFields && Object.keys(row.organizationProfile.extensionFields).length > 0 ? row.organizationProfile.extensionFields : "";
        this.setExtensionCells({
            rowIndex: this._generalRowIndex,
            colIndex: colIndex++,
            value: extensionFieldsData,
            ws: this._wsGeneral,
            border: generator_1.ReportGenerator.Constants.Styling.Dotted,
        });
        this._wsGeneral.getRow(this._generalRowIndex).commit();
        this._generalRowIndex++;
    }
    addOrganizationBasedRow(row) {
        row.organizationUnits
            .map((ou) => {
            return {
                organizationName: ou.name,
                userOrganizationUnitRole: ou.roleName,
            };
        })
            .forEach((params) => {
            let colIndex = 1;
            const rowData = [
                params.organizationName?.toUpperCase() ?? "",
                row.organizationProfile.name?.toUpperCase() ?? "",
                row.organizationProfile.surname?.toUpperCase() ?? "",
                row.organizationProfile.email ?? "",
                row.organizationProfile.phoneNumber ?? "",
                row.organizationProfile.address ?? "",
                params.userOrganizationUnitRole ? i18n_1.default.__({ phrase: params.userOrganizationUnitRole.toUpperCase(), locale: this._locale }) : "",
                row.organizationProfile.username ?? "",
                row.organizationProfile.uniqueId ?? "",
                (row.userGroups || [])
                    .map((ug) => ug.name)
                    .join(", ")
                    .toUpperCase() ?? "",
            ];
            colIndex = this.addCellsToRow(rowData, colIndex, this._organizationBasedRowIndex, this._wsOrganizationBased);
            const extensionFieldsData = row.organizationProfile.extensionFields && Object.keys(row.organizationProfile.extensionFields).length > 0 ? row.organizationProfile.extensionFields : "";
            this.setExtensionCells({
                rowIndex: this._organizationBasedRowIndex,
                colIndex: colIndex++,
                value: extensionFieldsData,
                ws: this._wsOrganizationBased,
                border: generator_1.ReportGenerator.Constants.Styling.Dotted,
            });
            this._wsOrganizationBased.getRow(this._organizationBasedRowIndex).commit();
            this._organizationBasedRowIndex++;
        });
    }
    addUserGroupBasedRow(row) {
        row.userGroups
            .map((ug) => ug.name)
            .forEach((userGroupName) => {
            let colIndex = 1;
            const rowData = [
                userGroupName?.toUpperCase() ?? "",
                row.organizationProfile.name?.toUpperCase() ?? "",
                row.organizationProfile.surname?.toUpperCase() ?? "",
                row.organizationProfile.email ?? "",
                row.organizationProfile.phoneNumber ?? "",
                row.organizationProfile.address ?? "",
                row.organizationProfile.roleName ? i18n_1.default.__({ phrase: row.organizationProfile.roleName.toUpperCase(), locale: this._locale }) : "",
                row.organizationProfile.username ?? "",
                row.organizationProfile.uniqueId ?? "",
                (row.organizationUnits || [])
                    .map((ou) => ou.name)
                    .join(", ")
                    .toUpperCase() ?? "",
            ];
            colIndex = this.addCellsToRow(rowData, colIndex, this._userGroupBasedRowIndex, this._wsUserGroupBased);
            const extensionFieldsData = row.organizationProfile.extensionFields && Object.keys(row.organizationProfile.extensionFields).length > 0 ? row.organizationProfile.extensionFields : "";
            this.setExtensionCells({
                rowIndex: this._userGroupBasedRowIndex,
                colIndex: colIndex++,
                value: extensionFieldsData,
                ws: this._wsUserGroupBased,
                border: generator_1.ReportGenerator.Constants.Styling.Dotted,
            });
            this._wsUserGroupBased.getRow(this._userGroupBasedRowIndex).commit();
            this._userGroupBasedRowIndex++;
        });
    }
    addAccessRightBasedRow(row) {
        row.accessRights
            .map((uar) => uar.accessControlPointName)
            .forEach((accessRight) => {
            let colIndex = 1;
            const rowData = [
                accessRight?.toUpperCase() ?? "",
                row.organizationProfile.name?.toUpperCase() ?? "",
                row.organizationProfile.surname?.toUpperCase() ?? "",
                row.organizationProfile.email ?? "",
                row.organizationProfile.phoneNumber ?? "",
                row.organizationProfile.address ?? "",
                row.organizationProfile.roleName ? i18n_1.default.__({ phrase: row.organizationProfile.roleName.toUpperCase(), locale: this._locale }) : "",
                row.organizationProfile.username ?? "",
                row.organizationProfile.uniqueId ?? "",
                (row.organizationUnits || [])
                    .map((ou) => ou.name)
                    .join(", ")
                    .toUpperCase() ?? "",
                (row.userGroups || [])
                    .map((ug) => ug.name)
                    .join(", ")
                    .toUpperCase() ?? "",
            ];
            colIndex = this.addCellsToRow(rowData, colIndex, this._accessRightBasedRowIndex, this._wsAccessRightBased);
            const extensionFieldsData = row.organizationProfile.extensionFields && Object.keys(row.organizationProfile.extensionFields).length > 0 ? row.organizationProfile.extensionFields : "";
            this.setExtensionCells({
                rowIndex: this._accessRightBasedRowIndex,
                colIndex: colIndex++,
                value: extensionFieldsData,
                ws: this._wsAccessRightBased,
                border: generator_1.ReportGenerator.Constants.Styling.Dotted,
            });
            this._wsAccessRightBased.getRow(this._accessRightBasedRowIndex).commit();
            this._accessRightBasedRowIndex++;
        });
    }
    setExtensionCells(params) {
        if (params.value) {
            Object.keys(params.value).forEach((fieldKey) => {
                let extIndex = this._extensionFields.map((e) => e["name"]).indexOf(fieldKey);
                let colIndex = extIndex + params.colIndex;
                let c = params.ws.getCell(params.rowIndex, colIndex);
                if (this._extensionFields[extIndex] && this._extensionFields[extIndex]["type"]) {
                    const formExtensionFieldDefinition = this._extensionFields[extIndex];
                    switch (formExtensionFieldDefinition.type) {
                        case dynamicFormCurrent_1.FormFieldType.Date:
                            c.value = (0, moment_1.default)(params.value[fieldKey]).format("DD/MM/YYYY");
                            break;
                        case dynamicFormCurrent_1.FormFieldType.Time:
                            c.value = (0, moment_1.default)(params.value[fieldKey]).format("HH:mm:ss");
                            break;
                        case dynamicFormCurrent_1.FormFieldType.DateTime:
                            c.value = (0, moment_1.default)(params.value[fieldKey]).format("DD/MM/YYYY HH:mm:ss");
                            break;
                        case dynamicFormCurrent_1.FormFieldType.Select:
                            c.value = formExtensionFieldDefinition.options.options.find((elem) => elem.value === params.value[fieldKey])?.captionLines?.join("\n") || "NOTFOUND";
                            break;
                        case dynamicFormCurrent_1.FormFieldType.DateRange:
                            const dateRange = (0, moment_1.default)(formExtensionFieldDefinition.options.startCaption).format("DD/MM/YYYY HH:mm:ss") +
                                " - " +
                                (0, moment_1.default)(formExtensionFieldDefinition.options.endCaption).format("DD/MM/YYYY HH:mm:ss");
                            c.value = dateRange;
                            break;
                        default:
                            c.value = params.value[fieldKey];
                            break;
                    }
                    c.alignment = {
                        horizontal: "left",
                        vertical: "middle",
                        wrapText: true,
                    };
                    if (params.border) {
                        c.border = params.border;
                    }
                }
            });
            this._extensionFields
                .map((e) => e["name"])
                .forEach((element) => {
                if (!Object.keys(params.value).includes(element)) {
                    let colIndex = this._extensionFields.map((e) => e["name"]).indexOf(element) + params.colIndex;
                    let c = params.ws.getCell(params.rowIndex, colIndex);
                    c.value = "";
                    c.alignment = {
                        horizontal: "left",
                        vertical: "middle",
                        wrapText: true,
                    };
                    if (params.border) {
                        c.border = params.border;
                    }
                }
            });
        }
        else {
            if (this._extensionFields) {
                for (let i = 0; i < this._extensionFields.length; i++) {
                    let c = params.ws.getCell(params.rowIndex, params.colIndex++);
                    c.value = "";
                    c.alignment = {
                        horizontal: "left",
                        vertical: "middle",
                        wrapText: true,
                    };
                    if (params.border) {
                        c.border = params.border;
                    }
                }
            }
        }
    }
    addFilterToSheet(ws, startRowIndex, startColIndex, filter) {
        this.addFilterCell11(ws, startRowIndex, startColIndex, this._organizationName + " " + i18n_1.default.__({ phrase: "EXCEL-REPORT.USER_LIST", locale: this._locale }), generator_1.ReportGenerator.Constants.Styling.AllThin, {
            bold: true,
        });
        this.addFilterCell11(ws, startRowIndex + 1, startColIndex, i18n_1.default.__({ phrase: "EXCEL-REPORT.FILE_CREATION_DATE", locale: this._locale }), generator_1.ReportGenerator.Constants.Styling.AllThin);
        this.addFilterCell11(ws, startRowIndex + 1, startColIndex + 1, {
            formula: "DATE(" + this._currentDate.year() + "," + (this._currentDate.month() + 1) + "," + this._currentDate.date() + ")",
        }, generator_1.ReportGenerator.Constants.Styling.AllThin, { italic: true }, generator_1.ReportGenerator.Constants.DateFormat);
        this.addFilterCell11(ws, startRowIndex + 2, startColIndex, i18n_1.default.__({ phrase: "EXCEL-REPORT.APPLIED_FILTERS", locale: this._locale }), generator_1.ReportGenerator.Constants.Styling.AllThin);
        ws.mergeCells(startRowIndex, startColIndex, startRowIndex, startColIndex + 2);
        ws.mergeCells(startRowIndex + 1, startColIndex + 1, startRowIndex + 1, startColIndex + 2);
        ws.mergeCells(startRowIndex + 2, startColIndex + 1, startRowIndex + 2, startColIndex + 2);
        this.addFilterCell11(ws, startRowIndex + 2, startColIndex, i18n_1.default.__({ phrase: "EXCEL-REPORT.APPLIED_FILTERS", locale: this._locale }), generator_1.ReportGenerator.Constants.Styling.AllThin);
        this.addFilterCell11(ws, startRowIndex + 2, startColIndex + 1, filter, generator_1.ReportGenerator.Constants.Styling.AllThin);
        let filterlength = filter.split("\r\n").length;
        let row = ws.getRow(startRowIndex + 2);
        row.height = filterlength * 20;
    }
    resolveHeaderCaption(phrase) {
        const matchedField = orgProfilePhraseFieldNames.find((item) => item.phrase === phrase);
        if (matchedField && this._overriddenFields.length) {
            const override = this._overriddenFields.find((field) => field.name === matchedField.fieldName);
            if (override?.caption) {
                return override.caption;
            }
        }
        return i18n_1.default.__({ phrase, locale: this._locale });
    }
    initGeneralSheet(wsRaw, startRowIndex, startColIndex) {
        let columns = [];
        const data = [
            {
                value: this.resolveHeaderCaption("EXCEL-REPORT.NAME"),
                cellWidth: 30,
            },
            {
                value: this.resolveHeaderCaption("EXCEL-REPORT.SURNAME"),
                cellWidth: 30,
            },
            {
                value: this.resolveHeaderCaption("EXCEL-REPORT.EMAIL"),
                cellWidth: 35,
            },
            {
                value: this.resolveHeaderCaption("EXCEL-REPORT.PHONE_NUMBER"),
                cellWidth: 22,
            },
            {
                value: this.resolveHeaderCaption("EXCEL-REPORT.ADDRESS"),
                cellWidth: 30,
            },
            {
                phrase: "EXCEL-REPORT.ROLE",
                cellWidth: 30,
            },
            {
                value: this.resolveHeaderCaption("EXCEL-REPORT.USERNAME"),
                cellWidth: 35,
            },
            {
                value: this.resolveHeaderCaption("EXCEL-REPORT.UNIQUE_ID"),
                cellWidth: 20,
            },
            {
                value: this.resolveHeaderCaption("EXCEL-REPORT.BIRTH_DATE"),
                cellWidth: 20,
            },
            {
                value: this.resolveHeaderCaption("EXCEL-REPORT.EMPLOYMENT_START_DATE"),
                cellWidth: 20,
            },
            {
                value: this.resolveHeaderCaption("EXCEL-REPORT.EMPLOYMENT_END_DATE"),
                cellWidth: 20,
            },
            {
                phrase: "EXCEL-REPORT.ORGANIZATION_UNITS",
                cellWidth: 30,
            },
            {
                phrase: "EXCEL-REPORT.USER_GROUPS",
                cellWidth: 30,
            },
            {
                phrase: "EXCEL-REPORT.CREDENTIALS",
                cellWidth: 30,
            },
        ];
        startColIndex = this.addInitCellsToRow(data, columns, startColIndex, startRowIndex, wsRaw);
        if (this._extensionFields) {
            this._extensionFields.forEach((field) => {
                this.addInitCell11(wsRaw, startRowIndex, startColIndex++, field["caption"], generator_1.ReportGenerator.Constants.Styling.AllThin);
                columns.push({ width: 30 });
            });
        }
        wsRaw.autoFilter = {
            from: "A" + startRowIndex,
            to: wsRaw.getColumn(columns.length).letter + startRowIndex,
        };
        wsRaw.columns = columns;
        wsRaw.views.push({ state: "frozen", ySplit: startRowIndex, activeCell: "A1", showGridLines: false });
    }
    initOrganizationBasedSheet(wsRaw, startRowIndex, startColIndex) {
        let columns = [];
        const data = [
            {
                phrase: "EXCEL-REPORT.ORGANIZATION_UNITS",
            },
            {
                value: this.resolveHeaderCaption("EXCEL-REPORT.NAME"),
            },
            {
                value: this.resolveHeaderCaption("EXCEL-REPORT.SURNAME"),
            },
            {
                value: this.resolveHeaderCaption("EXCEL-REPORT.EMAIL"),
                cellWidth: 35,
            },
            {
                value: this.resolveHeaderCaption("EXCEL-REPORT.PHONE_NUMBER"),
                cellWidth: 22,
            },
            {
                value: this.resolveHeaderCaption("EXCEL-REPORT.ADDRESS"),
            },
            {
                phrase: "EXCEL-REPORT.ROLE",
            },
            {
                value: this.resolveHeaderCaption("EXCEL-REPORT.USERNAME"),
                cellWidth: 35,
            },
            {
                value: this.resolveHeaderCaption("EXCEL-REPORT.UNIQUE_ID"),
                cellWidth: 20,
            },
            {
                phrase: "EXCEL-REPORT.USER_GROUPS",
            },
        ];
        startColIndex = this.addInitCellsToRow(data, columns, startColIndex, startRowIndex, wsRaw);
        if (this._extensionFields) {
            this._extensionFields.forEach((field) => {
                this.addInitCell11(wsRaw, startRowIndex, startColIndex++, field["caption"], generator_1.ReportGenerator.Constants.Styling.AllThin);
                columns.push({ width: 30 });
            });
        }
        wsRaw.autoFilter = {
            from: "A" + startRowIndex,
            to: wsRaw.getColumn(columns.length).letter + startRowIndex,
        };
        wsRaw.columns = columns;
        wsRaw.views.push({ state: "frozen", ySplit: startRowIndex, activeCell: "A1", showGridLines: false });
    }
    initUserGroupBasedSheet(wsRaw, startRowIndex, startColIndex) {
        let columns = [];
        const data = [
            {
                phrase: "EXCEL-REPORT.USER_GROUPS",
            },
            {
                value: this.resolveHeaderCaption("EXCEL-REPORT.NAME"),
            },
            {
                value: this.resolveHeaderCaption("EXCEL-REPORT.SURNAME"),
            },
            {
                value: this.resolveHeaderCaption("EXCEL-REPORT.EMAIL"),
                cellWidth: 35,
            },
            {
                value: this.resolveHeaderCaption("EXCEL-REPORT.PHONE_NUMBER"),
                cellWidth: 22,
            },
            {
                value: this.resolveHeaderCaption("EXCEL-REPORT.ADDRESS"),
            },
            {
                phrase: "EXCEL-REPORT.ROLE",
            },
            {
                value: this.resolveHeaderCaption("EXCEL-REPORT.USERNAME"),
                cellWidth: 35,
            },
            {
                value: this.resolveHeaderCaption("EXCEL-REPORT.UNIQUE_ID"),
                cellWidth: 20,
            },
            {
                phrase: "EXCEL-REPORT.ORGANIZATION_UNITS",
            },
        ];
        startColIndex = this.addInitCellsToRow(data, columns, startColIndex, startRowIndex, wsRaw);
        if (this._extensionFields) {
            this._extensionFields.forEach((field) => {
                this.addInitCell11(wsRaw, startRowIndex, startColIndex++, field["caption"], generator_1.ReportGenerator.Constants.Styling.AllThin);
                columns.push({ width: 30 });
            });
        }
        wsRaw.autoFilter = {
            from: "A" + startRowIndex,
            to: wsRaw.getColumn(columns.length).letter + startRowIndex,
        };
        wsRaw.columns = columns;
        wsRaw.views.push({ state: "frozen", ySplit: startRowIndex, activeCell: "A1", showGridLines: false });
    }
    initAccessRightBasedSheet(wsRaw, startRowIndex, startColIndex) {
        let columns = [];
        const data = [
            {
                phrase: "EXCEL-REPORT.ACCESS_RIGHT",
            },
            {
                value: this.resolveHeaderCaption("EXCEL-REPORT.NAME"),
            },
            {
                value: this.resolveHeaderCaption("EXCEL-REPORT.SURNAME"),
            },
            {
                value: this.resolveHeaderCaption("EXCEL-REPORT.EMAIL"),
                cellWidth: 35,
            },
            {
                value: this.resolveHeaderCaption("EXCEL-REPORT.PHONE_NUMBER"),
                cellWidth: 22,
            },
            {
                value: this.resolveHeaderCaption("EXCEL-REPORT.ADDRESS"),
            },
            {
                phrase: "EXCEL-REPORT.ROLE",
            },
            {
                value: this.resolveHeaderCaption("EXCEL-REPORT.USERNAME"),
                cellWidth: 35,
            },
            {
                value: this.resolveHeaderCaption("EXCEL-REPORT.UNIQUE_ID"),
                cellWidth: 20,
            },
            {
                phrase: "EXCEL-REPORT.ORGANIZATION_UNITS",
            },
            {
                phrase: "EXCEL-REPORT.USER_GROUPS",
            },
        ];
        startColIndex = this.addInitCellsToRow(data, columns, startColIndex, startRowIndex, wsRaw);
        if (this._extensionFields) {
            this._extensionFields.forEach((field) => {
                this.addInitCell11(wsRaw, startRowIndex, startColIndex++, field["caption"], generator_1.ReportGenerator.Constants.Styling.AllThin);
                columns.push({ width: 30 });
            });
        }
        wsRaw.autoFilter = {
            from: "A" + startRowIndex,
            to: wsRaw.getColumn(columns.length).letter + startRowIndex,
        };
        wsRaw.columns = columns;
        wsRaw.views.push({ state: "frozen", ySplit: startRowIndex, activeCell: "A1", showGridLines: false });
    }
    async generateUserListReport() {
        this._organizationName = (await dal_manager_1.dbManager.accessOrganization.getOrganizationBasic(this._request.organizationId)).name;
        app_logs_1.logger.debug("Organization Name : " + this._organizationName);
        await this.getExtensionFields();
        this._wsGeneral = this._wb.addWorksheet(i18n_1.default.__({ phrase: "EXCEL-REPORT.GENERAL", locale: this._locale }), {
            pageSetup: { fitToPage: true, fitToWidth: 1, fitToHeight: 0, paperSize: 9, orientation: "landscape" },
        });
        this._wsGeneral.properties.defaultRowHeight = 15;
        this._generalRowIndex = 6;
        this._wsOrganizationBased = this._wb.addWorksheet(i18n_1.default.__({ phrase: "EXCEL-REPORT.ORGANIZATION_UNIT_BASED_LIST", locale: this._locale }), {
            pageSetup: { fitToPage: true, fitToWidth: 1, fitToHeight: 0, paperSize: 9, orientation: "landscape" },
        });
        this._wsOrganizationBased.properties.defaultRowHeight = 15;
        this._organizationBasedRowIndex = 6;
        this._wsUserGroupBased = this._wb.addWorksheet(i18n_1.default.__({ phrase: "EXCEL-REPORT.USER_GROUP_BASED_LIST", locale: this._locale }), {
            pageSetup: { fitToPage: true, fitToWidth: 1, fitToHeight: 0, paperSize: 9, orientation: "landscape" },
        });
        this._wsUserGroupBased.properties.defaultRowHeight = 15;
        this._userGroupBasedRowIndex = 6;
        this._wsAccessRightBased = this._wb.addWorksheet(i18n_1.default.__({ phrase: "EXCEL-REPORT.ACCESS_RIGHT_BASED_LIST", locale: this._locale }), {
            pageSetup: { fitToPage: true, fitToWidth: 1, fitToHeight: 0, paperSize: 9, orientation: "landscape" },
        });
        this._wsAccessRightBased.properties.defaultRowHeight = 15;
        this._accessRightBasedRowIndex = 6;
        let { caption: requestedUserCaption } = await dal_manager_1.dbManager.accessRedisCache.getUserBadgeCache({ organizationId: this._request.organizationId, userId: this._request.requesterUserId });
        if (!requestedUserCaption) {
            throw new Error("Not found");
        }
        this._requestedUserCaption = requestedUserCaption[0].text[0];
        this._wb.creator = this._requestedUserCaption;
        let filters = [];
        let filter = this._options;
        if (filter.status) {
            filters.push(i18n_1.default.__({ phrase: "EXCEL-REPORT.STATUS", locale: this._locale }) +
                " : " +
                i18n_1.default.__({ phrase: "EXCEL-REPORT." + dal_constants_1.DalConstants.IdentityStatusType[filter.status].toUpperCase(), locale: this._locale }));
        }
        if (filter.user) {
            if (filter.user.accountEnabled) {
                filters.push(i18n_1.default.__({ phrase: "EXCEL-REPORT.USER_ACCOUNT", locale: this._locale }) + " : " + i18n_1.default.__({ phrase: "EXCEL-REPORT.ACTIVE", locale: this._locale }));
            }
            else {
                filters.push(i18n_1.default.__({ phrase: "EXCEL-REPORT.USER_ACCOUNT", locale: this._locale }) + " : " + i18n_1.default.__({ phrase: "EXCEL-REPORT.PASSIVE", locale: this._locale }));
            }
            if (filter.user.username) {
                filters.push(i18n_1.default.__({ phrase: "EXCEL-REPORT.USERNAME", locale: this._locale }) + " : " + filter.user.username);
            }
        }
        if (filter.profileFilter) {
            filters.push(i18n_1.default.__({ phrase: "EXCEL-REPORT.PROFILE_FILTER", locale: this._locale }) + " : " + filter.profileFilter);
        }
        if (filter.userOrganization) {
            if (!filter.userOrganization.isDisabled) {
                filters.push(i18n_1.default.__({ phrase: "EXCEL-REPORT.USER", locale: this._locale }) + " : " + i18n_1.default.__({ phrase: "EXCEL-REPORT.ACTIVE", locale: this._locale }));
            }
            else {
                filters.push(i18n_1.default.__({ phrase: "EXCEL-REPORT.USER", locale: this._locale }) + " : " + i18n_1.default.__({ phrase: "EXCEL-REPORT.PASSIVE", locale: this._locale }));
            }
            if (filter.userOrganization.roleId) {
                let roleName;
                let organizationRoles = await dal_manager_1.dbManager.accessUser.listRolesOfOrganization(this._request.organizationId);
                if (organizationRoles) {
                    organizationRoles.forEach((role) => {
                        if (role.id === filter.userOrganization.roleId) {
                            roleName = role.name;
                        }
                    });
                }
                app_logs_1.logger.debug("Role name : " + roleName);
                filters.push(i18n_1.default.__({ phrase: "EXCEL-REPORT.ROLE", locale: this._locale }) + " : " + i18n_1.default.__({ phrase: roleName, locale: this._locale }));
            }
        }
        if (filter.userOrganizationProfile) {
            if (filter.userOrganizationProfile.uniqueId) {
                filters.push(i18n_1.default.__({ phrase: "EXCEL-REPORT.UNIQUE_ID", locale: this._locale }) + " : " + filter.userOrganizationProfile.uniqueId);
            }
            if (filter.userOrganizationProfile.name) {
                filters.push(i18n_1.default.__({ phrase: "EXCEL-REPORT.NAME", locale: this._locale }) + " : " + filter.userOrganizationProfile.name);
            }
            if (filter.userOrganizationProfile.surname) {
                filters.push(i18n_1.default.__({ phrase: "EXCEL-REPORT.SURNAME", locale: this._locale }) + " : " + filter.userOrganizationProfile.surname);
            }
            if (filter.userOrganizationProfile.email) {
                filters.push(i18n_1.default.__({ phrase: "EXCEL-REPORT.EMAIL", locale: this._locale }) + " : " + filter.userOrganizationProfile.email);
            }
            if (filter.userOrganizationProfile.phoneNumber) {
                filters.push(i18n_1.default.__({ phrase: "EXCEL-REPORT.PHONE_NUMBER", locale: this._locale }) + " : " + filter.userOrganizationProfile.phoneNumber);
            }
            if (filter.userOrganizationProfile.address) {
                filters.push(i18n_1.default.__({ phrase: "EXCEL-REPORT.ADDRESS", locale: this._locale }) + " : " + filter.userOrganizationProfile.address);
            }
        }
        if (filter.organizationUnits && filter.organizationUnits[0]) {
            let organizationUnitInfo = await dal_manager_1.dbManager.accessOrganizationUnit.getIdNamePairsOfOrganizationUnits(this._request.organizationId, [
                filter.organizationUnits[0].id,
            ]);
            let organizationUnitName = organizationUnitInfo[0].name;
            app_logs_1.logger.debug("Organization Unit Name : " + organizationUnitName);
            filters.push(i18n_1.default.__({ phrase: "EXCEL-REPORT.ORGANIZATION_UNITS", locale: this._locale }) + " : " + organizationUnitName);
            if (filter.organizationUnits[0].includeDescendants) {
                filters.push(i18n_1.default.__({ phrase: "EXCEL-REPORT.INCLUDE_DESCENDANTS", locale: this._locale }) + " : " + i18n_1.default.__({ phrase: "EXCEL-REPORT.YES", locale: this._locale }));
            }
        }
        if (filter.organizationUnit && filter.organizationUnit.id) {
            let organizationUnitInfo = await dal_manager_1.dbManager.accessOrganizationUnit.getIdNamePairsOfOrganizationUnits(this._request.organizationId, [
                filter.organizationUnit.id,
            ]);
            let organizationUnitName = organizationUnitInfo[0].name;
            app_logs_1.logger.debug("Organization Unit Name : " + organizationUnitName);
            filters.push(i18n_1.default.__({ phrase: "EXCEL-REPORT.ORGANIZATION_UNITS", locale: this._locale }) + " : " + organizationUnitName);
            if (filter.organizationUnit.includeDescendants) {
                filters.push(i18n_1.default.__({ phrase: "EXCEL-REPORT.INCLUDE_DESCENDANTS", locale: this._locale }) + " : " + i18n_1.default.__({ phrase: "EXCEL-REPORT.YES", locale: this._locale }));
            }
        }
        if (filter.userOrganizationCredential) {
            if (filter.userOrganizationCredential.type) {
                filters.push(i18n_1.default.__({ phrase: "EXCEL-REPORT.USER_CREDENTIAL_TYPE", locale: this._locale }) +
                    " : " +
                    i18n_1.default.__({ phrase: "CREDENTIALTYPE." + enums_1.EnumsV2.CredentialType[filter.userOrganizationCredential.type].toUpperCase(), locale: this._locale }));
            }
            if (filter.userOrganizationCredential.data) {
                filters.push(i18n_1.default.__({ phrase: "EXCEL-REPORT.USER_CREDENTIAL_DATA", locale: this._locale }) + " : " + filter.userOrganizationCredential.data);
            }
        }
        if (filter.createdDate) {
            if (filter.createdDate.createdAtStart && filter.createdDate.createdAtEnd) {
                filters.push(i18n_1.default.__({ phrase: "EXCEL-REPORT.USER_CREATION_DATE", locale: this._locale }) +
                    " : " +
                    (0, moment_1.default)(filter.createdDate.createdAtStart).format("L") +
                    " - " +
                    (0, moment_1.default)(filter.createdDate.createdAtEnd).format("L"));
            }
        }
        if (filter.userGroup && filter.userGroup.id) {
            let userGroupName = (await dal_manager_1.dbManager.accessUserGroup.listUserGroupsByIds(this._request.organizationId, [filter.userGroup.id]))[0].name;
            app_logs_1.logger.debug("User group : " + userGroupName);
            filters.push(i18n_1.default.__({ phrase: "EXCEL-REPORT.USER_GROUPS", locale: this._locale }) + " : " + userGroupName);
        }
        if (filter.userGroupIds && filter.userGroupIds[0]) {
            let userGroupName = (await dal_manager_1.dbManager.accessUserGroup.listUserGroupsByIds(this._request.organizationId, filter.userGroupIds))[0].name;
            app_logs_1.logger.debug("User group : " + userGroupName);
            filters.push(i18n_1.default.__({ phrase: "EXCEL-REPORT.USER_GROUPS", locale: this._locale }) + " : " + userGroupName);
        }
        if (filters.length === 0) {
            filters.push(i18n_1.default.__({ phrase: "EXCEL-REPORT.NOT_EXISTING", locale: this._locale }));
        }
        app_logs_1.logger.debug("\n\nApplied Filters:\n");
        app_logs_1.logger.debug(filters.join("\r\n"));
        app_logs_1.logger.debug("\n\n");
        this.addFilterToSheet(this._wsGeneral, 1, 1, filters.join("\r\n"));
        this.initGeneralSheet(this._wsGeneral, 5, 1);
        this.addFilterToSheet(this._wsOrganizationBased, 1, 1, filters.join("\r\n"));
        this.initOrganizationBasedSheet(this._wsOrganizationBased, 5, 1);
        this.addFilterToSheet(this._wsUserGroupBased, 1, 1, filters.join("\r\n"));
        this.initUserGroupBasedSheet(this._wsUserGroupBased, 5, 1);
        this.addFilterToSheet(this._wsAccessRightBased, 1, 1, filters.join("\r\n"));
        this.initAccessRightBasedSheet(this._wsAccessRightBased, 5, 1);
        console.log("Generating report...");
        await dal_manager_1.dbManager.accessIdentity.fetchAllDataIdentity(this._options, this._request.organizationId, this.onDataCallbackFunction.bind(this));
        console.log("end of fetching");
        this._wsGeneral.pageSetup.printArea = "A6:" + this._wsGeneral.getColumn(this._wsGeneral.columns.length).letter + this._generalRowIndex;
        this._wsGeneral.pageSetup.printTitlesRow = "5:5";
        this._wsOrganizationBased.pageSetup.printArea = "A6:" + this._wsOrganizationBased.getColumn(this._wsOrganizationBased.columns.length).letter + this._organizationBasedRowIndex;
        this._wsOrganizationBased.pageSetup.printTitlesRow = "5:5";
        this._wsUserGroupBased.pageSetup.printArea = "A6:" + this._wsUserGroupBased.getColumn(this._wsUserGroupBased.columns.length).letter + this._userGroupBasedRowIndex;
        this._wsUserGroupBased.pageSetup.printTitlesRow = "5:5";
        this._wsAccessRightBased.pageSetup.printArea = "A6:" + this._wsAccessRightBased.getColumn(this._wsAccessRightBased.columns.length).letter + this._accessRightBasedRowIndex;
        this._wsAccessRightBased.pageSetup.printTitlesRow = "5:5";
        this._wsGeneral.commit();
        this._wsOrganizationBased.commit();
        this._wsUserGroupBased.commit();
        this._wsAccessRightBased.commit();
        await this._wb.commit();
        app_logs_1.logger.debug("File is ready -> " + this._filePath);
        return {
            format: app_enums_1.enums.ReportFormat.Excel,
            preparedReportId: this._fileId,
            filePath: this._filePath,
            notificationType: undefined,
            reportType: app_enums_1.enums.ReportCode.Identity,
            sendFileInAttachments: undefined,
        };
    }
    async onDataCallbackFunction(rows) {
        app_logs_1.logger.debug("Fetch data %s rows", rows.length);
        for (const row of rows) {
            this.addGeneralRow(row);
            this.addOrganizationBasedRow(row);
            this.addUserGroupBasedRow(row);
            this.addAccessRightBasedRow(row);
        }
        return Promise.resolve();
    }
    async generateReport() {
        return await this.generateUserListReport();
    }
}
exports.UserList = UserList;
