"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.FirstAndLastAccessLogsExcelReport = void 0;
const luxon_1 = require("luxon");
const i18n_1 = __importDefault(require("i18n"));
const app_logs_1 = require("../../../app.logs");
const app_enums_1 = require("../../../app.enums");
const generator_1 = require("../generator");
const business_report_export_1 = require("../business.report.export");
const excel_report_refactory_1 = require("../excel.report.refactory");
class FirstAndLastAccessLogsExcelReport extends generator_1.ReportGenerator {
    constructor(request, locale) {
        super(request, locale);
        this._options = request.filter;
    }
    async generateReport() {
        try {
            const data = await (0, business_report_export_1.generateFirstAndLastAccessLogsReportData)(this._request.organizationId, this._request.requesterUserId, this._options.hasOrganizationWide, this._options);
            const userIds = [...new Set(data.report.items.map((item) => item.user.id))];
            const generalSheetData = data.report.items.map((item) => ({
                userId: item.user.id,
                cells: [
                    {
                        value: item.firstUtc ? luxon_1.DateTime.fromJSDate(item.firstUtc).toFormat(generator_1.ReportGenerator.Constants.LuxonDateTimeFormat) + "\r\n" + item.firstAccessControlPoint.name : "",
                        alignment: {
                            wrapText: true,
                        },
                    },
                    {
                        value: item.lastUtc ? luxon_1.DateTime.fromJSDate(item.lastUtc).toFormat(generator_1.ReportGenerator.Constants.LuxonDateTimeFormat) + "\r\n" + item.lastAccessControlPoint.name : "",
                        alignment: {
                            wrapText: true,
                        },
                    },
                ],
            }));
            const sheetInfo = [
                {
                    sheetName: i18n_1.default.__({ phrase: "EXCEL-REPORT.GENERAL", locale: this._locale }),
                    headers: [
                        {
                            phrase: "EXCEL-REPORT.FIRST_ACCESS",
                            width: 25,
                        },
                        {
                            phrase: "EXCEL-REPORT.LAST_ACCESS",
                            width: 25,
                        },
                    ],
                    data: generalSheetData,
                },
            ];
            const reportInfoSheetData = {
                reportNamePhrase: "EXCEL-REPORT.FIRST_LAST_ACCESS_LOGS_REPORT",
                filters: {
                    orgUnitIds: this._options.organizationUnitIds,
                    userIds: this._options.userIds,
                    userGroupIds: this._options.userGroupIds,
                    userOrganizationStatus: this._options.status,
                    regionIds: this._options.regionIds,
                    accessControlPointIds: this._options.accessControlPointIds,
                    applyOrganizationUnitFilterHierarchically: this._options.filterOrganizationUnitMembersHierarchically,
                    dateTimeRange: { startDateTime: this._options.dateRange.startDateTime, endDateTime: this._options.dateRange.endDateTime },
                },
            };
            const reportFactory = new excel_report_refactory_1.ReportFactory(this._request.organizationId, this._locale);
            await reportFactory.initialize({
                requesterUserId: this._request.requesterUserId,
                userIds,
                reportInfoSheetData,
                sheetInfo,
            });
            reportFactory.appendData(sheetInfo);
            const fileObj = await reportFactory.finalize(reportInfoSheetData);
            return {
                format: app_enums_1.enums.ReportFormat.Excel,
                preparedReportId: fileObj.fileId,
                filePath: fileObj.filePath,
                notificationType: undefined,
                reportType: app_enums_1.enums.ReportCode.PacsMonthlyWorkingHoursSummaryExtraWork,
                sendFileInAttachments: undefined,
            };
        }
        catch (error) {
            app_logs_1.logger.error(`Error generating FirstAndLastAccessLogsExcelReport: ${error.message}`, {
                stack: error.stack,
                organizationId: this._request.organizationId,
                requesterUserId: this._request.requesterUserId,
            });
            throw new Error("Failed to generate FirstAndLastAccessLogsExcelReport");
        }
    }
}
exports.FirstAndLastAccessLogsExcelReport = FirstAndLastAccessLogsExcelReport;
