"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.CredentialsExcelReport = void 0;
const fs_1 = __importDefault(require("fs"));
const path_1 = __importDefault(require("path"));
const exceljs_1 = __importDefault(require("exceljs"));
const moment_1 = __importDefault(require("moment"));
const app_config_1 = require("../../../app.config");
const dal_manager_1 = require("../../../dal/dal.manager");
const dal_constants_1 = require("../../../dal/dal.constants");
const uuid_1 = __importDefault(require("uuid"));
const i18n_1 = __importDefault(require("i18n"));
const app_logs_1 = require("../../../app.logs");
const app_enums_1 = require("../../../app.enums");
const report_util_1 = require("../report.util");
const generator_1 = require("../generator");
class CredentialsExcelReport extends generator_1.ReportGenerator {
    constructor(request, locale) {
        super(request, locale);
        this._fileId = uuid_1.default.v4();
        this._filePath = path_1.default.join(app_config_1.appConfig.tmpDirectory, this._fileId + ".xlsx");
        if (fs_1.default.existsSync(this._filePath)) {
            fs_1.default.unlinkSync(this._filePath);
        }
        this._wb = new exceljs_1.default.stream.xlsx.WorkbookWriter({
            filename: this._filePath,
            useStyles: true,
            useSharedStrings: true,
        });
        this._currentDate = (0, moment_1.default)();
        this._options = request.filter;
    }
    addRow(row) {
        let colIndex = 1;
        let credentialInfo = this.createCellCredential(row);
        const rowData = [
            credentialInfo.description,
            (credentialInfo.dataText + "\r\n" + credentialInfo.decData),
            credentialInfo.expDate,
            (row.userProfile.name + " " + row.userProfile.surname),
            row.userProfile.uniqueId ? row.userProfile.uniqueId : "",
            row.userProfile.phoneNumber ? row.userProfile.phoneNumber : "",
            row.userProfile.email ? row.userProfile.email : "",
        ];
        if (this._organizationForms && this._organizationForms.length > 0) {
            let userOrganizationForm = this._organizationForms.find((f) => f.type === dal_constants_1.DalConstants.OrganizationFormType.UserOrganizationProfile);
            if (userOrganizationForm && userOrganizationForm.fields && userOrganizationForm.fields.length > 0) {
                for (const form of userOrganizationForm.fields) {
                    if (row.userProfile.extensionFields) {
                        let extensionField = row.userProfile.extensionFields.find((f) => f.name === form.name);
                        rowData.push(extensionField && extensionField.value ? extensionField.value : "");
                    }
                }
            }
        }
        rowData.push(this.createCellInformation(row));
        colIndex = this.addCellsToRow(rowData, colIndex, this._generalRowIndex, this._wsGeneral);
        this._wsGeneral.getRow(this._generalRowIndex).commit();
        this._generalRowIndex++;
    }
    async createFilters() {
        let locale = this._locale;
        let dateRange = "";
        if (this._options.dateRange && this._options.dateRange.startDateTime) {
            dateRange += (0, moment_1.default)(this._options.dateRange.startDateTime).locale(locale).format("DD/MM/YYYY HH:mm");
        }
        if (this._options.dateRange && this._options.dateRange.endDateTime) {
            dateRange += " - " + (0, moment_1.default)(this._options.dateRange.endDateTime).locale(locale).format("DD/MM/YYYY HH:mm");
        }
        let filterResult = await dal_manager_1.dbManager.accessReport.collectCredentialFilterInfo(this._request.organizationId, this._options);
        let users = "";
        if (filterResult.userCaptions != null && filterResult.userCaptions.length > 0) {
            let _res = " ";
            filterResult.userCaptions.forEach((userCaption, index) => {
                _res += " ";
                userCaption.captionLines.forEach((item, _index) => {
                    if (item.text != null && item.text != "__avatar") {
                        item.text.forEach((element) => {
                            if (element != null)
                                _res += element + (_index < filterResult.userCaptions.length - 1 ? " " : " ");
                        });
                    }
                });
                _res += index < filterResult.userCaptions.length - 1 ? ", " : " ";
            });
            users = _res;
        }
        let organizationUnits = "";
        if (filterResult.organizationUnits != null && filterResult.organizationUnits.length > 0) {
            let res = "";
            filterResult.organizationUnits.forEach((org, index) => {
                res += org.name + (index < filterResult.organizationUnits.length - 1 ? ", " : " ");
            });
            organizationUnits = res;
        }
        let userGroups = "";
        if (filterResult.userGroups != null && filterResult.userGroups.length > 0) {
            let res = "";
            filterResult.userGroups.forEach((userGroup, index) => {
                res += userGroup.name + (index < filterResult.userGroups.length - 1 ? "\r\n" : " ");
            });
            userGroups = res;
        }
        let data = "";
        if (this._options.data)
            data = this._options.data;
        let credentialType = "";
        if (this._options.credentialType != null) {
            let res = report_util_1.ReportConstants.CredentialTypes.find((credentialType) => credentialType.type == this._options.credentialType);
            if (res && res.description)
                credentialType += i18n_1.default.__({ phrase: "PDF-REPORT." + res.description, locale });
        }
        return {
            data: data,
            credentialType: credentialType,
            dateRange: dateRange,
            users: users,
            organizationUnits: organizationUnits,
            userGroups: userGroups,
        };
    }
    async addFilterToSheet(ws, startRowIndex, startColIndex) {
        let filterResult = await this.createFilters();
        let filtersArr = [];
        if (filterResult.data) {
            filtersArr.push(i18n_1.default.__({ phrase: "EXCEL-REPORT.CREDENTIAL_DATA", locale: this._locale }) + " : " + filterResult.data);
        }
        if (filterResult.credentialType) {
            filtersArr.push(i18n_1.default.__({ phrase: "EXCEL-REPORT.CREDENTIAL_TYPE", locale: this._locale }) + " : " + filterResult.credentialType);
        }
        if (filterResult.dateRange) {
            filtersArr.push(i18n_1.default.__({ phrase: "EXCEL-REPORT.EXPIRATION_DATE_RANGE", locale: this._locale }) + " : " + filterResult.dateRange);
        }
        if (filterResult.users) {
            filtersArr.push(i18n_1.default.__({ phrase: "EXCEL-REPORT.USER", locale: this._locale }) + " : " + filterResult.users);
        }
        if (filterResult.organizationUnits) {
            filtersArr.push(i18n_1.default.__({ phrase: "EXCEL-REPORT.ORGANIZATION_UNITS", locale: this._locale }) + " : " + filterResult.organizationUnits);
        }
        if (filterResult.userGroups) {
            filtersArr.push(i18n_1.default.__({ phrase: "PDF-REPORT.USER_GROUPS", locale: this._locale }) + " : " + filterResult.userGroups);
        }
        if (filtersArr.length === 0) {
            filtersArr.push(i18n_1.default.__({ phrase: "EXCEL-REPORT.NOT_EXISTING", locale: this._locale }));
        }
        app_logs_1.logger.debug("\n\nApplied Filters:\n");
        app_logs_1.logger.debug(filtersArr.join("\r\n"));
        app_logs_1.logger.debug("\n\n");
        this.addFilterCell11(ws, startRowIndex, startColIndex, this._organizationName + " " + i18n_1.default.__({ phrase: "EXCEL-REPORT.CREDENTIALS", locale: this._locale }), generator_1.ReportGenerator.Constants.Styling.AllThin, {
            bold: true,
        });
        this.addFilterCell11(ws, startRowIndex + 1, startColIndex, i18n_1.default.__({ phrase: "EXCEL-REPORT.FILE_CREATION_DATE", locale: this._locale }), generator_1.ReportGenerator.Constants.Styling.AllThin);
        this.addFilterCell11(ws, startRowIndex + 1, startColIndex + 1, this._currentDate.locale(this._locale).format("LL") + " " + this._currentDate.locale(this._locale).format("HH:mm:ss"), generator_1.ReportGenerator.Constants.Styling.AllThin, { italic: true }, generator_1.ReportGenerator.Constants.DateFormat);
        this.addFilterCell11(ws, startRowIndex + 2, startColIndex, i18n_1.default.__({ phrase: "EXCEL-REPORT.APPLIED_FILTERS", locale: this._locale }), generator_1.ReportGenerator.Constants.Styling.AllThin);
        ws.mergeCells(startRowIndex, startColIndex, startRowIndex, startColIndex + 2);
        ws.mergeCells(startRowIndex + 1, startColIndex + 1, startRowIndex + 1, startColIndex + 2);
        ws.mergeCells(startRowIndex + 2, startColIndex + 1, startRowIndex + 2, startColIndex + 2);
        this.addFilterCell11(ws, startRowIndex + 2, startColIndex, i18n_1.default.__({ phrase: "EXCEL-REPORT.APPLIED_FILTERS", locale: this._locale }), generator_1.ReportGenerator.Constants.Styling.AllThin);
        this.addFilterCell11(ws, startRowIndex + 2, startColIndex + 1, filtersArr.join("\r\n"), generator_1.ReportGenerator.Constants.Styling.AllThin);
        let filterlength = filtersArr.length;
        let row = ws.getRow(startRowIndex + 2);
        row.height = filterlength * 20;
    }
    initSheet(wsRaw, startRowIndex, startColIndex) {
        let columns = [];
        let data = [
            {
                phrase: "EXCEL-REPORT.CREDENTIAL_TYPE",
                cellWidth: 30
            },
            {
                phrase: "EXCEL-REPORT.CREDENTIAL_DATA",
                cellWidth: 30
            },
            {
                phrase: "EXCEL-REPORT.CREDENTIAL_EXPIRATION",
                cellWidth: 30
            },
            {
                phrase: "EXCEL-REPORT.ACCESS_LOGS_USER",
                cellWidth: 30
            },
            {
                phrase: "IDENTITY.UNIQUEID",
                cellWidth: 30
            },
            {
                phrase: "IDENTITY.PHONE_NUMBER",
                cellWidth: 30
            },
            {
                phrase: "IDENTITY.EMAIL",
                cellWidth: 30
            }
        ];
        let formData = [];
        if (this._organizationForms && this._organizationForms.length > 0) {
            let userOrganizationForm = this._organizationForms.find((f) => f.type === dal_constants_1.DalConstants.OrganizationFormType.UserOrganizationProfile);
            if (userOrganizationForm && userOrganizationForm.fields && userOrganizationForm.fields.length > 0) {
                for (const form of userOrganizationForm.fields) {
                    formData.push({
                        value: form.caption,
                        cellWidth: 25
                    });
                }
            }
        }
        data = [...data, ...formData];
        data.push({
            phrase: "EXCEL-REPORT.INFORMATION",
            cellWidth: 35
        });
        startColIndex = this.addInitCellsToRow(data, columns, startColIndex, startRowIndex, wsRaw);
        wsRaw.autoFilter = {
            from: "A" + startRowIndex,
            to: wsRaw.getColumn(columns.length).letter + startRowIndex,
        };
        wsRaw.columns = columns;
        wsRaw.views.push({ state: "frozen", ySplit: startRowIndex, activeCell: "A1", showGridLines: false });
    }
    createCellCredential(record) {
        let locale = this._locale;
        let credentialDescription = report_util_1.ReportConstants.CredentialTypes.find((credentialType) => credentialType.type == record.credentialType);
        let dataText = (0, report_util_1.getDataText)(record.data, record.credentialType);
        let decData = (0, report_util_1.reverseCredentialData)(record.data, record.credentialType);
        let description = "";
        let data = "";
        let decDataText = "";
        let expDate = "";
        if (credentialDescription && credentialDescription.description)
            description = i18n_1.default.__({ phrase: "PDF-REPORT." + credentialDescription.description, locale });
        if (dataText && credentialDescription.type !== app_enums_1.enums.CredentialType.QrCode) {
            if (credentialDescription.description === "CREDENTIALTYPE_FINGER_PRINT") {
                data = i18n_1.default.__({ phrase: "PDF-REPORT." + dataText, locale: this._locale });
            }
            else {
                data = dataText;
            }
        }
        if (decData && record.credentialType !== app_enums_1.enums.CredentialType.QrCode) {
            decDataText = decData;
        }
        if (record.expiresOn && record.credentialType !== app_enums_1.enums.CredentialType.QrCode)
            expDate = (0, moment_1.default)(record.expiresOn).locale(this._locale).format("LL dddd");
        return {
            description: description,
            dataText: data,
            decData: decDataText,
            expDate: expDate,
        };
    }
    createCellInformation(record) {
        let locale = this._locale;
        let hasForbid = true;
        let nowForbidden = false;
        let numberOfForbids;
        let note = "";
        if (record.forbids && record.forbids.length > 0) {
            hasForbid = true;
            nowForbidden = false;
            numberOfForbids = record.forbids.length;
            record.forbids.forEach((forbid) => {
                if ((0, moment_1.default)(new Date()).isBetween(forbid.startUtc, forbid.endUtc)) {
                    nowForbidden = true;
                }
                if (forbid.note) {
                    note += forbid.note + "\r\n";
                }
            });
        }
        else {
            hasForbid = false;
            nowForbidden = false;
        }
        let result = "";
        if (nowForbidden) {
            result += i18n_1.default.__({ phrase: "EXCEL-REPORT.USER_FORBIDDEN", locale });
        }
        if (hasForbid) {
            result += "\r\n" + i18n_1.default.__({ phrase: "EXCEL-REPORT.FORBIDDEN_IDENTITY_HAS_RECORDS", locale }, { numberOfForbids: numberOfForbids });
        }
        else {
            result += "\r\n" + i18n_1.default.__({ phrase: "EXCEL-REPORT.FORBIDDEN_IDENTITY_NO_RECORDS", locale });
        }
        if (note != "") {
            result += "\r\n" + i18n_1.default.__({ phrase: "EXCEL-REPORT.CREDENTIAL_TITLE_NOTE", locale });
            result += "\r\n" + note;
        }
        return result;
    }
    async generateCredentialReport() {
        this._organizationName = (await dal_manager_1.dbManager.accessOrganization.getOrganizationBasic(this._request.organizationId)).name;
        app_logs_1.logger.debug("Organization Name : " + this._organizationName);
        this._wsGeneral = this._wb.addWorksheet(i18n_1.default.__({ phrase: "EXCEL-REPORT.GENERAL", locale: this._locale }), {
            pageSetup: { fitToPage: true, fitToWidth: 1, fitToHeight: 0, paperSize: 9, orientation: "landscape" },
        });
        this._wsGeneral.properties.defaultRowHeight = 15;
        this._generalRowIndex = 6;
        let { caption: requestedUserCaption } = await dal_manager_1.dbManager.accessRedisCache.getUserBadgeCache({ organizationId: this._request.organizationId, userId: this._request.requesterUserId });
        if (!requestedUserCaption) {
            throw new Error("Not found");
        }
        this._requestedUserCaption = requestedUserCaption[0].text[0];
        this._wb.creator = this._requestedUserCaption;
        this._organizationForms = await dal_manager_1.dbManager.accessOrganization.getOrganizationForms(this._request.organizationId);
        await this.addFilterToSheet(this._wsGeneral, 1, 1);
        this.initSheet(this._wsGeneral, 5, 1);
        console.log("Generating report...");
        await dal_manager_1.dbManager.accessIdentity.searchCredentialsForExcel(this._request.organizationId, this._request.requesterUserId, this.onDataCallbackFunction.bind(this), this._options);
        console.log("end of fetching");
        this._wsGeneral.pageSetup.printArea = "A1:" + this._wsGeneral.getColumn(this._wsGeneral.columns.length).letter + this._generalRowIndex;
        this._wsGeneral.pageSetup.printTitlesRow = "5:5";
        this._wsGeneral.commit();
        await this._wb.commit();
        console.log("File is ready -> " + this._filePath);
        return {
            format: app_enums_1.enums.ReportFormat.Excel,
            preparedReportId: this._fileId,
            filePath: this._filePath,
            notificationType: undefined,
            reportType: app_enums_1.enums.ReportCode.Credential,
            sendFileInAttachments: undefined,
        };
    }
    async onDataCallbackFunction(rows) {
        app_logs_1.logger.debug("Fetch data %s rows", rows.length);
        for (const row of rows) {
            this.addRow(row);
        }
        return Promise.resolve();
    }
    async generateReport() {
        return await this.generateCredentialReport();
    }
}
exports.CredentialsExcelReport = CredentialsExcelReport;
