"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.AuditTerminalLogsExcelReport = void 0;
const i18n_1 = __importDefault(require("i18n"));
const app_enums_1 = require("../../../app.enums");
const app_logs_1 = require("../../../app.logs");
const dal_manager_1 = require("../../../dal/dal.manager");
const business_main_1 = require("../../business.main");
const generator_1 = require("../generator");
const excel_report_refactory_1 = require("../excel.report.refactory");
class AuditTerminalLogsExcelReport extends generator_1.ReportGenerator {
    constructor(request, locale) {
        super(request, locale);
        this._options = request.filter;
        this._reportFactory = new excel_report_refactory_1.ReportFactory(request.organizationId, locale);
    }
    async generateReport() {
        try {
            const sheetInfo = [
                {
                    sheetName: i18n_1.default.__({ phrase: "EXCEL-REPORT.GENERAL", locale: this._locale }),
                    headers: [
                        {
                            phrase: "EXCEL-REPORT.AUDIT_LOG_DATE_TIME",
                            cellWidth: 30,
                        },
                        {
                            phrase: "EXCEL-REPORT.AUDIT_TERMINAL",
                            cellWidth: 40,
                        },
                        {
                            phrase: "EXCEL-REPORT.AUDIT_LOG_ACTION_USER",
                            cellWidth: 35,
                        },
                        {
                            phrase: "EXCEL-REPORT.AUDIT_LOG_MESSAGE",
                            cellWidth: 60,
                        },
                    ],
                },
            ];
            const reportInfoSheetData = {
                reportNamePhrase: "EXCEL-REPORT.AUDIT_TERMINAL_REPORT",
                filters: {
                    userMainActionType: this._options.actionType,
                    terminalIds: this._options.terminalIds,
                    actionUserIds: this._options.actionUserIds,
                    dateTimeRange: { startDateTime: this._options.dateRange.startDateTime, endDateTime: this._options.dateRange.endDateTime },
                },
            };
            await this._reportFactory.initialize({
                requesterUserId: this._request.requesterUserId,
                userIds: null,
                reportInfoSheetData,
                sheetInfo,
            });
            const terminals = await dal_manager_1.dbManager.accessDevice.listTerminals(this._request.organizationId, {
                pagination: {
                    skip: 0,
                    take: 10000,
                },
            }, this._request.requesterUserId);
            let option = Object.assign(this._options, {
                pagination: {
                    take: undefined,
                    skip: undefined,
                },
            });
            const params = await dal_manager_1.dbManager.accessLog.listUsersTerminalActionHistoryQuery(this._request.organizationId, app_enums_1.enums.UserActionCategory.Main, this._options.actionType
                ? [this._options.actionType]
                : [
                    app_enums_1.enums.UserMainActionType.UpdateStatusSensor,
                    app_enums_1.enums.UserMainActionType.UpdateRelay,
                    app_enums_1.enums.UserMainActionType.UpdateReader,
                    app_enums_1.enums.UserMainActionType.UpdateCounterSensor,
                ], option, terminals.items.map((i) => i.id));
            await dal_manager_1.dbManager.accessLog.fetchAuditTerminalsData(params, this._request.organizationId, app_enums_1.enums.UserActionCategory.Main, this.onDataCallbackFunction.bind(this));
            const fileObj = await this._reportFactory.finalize(reportInfoSheetData);
            return {
                format: app_enums_1.enums.ReportFormat.Excel,
                preparedReportId: fileObj.fileId,
                filePath: fileObj.filePath,
                notificationType: undefined,
                reportType: app_enums_1.enums.ReportCode.AuditLogsTerminal,
                sendFileInAttachments: undefined,
            };
        }
        catch (error) {
            app_logs_1.logger.error(`Error generating AuditTerminalLogsExcelReport: ${error.message}`, {
                stack: error.stack,
                organizationId: this._request.organizationId,
                requesterUserId: this._request.requesterUserId,
            });
            throw new Error("Failed to generate AuditTerminalLogsExcelReport");
        }
    }
    async onDataCallbackFunction(rows) {
        app_logs_1.logger.debug("Fetching %s log records", rows.length);
        const localizedData = (0, business_main_1.transformAuditMainActionItemsLocalizedMessages)(rows, this._locale);
        const sheetInfo = [
            {
                sheetName: i18n_1.default.__({ phrase: "EXCEL-REPORT.GENERAL", locale: this._locale }),
                data: [],
            },
        ];
        for (const item of localizedData) {
            let action = "";
            if (item.action) {
                switch (item.action) {
                    case app_enums_1.enums.UserMainActionType.UpdateRelay:
                        action += "( " + i18n_1.default.__({ phrase: "EXCEL-REPORT.AUDIT_UPDATE_RELAY", locale: this._locale }) + " )";
                        break;
                    case app_enums_1.enums.UserMainActionType.UpdateStatusSensor:
                        action += "( " + i18n_1.default.__({ phrase: "EXCEL-REPORT.AUDIT_UPDATED_STATUS_SENSOR", locale: this._locale }) + " )";
                        break;
                    case app_enums_1.enums.UserMainActionType.UpdateCounterSensor:
                        action += "( " + i18n_1.default.__({ phrase: "EXCEL-REPORT.AUDIT_UPDATED_COUNTER_SENSOR", locale: this._locale }) + " )";
                        break;
                    case app_enums_1.enums.UserMainActionType.UpdateReader:
                        action += "( " + i18n_1.default.__({ phrase: "EXCEL-REPORT.AUDIT_UPDATED_READER", locale: this._locale }) + " )";
                        break;
                    default:
                        break;
                }
            }
            sheetInfo[0].data.push({
                cells: [
                    {
                        value: item.actionUtc ? this._reportFactory.formatDateTime(item.actionUtc, this._locale) : "",
                    },
                    {
                        value: item.terminalName ?? "",
                        alignment: {
                            wrapText: true,
                        },
                    },
                    {
                        value: item.actionUserCaptions && item.actionUserCaptions.length > 0 ? item.actionUserCaptions[0].text[0] + "\r\n( " + item.actionUserCaptions[1].text[0] + " )" : "",
                        alignment: {
                            wrapText: true,
                        },
                    },
                    {
                        value: item.message ? item.message + "\r\n" + action : action,
                        alignment: {
                            wrapText: true,
                        },
                    },
                ],
            });
        }
        this._reportFactory.appendData(sheetInfo);
        return Promise.resolve();
    }
}
exports.AuditTerminalLogsExcelReport = AuditTerminalLogsExcelReport;
