"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.AuditIdentityLogsExcelReport = void 0;
const i18n_1 = __importDefault(require("i18n"));
const app_enums_1 = require("../../../app.enums");
const app_logs_1 = require("../../../app.logs");
const dal_manager_1 = require("../../../dal/dal.manager");
const business_main_1 = require("../../business.main");
const generator_1 = require("../generator");
const excel_report_refactory_1 = require("../excel.report.refactory");
class AuditIdentityLogsExcelReport extends generator_1.ReportGenerator {
    constructor(request, locale) {
        super(request, locale);
        this._options = request.filter;
        this._reportFactory = new excel_report_refactory_1.ReportFactory(request.organizationId, locale);
    }
    async generateReport() {
        try {
            const sheetInfo = [
                {
                    sheetName: i18n_1.default.__({ phrase: "EXCEL-REPORT.GENERAL", locale: this._locale }),
                    headers: [
                        {
                            phrase: "EXCEL-REPORT.AUDIT_LOG_DATE_TIME",
                            width: 30,
                        },
                        {
                            phrase: "EXCEL-REPORT.AUDIT_LOG_MODIFIED_USER",
                            width: 30,
                        },
                        {
                            phrase: "EXCEL-REPORT.AUDIT_LOG_ACTION_USER",
                            width: 30,
                        },
                        {
                            phrase: "EXCEL-REPORT.AUDIT_LOG_MESSAGE",
                            width: 45,
                        },
                        {
                            phrase: "EXCEL-REPORT.AUDIT_LOG_CHANGED_FIELD",
                            width: 70,
                        },
                    ],
                },
            ];
            const reportInfoSheetData = {
                reportNamePhrase: "EXCEL-REPORT.AUDIT_IDENTITY_REPORT",
                filters: {
                    userMainActionType: this._options.actionType,
                    orgUnitIds: this._options.organizationUnitIds,
                    modifiedByUserIds: this._options.modifiedUserIds,
                    userGroupIds: this._options.userGroupIds,
                    actionUserIds: this._options.actionUserIds,
                    applyOrganizationUnitFilterHierarchically: this._options.filterOrganizationUnitMembersHierarchically,
                    dateTimeRange: { startDateTime: this._options.dateRange.startDateTime, endDateTime: this._options.dateRange.endDateTime },
                },
            };
            await this._reportFactory.initialize({
                requesterUserId: this._request.requesterUserId,
                userIds: null,
                reportInfoSheetData,
                sheetInfo,
            });
            this._roles = await dal_manager_1.dbManager.accessUser.listRolesOfOrganization(this._request.organizationId);
            let option = Object.assign(this._options, {
                pagination: {
                    take: undefined,
                    skip: undefined,
                },
            });
            const params = await dal_manager_1.dbManager.accessLog.listUsersIdentityActionHistoryQuery(this._request.organizationId, app_enums_1.enums.UserActionCategory.Main, this._options.actionType
                ? [this._options.actionType]
                : [
                    app_enums_1.enums.UserMainActionType.UpdateIdentity,
                    app_enums_1.enums.UserMainActionType.InsertIdentity,
                    app_enums_1.enums.UserMainActionType.AddAccessRight,
                    app_enums_1.enums.UserMainActionType.RemoveAccessRight,
                    app_enums_1.enums.UserMainActionType.AddCredential,
                    app_enums_1.enums.UserMainActionType.RemoveCredential,
                    app_enums_1.enums.UserMainActionType.UpdateProfile,
                    app_enums_1.enums.UserMainActionType.UpdateCredential,
                ], option, this._options.permittedUnitIds);
            await dal_manager_1.dbManager.accessLog.fetchAuditIdentityData(params, this._request.organizationId, app_enums_1.enums.UserActionCategory.Main, this.onDataCallbackFunction.bind(this));
            const fileObj = await this._reportFactory.finalize(reportInfoSheetData);
            return {
                format: app_enums_1.enums.ReportFormat.Excel,
                preparedReportId: fileObj.fileId,
                filePath: fileObj.filePath,
                notificationType: undefined,
                reportType: app_enums_1.enums.ReportCode.AuditLogsIdentity,
                sendFileInAttachments: undefined,
            };
        }
        catch (error) {
            app_logs_1.logger.error(`Error generating AuditIdentityLogsExcelReport: ${error.message}`, {
                stack: error.stack,
                organizationId: this._request.organizationId,
                requesterUserId: this._request.requesterUserId,
            });
            throw new Error("Failed to generate AuditIdentityLogsExcelReport");
        }
    }
    async onDataCallbackFunction(rows) {
        app_logs_1.logger.debug("Fetching %s log records", rows.length);
        const localizedData = (0, business_main_1.transformAuditMainActionItemsLocalizedMessages)(rows, this._locale, this._roles);
        const sheetInfo = [
            {
                sheetName: i18n_1.default.__({ phrase: "EXCEL-REPORT.GENERAL", locale: this._locale }),
                data: [],
            },
        ];
        for (const item of localizedData) {
            let changes = "";
            if (item.changes && item.changes.length > 0) {
                for (const change of item.changes) {
                    if (change.field) {
                        changes += change.field + " : ";
                    }
                    if (change.oldValue && change.newValue) {
                        changes += change.oldValue + " -> " + change.newValue;
                    }
                    else if (!change.oldValue && change.newValue) {
                        changes += "+ " + change.newValue;
                    }
                    changes += "\r\n";
                }
            }
            sheetInfo[0].data.push({
                cells: [
                    {
                        value: item.actionUtc ? this._reportFactory.formatDateTime(item.actionUtc, this._locale) : "",
                    },
                    {
                        value: item.modifiedUserCaptions && item.modifiedUserCaptions.length > 0
                            ? item.modifiedUserCaptions[0].text[0] + "\r\n( " + item.modifiedUserCaptions[1].text[0] + " )"
                            : "",
                        alignment: {
                            wrapText: true,
                        },
                    },
                    {
                        value: item.actionUserCaptions && item.actionUserCaptions.length > 0 ? item.actionUserCaptions[0].text[0] + "\r\n( " + item.actionUserCaptions[1].text[0] + " )" : "",
                    },
                    {
                        value: item?.message ?? "",
                        alignment: {
                            wrapText: true,
                        },
                    },
                    {
                        value: changes,
                        alignment: {
                            wrapText: true,
                        },
                    },
                ],
            });
        }
        this._reportFactory.appendData(sheetInfo);
        return Promise.resolve();
    }
}
exports.AuditIdentityLogsExcelReport = AuditIdentityLogsExcelReport;
