"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.AnalysisVisitorsReport = void 0;
const exceljs_1 = __importDefault(require("exceljs"));
const fs_1 = __importDefault(require("fs"));
const i18n_1 = __importDefault(require("i18n"));
const luxon_1 = require("luxon");
const path_1 = __importDefault(require("path"));
const uuid_1 = __importDefault(require("uuid"));
const app_config_1 = require("../../../app.config");
const app_enums_1 = require("../../../app.enums");
const app_logs_1 = require("../../../app.logs");
const dal_manager_1 = require("../../../dal/dal.manager");
const generator_1 = require("../generator");
class AnalysisVisitorsReport extends generator_1.ReportGenerator {
    constructor(request, locale) {
        super(request, locale);
        this._fileId = uuid_1.default.v4();
        this._filePath = path_1.default.join(app_config_1.appConfig.tmpDirectory, this._fileId + ".xlsx");
        if (fs_1.default.existsSync(this._filePath)) {
            fs_1.default.unlinkSync(this._filePath);
        }
        this._wb = new exceljs_1.default.stream.xlsx.WorkbookWriter({
            filename: this._filePath,
            useStyles: true,
            useSharedStrings: true,
        });
        this._options = request.filter;
    }
    addRow(row, startRowIndex, ws) {
        let colIndex = 1;
        let rowData = [row.date, row.pointOrReason, row.countOrDuration];
        colIndex = this.addCellsToRow(rowData, colIndex, startRowIndex, ws);
        ws.getRow(startRowIndex).commit();
    }
    addCombinedRow(row, startRowIndex, ws) {
        let colIndex = 1;
        let rowData = [row.date, row.registrationPoint, row.reason, row.countOrDuration];
        colIndex = this.addCellsToRow(rowData, colIndex, startRowIndex, ws);
        ws.getRow(startRowIndex).commit();
    }
    initSheet(ws, startRowIndex, startColIndex, data) {
        let columns = [];
        startColIndex = this.addInitCellsToRow(data, columns, startColIndex, startRowIndex, ws);
        ws.autoFilter = {
            from: "A" + startRowIndex,
            to: ws.getColumn(columns.length).letter + startRowIndex,
        };
        ws.columns = columns;
        ws.views.push({ state: "frozen", ySplit: startRowIndex, activeCell: "A1", showGridLines: false });
    }
    initSheets() {
        const datePhrase = "REPORT." + (this._options.segmentType === app_enums_1.enums.SegmentType.Month ? "MONTH" : "DATE");
        const headers = [
            { phrase: datePhrase, cellWidth: 40 },
            { phrase: "EXCEL-REPORT.REGISTRATION_POINT_COLUMN", cellWidth: 40 },
            { phrase: "EXCEL-REPORT.NUMBER", cellWidth: 40 },
        ];
        this.initSheet(this._wsVisitorPointCount, this._visitorPointCountRowIndex++, 1, headers);
        headers[2].phrase = "PACS.PPERMISSION_TYPE_UNIT_DURATION";
        this.initSheet(this._wsVisitorPointTime, this._visitorPointTimeRowIndex++, 1, headers);
        headers[1].phrase = "PDF-REPORT.ACCESS_REASONS";
        this.initSheet(this._wsReasonTime, this._reasonTimeRowIndex++, 1, headers);
        headers[2].phrase = "EXCEL-REPORT.NUMBER";
        this.initSheet(this._wsReasonCount, this._reasonCountRowIndex++, 1, headers);
        const combinedHeaders = [
            { phrase: datePhrase, cellWidth: 40 },
            { phrase: "EXCEL-REPORT.REGISTRATION_POINT_COLUMN", cellWidth: 40 },
            { phrase: "PDF-REPORT.ACCESS_REASONS", cellWidth: 40 },
            { phrase: "EXCEL-REPORT.NUMBER", cellWidth: 40 },
        ];
        this.initSheet(this._wsCombinedCount, this._combinedCountRowIndex++, 1, combinedHeaders);
        combinedHeaders[3].phrase = "PACS.PPERMISSION_TYPE_UNIT_DURATION";
        this.initSheet(this._wsCombinedTime, this._combinedTimeRowIndex++, 1, combinedHeaders);
    }
    async generateAnalysisVisitorsReport() {
        this._organizationName = (await dal_manager_1.dbManager.accessOrganization.getOrganizationBasic(this._request.organizationId)).name;
        app_logs_1.logger.debug("Organization Name : " + this._organizationName);
        this.initializeWorksheets();
        let { caption: requestedUserCaption } = await dal_manager_1.dbManager.accessRedisCache.getUserBadgeCache({ organizationId: this._request.organizationId, userId: this._request.requesterUserId });
        if (!requestedUserCaption) {
            throw new Error("Not found");
        }
        this._requestedUserCaption = requestedUserCaption[0].text[0];
        this._wb.creator = this._requestedUserCaption;
        this.initSheets();
        app_logs_1.logger.debug("Generating report...");
        this._registrationPoints = await dal_manager_1.dbManager.accessVisitor.listRegistrationPointsDetailed(this._request.organizationId, this._request.requesterUserId);
        this._visitReasons = await this.getVisitReasons(this._request.organizationId);
        const results = await Promise.all([
            dal_manager_1.dbManager.accessVisitor.getVisitCountsByFilter(this._request.organizationId, this._options, true),
            dal_manager_1.dbManager.accessVisitor.getVisitProcessTimeByFilter(this._request.organizationId, this._options, true),
            dal_manager_1.dbManager.accessVisitor.getVisitCountsByFilter(this._request.organizationId, this._options, false),
            dal_manager_1.dbManager.accessVisitor.getVisitProcessTimeByFilter(this._request.organizationId, this._options, false),
            dal_manager_1.dbManager.accessVisitor.getVisitCountsPointReason(this._request.organizationId, this._options),
            dal_manager_1.dbManager.accessVisitor.getVisitProcessTimePointReason(this._request.organizationId, this._options),
        ]);
        const data = this.parseData(results);
        this.fillSheets(data);
        app_logs_1.logger.debug("end of fetching");
        this.setPrintAreas();
        await this._wb.commit();
        app_logs_1.logger.debug("File is ready -> " + this._filePath);
        return {
            format: app_enums_1.enums.ReportFormat.Excel,
            preparedReportId: this._fileId,
            filePath: this._filePath,
            notificationType: undefined,
            reportType: app_enums_1.enums.ReportCode.AnalysisVisitorsReport,
            sendFileInAttachments: undefined,
        };
    }
    initializeWorksheets() {
        this._wsVisitorPointCount = this._wb.addWorksheet(i18n_1.default.__({ phrase: "EXCEL-REPORT.VISIT_COUNTS_VISITOR_POINT", locale: this._locale }), {
            pageSetup: { fitToPage: true, fitToWidth: 1, fitToHeight: 0, paperSize: 9, orientation: "landscape" },
        });
        this._wsVisitorPointCount.properties.defaultRowHeight = 15;
        this._visitorPointCountRowIndex = 1;
        this._wsVisitorPointTime = this._wb.addWorksheet(i18n_1.default.__({ phrase: "EXCEL-REPORT.VISITOR_PROCESS_TIME_POINT", locale: this._locale }), {
            pageSetup: { fitToPage: true, fitToWidth: 1, fitToHeight: 0, paperSize: 9, orientation: "landscape" },
        });
        this._wsVisitorPointTime.properties.defaultRowHeight = 15;
        this._visitorPointTimeRowIndex = 1;
        this._wsReasonCount = this._wb.addWorksheet(i18n_1.default.__({ phrase: "EXCEL-REPORT.VISIT_COUNTS_VISITOR_REASON", locale: this._locale }), {
            pageSetup: { fitToPage: true, fitToWidth: 1, fitToHeight: 0, paperSize: 9, orientation: "landscape" },
        });
        this._wsReasonCount.properties.defaultRowHeight = 15;
        this._reasonCountRowIndex = 1;
        this._wsReasonTime = this._wb.addWorksheet(i18n_1.default.__({ phrase: "EXCEL-REPORT.VISITOR_PROCESS_TIME_REASON", locale: this._locale }), {
            pageSetup: { fitToPage: true, fitToWidth: 1, fitToHeight: 0, paperSize: 9, orientation: "landscape" },
        });
        this._wsReasonTime.properties.defaultRowHeight = 15;
        this._reasonTimeRowIndex = 1;
        this._wsCombinedCount = this._wb.addWorksheet(i18n_1.default.__({ phrase: "EXCEL-REPORT.VISIT_COUNTS_VISITOR_POINT_AND_REASON", locale: this._locale }), {
            pageSetup: { fitToPage: true, fitToWidth: 1, fitToHeight: 0, paperSize: 9, orientation: "landscape" },
        });
        this._wsCombinedCount.properties.defaultRowHeight = 15;
        this._combinedCountRowIndex = 1;
        this._wsCombinedTime = this._wb.addWorksheet(i18n_1.default.__({ phrase: "EXCEL-REPORT.VISITOR_PROCESS_TIME_POINT_AND_REASON", locale: this._locale }), {
            pageSetup: { fitToPage: true, fitToWidth: 1, fitToHeight: 0, paperSize: 9, orientation: "landscape" },
        });
        this._wsCombinedTime.properties.defaultRowHeight = 15;
        this._combinedTimeRowIndex = 1;
    }
    setPrintAreas() {
        this._wsVisitorPointCount.pageSetup.printArea = "A2:" + this._wsVisitorPointCount.getColumn(this._wsVisitorPointCount.columns.length).letter + this._visitorPointCountRowIndex;
        this._wsVisitorPointCount.pageSetup.printTitlesRow = "1:1";
        this._wsVisitorPointTime.pageSetup.printArea = "A2:" + this._wsVisitorPointTime.getColumn(this._wsVisitorPointTime.columns.length).letter + this._visitorPointTimeRowIndex;
        this._wsVisitorPointTime.pageSetup.printTitlesRow = "1:1";
        this._wsReasonCount.pageSetup.printArea = "A2:" + this._wsReasonCount.getColumn(this._wsReasonCount.columns.length).letter + this._reasonCountRowIndex;
        this._wsReasonCount.pageSetup.printTitlesRow = "1:1";
        this._wsReasonTime.pageSetup.printArea = "A2:" + this._wsReasonTime.getColumn(this._wsReasonTime.columns.length).letter + this._reasonTimeRowIndex;
        this._wsReasonTime.pageSetup.printTitlesRow = "1:1";
        this._wsCombinedCount.pageSetup.printArea = "A2:" + this._wsCombinedCount.getColumn(this._wsCombinedCount.columns.length).letter + this._combinedCountRowIndex;
        this._wsCombinedCount.pageSetup.printTitlesRow = "1:1";
        this._wsCombinedTime.pageSetup.printArea = "A2:" + this._wsCombinedTime.getColumn(this._wsCombinedTime.columns.length).letter + this._combinedTimeRowIndex;
        this._wsCombinedTime.pageSetup.printTitlesRow = "1:1";
    }
    parseData(data) {
        let result = [];
        result.push(this.parseVisitAnalysisData(data[0], true));
        result.push(this.parseVisitAnalysisData(data[1], true, true));
        result.push(this.parseVisitAnalysisData(data[2], false));
        result.push(this.parseVisitAnalysisData(data[3], false, true));
        result.push(this.parseVisitAnalysisData(data[4], true, false, true));
        result.push(this.parseVisitAnalysisData(data[5], true, true, true));
        return result;
    }
    parseVisitAnalysisData(data, isPoint, isTime = false, isPointReason = false) {
        if (isPointReason) {
            return data.flatMap((item) => {
                const itemDate = this._options.segmentType === app_enums_1.enums.SegmentType.Month ? this.formatFullMonth(item.startDate) : this.formatDateForDayAndWeek(item.startDate);
                return item.data.map((dataItem) => ({
                    date: itemDate,
                    registrationPoint: this.getPointOrReason(dataItem.visitorRegistrationPointId, true),
                    reason: this.getPointOrReason(dataItem.reason, false),
                    countOrDuration: isTime ? dataItem.averageProcessTime.toFixed(2) : dataItem.count,
                }));
            });
        }
        else {
            return data.flatMap((item) => {
                const itemDate = this._options.segmentType === app_enums_1.enums.SegmentType.Month ? this.formatFullMonth(item.startDate) : this.formatDateForDayAndWeek(item.startDate);
                return item.data.map((dataItem) => ({
                    date: itemDate,
                    pointOrReason: this.getPointOrReason(dataItem.visitorRegistrationPointId || dataItem.reason, isPoint),
                    countOrDuration: isTime ? dataItem.averageProcessTime.toFixed(2) : dataItem.count,
                }));
            });
        }
    }
    findVisitReason(value) {
        if (!this._visitReasons) {
            return "";
        }
        const reason = this._visitReasons.options.options.find((element) => element.value === value);
        return reason ? reason.captionLines[0] : "";
    }
    async getVisitReasons(organizationId) {
        const { rows } = await dal_manager_1.dbManager.organizationTransaction(async (trx) => {
            return await trx.query(`
                    SELECT  "settings"
                    FROM "${organizationId}"."organizationVisitorModuleSettings" as ovm
                    WHERE ovm."organizationId" = $1;
                    `, [organizationId]);
        }, this._request.requesterUserId, this._request.organizationId);
        const visitorModuleSettings = rows[0]["settings"];
        return visitorModuleSettings?.visitFormFields?.find((vff) => vff.name === "reason") ?? null;
    }
    getPointOrReason(id, isPoint) {
        if (isPoint) {
            const filteredPoints = this._registrationPoints?.filter((point) => point.id === id);
            return filteredPoints && filteredPoints.length > 0 ? filteredPoints[0].name : "";
        }
        else {
            return this.findVisitReason(Number(id)) ?? "";
        }
    }
    formatDateForDayAndWeek(date) {
        return luxon_1.DateTime.fromJSDate(date).setLocale(this._locale).toFormat("d LLL");
    }
    formatFullMonth(date) {
        return luxon_1.DateTime.fromJSDate(date).setLocale(this._locale).toFormat("LLLL");
    }
    fillSheets(data) {
        data[0].forEach((row) => this.addRow(row, this._visitorPointCountRowIndex++, this._wsVisitorPointCount));
        data[1].forEach((row) => this.addRow(row, this._visitorPointTimeRowIndex++, this._wsVisitorPointTime));
        data[2].forEach((row) => this.addRow(row, this._reasonCountRowIndex++, this._wsReasonCount));
        data[3].forEach((row) => this.addRow(row, this._reasonTimeRowIndex++, this._wsReasonTime));
        data[4].forEach((row) => this.addCombinedRow(row, this._combinedCountRowIndex++, this._wsCombinedCount));
        data[5].forEach((row) => this.addCombinedRow(row, this._combinedTimeRowIndex++, this._wsCombinedTime));
    }
    async generateReport() {
        return await this.generateAnalysisVisitorsReport();
    }
}
exports.AnalysisVisitorsReport = AnalysisVisitorsReport;
