"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.AccessControlPointUsersExcelReport = void 0;
const path_1 = __importDefault(require("path"));
const exceljs_1 = __importDefault(require("exceljs"));
const app_config_1 = require("../../../app.config");
const dal_manager_1 = require("../../../dal/dal.manager");
const uuid_1 = __importDefault(require("uuid"));
const i18n_1 = __importDefault(require("i18n"));
const app_logs_1 = require("../../../app.logs");
const app_enums_1 = require("../../../app.enums");
const fs_1 = __importDefault(require("fs"));
const generator_1 = require("../generator");
class AccessControlPointUsersExcelReport extends generator_1.ReportGenerator {
    constructor(request, locale) {
        super(request, locale);
        this._fileId = uuid_1.default.v4();
        this._filePath = path_1.default.join(app_config_1.appConfig.tmpDirectory, this._fileId + ".xlsx");
        if (fs_1.default.existsSync(this._filePath)) {
            fs_1.default.unlinkSync(this._filePath);
        }
        this._wb = new exceljs_1.default.stream.xlsx.WorkbookWriter({
            filename: this._filePath,
            useStyles: true,
            useSharedStrings: true,
        });
        this._options = request.filter;
    }
    async generateReport() {
        this._organizationName = (await dal_manager_1.dbManager.accessOrganization.getOrganizationBasic(this._request.organizationId)).name;
        app_logs_1.logger.debug("Organization Name : " + this._organizationName);
        let { caption: requestedUserCaption } = await dal_manager_1.dbManager.accessRedisCache.getUserBadgeCache({ organizationId: this._request.organizationId, userId: this._request.requesterUserId });
        if (!requestedUserCaption) {
            throw new Error("Not found");
        }
        this._requestedUserCaption = requestedUserCaption[0].text[0];
        this._wb.creator = this._requestedUserCaption;
        app_logs_1.logger.debug("Generating acp users list excel report...");
        app_logs_1.logger.debug("Requester User : " + this._requestedUserCaption);
        this._wsGeneral = this._wb.addWorksheet(i18n_1.default.__({ phrase: "EXCEL-REPORT.GENERAL", locale: this._locale }), {
            pageSetup: { fitToPage: true, fitToWidth: 1, fitToHeight: 0, paperSize: 9, orientation: "landscape" },
        });
        this._wsGeneral.properties.defaultRowHeight = 15;
        this._generalRowIndex = 6;
        let acpBasics = await dal_manager_1.dbManager.accessAccessControlPoint.getAccessControlPointBasic(this._request.organizationId, this._options.accessControlPointId);
        this.initGeneralSheet(this._wsGeneral, 5, 1);
        this.addFilterToSheet(this._wsGeneral, 1, 1, acpBasics.name);
        this._data = await dal_manager_1.dbManager.accessAccessControlPoint.listUsersOnAccessPoint(this._request.organizationId, this._options.accessControlPointId);
        for (let index = 0; index < this._data.length; index++) {
            const acp = this._data[index];
            this.addGeneralRow(acp);
        }
        this._wsGeneral.pageSetup.printArea = "A5:" + this._wsGeneral.getColumn(this._wsGeneral.columns.length).letter + this._generalRowIndex;
        this._wsGeneral.pageSetup.printTitlesRow = "5:5";
        this._wsGeneral.commit();
        await this._wb.commit();
        app_logs_1.logger.debug("File is ready -> " + this._filePath);
        return {
            format: app_enums_1.enums.ReportFormat.Excel,
            preparedReportId: this._fileId,
            filePath: this._filePath,
            notificationType: undefined,
            reportType: app_enums_1.enums.ReportCode.AccessControlPointUsers,
            sendFileInAttachments: undefined,
        };
    }
    initGeneralSheet(wsRaw, startRowIndex, startColIndex) {
        let columns = [];
        let data = [
            {
                phrase: "EXCEL-REPORT.UNIQUE_ID",
                cellWidth: 20
            },
            {
                phrase: "EXCEL-REPORT.USER",
                cellWidth: 40
            }
        ];
        startColIndex = this.addInitCellsToRow(data, columns, startColIndex, startRowIndex, wsRaw);
        wsRaw.autoFilter = {
            from: "A" + startRowIndex,
            to: wsRaw.getColumn(columns.length).letter + startRowIndex,
        };
        wsRaw.columns = columns;
        app_logs_1.logger.debug("Sheet initializing ended...");
    }
    addGeneralRow(row) {
        let colIndex = 1;
        const rowData = [
            row.uniqueId,
            row.name.toUpperCase() + " " + row.surname.toUpperCase()
        ];
        colIndex = this.addCellsToRow(rowData, colIndex, this._generalRowIndex, this._wsGeneral);
        this._wsGeneral.getRow(this._generalRowIndex).commit();
        this._generalRowIndex++;
    }
    addFilterToSheet(ws, startRowIndex, startColIndex, acpName) {
        this.addFilterCell11(ws, startRowIndex, startColIndex, this._organizationName + " " + i18n_1.default.__({ phrase: "EXCEL-REPORT.ACCESS_CONTROL_POINT_USERS_REPORT", locale: this._locale }), generator_1.ReportGenerator.Constants.Styling.AllThin, { bold: true });
        ws.mergeCells(startRowIndex, startColIndex, startRowIndex, startColIndex + 1);
        this.addFilterCell11(ws, startRowIndex + 1, startColIndex, i18n_1.default.__({ phrase: "EXCEL-REPORT.AUDIT_ACCESS_CONTROL_POINT", locale: this._locale }), generator_1.ReportGenerator.Constants.Styling.AllThin, {
            bold: true,
        });
        this.addFilterCell11(ws, startRowIndex + 1, startColIndex + 1, acpName, generator_1.ReportGenerator.Constants.Styling.AllThin);
        ws.mergeCells(startRowIndex + 1, startColIndex + 1, startRowIndex + 1, startColIndex + 1);
    }
}
exports.AccessControlPointUsersExcelReport = AccessControlPointUsersExcelReport;
