"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.ExcelReportAccessControlPoints = void 0;
const path_1 = __importDefault(require("path"));
const exceljs_1 = __importDefault(require("exceljs"));
const app_config_1 = require("../../../app.config");
const dal_manager_1 = require("../../../dal/dal.manager");
const uuid_1 = __importDefault(require("uuid"));
const i18n_1 = __importDefault(require("i18n"));
const app_logs_1 = require("../../../app.logs");
const app_enums_1 = require("../../../app.enums");
const generator_1 = require("../generator");
const moment_1 = __importDefault(require("moment"));
const report_util_1 = require("../report.util");
class ExcelReportAccessControlPoints extends generator_1.ReportGenerator {
    constructor(request, locale) {
        super(request, locale);
        this._filter = this._request.filter;
        this._fileId = uuid_1.default.v4();
        this._filePath = path_1.default.join(app_config_1.appConfig.tmpDirectory, this._fileId + ".xlsx");
        this._currentDate = (0, moment_1.default)();
        this._wb = new exceljs_1.default.stream.xlsx.WorkbookWriter({
            filename: this._filePath,
            useStyles: true,
            useSharedStrings: true,
        });
        this._emptyData = true;
    }
    async generateReport() {
        const organizationId = this._request.organizationId;
        this._organizationName = (await dal_manager_1.dbManager.accessRedisCache.getOrganizationCache({ organizationId: organizationId })).n;
        app_logs_1.logger.debug("Organization Name : " + this._organizationName);
        this._ws = this._wb.addWorksheet(i18n_1.default.__({ phrase: "EXCEL-REPORT.ACCESS_CONTROL_POINTS", locale: this._locale }), {
            pageSetup: { fitToPage: true, fitToWidth: 1, fitToHeight: 0, paperSize: 9, orientation: "landscape" },
        });
        this._ws.properties.defaultRowHeight = 15;
        this._rowIndex = 10;
        let { caption: requestedUserCaption } = await dal_manager_1.dbManager.accessRedisCache.getUserBadgeCache({ organizationId: this._request.organizationId, userId: this._request.requesterUserId });
        if (!requestedUserCaption) {
            throw new Error("Not found");
        }
        this._requestedUserCaption = requestedUserCaption[0].text[0];
        app_logs_1.logger.debug("Requested User : " + this._requestedUserCaption);
        this._wb.creator = this._requestedUserCaption;
        await this.addFilterToSheet(this._ws, 1, 1);
        this.initSheet(this._ws, 9, 1);
        const filters = this._request.filter;
        ({ take: filters.take, skip: filters.skip } = { take: undefined, skip: undefined });
        let accessControlPointInfos = await dal_manager_1.dbManager.accessAccessControlPoint.searchAccessControlPoint(organizationId, filters, true, this._request.requesterUserId);
        for (const acp of accessControlPointInfos.items) {
            this.addRow(acp);
        }
        app_logs_1.logger.debug("Generating report...");
        this._ws.pageSetup.printArea = "A1:" + this._ws.getColumn(this._ws.columns.length).letter + this._rowIndex;
        this._ws.pageSetup.printTitlesRow = "5:5";
        this._ws.commit();
        await this._wb.commit();
        console.log("File is ready -> " + this._filePath);
        return {
            preparedReportId: this._fileId,
            filePath: this._filePath,
            format: app_enums_1.enums.ReportFormat.Excel,
            notificationType: undefined,
            reportType: app_enums_1.enums.ReportCode.AccessLogs,
            sendFileInAttachments: undefined,
            reportContainsEmptyData: this._request.reportTemplateId ? this._emptyData : undefined,
        };
    }
    async addFilterToSheet(ws, startRowIndex, startColIndex) {
        let filterResult = await this.createFilters();
        let filtersArr = [];
        if (filterResult.acpName) {
            filtersArr.push(i18n_1.default.__({ phrase: "COMMON.SEARCH_HEADER", locale: this._locale }) + ":  " + filterResult.acpName);
        }
        app_logs_1.logger.debug("Applied Filters:");
        app_logs_1.logger.debug(filtersArr);
        app_logs_1.logger.debug(filtersArr.join("\r\n"));
        app_logs_1.logger.debug("\n");
        this.addFilterCell11(ws, startRowIndex, startColIndex, this._organizationName + " " + i18n_1.default.__({ phrase: "EXCEL-REPORT.ACCESS_CONTROL_POINTS", locale: this._locale }), generator_1.ReportGenerator.Constants.Styling.AllThin, {
            bold: true,
        });
        this.addFilterCell11(ws, startRowIndex + 1, startColIndex, i18n_1.default.__({ phrase: "EXCEL-REPORT.FILE_CREATION_DATE", locale: this._locale }), generator_1.ReportGenerator.Constants.Styling.AllThin);
        this.addFilterCell11(ws, startRowIndex + 1, startColIndex + 1, this._currentDate.locale(this._locale).format("LL") + " " + this._currentDate.locale(this._locale).format("HH:mm:ss"), generator_1.ReportGenerator.Constants.Styling.AllThin, { italic: false });
        this.addFilterCell11(ws, startRowIndex + 2, startColIndex, i18n_1.default.__({ phrase: "EXCEL-REPORT.REQUESTED_USER", locale: this._locale }), generator_1.ReportGenerator.Constants.Styling.AllThin);
        this.addFilterCell11(ws, startRowIndex + 2, startColIndex + 1, this._requestedUserCaption, generator_1.ReportGenerator.Constants.Styling.AllThin);
        ws.mergeCells(startRowIndex, startColIndex, startRowIndex, startColIndex + 2);
        ws.mergeCells(startRowIndex + 1, startColIndex + 1, startRowIndex + 1, startColIndex + 2);
        ws.mergeCells(startRowIndex + 2, startColIndex + 1, startRowIndex + 2, startColIndex + 2);
        ws.mergeCells(startRowIndex + 3, startColIndex + 1, startRowIndex + 3, startColIndex + 2);
        this.addFilterCell11(ws, startRowIndex + 3, startColIndex, i18n_1.default.__({ phrase: "EXCEL-REPORT.APPLIED_FILTERS", locale: this._locale }), generator_1.ReportGenerator.Constants.Styling.AllThin);
        this.addFilterCell11(ws, startRowIndex + 3, startColIndex + 1, filtersArr.join("\r\n"), generator_1.ReportGenerator.Constants.Styling.AllThin);
        let filterlength = filtersArr.length;
        let row = ws.getRow(startRowIndex + 3);
        row.height = filterlength * 20;
    }
    async createFilters() {
        let filterResult = { acpName: undefined };
        if (this._filter.filter) {
            filterResult.acpName = this._filter.filter;
        }
        return filterResult;
    }
    initSheet(ws, startRowIndex, startColIndex) {
        let columns = [];
        const data = [
            {
                phrase: "EXCEL-REPORT.ACCESS_CONTROL_POINT_NAME",
                cellWidth: 30
            },
            {
                phrase: "EXCEL-REPORT.LOCATION",
                cellWidth: 30
            },
            {
                phrase: "EXCEL-REPORT.ACCESS_CONTROL_POINT_TYPE",
                cellWidth: 30
            }
        ];
        startColIndex = this.addInitCellsToRow(data, columns, startColIndex, startRowIndex, ws);
        ws.autoFilter = {
            from: "A" + startRowIndex,
            to: ws.getColumn(columns.length).letter + startRowIndex,
        };
        ws.columns = columns;
        ws.views.push({ state: "frozen", ySplit: startRowIndex, activeCell: "A1", showGridLines: false });
    }
    addRow(row) {
        let colIndex = 1;
        const rowData = [
            row.name,
            row.location,
            (0, report_util_1.getAccessControlPointTypeName)(row.type, this._locale)
        ];
        colIndex = this.addCellsToRow(rowData, colIndex, this._rowIndex, this._ws);
        this._ws.getRow(this._rowIndex).commit();
        this._rowIndex++;
    }
}
exports.ExcelReportAccessControlPoints = ExcelReportAccessControlPoints;
