"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.EmployeeBasedDataColumnTypes = exports.EmployeeDayTotalDataColumnTypes = exports.EmployeeDayDataColumnTypes = void 0;
const luxon_1 = require("luxon");
const app_logs_1 = require("../../app.logs");
const dal_constants_1 = require("../../dal/dal.constants");
const generator_1 = require("./generator");
const report_util_1 = require("./report.util");
const app_enums_1 = require("../../app.enums");
const i18n_1 = __importDefault(require("i18n"));
const MINUTES_IN_A_DAY = 24 * 60;
var EmployeeDayDataColumnTypes;
(function (EmployeeDayDataColumnTypes) {
    EmployeeDayDataColumnTypes[EmployeeDayDataColumnTypes["Date"] = 0] = "Date";
    EmployeeDayDataColumnTypes[EmployeeDayDataColumnTypes["FirstInTime"] = 1] = "FirstInTime";
    EmployeeDayDataColumnTypes[EmployeeDayDataColumnTypes["LastOutTime"] = 2] = "LastOutTime";
    EmployeeDayDataColumnTypes[EmployeeDayDataColumnTypes["AttendanceStatusText"] = 3] = "AttendanceStatusText";
    EmployeeDayDataColumnTypes[EmployeeDayDataColumnTypes["UsedPermissions"] = 4] = "UsedPermissions";
    EmployeeDayDataColumnTypes[EmployeeDayDataColumnTypes["WorkplanText"] = 5] = "WorkplanText";
    EmployeeDayDataColumnTypes[EmployeeDayDataColumnTypes["ExpectedEntrance"] = 6] = "ExpectedEntrance";
    EmployeeDayDataColumnTypes[EmployeeDayDataColumnTypes["ExpectedLeave"] = 7] = "ExpectedLeave";
    EmployeeDayDataColumnTypes[EmployeeDayDataColumnTypes["ExpectedWorkDuration"] = 8] = "ExpectedWorkDuration";
    EmployeeDayDataColumnTypes[EmployeeDayDataColumnTypes["NormalWorkDuration"] = 9] = "NormalWorkDuration";
    EmployeeDayDataColumnTypes[EmployeeDayDataColumnTypes["MissingWorkDuration"] = 10] = "MissingWorkDuration";
    EmployeeDayDataColumnTypes[EmployeeDayDataColumnTypes["TotalWorkDuration"] = 11] = "TotalWorkDuration";
    EmployeeDayDataColumnTypes[EmployeeDayDataColumnTypes["ExtraWorkDuration"] = 12] = "ExtraWorkDuration";
    EmployeeDayDataColumnTypes[EmployeeDayDataColumnTypes["PhysicallyInRegionDuration"] = 13] = "PhysicallyInRegionDuration";
    EmployeeDayDataColumnTypes[EmployeeDayDataColumnTypes["FixedBreakDuration"] = 14] = "FixedBreakDuration";
    EmployeeDayDataColumnTypes[EmployeeDayDataColumnTypes["Score"] = 15] = "Score";
    EmployeeDayDataColumnTypes[EmployeeDayDataColumnTypes["AccountedDeclaredWorkDuration"] = 16] = "AccountedDeclaredWorkDuration";
    EmployeeDayDataColumnTypes[EmployeeDayDataColumnTypes["AccountedIgnoredWorkDuration"] = 17] = "AccountedIgnoredWorkDuration";
    EmployeeDayDataColumnTypes[EmployeeDayDataColumnTypes["AccountedOffTimePermissionDuration"] = 18] = "AccountedOffTimePermissionDuration";
    EmployeeDayDataColumnTypes[EmployeeDayDataColumnTypes["EarlyLeave"] = 19] = "EarlyLeave";
    EmployeeDayDataColumnTypes[EmployeeDayDataColumnTypes["LateLeave"] = 20] = "LateLeave";
    EmployeeDayDataColumnTypes[EmployeeDayDataColumnTypes["EarlyStart"] = 21] = "EarlyStart";
    EmployeeDayDataColumnTypes[EmployeeDayDataColumnTypes["LateStart"] = 22] = "LateStart";
    EmployeeDayDataColumnTypes[EmployeeDayDataColumnTypes["ToleratedEarlyLeave"] = 23] = "ToleratedEarlyLeave";
    EmployeeDayDataColumnTypes[EmployeeDayDataColumnTypes["ToleratedEarlyStart"] = 24] = "ToleratedEarlyStart";
    EmployeeDayDataColumnTypes[EmployeeDayDataColumnTypes["ToleratedLateLeave"] = 25] = "ToleratedLateLeave";
    EmployeeDayDataColumnTypes[EmployeeDayDataColumnTypes["ToleratedLateStart"] = 26] = "ToleratedLateStart";
    EmployeeDayDataColumnTypes[EmployeeDayDataColumnTypes["ExpectedHolidayDuration"] = 27] = "ExpectedHolidayDuration";
    EmployeeDayDataColumnTypes[EmployeeDayDataColumnTypes["UsedHolidayDuration"] = 28] = "UsedHolidayDuration";
    EmployeeDayDataColumnTypes[EmployeeDayDataColumnTypes["AutomaticallyUsedClaimBreakDuration"] = 29] = "AutomaticallyUsedClaimBreakDuration";
    EmployeeDayDataColumnTypes[EmployeeDayDataColumnTypes["AutomaticallyUsedFixedBreakDuration"] = 30] = "AutomaticallyUsedFixedBreakDuration";
    EmployeeDayDataColumnTypes[EmployeeDayDataColumnTypes["ExpectedClaimBreakDuration"] = 31] = "ExpectedClaimBreakDuration";
    EmployeeDayDataColumnTypes[EmployeeDayDataColumnTypes["ExpectedFixedBreakDuration"] = 32] = "ExpectedFixedBreakDuration";
    EmployeeDayDataColumnTypes[EmployeeDayDataColumnTypes["ExpectedOffTimePermissionDuration"] = 33] = "ExpectedOffTimePermissionDuration";
    EmployeeDayDataColumnTypes[EmployeeDayDataColumnTypes["UsedClaimBreakDuration"] = 34] = "UsedClaimBreakDuration";
    EmployeeDayDataColumnTypes[EmployeeDayDataColumnTypes["UsedFixedBreakDuration"] = 35] = "UsedFixedBreakDuration";
    EmployeeDayDataColumnTypes[EmployeeDayDataColumnTypes["EntrancePenaltyTime"] = 36] = "EntrancePenaltyTime";
    EmployeeDayDataColumnTypes[EmployeeDayDataColumnTypes["ExitPenaltyTime"] = 37] = "ExitPenaltyTime";
})(EmployeeDayDataColumnTypes = exports.EmployeeDayDataColumnTypes || (exports.EmployeeDayDataColumnTypes = {}));
var EmployeeDayTotalDataColumnTypes;
(function (EmployeeDayTotalDataColumnTypes) {
    EmployeeDayTotalDataColumnTypes[EmployeeDayTotalDataColumnTypes["ExpectedWorkDuration"] = 65280] = "ExpectedWorkDuration";
    EmployeeDayTotalDataColumnTypes[EmployeeDayTotalDataColumnTypes["NormalWorkDuration"] = 65281] = "NormalWorkDuration";
    EmployeeDayTotalDataColumnTypes[EmployeeDayTotalDataColumnTypes["MissingWorkDuration"] = 65282] = "MissingWorkDuration";
    EmployeeDayTotalDataColumnTypes[EmployeeDayTotalDataColumnTypes["TotalWorkDuration"] = 65283] = "TotalWorkDuration";
    EmployeeDayTotalDataColumnTypes[EmployeeDayTotalDataColumnTypes["ExtraWorkDuration"] = 65284] = "ExtraWorkDuration";
})(EmployeeDayTotalDataColumnTypes = exports.EmployeeDayTotalDataColumnTypes || (exports.EmployeeDayTotalDataColumnTypes = {}));
var EmployeeBasedDataColumnTypes;
(function (EmployeeBasedDataColumnTypes) {
    EmployeeBasedDataColumnTypes[EmployeeBasedDataColumnTypes["ExpectedWorkDays"] = 4294967040] = "ExpectedWorkDays";
    EmployeeBasedDataColumnTypes[EmployeeBasedDataColumnTypes["DaysStartedWorkingOnTime"] = 4294967041] = "DaysStartedWorkingOnTime";
    EmployeeBasedDataColumnTypes[EmployeeBasedDataColumnTypes["DaysStartedWorkingEarlyAndOnTime"] = 4294967042] = "DaysStartedWorkingEarlyAndOnTime";
    EmployeeBasedDataColumnTypes[EmployeeBasedDataColumnTypes["DaysStartedWorkingLateInTolerance"] = 4294967043] = "DaysStartedWorkingLateInTolerance";
    EmployeeBasedDataColumnTypes[EmployeeBasedDataColumnTypes["DaysBeingLate"] = 4294967044] = "DaysBeingLate";
    EmployeeBasedDataColumnTypes[EmployeeBasedDataColumnTypes["DaysEarlyLeave"] = 4294967045] = "DaysEarlyLeave";
    EmployeeBasedDataColumnTypes[EmployeeBasedDataColumnTypes["DaysEarlyLeaveInTolerance"] = 4294967046] = "DaysEarlyLeaveInTolerance";
    EmployeeBasedDataColumnTypes[EmployeeBasedDataColumnTypes["DaysExtraWork"] = 4294967047] = "DaysExtraWork";
    EmployeeBasedDataColumnTypes[EmployeeBasedDataColumnTypes["UsedLeaveCount"] = 4294967048] = "UsedLeaveCount";
})(EmployeeBasedDataColumnTypes = exports.EmployeeBasedDataColumnTypes || (exports.EmployeeBasedDataColumnTypes = {}));
class ReportColumnExtractor {
    constructor() {
        this.registry = new Map();
        this.setRegistry();
    }
    setRegistry() {
        this.registry.set(EmployeeDayDataColumnTypes.Date, (data, context, trx) => {
            const date = luxon_1.DateTime.fromJSDate(context.date).setLocale(context.locale).toLocaleString(luxon_1.DateTime.DATE_FULL);
            return {
                value: date,
                border: generator_1.ReportGenerator.Constants.Styling.AllDotted,
            };
        });
        this.registry.set(EmployeeDayDataColumnTypes.FirstInTime, (data, context, trx) => {
            let firstEntryTime;
            data.ar.find((elem) => {
                let findingActualAccess = elem.a.find((innerElem) => innerElem.s && innerElem.si);
                if (findingActualAccess) {
                    firstEntryTime = {
                        firstStart: findingActualAccess.s,
                        firstStartId: findingActualAccess.si,
                    };
                }
                else {
                    let findingDummyAccess = elem.a.find((innerElem) => innerElem.s);
                    if (findingDummyAccess) {
                        firstEntryTime = {
                            firstStart: findingDummyAccess.s,
                            firstStartId: findingDummyAccess.si,
                        };
                    }
                }
            });
            let isDummyAccessLogForFirstIn = false;
            let firstInText;
            if (firstEntryTime && firstEntryTime.firstStart) {
                if (firstEntryTime.firstStartId) {
                    firstInText = (0, report_util_1.formatDateForExcel)(firstEntryTime.firstStart);
                }
                else {
                    isDummyAccessLogForFirstIn = true;
                }
            }
            return {
                value: isDummyAccessLogForFirstIn ? "__:__" : firstInText,
                border: generator_1.ReportGenerator.Constants.Styling.AllDotted,
                fillColor: isDummyAccessLogForFirstIn ? generator_1.ReportGenerator.Constants.ColorCodes.DUMMY_ACCESS_LOG : undefined,
                numFmt: isDummyAccessLogForFirstIn ? null : generator_1.ReportGenerator.Constants.TimeFormat,
                alignment: isDummyAccessLogForFirstIn ? "center" : "left",
            };
        });
        this.registry.set(EmployeeDayDataColumnTypes.LastOutTime, (data, context, trx) => {
            let lastLeaveTime;
            data.ar.reverse();
            data.ar.find((elem) => {
                elem.a.reverse();
                let findingActualAccess = elem.a.find((innerElem) => innerElem.e && innerElem.ei);
                if (findingActualAccess) {
                    lastLeaveTime = {
                        lastLeave: findingActualAccess.e,
                        lastLeaveId: findingActualAccess.ei,
                    };
                }
                else {
                    let findingActualAccess = elem.a.find((innerElem) => innerElem.e);
                    if (findingActualAccess) {
                        lastLeaveTime = {
                            lastLeave: findingActualAccess.e,
                            lastLeaveId: findingActualAccess.ei,
                        };
                    }
                }
            });
            let isDummyAccessLogForLastOut = false;
            let lastOutText;
            if (lastLeaveTime && lastLeaveTime.lastLeave) {
                if (lastLeaveTime.lastLeaveId) {
                    lastOutText = (0, report_util_1.formatDateForExcel)(lastLeaveTime.lastLeave);
                }
                else {
                    isDummyAccessLogForLastOut = true;
                }
            }
            return {
                value: isDummyAccessLogForLastOut ? "__:__" : lastOutText,
                border: generator_1.ReportGenerator.Constants.Styling.AllDotted,
                fillColor: isDummyAccessLogForLastOut ? generator_1.ReportGenerator.Constants.ColorCodes.DUMMY_ACCESS_LOG : undefined,
                numFmt: isDummyAccessLogForLastOut ? null : generator_1.ReportGenerator.Constants.TimeFormat,
                alignment: isDummyAccessLogForLastOut ? "center" : "left",
            };
        });
        this.registry.set(EmployeeDayDataColumnTypes.AttendanceStatusText, (data, context, trx) => {
            let statusText;
            if (data.s.pr > 0) {
                statusText = i18n_1.default.__({ phrase: "PDF-REPORT.arrived", locale: context.locale });
            }
            else if (data.s.wd > 0 && data.s.pr == 0) {
                statusText = i18n_1.default.__({ phrase: "PDF-REPORT.not_arrived", locale: context.locale });
            }
            else {
                statusText = "";
            }
            return {
                value: statusText,
                border: generator_1.ReportGenerator.Constants.Styling.AllDotted,
                numFmt: null,
                fillColor: this.getColorText(data),
                font: {
                    color: { argb: "FFFFFF" },
                    bold: true,
                },
            };
        });
        this.registry.set(EmployeeDayDataColumnTypes.UsedPermissions, (data, context, trx) => {
            let permissionArr = [];
            const holidayIdsOfTheDay = new Set(data.h.map((h) => h.h));
            const holidaysOfTheDay = context.holidays.filter((obj) => holidayIdsOfTheDay.has(obj.id));
            holidaysOfTheDay.forEach((holiday) => {
                let interval = luxon_1.DateTime.fromJSDate(holiday.startDateTime).setLocale(context.locale).toFormat("dd/MM/yyyy HH:mm") +
                    " - " +
                    luxon_1.DateTime.fromJSDate(holiday.endDateTime).setLocale(context.locale).toFormat("dd/MM/yyyy HH:mm");
                permissionArr.push(holiday.title + "\n(" + interval + ")");
            });
            const leaveIdsOfTheDay = new Set(data.p.map((l) => l.p));
            const leavesOfTheDay = context.leaves.filter((leave) => leaveIdsOfTheDay.has(leave.pid));
            leavesOfTheDay.forEach((permission) => {
                switch (permission?.status) {
                    case app_enums_1.enums.PPermissionStatus.Approved: {
                        let interval = luxon_1.DateTime.fromJSDate(permission.startDateTime).setLocale(context.locale).toFormat("dd/MM/yyyy HH:mm") +
                            " - " +
                            luxon_1.DateTime.fromJSDate(permission.endDateTime).setLocale(context.locale).toFormat("dd/MM/yyyy HH:mm");
                        permissionArr.push(permission.name + "\n(" + interval + ")");
                        break;
                    }
                    case app_enums_1.enums.PPermissionStatus.Rejected: {
                        permissionArr.push(i18n_1.default.__({ phrase: "EXCEL-REPORT.PPERMISSION_STATUS_REJECTED_INFO", locale: context.locale }));
                        break;
                    }
                    case app_enums_1.enums.PPermissionStatus.Waiting: {
                        permissionArr.push(i18n_1.default.__({ phrase: "EXCEL-REPORT.PPERMISSION_STATUS_WAITING_INFO", locale: context.locale }));
                        break;
                    }
                }
            });
            let permissionText = permissionArr.join("\n");
            return {
                value: permissionText,
                border: generator_1.ReportGenerator.Constants.Styling.AllDotted,
                alignment: {
                    wrapText: true,
                },
                numFmt: null,
                fillColor: this.getColorText(data),
                font: {
                    color: { argb: "FFFFFF" },
                    bold: true,
                },
            };
        });
        this.registry.set(EmployeeDayDataColumnTypes.WorkplanText, (data, context, trx) => {
            let wpText = "";
            if (data?.ws?.length && context?.workplans?.length) {
                const workingSegment = data.ws.find((segment) => segment?.wt === dal_constants_1.DalConstants.WorkPlanTimeRangeType.WorkingHours);
                if (workingSegment) {
                    const wp = context.workplans.find((wpp) => wpp?.id === workingSegment.wp);
                    wpText = wp?.name || "";
                }
            }
            if (!wpText && data?.w?.length && context?.workplans?.length) {
                const dayWpNames = data.w
                    .map((wpId) => {
                    const wp = context.workplans.find((wpp) => wpp?.id === wpId);
                    return wp?.name || "";
                })
                    .filter((name) => !!name);
                wpText = dayWpNames.join("\n");
            }
            return {
                value: wpText || "",
                border: generator_1.ReportGenerator.Constants.Styling.Dotted,
                alignment: {
                    wrapText: true,
                },
            };
        });
        this.registry.set(EmployeeDayDataColumnTypes.ExpectedEntrance, (data, context, trx) => {
            let workingSegments = data.ws.filter((segment) => segment.wt === dal_constants_1.DalConstants.WorkPlanTimeRangeType.WorkingHours);
            let expectedEntrance = workingSegments.map((val) => (0, report_util_1.formatDateToString)(new Date(val.s))).join("\n");
            return {
                value: expectedEntrance ?? "",
                border: generator_1.ReportGenerator.Constants.Styling.AllDotted,
                numFmt: expectedEntrance ? generator_1.ReportGenerator.Constants.TimeFormat : null,
            };
        });
        this.registry.set(EmployeeDayDataColumnTypes.ExpectedLeave, (data, context, trx) => {
            let workingSegments = data.ws.filter((segment) => segment.wt === dal_constants_1.DalConstants.WorkPlanTimeRangeType.WorkingHours);
            let expectedLeave = workingSegments.map((val) => (0, report_util_1.formatDateToString)(new Date(val.e))).join("\n");
            return {
                value: expectedLeave ?? "",
                border: generator_1.ReportGenerator.Constants.Styling.AllDotted,
                numFmt: expectedLeave ? generator_1.ReportGenerator.Constants.TimeFormat : null,
            };
        });
        this.registry.set(EmployeeDayDataColumnTypes.ExpectedWorkDuration, (data, context, trx) => {
            let expectedWorkDuration = data.s.ew;
            return {
                value: expectedWorkDuration ? (expectedWorkDuration / MINUTES_IN_A_DAY) : null,
                border: generator_1.ReportGenerator.Constants.Styling.Dotted,
                numFmt: expectedWorkDuration && this.getDurationFormatExcel(context.locale),
            };
        });
        this.registry.set(EmployeeDayDataColumnTypes.NormalWorkDuration, (data, context, trx) => {
            let normalWorkDuration = data.s.n ? data.s.n / MINUTES_IN_A_DAY : null;
            return {
                value: normalWorkDuration ? normalWorkDuration : null,
                border: generator_1.ReportGenerator.Constants.Styling.Dotted,
                numFmt: normalWorkDuration && this.getDurationFormatExcel(context.locale),
            };
        });
        this.registry.set(EmployeeDayDataColumnTypes.MissingWorkDuration, (data, context, trx) => {
            let missingWorkDuration = data.s.m ? data.s.m / MINUTES_IN_A_DAY : null;
            return {
                value: missingWorkDuration ? missingWorkDuration : null,
                border: generator_1.ReportGenerator.Constants.Styling.AllDotted,
                numFmt: missingWorkDuration && this.getDurationFormatExcel(context.locale),
            };
        });
        this.registry.set(EmployeeDayDataColumnTypes.ExtraWorkDuration, (data, context, trx) => {
            let extraWorkDuration = data.s.e ? data.s.e / MINUTES_IN_A_DAY : null;
            return {
                value: extraWorkDuration ? extraWorkDuration : null,
                border: generator_1.ReportGenerator.Constants.Styling.AllDotted,
                numFmt: extraWorkDuration && this.getDurationFormatExcel(context.locale),
            };
        });
        this.registry.set(EmployeeDayDataColumnTypes.TotalWorkDuration, (data, context, trx) => {
            let totalWorkDuration = data.s.e || data.s.n ? (data.s.e + data.s.n) / MINUTES_IN_A_DAY : null;
            return {
                value: totalWorkDuration ? totalWorkDuration : null,
                border: generator_1.ReportGenerator.Constants.Styling.AllDotted,
                numFmt: totalWorkDuration && this.getDurationFormatExcel(context.locale),
            };
        });
        this.registry.set(EmployeeDayDataColumnTypes.PhysicallyInRegionDuration, (data, context, trx) => {
            let physicallyInRegionDuration = data.s.pr ? data.s.pr / MINUTES_IN_A_DAY : null;
            return {
                value: physicallyInRegionDuration ? physicallyInRegionDuration : null,
                border: generator_1.ReportGenerator.Constants.Styling.AllDotted,
                numFmt: physicallyInRegionDuration && this.getDurationFormatExcel(context.locale),
            };
        });
        this.registry.set(EmployeeDayDataColumnTypes.FixedBreakDuration, (data, context, trx) => {
            let fixedBreakDuration = data.s.ufb + data.s.afb + data.s.ucb + data.s.acb;
            return {
                value: fixedBreakDuration !== 0 ? (fixedBreakDuration / MINUTES_IN_A_DAY) : null,
                border: generator_1.ReportGenerator.Constants.Styling.Dotted,
                numFmt: fixedBreakDuration && this.getDurationFormatExcel(context.locale),
            };
        });
        this.registry.set(EmployeeDayDataColumnTypes.Score, (data, context, trx) => {
            let score = data.s.s ?? null;
            return {
                value: score,
                border: generator_1.ReportGenerator.Constants.Styling.AllDotted,
            };
        });
        this.registry.set(EmployeeDayDataColumnTypes.AccountedDeclaredWorkDuration, (data, context, trx) => {
            let accountedDeclaredWorkDuration = data.s.dw ? data.s.dw / MINUTES_IN_A_DAY : null;
            return {
                value: accountedDeclaredWorkDuration ? accountedDeclaredWorkDuration : null,
                border: generator_1.ReportGenerator.Constants.Styling.AllDotted,
                numFmt: accountedDeclaredWorkDuration && this.getDurationFormatExcel(context.locale),
            };
        });
        this.registry.set(EmployeeDayDataColumnTypes.AccountedIgnoredWorkDuration, (data, context, trx) => {
            let accountedIgnoredWorkDuration = data.s.iw ? data.s.iw / MINUTES_IN_A_DAY : null;
            return {
                value: accountedIgnoredWorkDuration ? accountedIgnoredWorkDuration : null,
                border: generator_1.ReportGenerator.Constants.Styling.AllDotted,
                numFmt: accountedIgnoredWorkDuration && this.getDurationFormatExcel(context.locale),
            };
        });
        this.registry.set(EmployeeDayDataColumnTypes.AccountedOffTimePermissionDuration, (data, context, trx) => {
            let accountedOffTimePermissionDuration = data.s.ao ? data.s.ao / MINUTES_IN_A_DAY : null;
            return {
                value: accountedOffTimePermissionDuration ? accountedOffTimePermissionDuration : null,
                border: generator_1.ReportGenerator.Constants.Styling.AllDotted,
                numFmt: accountedOffTimePermissionDuration && this.getDurationFormatExcel(context.locale),
            };
        });
        this.registry.set(EmployeeDayDataColumnTypes.EarlyLeave, (data, context, trx) => {
            let earlyLeave = data.s.el ? data.s.el / MINUTES_IN_A_DAY : null;
            return {
                value: earlyLeave ? earlyLeave : null,
                border: generator_1.ReportGenerator.Constants.Styling.AllDotted,
                numFmt: earlyLeave && this.getDurationFormatExcel(context.locale),
            };
        });
        this.registry.set(EmployeeDayDataColumnTypes.LateLeave, (data, context, trx) => {
            let lateLeave = data.s.ll ? data.s.ll / MINUTES_IN_A_DAY : null;
            return {
                value: lateLeave ? lateLeave : null,
                border: generator_1.ReportGenerator.Constants.Styling.AllDotted,
                numFmt: lateLeave && this.getDurationFormatExcel(context.locale),
            };
        });
        this.registry.set(EmployeeDayDataColumnTypes.EarlyStart, (data, context, trx) => {
            let earlyStart = data.s.es ? data.s.es / MINUTES_IN_A_DAY : null;
            return {
                value: earlyStart ? earlyStart : null,
                border: generator_1.ReportGenerator.Constants.Styling.AllDotted,
                numFmt: earlyStart && this.getDurationFormatExcel(context.locale),
            };
        });
        this.registry.set(EmployeeDayDataColumnTypes.LateStart, (data, context, trx) => {
            let lateStart = data.s.ls ? data.s.ls / MINUTES_IN_A_DAY : null;
            return {
                value: lateStart ? lateStart : null,
                border: generator_1.ReportGenerator.Constants.Styling.AllDotted,
                numFmt: lateStart && this.getDurationFormatExcel(context.locale),
            };
        });
        this.registry.set(EmployeeDayDataColumnTypes.ToleratedEarlyLeave, (data, context, trx) => {
            let toleratedEarlyLeave = data.s.tel ? data.s.tel / MINUTES_IN_A_DAY : null;
            return {
                value: toleratedEarlyLeave ? toleratedEarlyLeave : null,
                border: generator_1.ReportGenerator.Constants.Styling.AllDotted,
                numFmt: toleratedEarlyLeave && this.getDurationFormatExcel(context.locale),
            };
        });
        this.registry.set(EmployeeDayDataColumnTypes.ToleratedEarlyStart, (data, context, trx) => {
            let toleratedEarlyStart = data.s.tes ? data.s.tes / MINUTES_IN_A_DAY : null;
            return {
                value: toleratedEarlyStart ? toleratedEarlyStart : null,
                border: generator_1.ReportGenerator.Constants.Styling.AllDotted,
                numFmt: toleratedEarlyStart && this.getDurationFormatExcel(context.locale),
            };
        });
        this.registry.set(EmployeeDayDataColumnTypes.ToleratedLateLeave, (data, context, trx) => {
            let toleratedLateLeave = data.s.tll ? data.s.tll / MINUTES_IN_A_DAY : null;
            return {
                value: toleratedLateLeave ? toleratedLateLeave : null,
                border: generator_1.ReportGenerator.Constants.Styling.AllDotted,
                numFmt: toleratedLateLeave && this.getDurationFormatExcel(context.locale),
            };
        });
        this.registry.set(EmployeeDayDataColumnTypes.ToleratedLateStart, (data, context, trx) => {
            let toleratedLateStart = data.s.tls ? data.s.tls / MINUTES_IN_A_DAY : null;
            return {
                value: toleratedLateStart ? toleratedLateStart : null,
                border: generator_1.ReportGenerator.Constants.Styling.AllDotted,
                numFmt: toleratedLateStart && this.getDurationFormatExcel(context.locale),
            };
        });
        this.registry.set(EmployeeDayDataColumnTypes.ExpectedHolidayDuration, (data, context, trx) => {
            let expectedHolidayDuration = data.s.eh ? data.s.eh / MINUTES_IN_A_DAY : null;
            return {
                value: expectedHolidayDuration ? expectedHolidayDuration : null,
                border: generator_1.ReportGenerator.Constants.Styling.AllDotted,
                numFmt: expectedHolidayDuration && this.getDurationFormatExcel(context.locale),
            };
        });
        this.registry.set(EmployeeDayDataColumnTypes.UsedHolidayDuration, (data, context, trx) => {
            let usedHolidayDuration = data.s.uh ? data.s.uh / MINUTES_IN_A_DAY : null;
            return {
                value: usedHolidayDuration ? usedHolidayDuration : null,
                border: generator_1.ReportGenerator.Constants.Styling.AllDotted,
                numFmt: usedHolidayDuration && this.getDurationFormatExcel(context.locale),
            };
        });
        this.registry.set(EmployeeDayDataColumnTypes.AutomaticallyUsedClaimBreakDuration, (data, context, trx) => {
            let automaticallyUsedClaimBreakDuration = data.s.acb ? data.s.acb / MINUTES_IN_A_DAY : null;
            return {
                value: automaticallyUsedClaimBreakDuration ? automaticallyUsedClaimBreakDuration : null,
                border: generator_1.ReportGenerator.Constants.Styling.AllDotted,
                numFmt: automaticallyUsedClaimBreakDuration && this.getDurationFormatExcel(context.locale),
            };
        });
        this.registry.set(EmployeeDayDataColumnTypes.AutomaticallyUsedFixedBreakDuration, (data, context, trx) => {
            let automaticallyUsedFixedBreakDuration = data.s.afb ? data.s.afb / MINUTES_IN_A_DAY : null;
            return {
                value: automaticallyUsedFixedBreakDuration ? automaticallyUsedFixedBreakDuration : null,
                border: generator_1.ReportGenerator.Constants.Styling.AllDotted,
                numFmt: automaticallyUsedFixedBreakDuration && this.getDurationFormatExcel(context.locale),
            };
        });
        this.registry.set(EmployeeDayDataColumnTypes.ExpectedClaimBreakDuration, (data, context, trx) => {
            let expectedClaimBreakDuration = data.s.ecb ? data.s.ecb / MINUTES_IN_A_DAY : null;
            return {
                value: expectedClaimBreakDuration ? expectedClaimBreakDuration : null,
                border: generator_1.ReportGenerator.Constants.Styling.AllDotted,
                numFmt: expectedClaimBreakDuration && this.getDurationFormatExcel(context.locale),
            };
        });
        this.registry.set(EmployeeDayDataColumnTypes.ExpectedFixedBreakDuration, (data, context, trx) => {
            let expectedFixedBreakDuration = data.s.efb ? data.s.efb / MINUTES_IN_A_DAY : null;
            return {
                value: expectedFixedBreakDuration ? expectedFixedBreakDuration : null,
                border: generator_1.ReportGenerator.Constants.Styling.AllDotted,
                numFmt: expectedFixedBreakDuration && this.getDurationFormatExcel(context.locale),
            };
        });
        this.registry.set(EmployeeDayDataColumnTypes.ExpectedOffTimePermissionDuration, (data, context, trx) => {
            let expectedOffTimePermissionDuration = data.s.eo ? data.s.eo / MINUTES_IN_A_DAY : null;
            return {
                value: expectedOffTimePermissionDuration ? expectedOffTimePermissionDuration : null,
                border: generator_1.ReportGenerator.Constants.Styling.AllDotted,
                numFmt: expectedOffTimePermissionDuration && this.getDurationFormatExcel(context.locale),
            };
        });
        this.registry.set(EmployeeDayDataColumnTypes.UsedClaimBreakDuration, (data, context, trx) => {
            let usedClaimBreakDuration = data.s.ucb ? data.s.ucb / MINUTES_IN_A_DAY : null;
            return {
                value: usedClaimBreakDuration ? usedClaimBreakDuration : null,
                border: generator_1.ReportGenerator.Constants.Styling.AllDotted,
                numFmt: usedClaimBreakDuration && this.getDurationFormatExcel(context.locale),
            };
        });
        this.registry.set(EmployeeDayDataColumnTypes.UsedFixedBreakDuration, (data, context, trx) => {
            let usedFixedBreakDuration = data.s.ufb ? data.s.ufb / MINUTES_IN_A_DAY : null;
            return {
                value: usedFixedBreakDuration ? usedFixedBreakDuration : null,
                border: generator_1.ReportGenerator.Constants.Styling.AllDotted,
                numFmt: usedFixedBreakDuration && this.getDurationFormatExcel(context.locale),
            };
        });
        this.registry.set(EmployeeDayDataColumnTypes.EntrancePenaltyTime, (data, context, trx) => {
            let entrancePenaltyTimeInSeconds = data.s.ls || data.s.tls ? (data.s.ls + data.s.tls) * 60 : null;
            return {
                value: entrancePenaltyTimeInSeconds ? (Math.round(entrancePenaltyTimeInSeconds / 60) / MINUTES_IN_A_DAY) : "",
                border: generator_1.ReportGenerator.Constants.Styling.AllDotted,
                numFmt: entrancePenaltyTimeInSeconds && this.getDurationFormatExcel(context.locale),
            };
        });
        this.registry.set(EmployeeDayDataColumnTypes.ExitPenaltyTime, (data, context, trx) => {
            let entrancePenaltyTimeInSeconds = data.s.el || data.s.tel ? (data.s.el + data.s.tel) * 60 : null;
            return {
                value: entrancePenaltyTimeInSeconds ? (Math.round(entrancePenaltyTimeInSeconds / 60) / MINUTES_IN_A_DAY) : "",
                border: generator_1.ReportGenerator.Constants.Styling.AllDotted,
                numFmt: entrancePenaltyTimeInSeconds && this.getDurationFormatExcel(context.locale),
            };
        });
        this.registry.set(EmployeeDayTotalDataColumnTypes.ExpectedWorkDuration, (data, context, trx) => {
            const totalExpectedWorkDuration = data.ew ? data.ew / MINUTES_IN_A_DAY : null;
            return {
                value: totalExpectedWorkDuration ? totalExpectedWorkDuration : null,
                border: generator_1.ReportGenerator.Constants.Styling.Dotted,
                numFmt: totalExpectedWorkDuration && this.getDurationFormatExcel(context.locale),
            };
        });
        this.registry.set(EmployeeDayTotalDataColumnTypes.NormalWorkDuration, (data, context, trx) => {
            const totalNormalWorkDuration = data.n ? data.n / MINUTES_IN_A_DAY : null;
            return {
                value: totalNormalWorkDuration ? totalNormalWorkDuration : null,
                border: generator_1.ReportGenerator.Constants.Styling.Dotted,
                numFmt: totalNormalWorkDuration && this.getDurationFormatExcel(context.locale),
            };
        });
        this.registry.set(EmployeeDayTotalDataColumnTypes.MissingWorkDuration, (data, context, trx) => {
            const totalMissingWorkDuration = data.m ? data.m / MINUTES_IN_A_DAY : null;
            return {
                value: totalMissingWorkDuration ? totalMissingWorkDuration : null,
                border: generator_1.ReportGenerator.Constants.Styling.AllDotted,
                numFmt: totalMissingWorkDuration && this.getDurationFormatExcel(context.locale),
            };
        });
        this.registry.set(EmployeeDayTotalDataColumnTypes.ExtraWorkDuration, (data, context, trx) => {
            const totalExtraWorkDuration = data.e ? data.e / MINUTES_IN_A_DAY : null;
            return {
                value: totalExtraWorkDuration ? totalExtraWorkDuration : null,
                border: generator_1.ReportGenerator.Constants.Styling.AllDotted,
                numFmt: totalExtraWorkDuration && this.getDurationFormatExcel(context.locale),
            };
        });
        this.registry.set(EmployeeDayTotalDataColumnTypes.TotalWorkDuration, (data, context, trx) => {
            let totalTotalWorkDuration = data.e || data.n ? (data.e + data.n) / MINUTES_IN_A_DAY : null;
            return {
                value: totalTotalWorkDuration ? totalTotalWorkDuration : null,
                border: generator_1.ReportGenerator.Constants.Styling.AllDotted,
                numFmt: totalTotalWorkDuration && this.getDurationFormatExcel(context.locale),
            };
        });
        this.registry.set(EmployeeBasedDataColumnTypes.ExpectedWorkDays, (data, context, trx) => {
            let expectedWorkDays = 0;
            for (let day of data) {
                if (day.s.ew > 0) {
                    expectedWorkDays++;
                }
            }
            return {
                value: expectedWorkDays ? expectedWorkDays : null,
                border: generator_1.ReportGenerator.Constants.Styling.AllDotted,
            };
        });
        this.registry.set(EmployeeBasedDataColumnTypes.DaysStartedWorkingOnTime, (data, context, trx) => {
            let daysStartedOnTime = 0;
            for (let day of data) {
                if (day.ar[0]?.a[0]?.s && +day.s.ls === 0) {
                    daysStartedOnTime++;
                }
            }
            return {
                value: daysStartedOnTime ? daysStartedOnTime : null,
                border: generator_1.ReportGenerator.Constants.Styling.AllDotted,
            };
        });
        this.registry.set(EmployeeBasedDataColumnTypes.DaysStartedWorkingEarlyAndOnTime, (data, context, trx) => {
            const days = data.filter((day) => day.ar[0]?.a[0]?.s && day.s.ls === 0 && day.s.tls === 0).length;
            return {
                value: days ? days : null,
                border: generator_1.ReportGenerator.Constants.Styling.AllDotted,
            };
        });
        this.registry.set(EmployeeBasedDataColumnTypes.DaysStartedWorkingLateInTolerance, (data, context, trx) => {
            const days = data.filter((day) => day.s.ls === 0 && day.s.tls > 0).length;
            return {
                value: days ? days : null,
                border: generator_1.ReportGenerator.Constants.Styling.AllDotted,
            };
        });
        this.registry.set(EmployeeBasedDataColumnTypes.DaysBeingLate, (data, context, trx) => {
            let lateStartedDays = 0;
            for (let day of data) {
                if (day.s.ls > 0) {
                    lateStartedDays++;
                }
            }
            return {
                value: lateStartedDays ? lateStartedDays : null,
                border: generator_1.ReportGenerator.Constants.Styling.AllDotted,
            };
        });
        this.registry.set(EmployeeBasedDataColumnTypes.DaysEarlyLeave, (data, context, trx) => {
            let daysLeftEarly = 0;
            for (let day of data) {
                if (day.s.el > 0) {
                    daysLeftEarly++;
                }
            }
            return {
                value: daysLeftEarly ? daysLeftEarly : null,
                border: generator_1.ReportGenerator.Constants.Styling.AllDotted,
            };
        });
        this.registry.set(EmployeeBasedDataColumnTypes.DaysEarlyLeaveInTolerance, (data, context, trx) => {
            const days = data.filter((day) => day.s.el === 0 && day.s.tel > 0).length;
            return {
                value: days ? days : null,
                border: generator_1.ReportGenerator.Constants.Styling.AllDotted,
            };
        });
        this.registry.set(EmployeeBasedDataColumnTypes.DaysExtraWork, (data, context, trx) => {
            let daysWorkedExtra = 0;
            for (let day of data) {
                if (day.s.ll > 0) {
                    daysWorkedExtra++;
                }
            }
            return {
                value: daysWorkedExtra ? daysWorkedExtra : null,
                border: generator_1.ReportGenerator.Constants.Styling.AllDotted,
            };
        });
        this.registry.set(EmployeeBasedDataColumnTypes.UsedLeaveCount, (data, context, trx) => {
            let usedLeaveCount = 0;
            for (let day of data) {
                for (let p of day.p) {
                    if (p.u > 0) {
                        usedLeaveCount++;
                    }
                }
            }
            return {
                value: usedLeaveCount ? usedLeaveCount : null,
                border: generator_1.ReportGenerator.Constants.Styling.AllDotted,
            };
        });
    }
    invoke(type, data, context, trx) {
        try {
            return this.registry.get(type)(data, context, trx);
        }
        catch (error) {
            app_logs_1.logger.error(error);
        }
    }
    getDurationFormatExcel(locale) {
        return ('[<0.000694444]"";[<0.0415][m]" ' +
            i18n_1.default.__({ phrase: "EXCEL-REPORT.MINUTE_ABBREVATION", locale: locale }) +
            '";[h] "' +
            i18n_1.default.__({ phrase: "EXCEL-REPORT.HOUR_ABBREVATION", locale: locale }) +
            ' " m "' +
            i18n_1.default.__({ phrase: "EXCEL-REPORT.MINUTE_ABBREVATION", locale: locale }) +
            '"');
    }
    getColorText(data) {
        let colorText = undefined;
        if (data.s.pr && data.s.pr > 0) {
            if (data.s.ls > 0) {
                colorText = generator_1.ReportGenerator.Constants.ColorCodes.NOT_TOLERATED;
            }
            else {
                if (data.s.tls && data.s.tls > 0) {
                    colorText = generator_1.ReportGenerator.Constants.ColorCodes.TOLERATED;
                }
                else {
                    colorText = generator_1.ReportGenerator.Constants.ColorCodes.ON_TIME;
                }
            }
        }
        else {
            if (data.s.wd && data.s.wd > 0) {
                if (data.s.ew && data.s.ew > 0) {
                    colorText = data.s.ao > 0 ? generator_1.ReportGenerator.Constants.ColorCodes.PERMISSION : generator_1.ReportGenerator.Constants.ColorCodes.NOT_TOLERATED_2;
                }
                else {
                    if (data.s.eh > 0 || data.s.ao > 0) {
                        colorText = generator_1.ReportGenerator.Constants.ColorCodes.PERMISSION;
                    }
                }
            }
            else {
                colorText = "AAAAAA";
                if (data.s.eh > 0) {
                    colorText = generator_1.ReportGenerator.Constants.ColorCodes.PERMISSION;
                }
            }
        }
        return colorText;
    }
}
const reportColumnExtractor = new ReportColumnExtractor();
exports.default = reportColumnExtractor;
