"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.getSensorEventsReport = exports.generateUnterminatedVisitReport = exports.generateUserTicketInfoReportData = exports.generateFirstAndLastAccessLogsForSingleUserReportData = exports.generateFirstAndLastAccessLogsReportData = exports.generatePACSDailyForSingleUserData = exports.generateListPPermissionsData = exports.generateVisitorListReportData = exports.generateTicketTransactionReportData = exports.generateUserTicketReportData = exports.generateRegionStateData = exports.generateForbiddenIdentityData = exports.getCaptionDefinitions = void 0;
const i18n_1 = __importDefault(require("i18n"));
const lodash_1 = __importDefault(require("lodash"));
const moment_1 = __importDefault(require("moment"));
const app_config_1 = require("../../app.config");
const app_enums_1 = require("../../app.enums");
const dal_manager_1 = require("../../dal/dal.manager");
const business_pacs_1 = require("../pacs/business.pacs");
const report_util_1 = require("./report.util");
function convertDateToUtc(d) {
    let utc = (0, moment_1.default)(d);
    let tz = utc.toDate().getTimezoneOffset();
    utc.add(-tz, "minute");
    return utc.toDate();
}
async function getCaptionDefinitions(organizationId, userId) {
    const { l: locale } = await dal_manager_1.dbManager.accessRedisCache.getUserNotificationCache({ organizationId, userId });
    let captionDefinitions = await dal_manager_1.dbManager.accessOrganization.getOrganizationUserProfileSettings(organizationId);
    for (let def of captionDefinitions.captionLineFields) {
        if (def.caption)
            def.caption = i18n_1.default.__({ phrase: "UIFIELDS.IDENTITY." + def.caption, locale });
    }
    return Promise.resolve(captionDefinitions);
}
exports.getCaptionDefinitions = getCaptionDefinitions;
async function generateForbiddenIdentityData(organizationId, userId, filter) {
    const result = {
        items: [],
        pagination: {
            take: undefined,
            skip: undefined,
            total: 0,
        },
        captionDefinitions: null,
    };
    const option = Object.assign(filter, {
        pagination: { take: app_config_1.appConfig.reportExportRowLimit, skip: undefined },
    });
    const dbResult = await dal_manager_1.dbManager.accessIdentity.listForbidden({ organizationId, filter: option, requesterUserId: userId });
    result.pagination.total = dbResult.pagination.total;
    result.captionDefinitions = await getCaptionDefinitions(organizationId, userId);
    result.organization = await dal_manager_1.dbManager.accessOrganization.getOrganizationLogoAndName(organizationId);
    const captions = await dal_manager_1.dbManager.accessUser.getUserOrganizationCaptionLines(organizationId, dbResult.items.map((i) => i.userId));
    result.items = dbResult.items.map((item) => {
        return {
            ...item,
            userCaptions: captions.find((c) => c.id === item.userId).captionLines ?? [],
        };
    });
    return result;
}
exports.generateForbiddenIdentityData = generateForbiddenIdentityData;
async function generateRegionStateData(options) {
    let result = {
        items: [],
        region: null,
        captionDefinitions: null,
    };
    let dbResult = await dal_manager_1.dbManager.accessRegion.getRegionStateReportNew(options.organizationId, {
        pagination: {
            take: app_config_1.appConfig.reportExportRowLimit + 1,
            skip: 0,
        },
        regionId: options.regionId,
        userIds: options.filter.userIds,
        organizationUnitIds: options.filter.organizationUnitIds,
        userGroupIds: options.filter.userGroupIds,
        applyOrganizationUnitFilterHierarchically: options.filter.applyOrganizationUnitFilterHierarchically,
        state: options.filter.state,
        status: options.filter.status,
    }, options.requesterUserId);
    result.region = await dal_manager_1.dbManager.accessRegion.getRegionDetailed({ organizationId: options.organizationId, regionId: options.regionId });
    let userCaptions = await dal_manager_1.dbManager.accessUser.getUserOrganizationCaptionLines(options.organizationId, dbResult.items.map((i) => i.user.id));
    for (const item of dbResult.items) {
        const captions = userCaptions.find((f) => f.id == item.user.id) || { captionLines: [] };
        result.items.push({
            ...item,
            userCaptions: captions.captionLines,
            userId: item.user.id,
        });
    }
    result.captionDefinitions = await getCaptionDefinitions(options.organizationId, options.requesterUserId);
    result.totalCount = dbResult.pagination.total;
    result.organization = await dal_manager_1.dbManager.accessOrganization.getOrganizationLogoAndName(options.organizationId);
    return Promise.resolve(result);
}
exports.generateRegionStateData = generateRegionStateData;
async function generateUserTicketReportData(organizationId, userId, filter) {
    let result = {
        items: [],
        captionDefinitions: null,
    };
    let dbResult = await dal_manager_1.dbManager.accessSocial.listRegionTicketUsageReport(organizationId, {
        take: app_config_1.appConfig.reportExportRowLimit + 1,
        skip: 0,
    }, {
        dateRange: filter.dateRange,
        userIds: filter.userIds,
        userGroupIds: filter.userGroupIds,
        organizationUnitIds: filter.organizationUnitIds,
        regionTicketIds: filter.regionTicketIds,
    });
    let userCaptions = await dal_manager_1.dbManager.accessUser.getUserOrganizationCaptionLines(organizationId, dbResult.items.map((i) => i.user.id));
    for (const item of dbResult.items) {
        let captions = userCaptions.find((f) => f.id == item.user.id);
        item.actionDate = convertDateToUtc(item.actionDate);
        if (captions) {
            result.items.push(lodash_1.default.assign({}, item, { userCaptions: captions.captionLines }));
        }
        else {
            result.items.push(lodash_1.default.assign({}, item, { userCaptions: [] }));
        }
    }
    result.captionDefinitions = await getCaptionDefinitions(organizationId, userId);
    result.totalCount = dbResult.pagination.total;
    result.organization = await dal_manager_1.dbManager.accessOrganization.getOrganizationLogoAndName(organizationId);
    return Promise.resolve(result);
}
exports.generateUserTicketReportData = generateUserTicketReportData;
async function generateTicketTransactionReportData(organizationId, userId, filter) {
    let result = {
        items: [],
        captionDefinitions: null,
    };
    let transactions = await dal_manager_1.dbManager.accessLog.listRegionTicketRawTransactions(organizationId, {
        actionUserId: filter.actionUserId,
        dateRange: filter.dateRange,
        targetUserId: filter.targetUserId,
    });
    let userIds = new Set();
    for (const item of transactions) {
        for (const targetUserId of item.userIds) {
            userIds.add(targetUserId);
        }
    }
    const userCaptions = await dal_manager_1.dbManager.accessUser.getUserOrganizationCaptionLines(organizationId, Array.from(userIds));
    const actionUsersCaptions = await dal_manager_1.dbManager.accessUser.getUserOrganizationCaptionLines(organizationId, transactions.map((t) => t.actionUserId));
    for (const item of transactions) {
        let userCaptionsGroups = userCaptions
            .filter((u) => item.userIds.some((s) => s == u.id))
            .map((u) => {
            return {
                id: u.id,
                userCaptions: u.captionLines,
            };
        });
        let newItem;
        let actionUserCaptions = actionUsersCaptions.find((f) => f.id == item.actionUserId);
        if (actionUserCaptions) {
            newItem = lodash_1.default.assign({}, item, { actionUserCaptions: actionUserCaptions.captionLines, userCaptions: userCaptionsGroups });
        }
        else {
            newItem = lodash_1.default.assign({}, item, { actionUserCaptions: [], userCaptions: userCaptionsGroups });
        }
        newItem.actionUtc = convertDateToUtc(item.actionUtc);
        result.items.push(newItem);
    }
    result.captionDefinitions = await getCaptionDefinitions(organizationId, userId);
    result.totalCount = transactions.length;
    result.organization = await dal_manager_1.dbManager.accessOrganization.getOrganizationLogoAndName(organizationId);
    return Promise.resolve(result);
}
exports.generateTicketTransactionReportData = generateTicketTransactionReportData;
async function generateVisitorListReportData(organizationId, userId, filter) {
    let organizationVisitorModuleSettings = await dal_manager_1.dbManager.accessVisitor.getVisitAndVisitorFormSettings(organizationId);
    let dbResult = await dal_manager_1.dbManager.accessVisitor.listVisitors(organizationId, {
        take: app_config_1.appConfig.reportExportRowLimit + 1,
        skip: 0,
        visitorProfileFilterFields: filter.visitorProfileFilterFields,
        showOnlyBlacklist: filter.showOnlyBlacklist,
        showOnlyTemporaryPermitted: filter.showOnlyTemporaryPermitted,
    }, organizationVisitorModuleSettings);
    if (filter.sensitiveFields) {
        (0, report_util_1.hideSensitiveFieldsOfVisitorForReport)(filter.sensitiveFields.visitorFields, dbResult.items, {
            hideUniqueIdForVisitor: filter.sensitiveFields.hideUniqueIdForVisitor,
            showNumberOfInitial: filter.sensitiveFields.showNumberOfInitial,
            showNumberOfLast: filter.sensitiveFields.showNumberOfLast,
        });
    }
    let result = {
        items: dbResult.items,
        organizationVisitorModuleSettings: organizationVisitorModuleSettings,
        totalCount: dbResult.pagination.total,
        captionDefinitions: await getCaptionDefinitions(organizationId, userId),
        organizationLogo: await dal_manager_1.dbManager.accessOrganization.getOrganizationLogoFile(organizationId),
    };
    return Promise.resolve(result);
}
exports.generateVisitorListReportData = generateVisitorListReportData;
async function generateListPPermissionsData(organizationId, userId, filter) {
    let result = {
        captionDefinitions: null,
        ppermissions: null,
        userCaptions: [],
    };
    let dbResult = await dal_manager_1.dbManager.systemTransaction((trx) => {
        return dal_manager_1.dbManager.accessPacs.listLeaves(trx, organizationId, userId, app_enums_1.enums.PPermissionInsertRole.HR, {
            ppermissionTypeId: filter.ppermissionTypeId,
            startDateTime: filter.startDateTime,
            endDateTime: filter.endDateTime,
            take: app_config_1.appConfig.reportExportRowLimit + 1,
            skip: 0,
            status: filter.status,
            userIds: filter.userIds,
            organizationUnitIds: filter.organizationUnitIds,
            organizationUnitHierarchically: filter.organizationUnitHierarchically,
            userGroupIds: filter.userGroupIds,
            ppermissionIds: [],
        });
    });
    result.ppermissions = (await (0, business_pacs_1.decoratePPermissionListWithCaptions)(organizationId, dbResult)).items;
    result.captionDefinitions = await getCaptionDefinitions(organizationId, userId);
    result.organization = await dal_manager_1.dbManager.accessOrganization.getOrganizationLogoAndName(organizationId);
    return Promise.resolve(result);
}
exports.generateListPPermissionsData = generateListPPermissionsData;
async function generatePACSDailyForSingleUserData(organizationId, userId, targetUserId, filter) {
    let result = {
        captionDefinitions: null,
        report: null,
    };
    let dbResult = await dal_manager_1.dbManager.accessPacs2.getDailySummaryReportForSingleUser(organizationId, targetUserId, filter);
    result.report = { items: dbResult.items, workPlans: dbResult.workPlans, summary: dbResult.summary, userCaptions: dbResult.userCaptions };
    result.totalCount = dbResult.pagination.total;
    result.captionDefinitions = await getCaptionDefinitions(organizationId, userId);
    result.organization = await dal_manager_1.dbManager.accessOrganization.getOrganizationLogoAndName(organizationId);
    return Promise.resolve(result);
}
exports.generatePACSDailyForSingleUserData = generatePACSDailyForSingleUserData;
async function generateFirstAndLastAccessLogsReportData(organizationId, userId, hasOrganizationWide, filter) {
    let result = {
        report: null,
        captionDefinitions: null,
    };
    result.report = await dal_manager_1.dbManager.accessReport.filterFirstAndtLastAccessLogs(organizationId, filter, userId, !hasOrganizationWide);
    result.captionDefinitions = await getCaptionDefinitions(organizationId, userId);
    result.organization = await dal_manager_1.dbManager.accessOrganization.getOrganizationLogoAndName(organizationId);
    return Promise.resolve(result);
}
exports.generateFirstAndLastAccessLogsReportData = generateFirstAndLastAccessLogsReportData;
async function generateFirstAndLastAccessLogsForSingleUserReportData(organizationId, userId, filter) {
    let result = {
        report: null,
        captionDefinitions: null,
    };
    result.report = await dal_manager_1.dbManager.accessReport.filterFirstAndtLastAccessLogsForSingleUser(organizationId, filter, { take: app_config_1.appConfig.reportExportRowLimit + 1, skip: 0 });
    result.captionDefinitions = await getCaptionDefinitions(organizationId, userId);
    result.organization = await dal_manager_1.dbManager.accessOrganization.getOrganizationLogoAndName(organizationId);
    return Promise.resolve(result);
}
exports.generateFirstAndLastAccessLogsForSingleUserReportData = generateFirstAndLastAccessLogsForSingleUserReportData;
async function generateUserTicketInfoReportData(organizationId, userId, filter) {
    let result = {
        items: [],
        captionDefinitions: null,
    };
    let dbResult = await dal_manager_1.dbManager.accessSocial.listUserRegionTicketUnits(organizationId, {
        take: app_config_1.appConfig.reportExportRowLimit + 1,
        skip: 0,
    }, filter);
    result.items = dbResult.items;
    result.captionDefinitions = await getCaptionDefinitions(organizationId, userId);
    result.totalCount = dbResult.pagination.total;
    result.organization = await dal_manager_1.dbManager.accessOrganization.getOrganizationLogoAndName(organizationId);
    return Promise.resolve(result);
}
exports.generateUserTicketInfoReportData = generateUserTicketInfoReportData;
async function generateUnterminatedVisitReport(organizationId, userId, request, settings) {
    let result = {
        report: null,
        captionDefinitions: null,
    };
    result.report = await dal_manager_1.dbManager.accessVisitor.listAllVisitsForVisitReport(organizationId, {
        sortStartUtcAsc: request.sortStartUtcAsc,
        take: app_config_1.appConfig.reportExportRowLimit + 1,
        skip: 0,
        startUtc: request.startUtc,
        endUtc: request.endUtc,
        visitorProfileIds: request.visitorProfileIds,
        visitorProfileFilterFields: request.visitorProfileFilterFields,
        visitFilterFields: request.visitFilterFields,
        type: request.type,
    }, request.visitorRegistrationPointId, settings, userId);
    result.captionDefinitions = await getCaptionDefinitions(organizationId, userId);
    result.organization = await dal_manager_1.dbManager.accessOrganization.getOrganizationLogoAndName(organizationId);
    return Promise.resolve(result);
}
exports.generateUnterminatedVisitReport = generateUnterminatedVisitReport;
async function getSensorEventsReport(params) {
    let result = await dal_manager_1.dbManager.accessDevice.reportSensorsPg({ organizationId: params.organizationId, filter: params.filter, requesterUserId: params.requesterUserId });
    let _result = {
        result: result,
        organization: await dal_manager_1.dbManager.accessOrganization.getOrganizationLogoAndName(params.organizationId),
    };
    return Promise.resolve(_result);
}
exports.getSensorEventsReport = getSensorEventsReport;
