"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.FlexibleWorkPlanDay = exports.WorkPlanDay = void 0;
const pacs_models_1 = require("./pacs.models");
const pacs_utils_1 = require("./pacs.utils");
const app_enums_1 = require("../../app.enums");
class WorkPlanDay {
    constructor(params) {
        this._params = params;
        this._workPlanDurationSegments = [
            {
                isWorkingHour: false,
                range: params.dateIntersection,
                workPlanTimeRangeType: app_enums_1.enums.WorkPlanTimeRangeType.OutOfWorkingHours,
                workPlanId: params.workPlanId,
                isBufferSegment: false,
            },
        ];
        this.invertBufferedSegments();
        this.invertWorkingHours(this._params.workingHours);
        if (this._params.holidays && this._params.holidays.length > 0) {
            this.invertHolidays(this._params.holidays);
        }
    }
    invertBufferedSegments() {
        const bufferRanges = this._params.bufferedDayRange.subtract(this._params.offsetDayRange);
        for (const buffer of bufferRanges) {
            let segments = [];
            for (const s of this._workPlanDurationSegments) {
                if (s.range.intersect(buffer)) {
                    let x = Object.assign({}, s);
                    let o = Object.assign({}, s, {
                        isBufferSegment: true,
                    });
                    segments.push(...(0, pacs_utils_1.splitDateRangeAccordingToIntersections)(s.range, buffer, x, o));
                }
                else {
                    segments.push(s);
                }
            }
            this._workPlanDurationSegments = segments;
        }
    }
    invertHolidays(holidays) {
        for (let holiday of holidays) {
            let segments = [];
            for (let s of this._workPlanDurationSegments) {
                if (!s.range.intersect(holiday.range)) {
                    segments.push(s);
                    continue;
                }
                let holidayIds = [holiday.id];
                if (s.workPlanTimeRangeType === app_enums_1.enums.WorkPlanTimeRangeType.Holiday) {
                    holidayIds.push(...(s.workPlanData && s.workPlanData.holidayIds ? s.workPlanData.holidayIds : []));
                }
                let x = Object.assign({}, s);
                let o = {
                    isWorkingHour: s.isWorkingHour,
                    range: undefined,
                    workPlanTimeRangeType: app_enums_1.enums.WorkPlanTimeRangeType.Holiday,
                    workPlanId: this._params.workPlanId,
                    workPlanData: {
                        holidayIds: holidayIds,
                    },
                    isBufferSegment: s.isBufferSegment,
                };
                segments.push(...(0, pacs_utils_1.splitDateRangeAccordingToIntersections)(s.range, holiday.range, x, o));
            }
            this._workPlanDurationSegments = segments;
        }
    }
    invertWorkingHours(ranges) {
        if (!ranges || ranges.length === 0) {
            return;
        }
        for (let i = 0; i < ranges.length; i++) {
            let ri = ranges[i];
            if (!this._params.dateIntersection.intersect(ri)) {
                continue;
            }
            for (let j = i + 1; j < ranges.length; j++) {
                let r = ranges[i].intersect(ranges[j]);
                if (r) {
                    throw new pacs_models_1.PacsModels.RangeError(pacs_models_1.PacsModels.PacsErrorCodes.MultiWorkingHoursIntersecting, r);
                }
            }
        }
        for (let range of ranges) {
            let segments = [];
            for (let s of this._workPlanDurationSegments) {
                if (s.workPlanTimeRangeType !== app_enums_1.enums.WorkPlanTimeRangeType.OutOfWorkingHours || !s.range.intersect(range)) {
                    segments.push(s);
                    continue;
                }
                let x = Object.assign({}, s, { range: undefined });
                let o = {
                    isWorkingHour: true,
                    range: undefined,
                    workPlanTimeRangeType: app_enums_1.enums.WorkPlanTimeRangeType.WorkingHours,
                    workPlanId: this._params.workPlanId,
                    workPlanData: undefined,
                    isBufferSegment: s.isBufferSegment,
                };
                segments.push(...(0, pacs_utils_1.splitDateRangeAccordingToIntersections)(s.range, range, x, o));
            }
            this._workPlanDurationSegments = segments;
        }
    }
    get workPlanDurationSegments() {
        return this._workPlanDurationSegments;
    }
    get tolerances() {
        return this._params.tolerances;
    }
    get fixedBreaks() {
        return this._params.fixedBreaks;
    }
    get extraWorkBreaks() {
        return this._params.extraWorkBreaks;
    }
    get claimBreak() {
        return this._params.claimBreak;
    }
    get workPlanType() {
        return this._params.type;
    }
    get attendanceControlMethod() {
        return this._params.attendanceControlMethod;
    }
    get mobileCheckinAllowed() {
        return this._params.allowMobileCheckins;
    }
    get unreliableCheckinAllowed() {
        return this._params.allowUnreliableCheckins;
    }
    get permissionRequiredForExtraWorking() {
        return this._params.permissionRequiredForExtraWorking;
    }
    get regionId() {
        return this._params.regionId;
    }
    get date() {
        return this._params.date;
    }
    get workPlanId() {
        return this._params.workPlanId;
    }
    get dateIntersection() {
        return this._params.dateIntersection;
    }
    get ignoreUnpairedAccesses() {
        return this._params.ignoreUnpairedAccesses;
    }
    get flexible() {
        return this._params.type === app_enums_1.enums.WorkPlanType.FULL_FLEXIBLE || this._params.type === app_enums_1.enums.WorkPlanType.HALF_FLEXIBLE;
    }
}
exports.WorkPlanDay = WorkPlanDay;
class FlexibleWorkPlanDay extends WorkPlanDay {
    constructor(params) {
        super(params);
        this._weeklyExpectedWorkDuration = params.weeklyExpectedWorkDuration;
    }
    get weeklyExpectedWorkingHours() {
        return this._weeklyExpectedWorkDuration;
    }
}
exports.FlexibleWorkPlanDay = FlexibleWorkPlanDay;
