"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.ValidateUserPPermission = void 0;
const moment_1 = __importDefault(require("moment"));
const app_enums_1 = require("../../app.enums");
const app_logs_1 = require("../../app.logs");
const business_hooks_1 = require("../business.hooks");
const dal_manager_1 = require("../../dal/dal.manager");
const business_pacs_ppermission_1 = require("./business.pacs.ppermission");
const business_pacs_claim_1 = require("./business.pacs.claim");
const i18n_1 = __importDefault(require("i18n"));
class ValidateUserPPermission {
    async validatePPermissionByUserSelectionSessionId(organizationId, ppermissionId, userSelectionSessionId, locale) {
        let ppermission = await dal_manager_1.dbManager.accessPacs.getPPermission(organizationId, ppermissionId);
        if (!ppermission)
            return [
                {
                    error: app_enums_1.enums.HttpStatusCode.NOT_FOUND,
                    message: "There is no ppermissions in db.",
                },
            ];
        let userIds = await dal_manager_1.dbManager.accessUser.listAddedUsersOfSelectionSession(organizationId, userSelectionSessionId);
        if (!userIds)
            return [
                {
                    error: app_enums_1.enums.HttpStatusCode.NOT_FOUND,
                    message: "There is no users in request.",
                },
            ];
        let errors = [];
        const userResults = await this.validateUsersPPermission(organizationId, ppermission.ppermissionType.id, userIds, locale, ppermission.dateRange.startDateTime, ppermission.dateRange.endDateTime, ppermissionId);
        const erroredUserIds = Array.from(userResults.entries())
            .filter(([_, result]) => !result.isOk)
            .map(([userId, _]) => userId);
        const userInfos = await dal_manager_1.dbManager.systemTransaction(async (trx) => {
            return erroredUserIds.length > 0
                ? await dal_manager_1.dbManager.accessRedisCache.getUsersNotificationCache({
                    organizationId,
                    userIds: erroredUserIds,
                    trx,
                })
                : new Map();
        });
        for (const userId of userIds) {
            const userResult = userResults.get(userId);
            if (!userResult?.isOk) {
                const userInfo = userInfos.get(userId);
                errors.push({
                    error: app_enums_1.enums.HttpStatusCode.CONFLICT,
                    message: userResult.errorMessage,
                    details: userInfo?.f || "User details not available",
                });
            }
        }
        return errors;
    }
    async validatePPermission(organizationId, ppermissionTypeId, userIds, locale, startDateTime, endDateTime, ppermissionId, performValidations = true) {
        let result = {
            items: [],
        };
        try {
            if (performValidations) {
                const userResults = await this.validateUsersPPermission(organizationId, ppermissionTypeId, userIds, locale, startDateTime, endDateTime, ppermissionId);
                for (let user of userIds) {
                    const userResult = userResults.get(user);
                    result.items.push({
                        userId: user,
                        workPlanIntersectingDates: userResult.workPlanIntersectingDates,
                        remainingPPermissionClaim: userResult.remainingPPermissionClaim,
                        isOk: userResult.isOk,
                        errorCode: userResult.errorCode,
                        errorMessage: userResult.errorMessage,
                    });
                }
            }
        }
        catch (error) {
            app_logs_1.logger.error(error);
        }
        return Promise.resolve(result);
    }
    async validateUserPPermission(organizationId, ppermissionTypeId, userId, locale, startDateTime, endDateTime, ppermissionId) {
        let result = {
            isOk: true,
            remainingPPermissionClaim: 0,
            usedPPermissionClaim: 0,
            workPlanIntersectingDates: [],
            errorCode: app_enums_1.enums.PPermissionValidationErrorCodes.None,
            accessLogDurations: [],
            errorMessage: undefined,
        };
        try {
            let ppermissionStatus = await (0, business_pacs_claim_1.getPPermissionClaimAndUsage)(organizationId, userId, new Date(), ppermissionTypeId);
            if (ppermissionStatus.length === 0) {
                result.isOk = false;
                result.errorCode = app_enums_1.enums.PPermissionValidationErrorCodes.PPermissionNotFound;
                result.errorMessage = i18n_1.default.__({ phrase: "ERRORS.PACS.PPERMISSIONTYPENOTFOUND", locale });
                return Promise.resolve(result);
            }
            let oldPPermissions = await dal_manager_1.dbManager.accessPacs.listOldPPermissionsForValidation(organizationId, userId, ppermissionStatus[0].type.id, startDateTime, endDateTime, ppermissionId);
            if (oldPPermissions) {
                result.isOk = false;
                result.errorCode = app_enums_1.enums.PPermissionValidationErrorCodes.InvalidDateRange;
                result.errorMessage = i18n_1.default.__({ phrase: "ERRORS.PACS.PPERMISSIONCONFLICTWITHSAMETYPE", locale });
                return Promise.resolve(result);
            }
            let vacations = await dal_manager_1.dbManager.accessPacs.listAllVacations(organizationId);
            let amount = 0;
            let usage = 0;
            if (ppermissionStatus[0].type.isDailyScheduled) {
                amount = await (0, business_pacs_ppermission_1.calculateDailyVacationUsage)(organizationId, ppermissionStatus[0].type.id, startDateTime, endDateTime, vacations, userId, ppermissionStatus[0].type);
                usage = ppermissionStatus[0].usage;
            }
            else {
                amount = (await (0, business_pacs_ppermission_1.estimateHourlyVacationUsage)(organizationId, ppermissionStatus[0].type.id, startDateTime, endDateTime, vacations, ppermissionStatus[0].type)) / 60;
                usage = ppermissionStatus[0].usage / 60;
            }
            if (!ppermissionStatus[0].type.isNegativeBalanceAllowed && amount + usage > ppermissionStatus[0].type.maxValue) {
                result.isOk = false;
                result.errorCode = app_enums_1.enums.PPermissionValidationErrorCodes.LimitError;
                result.errorMessage = i18n_1.default.__({ phrase: "ERRORS.PACS.PPERMISSIONUPPERLIMITERROR", locale });
            }
            if (amount < ppermissionStatus[0].type.minValue) {
                result.isOk = false;
                result.errorCode = app_enums_1.enums.PPermissionValidationErrorCodes.LimitError;
                result.errorMessage = i18n_1.default.__({ phrase: "ERRORS.PACS.PPERMISSIONLOWERLIMITERROR", locale });
            }
            amount = ppermissionStatus[0].type.isDailyScheduled ? amount : amount * 60;
            result.remainingPPermissionClaim = ppermissionStatus[0].claim - amount;
            result.usedPPermissionClaim = amount;
            if (!ppermissionStatus[0].type.isDailyScheduled && Math.abs((0, moment_1.default)(startDateTime).diff(endDateTime, "hours")) >= 24) {
                result.isOk = false;
                result.errorCode = app_enums_1.enums.PPermissionValidationErrorCodes.HourlyScheduledPPermissionCanNotBeDaily;
                result.errorMessage = i18n_1.default.__({ phrase: "ERRORS.PACS.PPERMISSIONHOURLYCANNOTBEDAILY", locale });
            }
            if (!(0, moment_1.default)(endDateTime).isAfter((0, moment_1.default)(startDateTime))) {
                result.isOk = false;
                result.errorCode = app_enums_1.enums.PPermissionValidationErrorCodes.InvalidDateRange;
                result.errorMessage = i18n_1.default.__({ phrase: "ERRORS.PACS.PPERMISSIONSTARTAFTEREND", locale });
            }
            let hook = business_hooks_1.armonHookManager.getValidateHook(organizationId);
            if (hook) {
                let hookResult = await hook({
                    jobId: "Validation",
                    organizationId: organizationId,
                    date: new Date(),
                    data: [
                        {
                            ppermissionTypeId: ppermissionTypeId,
                            userId: userId,
                            currentResult: result.isOk,
                            startDateTime: startDateTime,
                            endDateTime: endDateTime,
                            accessLogDurations: [],
                        },
                    ],
                });
                if (!hookResult && result.isOk) {
                    result.isOk = hookResult;
                    result.errorCode = app_enums_1.enums.PPermissionValidationErrorCodes.Custom;
                    result.errorMessage = i18n_1.default.__({ phrase: "ERRORS.PACS.PPERMISSIONHOOKERROR", locale });
                }
            }
        }
        catch (error) {
            result.isOk = false;
            result.errorCode = app_enums_1.enums.PPermissionValidationErrorCodes.Custom;
            result.errorMessage = i18n_1.default.__({ phrase: error.message, locale });
            app_logs_1.logger.error(error);
        }
        return Promise.resolve(result);
    }
    async validateUsersPPermission(organizationId, ppermissionTypeId, userIds, locale, startDateTime, endDateTime, ppermissionId) {
        const resultMap = new Map();
        const vacations = await dal_manager_1.dbManager.accessPacs.listAllVacations(organizationId);
        const oldPPermissionMap = await dal_manager_1.dbManager.accessPacs.listOldPPermissionsForValidationBulk(organizationId, userIds, ppermissionTypeId, startDateTime, endDateTime, ppermissionId);
        const ppermissionStatusMap = await dal_manager_1.dbManager.systemTransaction(async (trx) => {
            return await (0, business_pacs_claim_1.getPPermissionClaimAndUsageBulk)(organizationId, userIds, new Date(), trx, ppermissionTypeId);
        });
        for (const userId of userIds) {
            let result = {
                isOk: true,
                remainingPPermissionClaim: 0,
                usedPPermissionClaim: 0,
                workPlanIntersectingDates: [],
                errorCode: app_enums_1.enums.PPermissionValidationErrorCodes.None,
                accessLogDurations: [],
                errorMessage: undefined,
            };
            try {
                const ppermissionStatuses = ppermissionStatusMap.get(userId);
                if (!ppermissionStatuses) {
                    result.isOk = false;
                    result.errorCode = app_enums_1.enums.PPermissionValidationErrorCodes.Custom;
                    result.errorMessage = i18n_1.default.__({ phrase: "ERRORS.PACS.EMPLOYEESTARTDATEMISSING", locale });
                    resultMap.set(userId, result);
                    continue;
                }
                if (ppermissionStatuses.length == 0) {
                    result.isOk = false;
                    result.errorCode = app_enums_1.enums.PPermissionValidationErrorCodes.PPermissionNotFound;
                    result.errorMessage = i18n_1.default.__({ phrase: "ERRORS.PACS.PPERMISSIONTYPENOTFOUND", locale });
                    resultMap.set(userId, result);
                    continue;
                }
                const ppermissionStatus = ppermissionStatuses[0];
                const ppermissionType = ppermissionStatus.type;
                if (oldPPermissionMap.get(userId)) {
                    result.isOk = false;
                    result.errorCode = app_enums_1.enums.PPermissionValidationErrorCodes.InvalidDateRange;
                    result.errorMessage = i18n_1.default.__({ phrase: "ERRORS.PACS.PPERMISSIONCONFLICTWITHSAMETYPE", locale });
                    resultMap.set(userId, result);
                    continue;
                }
                let amount = 0;
                let usage = 0;
                if (ppermissionType.isDailyScheduled) {
                    amount = await (0, business_pacs_ppermission_1.calculateDailyVacationUsage)(organizationId, ppermissionType.id, startDateTime, endDateTime, vacations, userId, ppermissionType);
                    usage = ppermissionStatus.usage;
                }
                else {
                    amount = (await (0, business_pacs_ppermission_1.estimateHourlyVacationUsage)(organizationId, ppermissionType.id, startDateTime, endDateTime, vacations, ppermissionType)) / 60;
                    usage = ppermissionStatus.usage / 60;
                }
                if (!ppermissionType.isNegativeBalanceAllowed && amount + usage > ppermissionType.maxValue) {
                    result.isOk = false;
                    result.errorCode = app_enums_1.enums.PPermissionValidationErrorCodes.LimitError;
                    result.errorMessage = i18n_1.default.__({ phrase: "ERRORS.PACS.PPERMISSIONUPPERLIMITERROR", locale });
                }
                if (amount < ppermissionType.minValue) {
                    result.isOk = false;
                    result.errorCode = app_enums_1.enums.PPermissionValidationErrorCodes.LimitError;
                    result.errorMessage = i18n_1.default.__({ phrase: "ERRORS.PACS.PPERMISSIONLOWERLIMITERROR", locale });
                }
                if (!ppermissionType.isDailyScheduled && Math.abs((0, moment_1.default)(startDateTime).diff(endDateTime, "hours")) >= 24) {
                    result.isOk = false;
                    result.errorCode = app_enums_1.enums.PPermissionValidationErrorCodes.HourlyScheduledPPermissionCanNotBeDaily;
                    result.errorMessage = i18n_1.default.__({ phrase: "ERRORS.PACS.PPERMISSIONHOURLYCANNOTBEDAILY", locale });
                }
                if (!(0, moment_1.default)(endDateTime).isAfter((0, moment_1.default)(startDateTime))) {
                    result.isOk = false;
                    result.errorCode = app_enums_1.enums.PPermissionValidationErrorCodes.InvalidDateRange;
                    result.errorMessage = i18n_1.default.__({ phrase: "ERRORS.PACS.PPERMISSIONSTARTAFTEREND", locale });
                }
                const used = ppermissionType.isDailyScheduled ? amount : amount * 60;
                result.remainingPPermissionClaim = ppermissionStatus.claim - used;
                result.usedPPermissionClaim = used;
                const hook = business_hooks_1.armonHookManager.getValidateHook(organizationId);
                if (hook) {
                    const hookResult = await hook({
                        jobId: "Validation",
                        organizationId,
                        date: new Date(),
                        data: [
                            {
                                ppermissionTypeId,
                                userId,
                                currentResult: result.isOk,
                                startDateTime,
                                endDateTime,
                                accessLogDurations: [],
                            },
                        ],
                    });
                    if (!hookResult && result.isOk) {
                        result.isOk = false;
                        result.errorCode = app_enums_1.enums.PPermissionValidationErrorCodes.Custom;
                        result.errorMessage = i18n_1.default.__({ phrase: "ERRORS.PACS.PPERMISSIONHOOKERROR", locale });
                    }
                }
            }
            catch (error) {
                result.isOk = false;
                result.errorCode = app_enums_1.enums.PPermissionValidationErrorCodes.Custom;
                result.errorMessage = i18n_1.default.__({ phrase: error.message, locale });
                app_logs_1.logger.error(error);
            }
            resultMap.set(userId, result);
        }
        return resultMap;
    }
}
exports.ValidateUserPPermission = ValidateUserPPermission;
