"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.sendNextPPermissionNotification = exports.calculateDailyVacationUsage = exports.parsePPermissionListToUserInfoList = exports.calculateHourlyVacationUsage = exports.estimateHourlyVacationUsage = exports.getDefaultAnnualPPermissionIntervals = void 0;
const moment_1 = __importDefault(require("moment"));
const moment_range_1 = require("moment-range");
const app_enums_1 = require("../../app.enums");
const app_logs_1 = require("../../app.logs");
const dal_memcache_1 = require("../../dal/access/dal.memcache");
const dal_manager_1 = require("../../dal/dal.manager");
const predefined_permissions_1 = require("../../dal/db/predefined/predefined.permissions");
const messageBroker_notification_pub_1 = require("../../messageBroker/messageBroker.notification.pub");
const business_pacs_utils_1 = require("./business.pacs.utils");
function getDefaultAnnualPPermissionIntervals() {
    const defaultIntervals = [];
    defaultIntervals.push({
        lowerExperienceDuration: 5110,
        higherExperienceDuration: 100000,
        claim: 624,
    });
    defaultIntervals.push({
        lowerExperienceDuration: 1825,
        higherExperienceDuration: 5109,
        claim: 480,
    });
    defaultIntervals.push({
        lowerExperienceDuration: 365,
        higherExperienceDuration: 1824,
        claim: 336,
    });
    defaultIntervals.push({
        lowerExperienceDuration: 0,
        higherExperienceDuration: 364,
        claim: 0,
    });
    return defaultIntervals;
}
exports.getDefaultAnnualPPermissionIntervals = getDefaultAnnualPPermissionIntervals;
async function estimateHourlyVacationUsage(organizationId, ppermissionTypeId, startDateTime, endDateTime, holidays, ppermissionType) {
    if (!holidays) {
        holidays = await dal_manager_1.dbManager.accessPacs.listAllVacations(organizationId);
    }
    let range = new moment_range_1.DateRange(startDateTime, endDateTime);
    let ranges = [];
    if (!range.start.isSame(range.end, "day")) {
        ranges.push(new moment_range_1.DateRange(range.start.clone(), range.start.clone().add(1, "day").startOf("day")));
        let start = range.start.clone().add(1, "day").startOf("day");
        while (!start.isSame(range.end, "day")) {
            ranges.push(new moment_range_1.DateRange(start.clone(), start.clone().add(1, "day")));
            start.add(1, "day");
        }
        ranges.push(new moment_range_1.DateRange(range.end.clone().startOf("day"), range.end.clone()));
    }
    else {
        ranges.push(range);
    }
    const type = ppermissionType ?? (await dal_manager_1.dbManager.accessPacs.getPPermissionType(organizationId, ppermissionTypeId));
    const applicableDays = await dal_manager_1.dbManager.accessPacs.getApplicableDaysForPPermissionType(organizationId, ppermissionTypeId, type.method, type.isUnpaidLeave);
    if (!type.applyOnHolidays) {
        ranges = (0, business_pacs_utils_1.calculateRangesExceptHolidays)(ranges, holidays);
    }
    let result = 0;
    for (const r of ranges) {
        if (applicableDays.includes(r.start.isoWeekday())) {
            result += r.duration("minutes");
        }
    }
    return Promise.resolve(result);
}
exports.estimateHourlyVacationUsage = estimateHourlyVacationUsage;
async function calculateHourlyVacationUsage(organizationId, ppermissionTypeId, userId, ppermissionId, startDateTime, endDateTime, holidays) {
    return estimateHourlyVacationUsage(organizationId, ppermissionTypeId, startDateTime, endDateTime, holidays);
}
exports.calculateHourlyVacationUsage = calculateHourlyVacationUsage;
function parsePPermissionListToUserInfoList(permissionList) {
    const userInfoList = permissionList.items.flatMap((item) => item.userPPermissionInfo.map((userInfo) => ({
        userAndDeputyInfo: userInfo.userAndDeputyInfo,
        canListerApprove: !!userInfo.canListerApprove,
        status: userInfo.status,
        userSigned: userInfo.userSigned,
        approvementInfo: userInfo.approvementInfo.map((approver) => ({
            approverUsers: approver.approverUsers,
            approverUserId: approver.approverUserId,
            approvementDate: approver.approvementDate,
            note: approver.note,
            status: approver.status ? 1 : 0,
            organizationUnit: approver.organizationUnit,
            isHR: approver.isHR,
            order: approver.order,
            canListerApprove: approver.canListerApprove,
        })),
    })));
    return {
        pagination: {
            take: permissionList.pagination.take,
            skip: permissionList.pagination.skip,
            total: "userCount" in permissionList.items[0] ? permissionList.items[0].userCount : permissionList.pagination.total,
        },
        userInfoList,
    };
}
exports.parsePPermissionListToUserInfoList = parsePPermissionListToUserInfoList;
async function calculateDailyVacationUsage(organizationId, ppermissionTypeId, startDateTime, endDateTime, holidays, userId, ppermissionType) {
    const type = ppermissionType ?? (await dal_manager_1.dbManager.accessPacs.getPPermissionType(organizationId, ppermissionTypeId));
    const applicableDays = await dal_manager_1.dbManager.accessPacs.getApplicableDaysForPPermissionType(organizationId, ppermissionTypeId, type.method, type.isUnpaidLeave, userId);
    if (!holidays) {
        holidays = await dal_manager_1.dbManager.accessPacs.listAllVacations(organizationId);
    }
    let start = (0, moment_1.default)(startDateTime);
    let end = (0, moment_1.default)(endDateTime);
    let usage = 0;
    if (start.isSame(end, "day") && applicableDays.includes(start.isoWeekday())) {
        let holidayIntersects = (0, business_pacs_utils_1.intersectsWithHolidays)(start, end, holidays);
        if (!holidayIntersects.morning || type.applyOnHolidays) {
            usage = 0.5;
        }
    }
    else {
        if (start.hour() === 12) {
            if (applicableDays.includes(start.isoWeekday())) {
                let holidayIntersects = (0, business_pacs_utils_1.intersectsWithHolidays)(start.clone(), start.clone().add(12, "h"), holidays);
                if (!holidayIntersects.afternoon || type.applyOnHolidays) {
                    usage += 0.5;
                }
            }
            start = start.add(1, "day").startOf("day");
        }
        if (end.hour() === 12) {
            if (applicableDays.includes(end.isoWeekday())) {
                let holidayIntersects = (0, business_pacs_utils_1.intersectsWithHolidays)(end.clone().startOf("day"), end.clone(), holidays);
                if (!holidayIntersects.morning || type.applyOnHolidays) {
                    usage += 0.5;
                }
                end = end.startOf("day");
            }
        }
        while (start.isBefore(end)) {
            if (applicableDays.includes(start.isoWeekday())) {
                let holidayIntersects = (0, business_pacs_utils_1.intersectsWithHolidays)(start, start.clone().add(1, "d"), holidays);
                if (!holidayIntersects.morning || type.applyOnHolidays) {
                    usage += 0.5;
                }
                if (!holidayIntersects.afternoon || type.applyOnHolidays) {
                    usage += 0.5;
                }
            }
            start.add(1, "day");
        }
    }
    return Promise.resolve(usage);
}
exports.calculateDailyVacationUsage = calculateDailyVacationUsage;
async function sendNextPPermissionNotification(params) {
    const { organizationId, ppermissionId, requesterUserId } = params;
    const ppermission = await dal_manager_1.dbManager.accessPacs.getPPermission(organizationId, ppermissionId);
    if (!ppermission) {
        app_logs_1.logger.warn(`ppermission is not found! permissionId: ${ppermissionId}`);
        return;
    }
    if (!ppermission.ppermissionType.sendNotification) {
        app_logs_1.logger.debug(`notification sending is close for this ppermission type[${ppermission.ppermissionType}]`);
        return;
    }
    const isApprovementProcessCompleted = (data) => data.some((data) => data.approvementDate && data.status === false) || data.every((data) => data.approvementDate);
    const events = [];
    await dal_manager_1.dbManager.organizationTransaction(async (trx) => {
        try {
            let ppermissionApprovements = await dal_manager_1.dbManager.accessPacs.getUsersPPermissionApprovements({ organizationId, ppermissionId });
            ppermissionApprovements = ppermissionApprovements.filter((approvement) => params.userIds.includes(approvement.userId));
            const currentApprovementsUserIdOrganizationIdMapping = {};
            for (const { userId, data } of ppermissionApprovements) {
                if (isApprovementProcessCompleted(data)) {
                    const isApproved = data.every((data) => data.status);
                    const event = await createPPermissionApprovementNotificationEventForPPermissionOwner({
                        organizationId,
                        ppermissionId,
                        isApproved,
                        ppermissionOwnerUserId: userId,
                        trx,
                    });
                    events.push(event);
                    const eventSendedUserIds = [];
                    if (ppermission.ppermissionType.notifyManagers) {
                        eventSendedUserIds.push(...(await dal_manager_1.dbManager.accessPacs.getFirstManagerOfUsersOrganizationUnitTree(trx, organizationId, userId)));
                    }
                    if (ppermission.ppermissionType.notifyHR) {
                        eventSendedUserIds.push(...(await dal_manager_1.dbManager.accessPacs.getFirstHrOfUsersOrganizationUnitTree(trx, organizationId, userId)).filter((elem) => !eventSendedUserIds.includes(elem)));
                    }
                    if (eventSendedUserIds.length > 0) {
                        const event = await createPPermissionApprovementNotificationEventForManagersAndHrs({
                            organizationId,
                            ppermissionId,
                            isApproved,
                            userIds: eventSendedUserIds,
                            permissionOwnerUserId: userId,
                            trx,
                        });
                        events.push(event);
                    }
                }
                else {
                    const orderedPPermissinApprovement = data.sort((a, b) => a.order - b.order);
                    for (const approvement of orderedPPermissinApprovement) {
                        if (approvement.approvementDate === null) {
                            const organizationUnitOrHr = approvement.organizationUnitId !== null ? approvement.organizationUnitId : "HR";
                            currentApprovementsUserIdOrganizationIdMapping[organizationUnitOrHr]
                                ? currentApprovementsUserIdOrganizationIdMapping[organizationUnitOrHr].push(userId)
                                : (currentApprovementsUserIdOrganizationIdMapping[organizationUnitOrHr] = [userId]);
                            break;
                        }
                    }
                }
            }
            for (const organizationUnitId of Object.keys(currentApprovementsUserIdOrganizationIdMapping)) {
                if (organizationUnitId === "HR") {
                    const event = await createPendingPPermissionNotificationEventForHRs({
                        organizationId,
                        ppermissionId,
                        userIds: currentApprovementsUserIdOrganizationIdMapping[organizationUnitId],
                        trx,
                    });
                    events.push(event);
                }
                else {
                    const event = await createPendingPPermissionNotificationEventForNextOrganizationUnitManager({
                        organizationId,
                        ppermissionId,
                        userIds: currentApprovementsUserIdOrganizationIdMapping[organizationUnitId],
                        organizationUnitId,
                        trx,
                    });
                    events.push(event);
                }
            }
        }
        catch (error) {
            app_logs_1.logger.error("ppermission notifier error: " + error);
        }
    }, requesterUserId, organizationId);
    for (const event of events) {
        (0, messageBroker_notification_pub_1.publishToNotificationService)(event);
    }
}
exports.sendNextPPermissionNotification = sendNextPPermissionNotification;
const createPendingPPermissionNotificationEventForNextOrganizationUnitManager = async (params) => {
    const { manager } = await dal_manager_1.dbManager.accessOrganizationUnit.getCurrentOrganizationUnitManagerDeputy(params.organizationId, params.organizationUnitId);
    app_logs_1.logger.debug(`sending notification manager[${manager.id}]`);
    const notificationId = await (0, dal_memcache_1.getCacheUniqueNotificationIdOfOrganization)(params.organizationId, app_enums_1.enums.NotificationType.PPermissionWaitForApprovementToApprover, params.trx);
    const instanceData = {
        i: false,
        u: params.userIds,
        p: params.ppermissionId,
        receiverUserIds: [manager.id],
    };
    const notificationEventId = await dal_manager_1.dbManager.accessNotifications.addNotificationEvent({
        createdT: new Date(),
        notificationId,
        organizationId: params.organizationId,
        instanceData,
        trx: params.trx,
    });
    return {
        v: "1",
        n: 1,
        a: 0,
        i: notificationEventId,
        o: params.organizationId,
        d: instanceData,
    };
};
const createPendingPPermissionNotificationEventForHRs = async (params) => {
    app_logs_1.logger.debug("sending notification to HRs");
    const hrUserIds = await dal_manager_1.dbManager.accessUser.listUsersWithPermissionPg(params.organizationId, predefined_permissions_1.Permissions.ppermission.getHRApprovement());
    const notificationId = await (0, dal_memcache_1.getCacheUniqueNotificationIdOfOrganization)(params.organizationId, app_enums_1.enums.NotificationType.PPermissionWaitForApprovementToApprover, params.trx);
    const instanceData = {
        i: true,
        u: params.userIds,
        p: params.ppermissionId,
        receiverUserIds: hrUserIds,
    };
    const notificationEventId = await dal_manager_1.dbManager.accessNotifications.addNotificationEvent({
        createdT: new Date(),
        notificationId,
        organizationId: params.organizationId,
        trx: params.trx,
        instanceData,
    });
    return {
        v: "1",
        n: 1,
        a: 0,
        i: notificationEventId,
        o: params.organizationId,
        d: instanceData,
    };
};
const createPPermissionApprovementNotificationEventForPPermissionOwner = async (params) => {
    const notificationId = await (0, dal_memcache_1.getCacheUniqueNotificationIdOfOrganization)(params.organizationId, app_enums_1.enums.NotificationType.PPermissionApprovementToOwner, params.trx);
    const instanceData = {
        p: params.ppermissionId,
        isApproved: params.isApproved,
        receiverUserIds: [params.ppermissionOwnerUserId],
    };
    const notificationEventId = await dal_manager_1.dbManager.accessNotifications.addNotificationEvent({
        createdT: new Date(),
        notificationId,
        organizationId: params.organizationId,
        trx: params.trx,
        instanceData,
    });
    return {
        v: "1",
        n: 1,
        a: 0,
        i: notificationEventId,
        o: params.organizationId,
        d: instanceData,
    };
};
const createPPermissionApprovementNotificationEventForManagersAndHrs = async (params) => {
    const notificationId = await (0, dal_memcache_1.getCacheUniqueNotificationIdOfOrganization)(params.organizationId, app_enums_1.enums.NotificationType.PPermissionApprovementStatesCompleted, params.trx);
    const instanceData = {
        p: params.ppermissionId,
        isApproved: params.isApproved,
        permissionOwnerUserId: params.permissionOwnerUserId,
        receiverUserIds: params.userIds,
    };
    const notificationEventId = await dal_manager_1.dbManager.accessNotifications.addNotificationEvent({
        createdT: new Date(),
        notificationId,
        organizationId: params.organizationId,
        trx: params.trx,
        instanceData,
    });
    return {
        v: "1",
        n: 1,
        a: 0,
        i: notificationEventId,
        o: params.organizationId,
        d: instanceData,
    };
};
