"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.WorkPlanImportUploadExcel = void 0;
const exceljs_1 = __importDefault(require("exceljs"));
const fs_1 = __importDefault(require("fs"));
const i18n_1 = __importDefault(require("i18n"));
const moment_1 = __importDefault(require("moment"));
const path_1 = __importDefault(require("path"));
const uuid_1 = __importDefault(require("uuid"));
const app_config_1 = require("../../app.config");
const app_enums_1 = require("../../app.enums");
const app_logs_1 = require("../../app.logs");
const dal_memcache_1 = require("../../dal/access/dal.memcache");
const dal_manager_1 = require("../../dal/dal.manager");
const dal_utils_1 = require("../../dal/dal.utils");
const messageBroker_notification_pub_1 = require("../../messageBroker/messageBroker.notification.pub");
const pacs_models_1 = require("../pacs2/pacs.models");
const dal_db_armon_schema_1 = require("../../dal/db/armon/dal.db.armon.schema");
const dal_constants_1 = require("../../dal/dal.constants");
class WorkPlanImportUploadExcel {
    constructor(organizationId, userId, locale) {
        this._fileId = uuid_1.default.v4();
        this._requestedUserId = userId;
        this._organizationId = organizationId;
        this._locale = locale;
        this._headers = [
            i18n_1.default.__({ phrase: "IDENTITY.NAME", locale: this._locale }) + i18n_1.default.__({ phrase: "IDENTITY.SURNAME", locale: this._locale }),
            i18n_1.default.__({ phrase: "EXCEL-REPORT.UNIQUE_ID", locale: this._locale }),
        ];
        this._filePath = path_1.default.join(app_config_1.appConfig.tmpDirectory, this._fileId + ".xlsx");
        if (fs_1.default.existsSync(this._filePath)) {
            fs_1.default.unlinkSync(this._filePath);
        }
    }
    async loadExcel(data) {
        this._wbReader = new exceljs_1.default.Workbook();
        await this._wbReader.xlsx.load(data);
    }
    async read(importTemplate, type) {
        this._organizationName = (await dal_manager_1.dbManager.accessOrganization.getOrganizationBasic(this._organizationId)).name;
        let { caption: requestedUserCaption } = await dal_manager_1.dbManager.accessRedisCache.getUserBadgeCache({ organizationId: this._organizationId, userId: this._requestedUserId });
        if (!requestedUserCaption) {
            throw new Error("Not found");
        }
        this._requestedUserCaption = requestedUserCaption[0].text[0];
        let validation = await this.validateExcel(importTemplate, type);
        this._wbReader.getWorksheet(1).properties.defaultColWidth = 35;
        return validation;
    }
    async uploadExcel(params) {
        await dal_manager_1.dbManager.systemTransaction(async (trx) => {
            try {
                await dal_manager_1.dbManager.accessPacs2.commonShiftImport(this._organizationId, this._importTemplate, this._wbReader.getWorksheet(1), params);
                let event = await this.createWorkPlanExcelUploadedNotification(this._requestedUserId, true, trx);
                (0, messageBroker_notification_pub_1.publishToNotificationService)(event);
            }
            catch (err) {
                app_logs_1.logger.error(err);
                let event = await this.createWorkPlanExcelUploadedNotification(this._requestedUserId, false, trx);
                (0, messageBroker_notification_pub_1.publishToNotificationService)(event);
            }
        });
    }
    async createWorkPlanExcelUploadedNotification(userId, isSuccessful, trx) {
        app_logs_1.logger.debug(`sending notification user[${userId}]`);
        const notificationId = await (0, dal_memcache_1.getCacheUniqueNotificationIdOfOrganization)(this._organizationId, app_enums_1.enums.NotificationType.WorkPlanExcelImportResultNotification, trx);
        const instanceData = {
            s: isSuccessful,
            u: userId,
            t: new Date().toString(),
            receiverUserIds: [userId],
        };
        const notificationEventId = await dal_manager_1.dbManager.accessNotifications.addNotificationEvent({
            createdT: new Date(),
            notificationId,
            organizationId: this._organizationId,
            instanceData,
            trx: trx,
        });
        return {
            v: "1",
            n: 1,
            a: 0,
            i: notificationEventId,
            o: this._organizationId,
            d: instanceData,
        };
    }
    async validateExcel(importTemplate, type) {
        const sheet = this._wbReader.getWorksheet(1);
        let errorMessage;
        let errorCells = [];
        let errorDates = [];
        let errorUsers = [];
        let headerDateCount = 0;
        const datePattern = /^\d{2}\/\d{2}\/\d{4}$/;
        let uniqueIds = [];
        const hiddenSheet = this._wbReader.getWorksheet("HiddenData");
        if (!hiddenSheet) {
            return {
                validationSuccess: false,
                errorMessage: i18n_1.default.__({ phrase: "ACCESS_LOGS_EXCEL.ERROR.MISSING_HIDDENSHEET", locale: this._locale }),
                errorCells,
                errorDates,
                errorUsers,
            };
        }
        if (!sheet) {
            return {
                validationSuccess: false,
                errorMessage: i18n_1.default.__({ phrase: "ACCESS_LOGS_EXCEL.ERROR.MISSING_DATASHEET", locale: this._locale }),
                errorCells,
                errorDates,
                errorUsers,
            };
        }
        if (hiddenSheet.getCell("A1").value != type) {
            return {
                validationSuccess: false,
                errorMessage: i18n_1.default.__({ phrase: "WORKPLAN-EXCEL.ERROR.INCORRECT_TEMPLATE_TYPE", locale: this._locale }),
                errorCells,
                errorDates,
                errorUsers,
            };
        }
        importTemplate = await this.fillMissingRuleKeywords(importTemplate);
        this._importTemplate = importTemplate;
        sheet.eachRow((row, rowNumber) => {
            row.eachCell({ includeEmpty: true }, (cell, colNumber) => {
                const cellValue = cell.text;
                if (rowNumber === 1 && colNumber > 2) {
                    headerDateCount++;
                    if (type === dal_constants_1.DalConstants.WorkPlanImportTemplateType.DateRange) {
                        if (!cellValue || !datePattern.test(cellValue)) {
                            errorMessage = errorMessage ?? i18n_1.default.__({ phrase: "WORKPLAN-EXCEL.ERROR.INCORRECT_DATE", locale: this._locale });
                            errorDates.push(`${this.getColumnLetter(colNumber)}1`);
                        }
                        else {
                            const [day, month, year] = cellValue.split("/").map(Number);
                            const date = new Date(year, month - 1, day);
                            if (date.getFullYear() !== year || date.getMonth() + 1 !== month || date.getDate() !== day) {
                                errorMessage = errorMessage ?? i18n_1.default.__({ phrase: "WORKPLAN-EXCEL.ERROR.INCORRECT_DATE", locale: this._locale });
                                errorDates.push(`${this.getColumnLetter(colNumber)}1`);
                            }
                        }
                    }
                }
                if (rowNumber > 1 && colNumber === 2) {
                    uniqueIds.push(cellValue);
                }
                if (rowNumber > 1 && colNumber > 2) {
                    let checkWpMatch;
                    if (cellValue) {
                        checkWpMatch = this.checkWorkPlanMatch(importTemplate, cellValue);
                    }
                    if (checkWpMatch) {
                        errorMessage = checkWpMatch;
                        errorCells.push(`${this.getColumnLetter(colNumber)}${rowNumber}`);
                    }
                    if (row.actualCellCount - 2 > headerDateCount) {
                        errorMessage = i18n_1.default.__({ phrase: "WORKPLAN-EXCEL.ERROR.WORKPLANS_WO_DATE", locale: this._locale });
                        errorCells.push(`${this.getColumnLetter(colNumber)}${rowNumber}`);
                    }
                }
            });
        });
        let userResponse = await this.checkUsersOnDb(uniqueIds);
        errorMessage = errorMessage?.length > 0 ? errorMessage : userResponse.errorMessage;
        if (userResponse.errorMessage?.length > 0) {
            userResponse.errorIndexes.forEach((index) => {
                errorUsers.push(`${this.getColumnLetter(2)}${index}`);
            });
        }
        return {
            validationSuccess: errorMessage?.length > 0 ? false : true,
            errorMessage,
            errorCells,
            errorDates,
            errorUsers,
        };
    }
    async fillMissingRuleKeywords(importTemplate) {
        for (let rule of importTemplate.workPlanImportRules) {
            if (!rule.keyword) {
                let workPlan = await dal_manager_1.dbManager.accessPacs2.getWorkPlan(this._organizationId, rule.workPlanId);
                rule.keyword = workPlan.name;
            }
        }
        return importTemplate;
    }
    checkWorkPlanMatch(importTemplate, cellValue) {
        return importTemplate.workPlanImportRules.some((rule) => rule.keyword === cellValue) ? "" : i18n_1.default.__({ phrase: "WORKPLAN-EXCEL.ERROR.MISMATCHED_KEYWORD", locale: this._locale });
    }
    async checkUsersOnDb(uniqueIds) {
        let errorMessage;
        let errorIndexes = [];
        let users = await dal_manager_1.dbManager.accessUser.listUserIdsFromUniqueIds(this._organizationId, uniqueIds);
        uniqueIds.forEach((id, idIndex) => {
            let index = users.findIndex((user) => user.uniqueId === id);
            if (!id) {
                errorMessage = i18n_1.default.__({ phrase: "WORKPLAN-EXCEL.ERROR.MISSING_USER_ID", locale: this._locale });
                errorIndexes.push(idIndex + 2);
            }
            if (index < 0) {
                errorMessage = errorMessage?.length ? errorMessage : i18n_1.default.__({ phrase: "WORKPLAN-EXCEL.ERROR.MISMATCHED_UNIQUE_ID", locale: this._locale });
                errorIndexes.push(idIndex + 2);
            }
        });
        return { errorMessage, errorIndexes };
    }
    async generateExcel(params, importTemplate) {
        this._organizationName = (await dal_manager_1.dbManager.accessOrganization.getOrganizationBasic(this._organizationId)).name;
        this.createWorkbook();
        this._sheetNames = {
            [pacs_models_1.PacsModels.WorkPlanExcelSheetType.Input]: null,
        };
        importTemplate = await this.fillMissingRuleKeywords(importTemplate);
        await this.createSheets(params.userFilterId, params.dateRange, params.days, params.type, importTemplate);
        for (let sheetName in this._sheetNames) {
            this._wbWriter.getWorksheet(this._sheetNames[sheetName]).commit();
        }
        await this._wbWriter.commit();
        return {
            filePath: this._filePath,
            filename: this._organizationName
                .toLocaleLowerCase()
                .replace(new RegExp(/[&\/\\#,+()$~%.'":*?<>{}`\r\n\t]/g), "")
                .replace(new RegExp(/[\s]+/g), "_")
                .replace(new RegExp(/ç/g), "c")
                .replace(new RegExp(/ğ/g), "g")
                .replace(new RegExp(/ı/g), "i")
                .replace(new RegExp(/ö/g), "o")
                .replace(new RegExp(/ş/g), "s")
                .replace(new RegExp(/ü/g), "u") +
                "_" +
                i18n_1.default.__({ phrase: "WORKPLAN-EXCEL.FILENAME", locale: this._locale }) +
                ".xlsx",
        };
    }
    async createSheets(userFilterId, dateRange, days, type, importTemplate) {
        let dayCount;
        if (type === dal_constants_1.DalConstants.WorkPlanImportTemplateType.DateRange) {
            dayCount = this.getHeaders(dateRange);
        }
        else {
            for (let i = 1; i <= days; i++) {
                this._headers.push((i + ". " + i18n_1.default.__({ phrase: `EXCEL-REPORT.DAY`, locale: this._locale })));
                dayCount = days;
            }
        }
        for (let sheetName in this._sheetNames) {
            this.addSheetToWorkbook(sheetName);
            this._wbWriter.getWorksheet(this._sheetNames[sheetName]).properties.defaultRowHeight = 65;
            this._wbWriter.getWorksheet(this._sheetNames[sheetName]).properties.defaultColWidth = 30;
            this.setColumnKeys(this._wbWriter.getWorksheet(this._sheetNames[sheetName]));
            const headerRow = this._wbWriter.getWorksheet(this._sheetNames[sheetName]).addRow(this._headers);
            headerRow.eachCell((cell) => {
                cell.fill = {
                    type: "pattern",
                    pattern: "solid",
                    fgColor: { argb: "FFFF00" },
                };
                cell.font = {
                    bold: true,
                };
                cell.border = {
                    top: { style: "thin", color: { argb: "FF000000" } },
                    left: { style: "thin", color: { argb: "FF000000" } },
                    bottom: { style: "thin", color: { argb: "FF000000" } },
                    right: { style: "thin", color: { argb: "FF000000" } },
                };
                cell.alignment = { horizontal: "center" };
            });
            let rowCount = 0;
            await dal_manager_1.dbManager.systemTransaction(async (trx) => {
                const bindingKeys = [];
                let query = (0, dal_utils_1.generateUserFilterQuery)(this._organizationId, userFilterId, bindingKeys);
                let userListForDefinition = await trx.query(`SELECT "uop"."name", "uop"."surname", "uop"."uniqueId" 
					FROM "${this._organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userOrganizationProfiles}" AS uop
					INNER JOIN (
						${query}
					) subq 
					ON uop."userId" = subq."userId"`, bindingKeys);
                let excelRow = [];
                rowCount = userListForDefinition.rowCount;
                userListForDefinition.rows.forEach((row) => {
                    excelRow.push(`${row.name} ${row.surname}`);
                    excelRow.push(`${row.uniqueId}`);
                    this._wbWriter.getWorksheet(this._sheetNames[sheetName]).addRow(excelRow);
                    excelRow = [];
                });
            });
            const colStart = 3;
            const rowStart = 2;
            const colEnd = dayCount + 3;
            const rowEnd = rowCount + 1;
            let formula = "";
            importTemplate.workPlanImportRules.forEach((rule, index) => {
                if (index === 0)
                    formula += `AND(NOT(C2="${rule.keyword}")`;
                else
                    formula += `, NOT(C2="${rule.keyword}")`;
                if (index === importTemplate.workPlanImportRules.length - 1)
                    formula += `, NOT(C2=""))`;
            });
            const range = `${this.getColumnLetter(colStart)}${rowStart}:${this.getColumnLetter(colEnd)}${rowEnd}`;
            this._wbWriter.getWorksheet(this._sheetNames[sheetName]).addConditionalFormatting({
                ref: range,
                rules: [
                    {
                        type: "expression",
                        formulae: [`${formula}`],
                        priority: 1,
                        style: {
                            fill: {
                                type: "pattern",
                                pattern: "solid",
                                bgColor: { argb: "FFFF0000" },
                            },
                        },
                    },
                ],
            });
        }
        const hiddenSheet = this._wbWriter.addWorksheet("HiddenData");
        hiddenSheet.getCell("A1").value = type;
        hiddenSheet.state = "veryHidden";
    }
    getColumnLetter(col) {
        let letter = "";
        while (col > 0) {
            const temp = (col - 1) % 26;
            letter = String.fromCharCode(temp + 65) + letter;
            col = Math.floor((col - temp) / 26);
        }
        return letter;
    }
    addSheetToWorkbook(key) {
        let translatedName = i18n_1.default.__({ phrase: `WORKPLAN-EXCEL.SHEET-NAMES.${key.toUpperCase()}`, locale: this._locale });
        let ws = this._wbWriter.addWorksheet(translatedName, {
            pageSetup: { fitToPage: true, fitToWidth: 1, fitToHeight: 0, paperSize: 9, orientation: "landscape" },
        });
        this._sheetNames[key] = ws.id;
    }
    createWorkbook() {
        const options = {
            filename: this._filePath,
            useSharedStrings: true,
            useStyles: true,
        };
        this._wbWriter = new exceljs_1.default.stream.xlsx.WorkbookWriter(options);
        this._wbWriter.creator = this._requestedUserCaption;
        this._wbWriter.company = this._organizationName;
    }
    setColumnKeys(ws) {
        let columns = [];
        for (let element of this._headers) {
            columns.push({
                key: element,
                width: ws.properties.defaultColWidth,
            });
        }
        columns.push({
            key: "empty1",
            width: ws.properties.defaultColWidth,
        });
        columns.push({
            key: "empty2",
            width: ws.properties.defaultColWidth,
        });
        ws.columns = columns;
    }
    getHeaders(dateRange) {
        const { startDateTime, endDateTime } = dateRange;
        const dayCount = (0, moment_1.default)(endDateTime).diff(startDateTime, "days");
        for (let i = 0; i <= dayCount; i++) {
            this._headers.push((0, moment_1.default)(startDateTime).add(i, "d").format("DD/MM/YYYY"));
        }
        return dayCount;
    }
}
exports.WorkPlanImportUploadExcel = WorkPlanImportUploadExcel;
