"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.PregistrationInfo = exports.RegionInfo = exports.UserInfoSummary = void 0;
const moment_1 = __importDefault(require("moment"));
const util_1 = require("util");
const app_enums_1 = require("../app.enums");
const dal_manager_1 = require("../dal/dal.manager");
const predefined_roles_1 = require("../dal/db/predefined/predefined.roles");
class UserInfoSummary {
    constructor(organizationId, userId) {
        this._organizationId = organizationId;
        this._userId = userId;
    }
    async getAccessAndLoginSummary(organizationId) {
        let result = {
            lastAccess: null,
            lastLogin: null,
            currentRegions: [],
        };
        let logins = await dal_manager_1.dbManager.accessUser.listUserActiveDevices(organizationId, this._userId);
        if (logins.items && logins.items.length > 0) {
            let lastLogin = logins.items.sort((x, y) => ((0, moment_1.default)(x.lastGrantSessionUtc).isSameOrBefore((0, moment_1.default)(y.lastGrantSessionUtc)) ? 1 : -1))[0];
            result.lastLogin = {
                device: lastLogin.device,
                id: lastLogin.id,
                lastGrantSessionUtc: lastLogin.lastGrantSessionUtc,
                os: lastLogin.os,
            };
        }
        let now = new Date();
        let oneWeekEarly = (0, moment_1.default)(now).add("week", -1).toDate();
        let latestLogs = await dal_manager_1.dbManager.accessLog.getAccessLogsReport(this._organizationId, {
            startUtc: oneWeekEarly,
            endUtc: now,
            userIds: [this._userId],
            sortDateDesc: true,
            accessResult: app_enums_1.enums.AccessReportFilterAccessResultType.Success,
            pagination: { take: 100, skip: 0 },
        }, false);
        let latestAccessLog = latestLogs.logs[0];
        if (latestAccessLog) {
            result.lastAccess = {
                accessControlPointName: latestAccessLog.an,
                lastAccessLog: latestAccessLog,
                direction: latestAccessLog.d,
                lastAccessUtc: new Date(latestAccessLog.u),
                regionNames: latestAccessLog.rg && latestAccessLog.rg.length > 0 ? latestAccessLog.rg.filter((r) => r.s == app_enums_1.enums.AntiPassbackState.In).map((r) => r.n) : [],
            };
            for (const regionLog of latestAccessLog.rg.filter((r) => r.s == app_enums_1.enums.AntiPassbackState.In)) {
                let lastExit = latestLogs.logs.find((l) => l.rg.some((r) => r.i == regionLog.i) && l.d == app_enums_1.enums.AccessDirection.Exit);
                result.currentRegions.push({
                    lastEntranceUtc: latestAccessLog.u,
                    lastExitUtc: lastExit ? lastExit.u : util_1.isNull,
                    regionName: regionLog.n,
                });
            }
        }
        return Promise.resolve(result);
    }
    async getUserSocialDetails() {
        let result = {
            lastSpent: null,
            lastTransaction: null,
            currentBalance: 0,
            totalSpentAllTime: 0,
            month: {
                totalEarnings: 0,
                totalSpent: 0,
            },
            year: {
                totalEarnings: 0,
                totalSpent: 0,
            },
        };
        let now = new Date();
        let oneYearEarly = (0, moment_1.default)(now).add(-1, "year").toDate();
        let ticketTransactions = await dal_manager_1.dbManager.accessLog.listRegionTicketTransactions(this._organizationId, {
            take: 1000,
            skip: 0,
        }, {
            startDateTime: oneYearEarly,
            endDateTime: now,
        }, null, this._userId);
        let spendings = await dal_manager_1.dbManager.accessLog.listRegionTicketReport(this._organizationId, {
            take: 1000,
            skip: 0,
        }, {
            startDateTime: oneYearEarly,
            endDateTime: now,
        }, [this._userId]);
        let lastSpent = spendings.items[0];
        if (lastSpent) {
            result.lastSpent = {
                amount: lastSpent.rg.reduce((x, y) => {
                    return x + y.rti.c;
                }, 0),
                actionUtc: new Date(lastSpent.u),
            };
        }
        let lastTransaction = ticketTransactions.items[0];
        if (lastTransaction) {
            result.lastTransaction = {
                actionUtc: lastTransaction.date,
                amount: lastTransaction.amount,
                user: lastTransaction.actionUser,
            };
        }
        let oneMonthEarly = (0, moment_1.default)(now).add(-1, "month").toDate();
        result.month = {
            totalEarnings: ticketTransactions.items
                .filter((t) => (0, moment_1.default)(t.date).isAfter(oneMonthEarly) && (0, moment_1.default)(t.date).isBefore(now))
                .reduce((x, y) => {
                return x + y.amount;
            }, 0),
            totalSpent: spendings.items
                .filter((t) => (0, moment_1.default)(t.u).isAfter(oneMonthEarly) && (0, moment_1.default)(t.u).isBefore(now))
                .reduce((x, y) => {
                return (x +
                    y.rg.reduce((x, y) => {
                        return x + y.rti.c;
                    }, 0));
            }, 0),
        };
        result.year = {
            totalEarnings: ticketTransactions.items
                .filter((t) => (0, moment_1.default)(t.date).isBefore(now))
                .reduce((x, y) => {
                return x + y.amount;
            }, 0),
            totalSpent: spendings.items
                .filter((t) => (0, moment_1.default)(t.u).isBefore(now))
                .reduce((x, y) => {
                return (x +
                    y.rg.reduce((x, y) => {
                        return x + y.rti.c;
                    }, 0));
            }, 0),
        };
        let currentBalance = await dal_manager_1.dbManager.accessSocial.getUserCurrentBalance(this._organizationId, this._userId);
        if (currentBalance) {
            result.currentBalance = currentBalance.remainingUnits;
            result.totalSpentAllTime = currentBalance.totalSpentUnits;
        }
        return Promise.resolve(result);
    }
    async getUserVisitorDetails() {
        let result = {
            current: null,
            past: null,
        };
        let userOrganizationId = await dal_manager_1.dbManager.accessUser.getUserOrganizationId({ organizationId: this._organizationId, userId: this._userId });
        let activeVisits = await dal_manager_1.dbManager.accessVisitor.listUserActiveVisits(this._organizationId, userOrganizationId);
        result.current = {
            currentVisitorCount: activeVisits.filter((v) => v.state == app_enums_1.enums.UnterminatedVisitState.Active).length,
            expectedVisitorCount: activeVisits.filter((v) => v.state == app_enums_1.enums.UnterminatedVisitState.Expected).length,
        };
        let now = new Date();
        let startOfToday = (0, moment_1.default)(now).startOf("day").toDate();
        let startOfMonth = (0, moment_1.default)(now).startOf("month").toDate();
        let startOfWeek = (0, moment_1.default)(now).startOf("week").toDate();
        let oneYearEarly = (0, moment_1.default)(now).add(-1, "year").toDate();
        let pastVisits = await dal_manager_1.dbManager.accessLog.listUserPastVisits(this._organizationId, {
            startDateTime: oneYearEarly,
            endDateTime: now,
        }, this._userId);
        result.past = {
            currentDay: pastVisits.filter((t) => (0, moment_1.default)(t.startUtc).isAfter(startOfToday)).length,
            currentMonth: pastVisits.filter((t) => (0, moment_1.default)(t.startUtc).isAfter(startOfMonth)).length,
            currentWeek: pastVisits.filter((t) => (0, moment_1.default)(t.startUtc).isAfter(startOfWeek)).length,
        };
        return Promise.resolve(result);
    }
}
exports.UserInfoSummary = UserInfoSummary;
class RegionInfo {
    constructor(organizationId) {
        this._organizationId = organizationId;
    }
    async getRegionSummary() {
        let result = {
            regions: [],
        };
        let regionsOfOrganization = await dal_manager_1.dbManager.accessRegion.listOrganizationRegions(this._organizationId);
        let userStates = await dal_manager_1.dbManager.accessRegion.listRegionUserStates(this._organizationId);
        for (const region of regionsOfOrganization) {
            result.regions.push({
                name: region.name,
                totalCount: userStates.filter((u) => u.regionId == region.id).length,
                visitorCount: userStates.filter((u) => u.regionId == region.id && [predefined_roles_1.PredefinedRoles.OrganizationVisitor.id, predefined_roles_1.PredefinedRoles.OrganizationUnitVisitor.id].some((p) => p == u.typeId)).length,
                userCount: userStates.filter((u) => u.regionId == region.id && ![predefined_roles_1.PredefinedRoles.OrganizationVisitor.id, predefined_roles_1.PredefinedRoles.OrganizationUnitVisitor.id].some((p) => p == u.typeId)).length,
            });
        }
        return Promise.resolve(result);
    }
}
exports.RegionInfo = RegionInfo;
class PregistrationInfo {
    constructor(organizationId) {
        this._organizationId = organizationId;
    }
    async getPregistrationSummary(dateRange, segment) {
        let result = {
            items: [],
        };
        let startDate = dateRange.startDateTime;
        let endDate = dateRange.endDateTime;
        let indexJump;
        switch (segment) {
            case app_enums_1.enums.SegmentType.Day:
                startDate = (0, moment_1.default)(startDate).startOf("day").toDate();
                endDate = (0, moment_1.default)(endDate).endOf("day").toDate();
                indexJump = "day";
                break;
            case app_enums_1.enums.SegmentType.Week:
                startDate = (0, moment_1.default)(startDate).startOf("isoWeek").toDate();
                endDate = (0, moment_1.default)(endDate).endOf("isoWeek").toDate();
                indexJump = "week";
                break;
            case app_enums_1.enums.SegmentType.Month:
                startDate = (0, moment_1.default)(startDate).startOf("month").toDate();
                endDate = (0, moment_1.default)(endDate).endOf("month").toDate();
                indexJump = "month";
                break;
            default:
                break;
        }
        let dateIndexStart = (0, moment_1.default)(startDate);
        let dateIndexEnd = (0, moment_1.default)(endDate);
        let loopGuard = 0;
        let pregistrations = await dal_manager_1.dbManager.accessLog.listPregistrationReport(this._organizationId, dateRange);
        while (dateIndexStart.isSameOrBefore(dateIndexEnd) && loopGuard < 1000) {
            loopGuard++;
            let lookupEndDate = (0, moment_1.default)(dateIndexStart).toDate();
            switch (segment) {
                case app_enums_1.enums.SegmentType.Day:
                    lookupEndDate = (0, moment_1.default)(dateIndexStart).endOf("day").toDate();
                    break;
                case app_enums_1.enums.SegmentType.Week:
                    lookupEndDate = (0, moment_1.default)(dateIndexStart).endOf("isoWeek").toDate();
                    break;
                case app_enums_1.enums.SegmentType.Month:
                    lookupEndDate = (0, moment_1.default)(dateIndexStart).endOf("month").toDate();
                    break;
                default:
                    break;
            }
            result.items.push({
                startDate: dateIndexStart.toDate(),
                endDate: lookupEndDate,
                data: {
                    activeVisitsFromPregistration: pregistrations.filter((p) => p.turnedIntoVisit && (0, moment_1.default)(p.actionUtc).isSameOrAfter(dateIndexStart) && (0, moment_1.default)(p.actionUtc).isSameOrBefore(dateIndexEnd)).length,
                    preregistrationCount: pregistrations.filter((p) => (0, moment_1.default)(p.actionUtc).isSameOrAfter(dateIndexStart) && (0, moment_1.default)(p.actionUtc).isSameOrBefore(dateIndexEnd)).length,
                },
            });
            switch (segment) {
                case app_enums_1.enums.SegmentType.Day:
                    dateIndexStart = (0, moment_1.default)(dateIndexStart).add(1, "day");
                    break;
                case app_enums_1.enums.SegmentType.Week:
                    dateIndexStart = (0, moment_1.default)(dateIndexStart).add(1, "week");
                    break;
                case app_enums_1.enums.SegmentType.Month:
                    dateIndexStart = (0, moment_1.default)(dateIndexStart).add(1, "month");
                    break;
                default:
                    break;
            }
        }
        return Promise.resolve(result);
    }
}
exports.PregistrationInfo = PregistrationInfo;
