"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.BusinessOrganizationDefinitionFile = exports.BusinessOrganizationLicence = void 0;
const crypto_1 = __importDefault(require("crypto"));
const got_1 = __importDefault(require("got"));
const app_config_1 = require("../app.config");
const app_constants_1 = require("../app.constants");
const app_enums_1 = require("../app.enums");
const app_logs_1 = require("../app.logs");
const dal_access_error_1 = require("../dal/access/dal.access.error");
const dal_manager_1 = require("../dal/dal.manager");
const messageBroker_cli_1 = require("../messageBroker/messageBroker.cli");
const business_pacs_1 = require("./pacs/business.pacs");
const business_visitor_1 = require("./visitor/business.visitor");
class BusinessOrganizationLicence {
    constructor(organizationPublicKey, organizationPrivateKey, fileData) {
        this._organizationPublicKey = organizationPublicKey;
        this._organizationPrivateKey = organizationPrivateKey;
        this._licenceFile = this.verifyAndParseOrganizationLicenceFile(fileData);
    }
    verifyAndParseOrganizationLicenceFile(fileData) {
        let ecdh = crypto_1.default.createECDH("prime256v1");
        ecdh.setPrivateKey(this._organizationPrivateKey, "base64");
        let file = JSON.parse(fileData.toString());
        let sharedSecret = ecdh.computeSecret(app_constants_1.armonPublicKey, "base64");
        let signatureKey = crypto_1.default
            .createHmac("sha256", sharedSecret)
            .update(sharedSecret.toString("base64") + file.content.id)
            .digest("base64");
        let signature = crypto_1.default.createHmac("sha256", signatureKey).update(JSON.stringify(file.content)).digest("base64");
        if (file.signature !== signature) {
            throw new Error("Organization licence file integrity is not valid");
        }
        let now = new Date();
        if (file.content.licenceStartDate > now) {
            throw new Error("The licence file can not be used before " + file.content.licenceStartDate.toString());
        }
        if (file.content.licenceEndDate && file.content.licenceEndDate < now) {
            throw new Error("The licence file is expired at " + file.content.licenceEndDate.toString());
        }
        return file;
    }
    async handleLicenceFile() {
        await dal_manager_1.dbManager.transaction(async (trx) => {
            const conflictingDevices = await dal_manager_1.dbManager.accessDevice.checkDeviceOrganizationMappings(this._licenceFile.content.id, this._licenceFile.content.devices.map((d) => d.id), trx);
            if (conflictingDevices.length) {
                let errorMessage = `Yüklenen lisans dosyasında bulunan aşağıdaki cihazlar başka bir organizasyonda kayıtlı!`;
                for (const cd of conflictingDevices) {
                    errorMessage += `\n Ad: ${cd.deviceName}, SN: ${cd.deviceSerialNumber}, Org Name: ${cd.organizationName}`;
                }
                (0, dal_access_error_1.throwDbAccessConflictError)(errorMessage);
            }
            let disabledModules = await dal_manager_1.dbManager.accessOrganization.disableModulesExcept(this._licenceFile.content.id, (this._licenceFile.content.modules || []).map((m) => m.type), trx);
            await this.disableModuleActions(this._licenceFile.content.id, disabledModules, trx);
            let enabledModules = await dal_manager_1.dbManager.accessOrganization.enableModules(this._licenceFile.content.id, this._licenceFile.content.modules || [], trx);
            await this.enableModuleActions(this._licenceFile.content.id, enabledModules, trx);
            let actionedAdapterIds = await dal_manager_1.dbManager.accessDevice.upsertOrganizationAdapters(this._licenceFile.content.id, this._licenceFile.content.adapters, trx);
            let actionedDeviceIds = await dal_manager_1.dbManager.accessDevice.upsertOrganizationDevices(this._licenceFile.content.id, this._licenceFile.content.devices, trx);
            await this.deleteRabbitMqUsers(actionedAdapterIds.removed.concat(...actionedDeviceIds.removed));
            await this.putRabbitMqUsers(actionedAdapterIds.added.concat(...actionedDeviceIds.added));
            setTimeout(() => {
                (0, messageBroker_cli_1.refreshTerminalConnectionStates)({
                    hostname: app_config_1.appConfig.amqpClient.hostname,
                    port: app_config_1.appConfig.amqpClient.managementApiPort,
                    authorization: app_config_1.appConfig.amqpManagementApiAuth,
                    organizationId: this._licenceFile.content.id,
                });
            }, 30000);
        });
    }
    async putRabbitMqUsers(ids) {
        for (const id of ids) {
            await got_1.default.put(`http://${app_config_1.appConfig.amqpClient.hostname}:${app_config_1.appConfig.amqpClient.managementApiPort}/api/users/${id}`, {
                json: true,
                headers: {
                    Authorization: app_config_1.appConfig.amqpManagementApiAuth,
                },
                body: {
                    tags: "",
                    password_hash: "",
                },
            });
            await got_1.default.put(`http://${app_config_1.appConfig.amqpClient.hostname}:${app_config_1.appConfig.amqpClient.managementApiPort}/api/permissions/${encodeURIComponent(app_config_1.appConfig.amqpClient.vhost)}/${id}`, {
                json: true,
                headers: {
                    Authorization: app_config_1.appConfig.amqpManagementApiAuth,
                },
                body: {
                    configure: ".*",
                    write: ".*",
                    read: ".*",
                },
            });
        }
    }
    async deleteRabbitMqUsers(ids) {
        let response = await got_1.default.get(`http://${app_config_1.appConfig.amqpClient.hostname}:${app_config_1.appConfig.amqpClient.managementApiPort}/api/users`, {
            json: true,
            headers: {
                Authorization: app_config_1.appConfig.amqpManagementApiAuth,
            },
        });
        const usersInRabbitMq = ids.filter((i) => response.body.some((r) => r.name == i));
        if (usersInRabbitMq.length > 0) {
            await got_1.default.post(`http://${app_config_1.appConfig.amqpClient.hostname}:${app_config_1.appConfig.amqpClient.managementApiPort}/api/users/bulk-delete`, {
                json: true,
                headers: {
                    Authorization: app_config_1.appConfig.amqpManagementApiAuth,
                },
                body: {
                    users: usersInRabbitMq,
                },
            });
        }
    }
    async disableModuleActions(organizationId, modules, trx) {
        if (!modules || modules.length === 0) {
            return Promise.resolve();
        }
        for (const module of modules) {
            switch (module) {
                case app_enums_1.enums.ArmonApplicationModule.PACS:
                    await (0, business_pacs_1.disablePACSModule)(organizationId, trx);
                    break;
                case app_enums_1.enums.ArmonApplicationModule.SocialArea:
                    break;
                case app_enums_1.enums.ArmonApplicationModule.Visitor:
                    await (0, business_visitor_1.disableVisitorModule)(organizationId, trx);
                    break;
                case app_enums_1.enums.ArmonApplicationModule.Base:
                    break;
                default:
                    app_logs_1.logger.error("Undefined module " + module);
                    break;
            }
        }
    }
    async enableModuleActions(organizationId, modules, trx) {
        if (!modules || modules.length === 0) {
            return Promise.resolve();
        }
        for (const module of modules) {
            switch (module) {
                case app_enums_1.enums.ArmonApplicationModule.PACS:
                    await (0, business_pacs_1.enablePACSModule)(organizationId, trx);
                    break;
                case app_enums_1.enums.ArmonApplicationModule.SocialArea:
                    break;
                case app_enums_1.enums.ArmonApplicationModule.Visitor:
                    await (0, business_visitor_1.enableVisitorModule)(organizationId, trx);
                    break;
                case app_enums_1.enums.ArmonApplicationModule.Base:
                    break;
                default:
                    app_logs_1.logger.error("Undefined module " + module);
                    break;
            }
        }
    }
}
exports.BusinessOrganizationLicence = BusinessOrganizationLicence;
class BusinessOrganizationDefinitionFile {
    constructor(organizaitonId) {
        this._organizationId = organizaitonId;
    }
    async createOrganizationDefinitionFile(version) {
        let organization = await dal_manager_1.dbManager.accessSystem.getOrganizationDefinitionFileContentV3(this._organizationId);
        let organizationPrivateKey = organization.organizationPrivateKey;
        let content = organization.content;
        let ecdh = crypto_1.default.createECDH("prime256v1");
        ecdh.setPrivateKey(organizationPrivateKey, "base64");
        let file = {
            version: version,
            content: content,
            signature: null,
        };
        let sharedSecret = ecdh.computeSecret(app_constants_1.armonPublicKey, "base64");
        let signatureKey = crypto_1.default
            .createHmac("sha256", sharedSecret)
            .update(sharedSecret.toString("base64") + file.content.id)
            .digest("base64");
        let signature = crypto_1.default.createHmac("sha256", signatureKey).update(JSON.stringify(file.content)).digest("base64");
        file.signature = signature;
        return file;
    }
}
exports.BusinessOrganizationDefinitionFile = BusinessOrganizationDefinitionFile;
