var fs = require("fs");
var path = require("path");
var child_process = require("child_process");

function checkAndCreateDir(dirpath) {
	try {
		console.log("Re/creating " + dirpath);
		if (!fs.existsSync(dirpath)) {
			fs.mkdirSync(dirpath, { recursive: true });
		}
	} catch (error) {
		console.log("	" + error);
		process.exit(1);
	}
}

function refreshFile(inputPath, outputPath, replacements) {
	try {
		console.log("Refreshing " + outputPath);
		let contents = fs.readFileSync(inputPath).toString();
		for (const rule of replacements || []) {
			contents = contents.replace(new RegExp(rule.regex, "g"), rule.replacement);
		}
		fs.writeFileSync(outputPath, contents);
	} catch (error) {
		console.log("Error while refreshing file " + outputPath);
		if (replacements) {
			console.log("	replacements: " + JSON.stringify(replacements, null, 4));
		}
		console.log("	" + error);
		process.exit(1);
	}
}

function copyIfNotExists(inputPath, outputPath) {
	try {
		console.log("Copying if not exists " + outputPath);
		if (!fs.existsSync(outputPath)) {
			fs.copyFileSync(inputPath, outputPath);
		}
	} catch (error) {
		console.log("	" + error);
		process.exit(1);
	}
}
// Create volume directory tree if it does not exist"checkAndCreateDir(path.join("/volroot", "config"));
checkAndCreateDir(path.join("/volroot", "config", "rabbitmq"));
checkAndCreateDir(path.join("/volroot", "config", "certs"));
checkAndCreateDir(path.join("/volroot", "config", "domain-certs"));
checkAndCreateDir(path.join("/volroot", "config", "gateway"));
checkAndCreateDir(path.join("/volroot", "config", "postgresql"));
checkAndCreateDir(path.join("/volroot", "config", "redis"));
checkAndCreateDir(path.join("/volroot", "data"));
checkAndCreateDir(path.join("/volroot", "data", "pg"));
checkAndCreateDir(path.join("/volroot", "data", "rabbitmq"));
checkAndCreateDir(path.join("/volroot", "data", "redis"));
checkAndCreateDir(path.join("/volroot", "var"));
checkAndCreateDir(path.join("/volroot", "var", "app-web"));
checkAndCreateDir(path.join("/volroot", "var", "rabbitmq"));

// adjust rabbitmq var folder ownership otherwise rmq cannot write here
try {
	console.log("Adjusting rabbitmq and redis owner/group");
	fs.chownSync(path.join("/volroot", "var", "rabbitmq"), 100, 101);
	fs.chownSync(path.join("/volroot", "data", "redis"), 999, 1000);
} catch (error) {
	console.log("	" + error);
	process.exit(1);
}

const defaultAssetDir = "/opt/armon/webapp/assets/docker-defaults";
const targetConfDir = "/volroot/config";

// Copy certificates if they do not exist. This section is for dev environments
copyIfNotExists(path.join(defaultAssetDir, "certs", "ca.crt.pem"), path.join(targetConfDir, "certs", "ca.crt.pem"));
copyIfNotExists(path.join(defaultAssetDir, "certs", "ca.key.pem"), path.join(targetConfDir, "certs", "ca.key.pem"));
copyIfNotExists(path.join(defaultAssetDir, "certs", "chain.ca.crt.pem"), path.join(targetConfDir, "certs", "chain.ca.crt.pem"));
copyIfNotExists(path.join(defaultAssetDir, "certs", "server.crt.pem"), path.join(targetConfDir, "certs", "server.crt.pem"));
copyIfNotExists(path.join(defaultAssetDir, "certs", "server.key.pem"), path.join(targetConfDir, "certs", "server.key.pem"));
copyIfNotExists(path.join(defaultAssetDir, "certs", "webappclient.crt.pem"), path.join(targetConfDir, "certs", "webappclient.crt.pem"));
copyIfNotExists(path.join(defaultAssetDir, "certs", "webappclient.key.pem"), path.join(targetConfDir, "certs", "webappclient.key.pem"));

// Refresh config files with values from .env file
copyIfNotExists(path.join(defaultAssetDir, "assets", "rabbitmq.conf"), path.join(targetConfDir, "rabbitmq", "rabbitmq.conf"));
copyIfNotExists(path.join(defaultAssetDir, "assets", "redis.conf"), path.join(targetConfDir, "redis", "redis.conf"));
copyIfNotExists(path.join(defaultAssetDir, "assets", "definitions.json"), path.join(targetConfDir, "rabbitmq", "definitions.json"));
refreshFile(path.join(defaultAssetDir, "assets", "agent_config.json"), path.join(targetConfDir, "gateway", "configuration.json"), [
	{ regex: "SERVER_ADDRESS", replacement: process.env.HTTP_PROTOCOL + "://" + process.env.DOMAIN_NAME },
	{ regex: "APIPORT", replacement: process.env.APIPORT },
]);

// We need to handle config.json specially to not lose custom configurations
copyIfNotExists(path.join(defaultAssetDir, "assets", "config.json"), path.join(targetConfDir, "config.json"));
let configContents = JSON.parse(fs.readFileSync(path.join(targetConfDir, "config.json")).toString());
if (configContents.db?.main) {
	configContents.db.main = Object.assign(configContents.db.main, {
		host: process.env.PGHOST,
		port: process.env.PGPORT,
		name: process.env.PGDATABASE,
		username: process.env.PGUSER,
		password: process.env.PGUSER_PASSWORD,
	});
}
if (configContents.db?.log) {
	configContents.db.log = Object.assign(configContents.db.log, {
		host: process.env.PGHOST,
		port: process.env.PGPORT,
		name: process.env.PGDATABASE + "log",
		username: process.env.PGUSER,
		password: process.env.PGUSER_PASSWORD,
	});
}
if (configContents.amqpClient?.hostname) {
	configContents.amqpClient.hostname = process.env.AMQP_HOSTNAME;
}
if (configContents.httpServer) {
	configContents.httpServer.port = parseInt(process.env.APIPORT);
}
fs.writeFileSync(path.join(targetConfDir, "config.json"), JSON.stringify(configContents, null, 4));

copyIfNotExists(path.join(defaultAssetDir, "assets", "gateway.template"), path.join(targetConfDir, "gateway", "gateway.template"));

copyIfNotExists(path.join(defaultAssetDir, "assets", "init-db.sh"), path.join(targetConfDir, "postgresql", "init-db.sh"));
copyIfNotExists(path.join(defaultAssetDir, "assets", "scaffold.sql"), path.join(targetConfDir, "postgresql", "scaffold.sql"));

try {
	console.log("Refreshing .pgpass file");
	const apiline = [process.env.PGHOST, process.env.PGPORT, process.env.PGDATABASE, process.env.PGUSER, process.env.PGUSER_PASSWORD].join(":");
	let superline = [process.env.PGHOST, process.env.PGPORT, process.env.PGDATABASE, process.env.PGSUPERUSER, process.env.PGSUPERUSER_PASSWORD].join(":");
	fs.writeFileSync(path.join(targetConfDir, "postgresql", ".pgpass"), apiline + "\n" + superline + "\n");
	fs.chmodSync(path.join(targetConfDir, "postgresql", ".pgpass"), "0600");
} catch (error) {
	console.log("	" + error);
	process.exit(1);
}

console.log("Setup completed successfully");
