"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.AppReport = void 0;
const elastic_apm_node_1 = __importDefault(require("elastic-apm-node"));
if (process.env.ELASTIC_APM_SERVER_URL &&
    process.env.ELASTIC_APM_SECRET_TOKEN &&
    process.env.ELASTIC_APM_PREFIX &&
    ((process.env.ELASTIC_APM_VERIFY_SERVER_CERT === "true" && process.env.ELASTIC_APM_SERVER_CA_CERT_FILE) || process.env.ELASTIC_APM_VERIFY_SERVER_CERT === "false")) {
    elastic_apm_node_1.default.start({
        serviceName: `${process.env.ELASTIC_APM_PREFIX}-report`,
        serverUrl: process.env.ELASTIC_APM_SERVER_URL,
        secretToken: process.env.ELASTIC_APM_SECRET_TOKEN,
        environment: "production",
        serverCaCertFile: process.env.ELASTIC_APM_SERVER_CA_CERT_FILE ?? null,
        verifyServerCert: process.env.ELASTIC_APM_VERIFY_SERVER_CERT === "true",
    });
}
else {
    console.warn(`Elastic APM is not enabled. Please set ELASTIC_APM_SERVER_URL, ELASTIC_APM_SECRET_TOKEN, ELASTIC_APM_PREFIX, ELASTIC_APM_VERIFY_SERVER_CERT, and ELASTIC_APM_SERVER_CA_CERT_FILE environment variables.`);
}
const cluster_1 = __importDefault(require("cluster"));
const os_1 = __importDefault(require("os"));
const app_config_1 = require("./app.config");
const app_constants_1 = require("./app.constants");
const app_logs_1 = require("./app.logs");
const dal_manager_1 = require("./dal/dal.manager");
const messageBroker_manager_1 = require("./messageBroker/messageBroker.manager");
const business_hooks_1 = require("./business/business.hooks");
class AppReport {
    static main(args) {
        let app = new AppReport();
        app.start(args);
    }
    constructor() { }
    async start(args) {
        try {
            await app_config_1.appConfig.init(app_constants_1.ServiceNames.ReportService);
            app_logs_1.appLogger.init(app_config_1.appConfig.appLogDirectory);
            if (cluster_1.default.isMaster) {
                await this.startMaster();
            }
            else {
                await this.startChild();
            }
            app_logs_1.logger.info("armon-report-app is started");
        }
        catch (error) {
            app_logs_1.logger.error(error);
            process.exit(1);
        }
    }
    async startMaster() {
        try {
            cluster_1.default.on("exit", (worker, code, signal) => {
                app_logs_1.logger.error("child process %d died (%s). restarting...", worker.process.pid, signal || code);
                cluster_1.default.fork();
            });
            let childCountInfoLog = `There are #${os_1.default.cpus().length} cpus`;
            if (app_config_1.appConfig.maxCpuCore > 0) {
                childCountInfoLog += `but max available cpu count config is set as ${app_config_1.appConfig.maxCpuCore}`;
            }
            let childProcessCount = app_config_1.appConfig.maxCpuCore > 0 ? app_config_1.appConfig.maxCpuCore : os_1.default.cpus().length;
            childCountInfoLog += `. So there will be ${childProcessCount} child process will be ran.`;
            app_logs_1.logger.info(childCountInfoLog);
            for (let i = 0; i < childProcessCount; i++) {
                cluster_1.default.fork();
            }
        }
        catch (error) {
            app_logs_1.logger.error("Error while starting app. %j", error);
            process.exit(1);
        }
        app_logs_1.logger.info("Master process is ready");
    }
    async startChild() {
        try {
            await dal_manager_1.dbManager.init({
                main: app_config_1.appConfig.db.main,
                log: app_config_1.appConfig.db.log,
                environment: app_config_1.appConfig.nodeEnv,
                ignoreMigrations: true,
                logDirectory: app_config_1.appConfig.dbLogDirectory,
                redis: app_config_1.appConfig.db.redis,
            }, false, false, false, false);
            business_hooks_1.armonHookManager.init();
            if (!app_config_1.appConfig.disableMessageQueue) {
                await messageBroker_manager_1.messageBrokerManager.initForReportServerChild();
            }
        }
        catch (error) {
            app_logs_1.logger.error("Error while starting child process %j", error);
            process.exit(1);
        }
        app_logs_1.logger.info("Child process %s is ready", cluster_1.default.worker ? cluster_1.default.worker.id : " single node mode");
    }
}
exports.AppReport = AppReport;
AppReport.main(process.argv);
