"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.rateLimiterMiddleware = exports.authRateLimitReset = exports.authRateLimiter = exports.initLimiters = void 0;
const rate_limiter_flexible_1 = require("rate-limiter-flexible");
const dal_manager_1 = require("./dal/dal.manager");
const dal_db_armon_schema_1 = require("./dal/db/armon/dal.db.armon.schema");
const app_config_1 = require("./app.config");
const dal_access_cache_client_1 = __importDefault(require("./dal/access/redis/dal.access.cache.client"));
const jsonwebtoken_1 = __importDefault(require("jsonwebtoken"));
const app_auth_1 = require("./app.auth");
const app_logs_1 = require("./app.logs");
const i18n_1 = __importDefault(require("i18n"));
const app_enums_1 = require("./app.enums");
const app_constants_1 = require("./app.constants");
let authRateLimiterSlowBruteByIP = undefined;
let authRateLimiterConsecutiveFailsByUsernameAndIP = undefined;
let authRateLimiterSlowBruteByUsername = undefined;
let genericRateLimiter = undefined;
const userBasedRateLimiters = new Map();
const secondsInMinute = 60;
const initLimiters = async () => {
    authRateLimiterSlowBruteByUsername = new rate_limiter_flexible_1.RateLimiterPostgres({
        storeClient: dal_manager_1.dbManager.poolMain,
        points: app_config_1.appConfig.rateLimits.auth.byUsername.maxAttempts,
        duration: secondsInMinute * app_config_1.appConfig.rateLimits.auth.byUsername.windowDurationInMinutes,
        blockDuration: secondsInMinute * app_config_1.appConfig.rateLimits.auth.byUsername.blockDurationInMinutes,
        keyPrefix: "login_fail_username_per_day",
        schemaName: "public",
        tableName: dal_db_armon_schema_1.ArmonSchema.tableNames.rateLimits,
        tableCreated: true,
    });
    authRateLimiterConsecutiveFailsByUsernameAndIP = new rate_limiter_flexible_1.RateLimiterPostgres({
        storeClient: dal_manager_1.dbManager.poolMain,
        points: app_config_1.appConfig.rateLimits.auth.byUsernameAndIP.maxAttempts,
        duration: secondsInMinute * app_config_1.appConfig.rateLimits.auth.byUsernameAndIP.windowDurationInMinutes,
        blockDuration: secondsInMinute * app_config_1.appConfig.rateLimits.auth.byUsernameAndIP.blockDurationInMinutes,
        keyPrefix: "login_fail_consecutive_username_and_ip",
        schemaName: "public",
        tableName: dal_db_armon_schema_1.ArmonSchema.tableNames.rateLimits,
        tableCreated: true,
    });
    authRateLimiterSlowBruteByIP = new rate_limiter_flexible_1.RateLimiterPostgres({
        storeClient: dal_manager_1.dbManager.poolMain,
        points: app_config_1.appConfig.rateLimits.auth.byIP.maxAttempts,
        duration: secondsInMinute * app_config_1.appConfig.rateLimits.auth.byIP.windowDurationInMinutes,
        blockDuration: secondsInMinute * app_config_1.appConfig.rateLimits.auth.byIP.blockDurationInMinutes,
        keyPrefix: "login_fail_ip_per_day",
        schemaName: "public",
        tableName: dal_db_armon_schema_1.ArmonSchema.tableNames.rateLimits,
        tableCreated: true,
    });
    const rClient = await dal_access_cache_client_1.default.connect({
        url: `redis://${app_config_1.appConfig.db.redis.host}:${app_config_1.appConfig.db.redis.port}`,
    });
    genericRateLimiter = new rate_limiter_flexible_1.RateLimiterRedis({
        storeClient: rClient,
        keyPrefix: "api",
        points: app_config_1.appConfig.rateLimits.generic.maxAttempts,
        duration: secondsInMinute * app_config_1.appConfig.rateLimits.generic.windowDurationInMinutes,
        blockDuration: secondsInMinute * app_config_1.appConfig.rateLimits.generic.blockDurationInMinutes,
    });
    if (app_config_1.appConfig.rateLimits.customUserParams && Object.keys(app_config_1.appConfig.rateLimits.customUserParams).length) {
        for (const commaSeperatedUserIds of Object.keys(app_config_1.appConfig.rateLimits.customUserParams)) {
            const userIds = commaSeperatedUserIds.replace(/\s/g, "").split(",");
            const rateLimitParams = app_config_1.appConfig.rateLimits.customUserParams[commaSeperatedUserIds];
            for (const userId of userIds) {
                const customRateLimiter = new rate_limiter_flexible_1.RateLimiterRedis({
                    storeClient: rClient,
                    keyPrefix: "custom_",
                    points: rateLimitParams.maxAttempts,
                    duration: secondsInMinute * rateLimitParams.windowDurationInMinutes,
                    blockDuration: secondsInMinute * rateLimitParams.blockDurationInMinutes,
                });
                userBasedRateLimiters.set(userId, customRateLimiter);
            }
        }
    }
};
exports.initLimiters = initLimiters;
const getUsernameIPkey = (username, ip) => `${username}_${ip}`;
const authRateLimiter = async (username, ipAddress) => {
    const usernameIpKey = getUsernameIPkey(username, ipAddress);
    const [resUsernameAndIP, resSlowByIP, resSlowUsername] = await Promise.all([
        authRateLimiterConsecutiveFailsByUsernameAndIP.get(usernameIpKey),
        authRateLimiterSlowBruteByIP.get(ipAddress),
        authRateLimiterSlowBruteByUsername.get(username),
    ]);
    let retrySecs = 0;
    if (resSlowByIP !== null && resSlowByIP.consumedPoints > app_config_1.appConfig.rateLimits.auth.byIP.maxAttempts) {
        retrySecs = Math.round(resSlowByIP.msBeforeNext / 1000) || 1;
    }
    else if (resUsernameAndIP !== null && resUsernameAndIP.consumedPoints > app_config_1.appConfig.rateLimits.auth.byUsernameAndIP.maxAttempts) {
        retrySecs = Math.round(resUsernameAndIP.msBeforeNext / 1000) || 1;
    }
    else if (resSlowUsername !== null && resSlowUsername.consumedPoints > app_config_1.appConfig.rateLimits.auth.byUsername.maxAttempts) {
        retrySecs = Math.round(resSlowUsername.msBeforeNext / 1000) || 1;
    }
    if (retrySecs > 0) {
        return {
            success: false,
            retryInSecs: retrySecs,
        };
    }
    else {
        try {
            const limiterPromises = [];
            limiterPromises.push(authRateLimiterSlowBruteByIP.consume(ipAddress));
            limiterPromises.push(authRateLimiterConsecutiveFailsByUsernameAndIP.consume(usernameIpKey));
            limiterPromises.push(authRateLimiterSlowBruteByUsername.consume(username));
            await Promise.all(limiterPromises);
            return {
                success: true,
            };
        }
        catch (rlRejected) {
            if (rlRejected instanceof Error) {
                throw rlRejected;
            }
            else {
                return {
                    success: false,
                    retryInSecs: Math.round(rlRejected.msBeforeNext / 1000) || 60,
                };
            }
        }
    }
};
exports.authRateLimiter = authRateLimiter;
const authRateLimitReset = async (username, ipAddress) => {
    const usernameIpKey = getUsernameIPkey(username, ipAddress);
    const [resUsernameAndIP, resSlowByIP, resSlowUsername] = await Promise.all([
        authRateLimiterConsecutiveFailsByUsernameAndIP.get(usernameIpKey),
        authRateLimiterSlowBruteByIP.get(ipAddress),
        authRateLimiterSlowBruteByUsername.get(username),
    ]);
    if (resUsernameAndIP !== null && resUsernameAndIP.consumedPoints > 0) {
        await authRateLimiterConsecutiveFailsByUsernameAndIP.delete(usernameIpKey);
    }
    if (resSlowUsername !== null && resSlowUsername.consumedPoints > 0) {
        await authRateLimiterSlowBruteByUsername.delete(username);
    }
    if (resSlowByIP !== null && resSlowByIP.consumedPoints > 0) {
        await authRateLimiterSlowBruteByIP.delete(ipAddress);
    }
};
exports.authRateLimitReset = authRateLimitReset;
const rateLimiterMiddleware = async (req, res, next) => {
    let jwtPayload = null;
    try {
        let authHeader = req.header(app_constants_1.ArmonHeaders.Authorization);
        let bearerToken = authHeader.substr(7);
        jwtPayload = new app_auth_1.UserJwtPayload(jsonwebtoken_1.default.decode(bearerToken));
    }
    catch (err) {
        app_logs_1.logger.error(`Error while checking rate limits!`);
        app_logs_1.logger.error(err);
    }
    if (!jwtPayload || !jwtPayload.userId) {
        res.status(400).send("Bad Request");
        return;
    }
    const userId = jwtPayload.userId;
    try {
        const customRateLimiterForUser = userBasedRateLimiters.get(userId);
        if (customRateLimiterForUser) {
            await customRateLimiterForUser.consume(userId);
        }
        else {
            await genericRateLimiter.consume(userId);
        }
        next();
    }
    catch (rateLimiterRes) {
        let retrySecs = Math.round(rateLimiterRes.msBeforeNext / 1000) || 1;
        res.set("Retry-After", String(retrySecs));
        let message = i18n_1.default.__({ phrase: "ERRORS.GENERAL.TOO_MANY_REQUESTS", locale: req.locale ?? "tr" }, { waittime: Math.ceil(retrySecs / 60).toString() });
        res.status(app_enums_1.enums.HttpStatusCode.TOO_MANY_REQUEST).json({ message: message, noReport: true, showAsModal: true });
        return;
    }
};
exports.rateLimiterMiddleware = rateLimiterMiddleware;
