"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
const elastic_apm_node_1 = __importDefault(require("elastic-apm-node"));
if (process.env.ELASTIC_APM_SERVER_URL &&
    process.env.ELASTIC_APM_SECRET_TOKEN &&
    process.env.ELASTIC_APM_PREFIX &&
    ((process.env.ELASTIC_APM_VERIFY_SERVER_CERT === "true" && process.env.ELASTIC_APM_SERVER_CA_CERT_FILE) || process.env.ELASTIC_APM_VERIFY_SERVER_CERT === "false")) {
    elastic_apm_node_1.default.start({
        serviceName: `${process.env.ELASTIC_APM_PREFIX}-pacs`,
        serverUrl: process.env.ELASTIC_APM_SERVER_URL,
        secretToken: process.env.ELASTIC_APM_SECRET_TOKEN,
        environment: "production",
        serverCaCertFile: process.env.ELASTIC_APM_SERVER_CA_CERT_FILE ?? null,
        verifyServerCert: process.env.ELASTIC_APM_VERIFY_SERVER_CERT === "true",
    });
}
else {
    console.warn(`Elastic APM is not enabled. Please set ELASTIC_APM_SERVER_URL, ELASTIC_APM_SECRET_TOKEN, ELASTIC_APM_PREFIX, ELASTIC_APM_VERIFY_SERVER_CERT, and ELASTIC_APM_SERVER_CA_CERT_FILE environment variables.`);
}
const app_config_1 = require("./app.config");
const app_logs_1 = require("./app.logs");
const dal_manager_1 = require("./dal/dal.manager");
const messageBroker_manager_1 = require("./messageBroker/messageBroker.manager");
const moment = require("moment");
const employee_day_1 = require("./business/pacs2/employee.day");
const dal_constants_1 = require("./dal/dal.constants");
const business_hooks_1 = require("./business/business.hooks");
const messageBroker_server_to_pacs_sub_1 = require("./messageBroker/messageBroker.server-to-pacs.sub");
const app_constants_1 = require("./app.constants");
class AppPacs2Service {
    constructor() {
        this._mutexes = {};
        this._liveCache = {};
        this._liveRecalculationInterval = 60 * 1000;
        this._recalculatePeriod = 2 * 60 * 1000;
        this._recalculateStartTime = null;
        this._liveCounter = 0;
    }
    static main(args) {
        let app = new AppPacs2Service();
        app.start(args);
    }
    async start(args) {
        try {
            await app_config_1.appConfig.init(app_constants_1.ServiceNames.PACSService);
            app_logs_1.appLogger.init(app_config_1.appConfig.appLogDirectory);
            await dal_manager_1.dbManager.init({
                main: app_config_1.appConfig.db.main,
                log: app_config_1.appConfig.db.log,
                environment: app_config_1.appConfig.nodeEnv,
                logDirectory: app_config_1.appConfig.dbLogDirectory,
                redis: app_config_1.appConfig.db.redis,
            }, false, false, false, true);
            business_hooks_1.armonHookManager.init();
            await messageBroker_manager_1.messageBrokerManager.initForPacs2Server(this.onPacsLiveQueueMessage.bind(this));
            setTimeout(this.processRecalculationQueue.bind(this), 5 * 1 * 1000);
            await this.startScaffolding();
            this.generateMonthlyReportsForPastDates();
            app_logs_1.logger.info("armon-pacs2 is started");
        }
        catch (error) {
            app_logs_1.logger.error(error);
            process.exit(1);
        }
    }
    async processRecalculationQueue() {
        app_logs_1.logger.info("Triggered past recalculation");
        this._recalculateStartTime = moment();
        let monthlyReportsToInvalidate = [];
        try {
            let startOfToday = moment().startOf("day");
            let startOfThisMonth = moment().startOf("month");
            while (true) {
                let entry = await dal_manager_1.dbManager.accessPacs2.getFirstRecalculateTask();
                if (!entry) {
                    break;
                }
                let serials = [];
                let dateToProcess = moment(entry.startDate);
                while (dateToProcess.isBefore(startOfToday)) {
                    try {
                        let employeeDay = await employee_day_1.EmployeeDay.createEmployeeDay(entry.organizationId, entry.userId, dateToProcess.format("YYYY-MM-DD"), serials);
                        if (employeeDay) {
                            employeeDay.process();
                            let serialized = employeeDay.getSerialized();
                            serials.push(serialized);
                        }
                    }
                    catch (error) {
                        app_logs_1.logger.info(`Could not calculate ${dateToProcess.format("YYYY-MM-DD")} for ${entry.userId} because: ` + error);
                    }
                    dateToProcess.add(1, "day");
                }
                await dal_manager_1.dbManager.accessPacs2.insertEmployeeDaySummary(entry.userId, entry.organizationId, entry.startDate, serials, entry.id);
                let reportMonth = moment(entry.startDate).startOf("month");
                while (reportMonth.isBefore(startOfThisMonth)) {
                    let monthlyEntry = monthlyReportsToInvalidate.find((r) => r.organizationId === entry.organizationId && r.date.isSame(reportMonth, "month"));
                    if (!monthlyEntry) {
                        monthlyEntry = {
                            organizationId: entry.organizationId,
                            date: reportMonth.clone(),
                            users: [],
                        };
                        monthlyReportsToInvalidate.push(monthlyEntry);
                    }
                    monthlyEntry.users.push(entry.userId);
                    reportMonth.add(1, "month");
                }
            }
            for (const row of monthlyReportsToInvalidate) {
                app_logs_1.logger.info("Report " + row.organizationId + " " + row.date.format("YYYY-MM") + " will be invalidated, " + row.users.length + " users' data will be recalculated");
                await dal_manager_1.dbManager.accessPacs2.createOrInvalidateMonthlyReport(row.organizationId, row.date);
                for (const userId of row.users) {
                    await dal_manager_1.dbManager.accessPacs2.upsertUserMonthlyReport(row.organizationId, userId, row.date);
                }
            }
        }
        catch (error) {
            app_logs_1.logger.info("There was a problem recalculating past employee days: " + error);
        }
        let remaining = this._recalculatePeriod - moment().diff(this._recalculateStartTime, "milliseconds");
        app_logs_1.logger.info("There is " + remaining + " milliseconds before the next past recalculation");
        if (remaining > 0) {
            setTimeout(this.processRecalculationQueue.bind(this), remaining);
        }
        else {
            this.processRecalculationQueue();
        }
    }
    async onPacsLiveQueueMessage(msg, replyTo, correlationId) {
        let payload = msg.d;
        let totalUpdated = 0;
        const recordSetHandler = async (organizationId, userIds) => {
            let updated = [];
            for (const row of userIds) {
                if (this._liveCache[organizationId]) {
                    let dayInCache = this._liveCache[organizationId][row];
                    if (dayInCache) {
                        dayInCache.summarize();
                        dayInCache.serialize();
                        let serialized = dayInCache.getSerialized();
                        updated.push({ userId: row, employeeDayShort: serialized });
                    }
                }
            }
            totalUpdated += updated.length;
            if (updated.length > 0) {
                try {
                    await dal_manager_1.dbManager.accessPacs2.updateLiveEmployeeDays(organizationId, updated);
                }
                catch (error) {
                    app_logs_1.logger.error("Error while refreshing records for report request: " + error);
                }
            }
        };
        await dal_manager_1.dbManager.accessPacs2.mapFunctionToUpdateList(payload, recordSetHandler);
        app_logs_1.logger.info(`Refreshed ${totalUpdated} live records for report request`);
        (0, messageBroker_server_to_pacs_sub_1.replyPacsMessage)(replyTo, {}, correlationId);
    }
    async onLiveRecalculationTriggered(payload) {
        let liveid = this._liveCounter++;
        app_logs_1.logger.info("Live recalculation starting: " + liveid + " - Organization: " + (payload ? payload : "All"));
        let calculated = 0;
        let organizationList = [];
        if (payload) {
            organizationList = [payload];
        }
        else {
            organizationList = await dal_manager_1.dbManager.accessOrganization.listOrganizations();
        }
        for (const organization of organizationList) {
            if (this._mutexes[organization] === true) {
                app_logs_1.logger.info(liveid + " - This organization is already being processed, skipping.");
                continue;
            }
            else {
                this._mutexes[organization] = true;
            }
            let sessionId = null;
            try {
                sessionId = await dal_manager_1.dbManager.accessPacs2.markEmployeeLiveDaysForRecalculation(organization);
                if (sessionId) {
                    while (true) {
                        let daysToBeCalculated = await dal_manager_1.dbManager.accessPacs2.fetchEmployeeLiveDaysForRecalculation(organization, sessionId, 5);
                        if (daysToBeCalculated.length === 0) {
                            break;
                        }
                        for (const day of daysToBeCalculated) {
                            var dayInCache = undefined;
                            if (this._liveCache[day.organizationId]) {
                                dayInCache = this._liveCache[day.organizationId][day.userId];
                            }
                            var calculatedDay = undefined;
                            if (day.updateRequiredReason) {
                                if (day.updateRequiredReason & dal_constants_1.DalConstants.RecalculateWorkReason.Initialization ||
                                    day.updateRequiredReason & dal_constants_1.DalConstants.RecalculateWorkReason.Holiday ||
                                    day.updateRequiredReason & dal_constants_1.DalConstants.RecalculateWorkReason.ManuallyTriggered ||
                                    day.updateRequiredReason & dal_constants_1.DalConstants.RecalculateWorkReason.UserWorkPlanChange ||
                                    day.updateRequiredReason & dal_constants_1.DalConstants.RecalculateWorkReason.WorkPlanDefinitionChange ||
                                    !dayInCache) {
                                    calculatedDay = await employee_day_1.EmployeeDay.createEmployeeDay(day.organizationId, day.userId, moment(day.startAt).add(12, "hours").format("YYYY-MM-DD"));
                                    if (calculatedDay) {
                                        calculatedDay.setLive(true);
                                    }
                                }
                                else {
                                    calculatedDay = dayInCache;
                                    if (day.updateRequiredReason & dal_constants_1.DalConstants.RecalculateWorkReason.ManuelLogChange ||
                                        day.updateRequiredReason & dal_constants_1.DalConstants.RecalculateWorkReason.OfflineLogReceived ||
                                        day.updateRequiredReason & dal_constants_1.DalConstants.RecalculateWorkReason.NewAccessLog) {
                                        await calculatedDay.accessTimesUpdated();
                                    }
                                    if (day.updateRequiredReason & dal_constants_1.DalConstants.RecalculateWorkReason.PermissionChange) {
                                        await calculatedDay.permissionUpdated();
                                    }
                                }
                            }
                            if (calculatedDay) {
                                let now = moment();
                                calculatedDay.setProcessTime(now);
                                calculatedDay.process();
                                if (!this._liveCache[day.organizationId]) {
                                    this._liveCache[day.organizationId] = {};
                                }
                                this._liveCache[day.organizationId][day.userId] = calculatedDay;
                                await dal_manager_1.dbManager.accessPacs2.updateLiveEmployeeDay(organization, {
                                    id: day.id,
                                    employeeDayShort: calculatedDay.getSerialized(),
                                    time: now,
                                    sessionId: sessionId,
                                });
                                calculated++;
                            }
                            else {
                                app_logs_1.logger.error(`Calculated day for ${organization} - ${day.userId} at ${day.startAt}-${day.expiredAt} is empty.This is probably due to timezone mis-handling. Please review.`);
                                await dal_manager_1.dbManager.accessPacs2.deleteLiveDayEntry(organization, day.userId);
                            }
                        }
                    }
                }
            }
            catch (error) {
                app_logs_1.logger.error("There was a problem updating live days: " + error);
                if (!sessionId) {
                    app_logs_1.logger.error("There might be erroneous live day entries left in the database for session: " + sessionId);
                }
            }
            this._mutexes[organization] = false;
        }
        app_logs_1.logger.info("Recalculated " + calculated + " live records: " + liveid);
        app_logs_1.logger.info("Live recalculation completed: " + liveid);
    }
    async startScaffolding() {
        let organizations = await dal_manager_1.dbManager.accessOrganization.listOrganizations();
        for (const organization of organizations) {
            await dal_manager_1.dbManager.accessPacs2.scaffoldEmployeeLiveData(organization);
        }
        dal_manager_1.dbManager.accessPacs2.registerLiveRecalculationListener(this.onLiveRecalculationTriggered.bind(this));
        this._rescaffoldTimer = setTimeout(this.rescaffoldTimeout.bind(this), 5000);
    }
    async rescaffoldTimeout() {
        if (this._rescaffoldTimer) {
            clearTimeout(this._rescaffoldTimer);
            this._rescaffoldTimer = undefined;
        }
        app_logs_1.logger.info("Rescaffold and live update starting because of timeout");
        try {
            let organizations = await dal_manager_1.dbManager.accessOrganization.listOrganizations();
            let currentTime = moment();
            if (currentTime.hour() === 3 && currentTime.minute() === 10) {
                for (const organization of organizations) {
                    await dal_manager_1.dbManager.accessPacs2.mergeEmployeeWorkPlanMemberships(organization);
                }
            }
            for (const organization of organizations) {
                let purgedDays = await dal_manager_1.dbManager.accessPacs2.rescaffoldEmployeeLiveData(organization);
                for (const day of purgedDays) {
                    if (this._liveCache[day.organizationId]) {
                        let existingDay = this._liveCache[day.organizationId][day.userId];
                        if (existingDay) {
                            existingDay.setLive(false);
                            existingDay.setProcessTime(null);
                            existingDay.process();
                            let existingSerialized = existingDay.getSerialized();
                            await dal_manager_1.dbManager.accessPacs2.insertEmployeeDaySummary(day.userId, day.organizationId, existingSerialized.d, [existingSerialized], null);
                            delete this._liveCache[day.organizationId][day.userId];
                        }
                        else {
                            app_logs_1.logger.info("Expired live day is not found in the cache: " + day.organizationId + " " + day.userId);
                        }
                    }
                    else {
                        app_logs_1.logger.info("Organization for expired live day is not found in the cache: " + day.organizationId);
                    }
                }
            }
            await this.onLiveRecalculationTriggered();
        }
        catch (error) {
            app_logs_1.logger.error("There was a problem updating live days: " + error);
        }
        this._rescaffoldTimer = setTimeout(this.rescaffoldTimeout.bind(this), this._liveRecalculationInterval);
    }
    async generateMonthlyReportsForPastDates() {
        let nextTimeout = undefined;
        let orgIds = await dal_manager_1.dbManager.accessOrganization.listOrganizations();
        for (const oid of orgIds) {
            try {
                let nextReportDate = await dal_manager_1.dbManager.accessPacs2.getUnactedMonthlyReportDate(oid);
                for (const reportDate of nextReportDate) {
                    app_logs_1.logger.info("Generating report date is: " + reportDate.year + "-" + reportDate.month);
                    await dal_manager_1.dbManager.accessPacs2.generateMonthlyReports(oid, reportDate.year, reportDate.month);
                    app_logs_1.logger.info("Successfully generated reports for " + reportDate.year + "-" + reportDate.month);
                }
                nextTimeout = moment().startOf("day").add(1, "day").hour(4);
            }
            catch (error) {
                app_logs_1.logger.info("There was a problem generating monthly reports, Will trigger again in 10 minutes");
                nextTimeout = moment().add(10, "minutes");
            }
        }
        app_logs_1.logger.info("Next monthly report generation will trigger at " + nextTimeout.format("YYYY-MM-DD HH:mm:ss"));
        setTimeout(this.generateMonthlyReportsForPastDates.bind(this), moment.duration(nextTimeout.diff(moment())).asMilliseconds());
    }
}
AppPacs2Service.main(process.argv);
