"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.AppNotification = void 0;
const elastic_apm_node_1 = __importDefault(require("elastic-apm-node"));
if (process.env.ELASTIC_APM_SERVER_URL &&
    process.env.ELASTIC_APM_SECRET_TOKEN &&
    process.env.ELASTIC_APM_PREFIX &&
    ((process.env.ELASTIC_APM_VERIFY_SERVER_CERT === "true" && process.env.ELASTIC_APM_SERVER_CA_CERT_FILE) || process.env.ELASTIC_APM_VERIFY_SERVER_CERT === "false")) {
    elastic_apm_node_1.default.start({
        serviceName: `${process.env.ELASTIC_APM_PREFIX}-notification`,
        serverUrl: process.env.ELASTIC_APM_SERVER_URL,
        secretToken: process.env.ELASTIC_APM_SECRET_TOKEN,
        environment: "production",
        serverCaCertFile: process.env.ELASTIC_APM_SERVER_CA_CERT_FILE ?? null,
        verifyServerCert: process.env.ELASTIC_APM_VERIFY_SERVER_CERT === "true",
    });
}
else {
    console.warn(`Elastic APM is not enabled. Please set ELASTIC_APM_SERVER_URL, ELASTIC_APM_SECRET_TOKEN, ELASTIC_APM_PREFIX, ELASTIC_APM_VERIFY_SERVER_CERT, and ELASTIC_APM_SERVER_CA_CERT_FILE environment variables.`);
}
const cluster_1 = __importDefault(require("cluster"));
const app_config_1 = require("./app.config");
const app_constants_1 = require("./app.constants");
const app_logs_1 = require("./app.logs");
const business_hooks_1 = require("./business/business.hooks");
const dal_manager_1 = require("./dal/dal.manager");
const messageBroker_manager_1 = require("./messageBroker/messageBroker.manager");
const messageBroker_server_to_app_pub_1 = require("./messageBroker/messageBroker.server-to-app.pub");
const messageBroker_server_to_app_sub_1 = require("./messageBroker/messageBroker.server-to-app.sub");
const ws_usernsp_1 = require("./ws/ws.usernsp");
const firebase_1 = require("./business/notification/firebase");
class AppNotification {
    static main(args) {
        let app = new AppNotification();
        app.start(args);
    }
    constructor() { }
    async start(args) {
        try {
            app_config_1.appConfig.init(app_constants_1.ServiceNames.NotificationService);
            app_logs_1.appLogger.init(app_config_1.appConfig.appLogDirectory);
        }
        catch (error) {
            app_logs_1.logger.error(error);
            process.exit(1);
        }
        if (cluster_1.default.isMaster) {
            app_logs_1.logger.info(`app.notification master ${process.pid} is running`);
            for (let i = 0; i < app_config_1.appConfig.maxCpuCore; i++) {
                cluster_1.default.fork();
            }
            cluster_1.default.on("exit", (worker, code, signal) => {
                app_logs_1.logger.info(`app.notification worker ${worker.process.pid} died`);
            });
        }
        else {
            app_logs_1.logger.info(`app.notification Worker ${process.pid} started`);
            try {
                await dal_manager_1.dbManager.init({
                    main: app_config_1.appConfig.db.main,
                    log: app_config_1.appConfig.db.log,
                    environment: app_config_1.appConfig.nodeEnv,
                    ignoreMigrations: true,
                    logDirectory: app_config_1.appConfig.dbLogDirectory,
                    redis: app_config_1.appConfig.db.redis,
                }, false, false, true, false);
                business_hooks_1.armonHookManager.init();
                (0, firebase_1.initializeFirebase)();
                await messageBroker_manager_1.messageBrokerManager.initForNotificationService();
                await this.setAvailableMediums();
            }
            catch (error) {
                app_logs_1.logger.error(error);
                process.exit(1);
            }
        }
    }
    async setAvailableMediums() {
        let hasChange = false;
        const organizationIds = await dal_manager_1.dbManager.accessOrganization.listOrganizationPg();
        for (const organizationId of organizationIds) {
            const availableMediums = {
                email: false,
                pushNotification: false,
                sms: false,
                web: true,
            };
            await dal_manager_1.dbManager.systemTransaction(async (trx) => {
                const { notification } = await dal_manager_1.dbManager.accessOrganization.getOrganizationSettings({ organizationId, trx });
                if (notification.smsSettings) {
                    availableMediums.sms = true;
                }
                if (notification.smtpSettings && notification.smtpSettings.enabled) {
                    availableMediums.email = true;
                }
                if (app_config_1.appConfig.firebasePrivateKey) {
                    availableMediums.pushNotification = true;
                }
                if (availableMediums.email !== notification.availableMediums.email ||
                    availableMediums.sms !== notification.availableMediums.sms ||
                    availableMediums.pushNotification !== notification.availableMediums.pushNotification ||
                    availableMediums.web !== notification.availableMediums.web) {
                    hasChange = true;
                    await dal_manager_1.dbManager.accessOrganization.setOrganizationAvailableMedium({ organizationId, availableMediums, trx });
                }
            });
            if (hasChange) {
                messageBroker_server_to_app_pub_1.amqpServerToAppPub.sendToExchange(organizationId + "." + ws_usernsp_1.room.systemStatus + ".*", {
                    e: messageBroker_server_to_app_sub_1.amqpServerToAppSubUserEventNames.organizationSettingsChanged,
                    p: {},
                });
            }
        }
    }
}
exports.AppNotification = AppNotification;
AppNotification.main(process.argv);
