"use strict";
var __createBinding = (this && this.__createBinding) || (Object.create ? (function(o, m, k, k2) {
    if (k2 === undefined) k2 = k;
    var desc = Object.getOwnPropertyDescriptor(m, k);
    if (!desc || ("get" in desc ? !m.__esModule : desc.writable || desc.configurable)) {
      desc = { enumerable: true, get: function() { return m[k]; } };
    }
    Object.defineProperty(o, k2, desc);
}) : (function(o, m, k, k2) {
    if (k2 === undefined) k2 = k;
    o[k2] = m[k];
}));
var __setModuleDefault = (this && this.__setModuleDefault) || (Object.create ? (function(o, v) {
    Object.defineProperty(o, "default", { enumerable: true, value: v });
}) : function(o, v) {
    o["default"] = v;
});
var __importStar = (this && this.__importStar) || function (mod) {
    if (mod && mod.__esModule) return mod;
    var result = {};
    if (mod != null) for (var k in mod) if (k !== "default" && Object.prototype.hasOwnProperty.call(mod, k)) __createBinding(result, mod, k);
    __setModuleDefault(result, mod);
    return result;
};
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.authManager = exports.AuthManager = exports.AccessTokenErrorType = exports.GrantTypes = void 0;
const crypto_1 = __importDefault(require("crypto"));
const express_1 = __importDefault(require("express"));
const express_ntlm_1 = __importDefault(require("express-ntlm"));
const fs_1 = __importDefault(require("fs"));
const handlebars_1 = __importDefault(require("handlebars"));
const https_1 = __importDefault(require("https"));
const i18n_1 = __importDefault(require("i18n"));
const jsonwebtoken_1 = __importDefault(require("jsonwebtoken"));
const lodash_1 = __importDefault(require("lodash"));
const openid_client_1 = require("openid-client");
const passport_1 = __importDefault(require("passport"));
const passport_oauth2_1 = __importDefault(require("passport-oauth2"));
const path_1 = __importDefault(require("path"));
const saml = __importStar(require("saml2-js"));
const url_1 = __importDefault(require("url"));
const xml2js = __importStar(require("xml2js"));
const api_error_1 = require("./api/api.error");
const api_validatorhelper_1 = require("./api/api.validatorhelper");
const app_auth_1 = require("./app.auth");
const app_config_1 = require("./app.config");
const app_constants_1 = require("./app.constants");
const app_enums_1 = require("./app.enums");
const app_logs_1 = require("./app.logs");
const app_middleware_1 = require("./app.middleware");
const app_rate_limiter_1 = require("./app.rate-limiter");
const dal_access_error_1 = require("./dal/access/dal.access.error");
const dal_constants_1 = require("./dal/dal.constants");
const dal_manager_1 = require("./dal/dal.manager");
const dal_db_armon_schema_1 = require("./dal/db/armon/dal.db.armon.schema");
const predefined_oAuthClients_1 = require("./dal/db/predefined/predefined.oAuthClients");
const dal_memcache_1 = require("./dal/access/dal.memcache");
const messageBroker_notification_pub_1 = require("./messageBroker/messageBroker.notification.pub");
var LdapStrategy = require("passport-ldapauth");
const asyncHandler = (fn) => (req, res, next) => Promise.resolve(fn(req, res, next)).catch(next);
var GrantTypes;
(function (GrantTypes) {
    GrantTypes[GrantTypes["password"] = 0] = "password";
    GrantTypes[GrantTypes["refresh_token"] = 1] = "refresh_token";
})(GrantTypes = exports.GrantTypes || (exports.GrantTypes = {}));
var AccessTokenErrorType;
(function (AccessTokenErrorType) {
    AccessTokenErrorType[AccessTokenErrorType["invalid_request"] = 0] = "invalid_request";
    AccessTokenErrorType[AccessTokenErrorType["invalid_client"] = 1] = "invalid_client";
    AccessTokenErrorType[AccessTokenErrorType["invalid_grant"] = 2] = "invalid_grant";
    AccessTokenErrorType[AccessTokenErrorType["unauthorized_client"] = 3] = "unauthorized_client";
    AccessTokenErrorType[AccessTokenErrorType["unsupported_grant_type"] = 4] = "unsupported_grant_type";
    AccessTokenErrorType[AccessTokenErrorType["invalid_scope"] = 5] = "invalid_scope";
    AccessTokenErrorType[AccessTokenErrorType["mobile_device_mismatch"] = 6] = "mobile_device_mismatch";
    AccessTokenErrorType[AccessTokenErrorType["missing_refresh_token"] = 7] = "missing_refresh_token";
    AccessTokenErrorType[AccessTokenErrorType["missing_organization_id"] = 8] = "missing_organization_id";
})(AccessTokenErrorType = exports.AccessTokenErrorType || (exports.AccessTokenErrorType = {}));
async function checkNTLMForOrganizationAlias(organizationAuthMethods, req, res, next) {
    try {
        if (organizationAuthMethods.length > 0 &&
            lodash_1.default.uniq(organizationAuthMethods.map((o) => o.organizationId)).length == 1 &&
            organizationAuthMethods.some((a) => a.authenticationMethod == app_enums_1.enums.AuthMethod.NTLM)) {
            let ntlmAuth = organizationAuthMethods.find((a) => a.authenticationMethod == app_enums_1.enums.AuthMethod.NTLM);
            if (!ntlmAuth || !ntlmAuth.config) {
                return Promise.resolve(false);
            }
            let ntlmOptions = ntlmAuth.config;
            if (ntlmOptions.enabled) {
                if (!ntlmOptions.domain || !ntlmOptions.domaincontroller) {
                    return Promise.resolve(false);
                }
                let ntlmTls = undefined;
                if (ntlmOptions.tlsOptions) {
                    let ca = fs_1.default.readFileSync(ntlmOptions.tlsOptions.caFilePath);
                    ntlmTls = {
                        ca: ca,
                        rejectUnauthorized: ntlmOptions.tlsOptions.rejectUnauthorized,
                    };
                }
                let _ntlmMiddleWare = (0, express_ntlm_1.default)({
                    debug: ntlmOptions.debug
                        ? function () {
                            var args = Array.prototype.slice.apply(arguments);
                            console.log.apply(null, args);
                        }
                        : undefined,
                    domain: ntlmOptions.domain,
                    tlsOptions: ntlmTls,
                    domaincontroller: ntlmOptions.domaincontroller,
                    forbidden: function (request, response, next) {
                        next();
                    },
                    badrequest: function (request, response, next) {
                        next();
                    },
                    internalservererror: function (request, response, next) {
                        next();
                    },
                });
                await (0, app_middleware_1.invokeMiddleware)([_ntlmMiddleWare], req, res);
                let hasNtlm = req.ntlm;
                if (hasNtlm && hasNtlm.Authenticated) {
                    return Promise.resolve(true);
                }
            }
            return Promise.resolve(false);
        }
    }
    catch (error) {
        app_logs_1.logger.error("ntlm control error " + error);
    }
    return Promise.resolve(false);
}
class AuthManager {
    constructor() {
        this._router = express_1.default.Router();
    }
    async init(app) {
        (0, app_rate_limiter_1.initLimiters)();
        this._urlencodedParser = express_1.default.urlencoded({ extended: false });
        await dal_manager_1.dbManager.systemTransaction(async (trx) => {
            await this.initOauthMethods(trx);
            await this.initLdapMethods(trx);
            await this.initSAMLMethods(trx);
        });
        this._router.use(passport_1.default.initialize());
        this._router.post("/usernamepass/:grantTypeId", this._urlencodedParser, asyncHandler(await AuthManager.authUser(app_enums_1.enums.AuthGrantType.UsernamePassword)));
        this._router.get("/usernamepass/:grantTypeId", this._urlencodedParser, asyncHandler(await AuthManager.authUser(app_enums_1.enums.AuthGrantType.UsernamePassword)));
        this._router.post("/sso/:grantTypeId", this._urlencodedParser, asyncHandler(await AuthManager.authUser(app_enums_1.enums.AuthGrantType.SSO)));
        this._router.get("/sso/:grantTypeId", this._urlencodedParser, asyncHandler(await AuthManager.authUser(app_enums_1.enums.AuthGrantType.SSO)));
        this._router.post("/mobile/sso/:grantTypeId", this._urlencodedParser, asyncHandler(await AuthManager.authUser(app_enums_1.enums.AuthGrantType.SSO, true)));
        this._router.get("/mobile/sso/:grantTypeId", this._urlencodedParser, asyncHandler(await AuthManager.authUser(app_enums_1.enums.AuthGrantType.SSO, true)));
        this._router.get("/check", this._urlencodedParser, (req, res) => {
            return res.sendStatus(app_enums_1.enums.HttpStatusCode.SUCCESS);
        });
        this._router.post("/info/", this._urlencodedParser, async function (req, res, next) {
            try {
                await dal_manager_1.dbManager.systemTransaction(async (trx) => {
                    if (req.body && req.body.alias) {
                        let organizationAuthMethods = await dal_manager_1.dbManager.accessOrganization.searchOrganizationAuthByAliasOrName(req.body.alias, trx);
                        if (organizationAuthMethods.length > 0) {
                            let ntlmCheck = await checkNTLMForOrganizationAlias(organizationAuthMethods[0].authentications, req, res, next);
                            if (ntlmCheck) {
                                let ntlm = req.ntlm;
                                let user = await app_auth_1.authHelper.authenticateWithUsername(organizationAuthMethods[0].id, res, ntlm.UserName, trx);
                                let scopes = req.query.scope ? req.query.scope.split(" ") : undefined;
                                let token = await app_auth_1.authHelper.generateToken(req, { user: user, clientId: undefined, scopes: scopes, orgIds: [organizationAuthMethods[0].id], custom: undefined }, trx);
                                return res.json({
                                    session: {
                                        tokenId: token.tokenId,
                                        token: token.access_token,
                                        expiresIn: token.expires_in,
                                        refreshToken: token.refresh_token,
                                        organization: user.organizations.find((o) => o.organization.id == organizationAuthMethods[0].id),
                                    },
                                    organizations: [
                                        {
                                            id: organizationAuthMethods[0].id,
                                            name: organizationAuthMethods[0].name,
                                            code: organizationAuthMethods[0].code,
                                            alias: organizationAuthMethods[0].alias,
                                            logo: organizationAuthMethods[0].logo,
                                            authentications: organizationAuthMethods[0].authentications.filter((a) => a.authenticationMethod == app_enums_1.enums.AuthMethod.NTLM),
                                        },
                                    ],
                                });
                            }
                            else {
                                let result = {
                                    organizations: organizationAuthMethods,
                                };
                                return res.json(result);
                            }
                        }
                        else {
                            return res.status(app_enums_1.enums.HttpStatusCode.NOT_FOUND).json({
                                message: "No authentication method found for alias " + req.body.alias,
                            });
                        }
                    }
                    else {
                        if (req.cookies &&
                            req.cookies.xopenidaccesstoken &&
                            req.cookies.xopenididtoken &&
                            req.cookies.xopenidrefreshtoken &&
                            req.cookies.xopenidusername &&
                            req.cookies.xopenidgrantid) {
                            let grantTypeId = (0, api_validatorhelper_1.validateUUID)(req.cookies.xopenidgrantid, "grantTypeId");
                            let authMethod = await dal_manager_1.dbManager.accessOrganization.getAuthenticationMethodById(grantTypeId, trx);
                            const openIdConfig = authMethod.config;
                            const result = await AuthManager.authenticateOpenID(authMethod.organizationId, req, res, openIdConfig, trx);
                            if (result?.responseStatus === app_enums_1.enums.HttpStatusCode.SUCCESS && result?.body?.tokenId) {
                                res.json({
                                    session: {
                                        tokenId: result.body.tokenId,
                                        token: result.body.token,
                                        expiresIn: result.body.expiresIn,
                                        refreshToken: result.body.refreshToken,
                                        organization: result.body.organization,
                                    },
                                    organizations: [
                                        {
                                            id: authMethod.organizationId,
                                            name: authMethod.orgName,
                                            code: authMethod.orgCode,
                                            alias: authMethod.orgAlias,
                                            logo: authMethod.orgLogo,
                                            authentications: [
                                                {
                                                    id: authMethod.id,
                                                    isDefault: authMethod.isDefault,
                                                    name: authMethod.name,
                                                    grantType: authMethod.grantType,
                                                    organizationId: authMethod.organizationId,
                                                    authenticationMethod: authMethod.authenticationMethod,
                                                },
                                            ],
                                        },
                                    ],
                                });
                            }
                            return;
                        }
                        else if (req.cookies && req.cookies.xsamluser && req.cookies.xsamlhash && req.cookies.xsamlgrantid) {
                            let grantTypeId = (0, api_validatorhelper_1.validateUUID)(req.cookies.xsamlgrantid, "grantTypeId");
                            let authMethod = await dal_manager_1.dbManager.accessOrganization.getAuthenticationMethodById(grantTypeId, trx);
                            const samlConfig = authMethod.config;
                            const result = await AuthManager.authenticateSAML(authMethod.organizationId, req, res, next, samlConfig, trx);
                            if (result?.responseStatus === app_enums_1.enums.HttpStatusCode.SUCCESS && result?.body?.tokenId) {
                                res.json({
                                    session: {
                                        tokenId: result.body.tokenId,
                                        token: result.body.token,
                                        expiresIn: result.body.expiresIn,
                                        refreshToken: result.body.refreshToken,
                                        organization: result.body.organization,
                                    },
                                    organizations: [
                                        {
                                            id: authMethod.organizationId,
                                            name: authMethod.orgName,
                                            code: authMethod.orgCode,
                                            alias: authMethod.orgAlias,
                                            logo: authMethod.orgLogo,
                                            authentications: [
                                                {
                                                    id: authMethod.id,
                                                    isDefault: authMethod.isDefault,
                                                    name: authMethod.name,
                                                    grantType: authMethod.grantType,
                                                    organizationId: authMethod.organizationId,
                                                    authenticationMethod: authMethod.authenticationMethod,
                                                },
                                            ],
                                        },
                                    ],
                                });
                            }
                            return;
                        }
                        else if (req.body && req.body.accessToken && req.body.refreshToken && req.body.idToken && req.body.grantTypeId) {
                            let grantTypeId = (0, api_validatorhelper_1.validateUUID)(req.body.grantTypeId, "grantTypeId");
                            let authMethod = await dal_manager_1.dbManager.accessOrganization.getAuthenticationMethodById(grantTypeId, trx);
                            const openIdClientConfig = authMethod.config;
                            req.body.organizationId = authMethod.organizationId;
                            const result = await AuthManager.authenticateOpenIdByClientSide(req, res, openIdClientConfig, trx);
                            if (result?.responseStatus === app_enums_1.enums.HttpStatusCode.SUCCESS && result?.body?.tokenId) {
                                res.json({
                                    session: {
                                        tokenId: result.body.tokenId,
                                        token: result.body.token,
                                        expiresIn: result.body.expiresIn,
                                        refreshToken: result.body.refreshToken,
                                        organization: result.body.organization,
                                    },
                                    organizations: [
                                        {
                                            id: authMethod.organizationId,
                                            name: authMethod.orgName,
                                            code: authMethod.orgCode,
                                            alias: authMethod.orgAlias,
                                            logo: authMethod.orgLogo,
                                            authentications: [
                                                {
                                                    id: authMethod.id,
                                                    isDefault: authMethod.isDefault,
                                                    name: authMethod.name,
                                                    grantType: authMethod.grantType,
                                                    organizationId: authMethod.organizationId,
                                                    authenticationMethod: authMethod.authenticationMethod,
                                                },
                                            ],
                                        },
                                    ],
                                });
                            }
                            return;
                        }
                        if (app_config_1.appConfig.isCloudServer && app_config_1.appConfig.nodeEnv !== "development") {
                            let authHeader = req.header(app_constants_1.ArmonHeaders.Authorization);
                            if (!authHeader) {
                                return res.json({
                                    organizations: null,
                                });
                            }
                        }
                        let showHidden = req.body.showHidden == "true" || false;
                        let authMethods = await dal_manager_1.dbManager.accessOrganization.listAllOrganizationAuthMethods(trx, showHidden);
                        let availableAuths = [];
                        for (const organization of authMethods) {
                            for (const authMethod of organization.authentications) {
                                if (authMethod.authenticationMethod === dal_constants_1.DalConstants.AuthenticationMethod.LDAP ||
                                    authMethod.authenticationMethod === dal_constants_1.DalConstants.AuthenticationMethod.OPENID ||
                                    authMethod.authenticationMethod === dal_constants_1.DalConstants.AuthenticationMethod.SAML) {
                                    delete authMethod.config;
                                }
                                availableAuths.push(authMethod);
                            }
                        }
                        if (availableAuths.length == 1 || availableAuths[0].authenticationMethod == app_enums_1.enums.AuthMethod.CAS) {
                            let targetOrganization = authMethods.find((s) => s.authentications.some((a) => a.id == availableAuths[0].id));
                            let result = {
                                organizations: [
                                    {
                                        id: targetOrganization.id,
                                        name: targetOrganization.name,
                                        logo: targetOrganization.logo,
                                        code: targetOrganization.code,
                                        alias: targetOrganization.alias,
                                        authentications: [availableAuths[0]],
                                    },
                                ],
                            };
                            res.json(result);
                        }
                        else {
                            let result = {
                                organizations: authMethods,
                            };
                            res.json(result);
                        }
                    }
                });
            }
            catch (error) {
                app_logs_1.logger.error("auth info error " + error);
                app_logs_1.logger.error(error);
                res.status(app_enums_1.enums.HttpStatusCode.NOT_FOUND).json({
                    message: error,
                });
            }
        });
        this._router.post("/user", this._urlencodedParser, async function (req, res, next) {
            if (!req.body.username) {
                return res.status(app_enums_1.enums.HttpStatusCode.NOT_FOUND).json({
                    error: "null username",
                });
            }
            const rateLimitCheck = await (0, app_rate_limiter_1.authRateLimiter)(req.body.username ?? req.ntlm.UserName, req.socket.remoteAddress);
            if (!rateLimitCheck.success) {
                res.set("Retry-After", rateLimitCheck.retryInSecs?.toString());
                let message = i18n_1.default.__({ phrase: "ERRORS.GENERAL.TOO_MANY_REQUESTS", locale: req.locale ?? "tr" }, { waittime: Math.ceil(rateLimitCheck.retryInSecs / 60).toString() });
                res.status(app_enums_1.enums.HttpStatusCode.TOO_MANY_REQUEST).json({ message: message, noReport: true, showAsModal: true });
                return;
            }
            await dal_manager_1.dbManager.systemTransaction(async (trx) => {
                let userOrganizationAuthMethods = await dal_manager_1.dbManager.accessUser.listUserOrganizationAuthMethods(req.body.username, req.body.showHidden === "true", trx);
                if (userOrganizationAuthMethods.length > 0) {
                    for (const uoam of userOrganizationAuthMethods) {
                        for (const authMethod of uoam.authentications) {
                            if (authMethod.authenticationMethod === dal_constants_1.DalConstants.AuthenticationMethod.LDAP ||
                                authMethod.authenticationMethod === dal_constants_1.DalConstants.AuthenticationMethod.OPENID ||
                                authMethod.authenticationMethod === dal_constants_1.DalConstants.AuthenticationMethod.SAML) {
                                delete authMethod.config;
                            }
                        }
                    }
                }
                if (userOrganizationAuthMethods.some((s) => s.authentications.some((n) => n.authenticationMethod == app_enums_1.enums.AuthMethod.NTLM))) {
                    let authOrg = userOrganizationAuthMethods.find((s) => s.authentications.some((n) => n.authenticationMethod == app_enums_1.enums.AuthMethod.NTLM));
                    let ntlmCheck = await checkNTLMForOrganizationAlias(authOrg.authentications, req, res, next);
                    if (ntlmCheck) {
                        let ntlm = req.ntlm;
                        let user = await app_auth_1.authHelper.authenticateWithUsername(authOrg.id, res, ntlm.UserName, trx);
                        let scopes = req.query.scope ? req.query.scope.split(" ") : undefined;
                        let token = await app_auth_1.authHelper.generateToken(req, { user: user, clientId: undefined, scopes: scopes, orgIds: [authOrg.id], custom: undefined }, trx);
                        await (0, app_rate_limiter_1.authRateLimitReset)(ntlm.UserName, req.socket.remoteAddress);
                        return res.json({
                            session: {
                                tokenId: token.tokenId,
                                token: token.access_token,
                                expiresIn: token.expires_in,
                                refreshToken: token.refresh_token,
                                organization: user.organizations.find((o) => o.organization.id == authOrg.id),
                            },
                            organizations: [
                                {
                                    id: authOrg.id,
                                    name: authOrg.name,
                                    code: authOrg.code,
                                    alias: authOrg.alias,
                                    logo: authOrg.logo,
                                    authentications: authOrg.authentications.filter((a) => a.authenticationMethod == app_enums_1.enums.AuthMethod.NTLM),
                                },
                            ],
                        });
                    }
                }
                if (userOrganizationAuthMethods.length) {
                    await (0, app_rate_limiter_1.authRateLimitReset)(req.body.username ?? req.ntlm.UserName, req.socket.remoteAddress);
                }
                return res.json({ organizations: userOrganizationAuthMethods });
            });
        });
        this._router.post("/refresh", this._urlencodedParser, async (req, res, next) => {
            try {
                if (!req.body || !req.body.refreshToken) {
                    return res.status(app_enums_1.enums.HttpStatusCode.BAD_REQUEST).json({
                        error: AccessTokenErrorType[AccessTokenErrorType.unsupported_grant_type],
                        error_description: req.body.grant_type,
                    });
                }
                let authHeader = req.header(app_constants_1.ArmonHeaders.Authorization);
                if (!authHeader) {
                    return res.redirect(app_enums_1.enums.HttpStatusCode.SUCCESS, app_config_1.appConfig.hostBaseRedirectUrl);
                }
                let headerTokens = authHeader.split(" ");
                if (headerTokens[0] != "Bearer") {
                    return res.status(app_enums_1.enums.HttpStatusCode.BAD_REQUEST).json({
                        error: AccessTokenErrorType[AccessTokenErrorType.invalid_request],
                        error_description: "no bearer",
                    });
                }
                if (!req.body.refreshToken) {
                    return res.status(app_enums_1.enums.HttpStatusCode.BAD_REQUEST).json({
                        error: AccessTokenErrorType[AccessTokenErrorType.missing_refresh_token],
                        error_description: req.body.refreshToken,
                    });
                }
                if (!req.body.organizationId) {
                    return res.status(app_enums_1.enums.HttpStatusCode.BAD_REQUEST).json({
                        error: AccessTokenErrorType[AccessTokenErrorType.missing_organization_id],
                        error_description: req.body.organizationId,
                    });
                }
                await dal_manager_1.dbManager.systemTransaction(async (trx) => {
                    let clientSpecified = req.query.client_id !== undefined;
                    let client = null;
                    if (clientSpecified) {
                        client = await AuthManager.validateClientForTokenRequest(req, res, trx);
                        if (res.finished) {
                            return;
                        }
                    }
                    const userId = await dal_manager_1.dbManager.accessOAuth.getUserIdFromRefreshToken(req.body.organizationId, req.body.refreshToken, trx);
                    if (userId) {
                        const mobileInstallIdCheckResult = await app_auth_1.authHelper.checkMobileDeviceMismatch(req.body.organizationId, {
                            userAgent: req.header(app_constants_1.ArmonHeaders.UserAgent),
                            mobileInstallId: req.header(app_constants_1.ArmonHeaders.MobileInstallId),
                            userId,
                        }, trx);
                        if (!mobileInstallIdCheckResult) {
                            return res.status(app_enums_1.enums.HttpStatusCode.BAD_REQUEST).json({
                                error: AccessTokenErrorType[AccessTokenErrorType.mobile_device_mismatch],
                                error_description: req.header(app_constants_1.ArmonHeaders.MobileInstallId),
                            });
                        }
                    }
                    let customData = null;
                    if (req.cookies.xopenidrefreshtoken && req.cookies.xopenidgrantid && req.cookies.xopenidgrantid !== "undefined") {
                        const openIdConfig = (await dal_manager_1.dbManager.accessOrganization.getAuthenticationMethodById(req.cookies.xopenidgrantid, trx)).config;
                        const openIdIssuer = await openid_client_1.Issuer.discover(openIdConfig.wellKnownEndpoint);
                        const client = new openIdIssuer.Client({
                            client_id: openIdConfig.clientId,
                            client_secret: openIdConfig.clientSecret,
                            redirect_uris: openIdConfig.redirectUris,
                            response_types: openIdConfig.responseTypes,
                        });
                        let refreshedToken = null;
                        try {
                            refreshedToken = await client.refresh(req.cookies.xopenidrefreshtoken);
                        }
                        catch (error) {
                            app_logs_1.logger.error(`Error while refresh token for user with OpenID at OpenID provider!`);
                            app_logs_1.logger.error(error);
                            invalidateOpenIdCookies(res);
                            throw error;
                        }
                        const userinfo = await client.userinfo(refreshedToken.access_token);
                        const accessTokenDecoded = jsonwebtoken_1.default.decode(refreshedToken.access_token, { complete: true }).payload;
                        const idTokenDecoded = jsonwebtoken_1.default.decode(refreshedToken.id_token, { complete: true }).payload;
                        if (userinfo.sub !== idTokenDecoded["sub"]) {
                            throw new Error("Malformed user info.");
                        }
                        setOpenIdCookies(req.cookies.xopenidgrantid, res, refreshedToken, accessTokenDecoded, openIdConfig.binding.externalUserInfoKey);
                        customData = app_auth_1.authHelper.calculateCustomTokenExpiration(refreshedToken.access_token, refreshedToken.refresh_token);
                    }
                    if (req.body.openIdAccessToken && req.body.openIdRefreshToken && req.body.grantTypeId) {
                        const openIdClientConfig = (await dal_manager_1.dbManager.accessOrganization.getAuthenticationMethodById(req.body.grantTypeId, trx)).config;
                        if (!openIdClientConfig) {
                            res.status(app_enums_1.enums.HttpStatusCode.NOT_FOUND).send(`Authentication Method not found`);
                            return;
                        }
                        if (!userId) {
                            app_logs_1.logger.error(`User Id cannot be obtained from refreshToken ${req.body.refreshToken} for organization with id ${req.body.organizationId}`);
                            app_logs_1.logger.error(`Client probably lost new refresh token, therefor unable to obtain access token with (probably) old refresh token`);
                            res.status(app_enums_1.enums.HttpStatusCode.NOT_FOUND).send(`Refresh token not found`);
                            return;
                        }
                        const accessTokenOpen = jsonwebtoken_1.default.decode(req.body.openIdAccessToken, { complete: true });
                        const externalValueToMatch = accessTokenOpen.payload[openIdClientConfig.binding.externalUserInfoKey]?.toLowerCase();
                        const binding = [];
                        let qCondition = ` u.id = $1 AND `;
                        binding.push(userId);
                        if (openIdClientConfig.binding.isExtension) {
                            qCondition += `unaccent(lower(uop."extensionFields"->>'${openIdClientConfig.binding.internalPropertyToMatch}')) = unaccent(lower($${binding.length + 1}))`;
                            binding.push(externalValueToMatch);
                        }
                        else {
                            if (openIdClientConfig.binding.internalPropertyToMatch === "username") {
                                qCondition += `unaccent(lower(u."${openIdClientConfig.binding.internalPropertyToMatch}")) = unaccent(lower($${binding.length + 1}))`;
                                binding.push(externalValueToMatch);
                            }
                            else {
                                qCondition += `unaccent(lower(uop."${openIdClientConfig.binding.internalPropertyToMatch}")) = unaccent(lower($${binding.length + 1}))`;
                                binding.push(externalValueToMatch);
                            }
                        }
                        const { rowCount } = await trx.query(`SELECT u.id FROM "${req.body.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.users}" u
								INNER JOIN "${req.body.organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userOrganizationProfiles}" uop
							ON uop."userId" = u.id
							WHERE ${qCondition}`, binding);
                        if (rowCount) {
                            customData = app_auth_1.authHelper.calculateCustomTokenExpiration(req.body.openIdAccessToken, req.body.openIdRefreshToken, -30, -30);
                        }
                        else {
                            app_logs_1.logger.error(`Intruder Alert! Open ID Access Token and provided user name does not match while refresh,
accessToken: ${req.body.openIdAccessToken}, accessTokenOpen: ${JSON.stringify(accessTokenOpen, null, 4)},
userId: ${userId}, external user value from token: ${accessTokenOpen.payload[openIdClientConfig.binding.externalUserInfoKey]}`);
                            res.status(app_enums_1.enums.HttpStatusCode.UNAUTHORIZED_ACCESS).send({ message: "Intruder Alert!" });
                            return;
                        }
                    }
                    let refreshResult = await app_auth_1.authHelper.refreshToken(req, {
                        refreshToken: req.body.refreshToken,
                        authToken: headerTokens[1],
                        isMobileApplication: req.header(app_constants_1.ArmonHeaders.UserAgent).startsWith("mobilemanager"),
                        clientId: client ? client.id : undefined,
                        orgIds: [req.body.organizationId],
                        custom: customData,
                    }, trx);
                    let user = refreshResult.user;
                    let token = refreshResult.token;
                    return res.json({
                        tokenId: token.tokenId,
                        token: token.access_token,
                        expiresIn: token.expires_in,
                        refreshToken: token.refresh_token,
                        organization: user.organizations.find((o) => o.organization.id == req.body.organizationId),
                    });
                });
            }
            catch (error) {
                if (typeof error === "object") {
                    app_logs_1.logger.error(JSON.stringify(error, null, 4));
                }
                else {
                    app_logs_1.logger.error(error);
                }
                app_logs_1.logger.error(`Error while refreshing token`);
                app_logs_1.logger.error(error?.message ?? error?.body?.message ?? error);
                return res.status(error?.statusCode ?? error?.errorCode ?? 500).json({
                    error: AccessTokenErrorType[AccessTokenErrorType.invalid_grant],
                    error_description: error?.body?.message ?? error?.message ?? error,
                });
            }
        });
        this._router.post("/verifyToken/:grantTypeId", this._urlencodedParser, async (req, res, next) => {
            if (!req.body.username) {
                return res.status(app_enums_1.enums.HttpStatusCode.NOT_FOUND).json({
                    error: "null username",
                });
            }
            if (!req.body.token) {
                return res.status(app_enums_1.enums.HttpStatusCode.NOT_FOUND).json({
                    error: "token required",
                });
            }
            let grantTypeId = (0, api_validatorhelper_1.validateUUID)(req.params.grantTypeId, "grantTypeId");
            let authMethod = null;
            await dal_manager_1.dbManager.systemTransaction(async (trx) => {
                authMethod = await dal_manager_1.dbManager.accessOrganization.getAuthenticationMethodById(grantTypeId, trx);
            });
            if (authMethod?.organizationId) {
                const organizationId = authMethod.organizationId;
                req.body.organizationId = authMethod.organizationId;
                const user = await dal_manager_1.dbManager.armondb.withSchema(organizationId).from("users").whereRaw("lower(username) = lower(?)", req.body.username).first();
                let userSettings = await dal_manager_1.dbManager.organizationTransaction(async (trx) => {
                    const userSettings = await dal_manager_1.dbManager.accessUser.getUserSettings({ userId: user.id, organizationId, trx });
                    return userSettings;
                }, user.id, organizationId);
                const getUsersStoredVerificationToken = await dal_manager_1.dbManager.accessRedisCache.getVerificationToken(req.body.username);
                if (!getUsersStoredVerificationToken) {
                    return res.status(app_enums_1.enums.MultiFactorHttpsStatusCodes.TOKEN_EXPIRED).json({
                        errorMessage: i18n_1.default.__({ phrase: "ERRORS.AUTH.TOKEN_EXPIRED", locale: userSettings.locale }),
                    });
                }
                if (getUsersStoredVerificationToken.b) {
                    return res.status(app_enums_1.enums.MultiFactorHttpsStatusCodes.BLOCKED_USER).json({
                        errorMessage: i18n_1.default.__({ phrase: "ERRORS.AUTH.BLOCKED_USER", locale: userSettings.locale }),
                    });
                }
                if (getUsersStoredVerificationToken.t === parseInt(req.body.token) && !getUsersStoredVerificationToken.b) {
                    let armonApplicationUser = null;
                    switch (authMethod.authenticationMethod) {
                        case app_enums_1.enums.AuthMethod.LDAP:
                            await dal_manager_1.dbManager.systemTransaction(async (trx) => {
                                armonApplicationUser = await app_auth_1.authHelper.authenticateWithUsername(req.body.organizationId, res, req.body.username, trx);
                                return await AuthManager.authenticateToken(req, res, next, trx, armonApplicationUser);
                            });
                            break;
                        case app_enums_1.enums.AuthMethod.Native:
                            await dal_manager_1.dbManager.systemTransaction(async (trx) => {
                                armonApplicationUser = await app_auth_1.authHelper.authenticateWithPasswordCredentialsByUrlEncodedRequest(req, res, trx);
                                return await AuthManager.authenticateToken(req, res, next, trx, armonApplicationUser);
                            });
                            break;
                        default:
                            res.status(app_enums_1.enums.HttpStatusCode.FORBIDDEN).json({
                                errorMessage: "MFA is not available for SSO methods.",
                            });
                            break;
                    }
                }
                else {
                    getUsersStoredVerificationToken.nfa += 1;
                    await dal_manager_1.dbManager.accessRedisCache.setVerificationToken(req.body.username, { ...getUsersStoredVerificationToken, ets: undefined });
                    if (getUsersStoredVerificationToken.nfa > 3) {
                        getUsersStoredVerificationToken.b = true;
                        getUsersStoredVerificationToken.ets = 120;
                        getUsersStoredVerificationToken.nfa = 0;
                        await dal_manager_1.dbManager.accessRedisCache.setVerificationToken(req.body.username, { ...getUsersStoredVerificationToken });
                        return res.status(app_enums_1.enums.MultiFactorHttpsStatusCodes.BLOCKED_USER).json({
                            errorMessage: i18n_1.default.__({ phrase: "ERRORS.AUTH.BLOCKED_USER", locale: userSettings.locale }),
                        });
                    }
                    return res.status(app_enums_1.enums.MultiFactorHttpsStatusCodes.TOKEN_NOT_VERIFIED).json({
                        errorMessage: i18n_1.default.__({ phrase: "ERRORS.AUTH.TOKEN_NOT_VERIFIED", locale: userSettings.locale }),
                    });
                }
            }
        });
        this._router.post("/logout", this._urlencodedParser, asyncHandler(async (req, res) => {
            if (!req.body.grantTypeId) {
                throw (0, api_error_1.generateBadRequestApiError)({ message: "null grant type Id" });
            }
            await dal_manager_1.dbManager.systemTransaction(async (trx) => {
                let grantTypeId = (0, api_validatorhelper_1.validateUUID)(req.body.grantTypeId, "grantTypeId");
                let authMethod = await dal_manager_1.dbManager.accessOrganization.getAuthenticationMethodById(grantTypeId, trx);
                switch (authMethod.authenticationMethod) {
                    case app_enums_1.enums.AuthMethod.CAS:
                        if (!req.query.ssodisable) {
                            res.clearCookie("xcasusername", {
                                domain: undefined,
                                path: undefined,
                                httpOnly: true,
                            });
                            return res.json({
                                redirectUrl: authMethod.config.casUrl +
                                    url_1.default.format({
                                        pathname: "/logout",
                                    }),
                            });
                        }
                        break;
                    case app_enums_1.enums.AuthMethod.OPENID: {
                        const openIdConfig = authMethod.config;
                        const openIdIssuer = await openid_client_1.Issuer.discover(openIdConfig.wellKnownEndpoint);
                        const client = new openIdIssuer.Client({
                            client_id: openIdConfig.clientId,
                            client_secret: openIdConfig.clientSecret,
                            redirect_uris: openIdConfig.redirectUris,
                            response_types: openIdConfig.responseTypes,
                        });
                        const hint = new openid_client_1.TokenSet({
                            id_token: req.cookies.xopenididtoken,
                            refresh_token: req.cookies.xopenidrefreshtoken,
                            access_token: req.cookies.xopenidaccesstoken,
                        });
                        const url = client.endSessionUrl({
                            id_token_hint: hint,
                            post_logout_redirect_uri: app_config_1.appConfig.httpServer.hostBaseRedirectUrl,
                            client_id: openIdConfig.clientId,
                        });
                        if (!req.query.ssodisable) {
                            invalidateOpenIdCookies(res);
                            return res.json({
                                redirectUrl: url,
                            });
                        }
                        break;
                    }
                    case app_enums_1.enums.AuthMethod.SAML: {
                        await new Promise((resolve) => {
                            AuthManager._samlServiceProviderMap
                                .get(grantTypeId)
                                .create_logout_request_url(AuthManager._samlIdentityProviderMap.get(grantTypeId), { name_id: req.cookies.xsamluserId }, (err, logoutUrl) => {
                                if (err) {
                                    return res.status(app_enums_1.enums.HttpStatusCode.INTERNAL_ERROR).send("Error creating logout URL: " + err);
                                }
                                invalidateSamlCookies(res);
                                res.status(app_enums_1.enums.HttpStatusCode.CONFLICT).json({
                                    url: logoutUrl,
                                });
                                resolve({});
                            });
                        });
                        break;
                    }
                }
                let authHeader = req.header(app_constants_1.ArmonHeaders.Authorization);
                if (!authHeader) {
                    return res.redirect(app_enums_1.enums.HttpStatusCode.SUCCESS, app_config_1.appConfig.hostBaseRedirectUrl);
                }
                let headerTokens = authHeader.split(" ");
                if (headerTokens[0] != "Bearer") {
                    res.status(app_enums_1.enums.HttpStatusCode.BAD_REQUEST).json({
                        error: AccessTokenErrorType[AccessTokenErrorType.invalid_request],
                        error_description: "no bearer",
                    });
                }
                try {
                    let decodedToken = jsonwebtoken_1.default.verify(headerTokens[1], app_config_1.appConfig.jwtSecret);
                    const organizationId = decodedToken.o[0].i;
                    await dal_manager_1.dbManager.accessUser.logoutUserWithTokenId(organizationId, decodedToken.i, req.header(app_constants_1.ArmonHeaders.UserAgent), trx);
                    if (req.session) {
                        req.session.destroy((err) => {
                            if (err)
                                app_logs_1.logger.error(err);
                        });
                    }
                }
                catch (error) {
                    app_logs_1.logger.error(`Error while logging out user`);
                    app_logs_1.logger.error(error?.message ?? error?.body?.message ?? error);
                    return res.status(app_enums_1.enums.HttpStatusCode.BAD_REQUEST).json({
                        error: AccessTokenErrorType[AccessTokenErrorType.invalid_request],
                        error_description: error.message || error.body.message,
                    });
                }
                if (!res.headersSent) {
                    res.redirect(app_enums_1.enums.HttpStatusCode.SUCCESS, app_config_1.appConfig.hostBaseRedirectUrl);
                }
            });
        }));
        app.use("/auth", this._router);
    }
    async initSAMLMethods(trx) {
        let allOrganizationAuthMethods = await dal_manager_1.dbManager.accessOrganization.listAllOrganizationAuthMethods(trx);
        for (const organization of allOrganizationAuthMethods) {
            let samlMethods = organization.authentications.filter((a) => a.authenticationMethod == app_enums_1.enums.AuthMethod.SAML);
            for (const samlMethod of samlMethods) {
                const samlConfig = samlMethod.config;
                let ip_options = {
                    sso_login_url: samlConfig.sso_login_url,
                    sso_logout_url: samlConfig.sso_logout_url,
                    certificates: fs_1.default.readFileSync(samlConfig.certFilePath, "utf8"),
                };
                let sp_options = {
                    entity_id: samlConfig.entity_id,
                    private_key: fs_1.default.readFileSync(app_config_1.appConfig.httpServer.ssl.keyFileName, "utf8"),
                    certificate: fs_1.default.readFileSync(app_config_1.appConfig.httpServer.ssl.certFileName, "utf8"),
                    assert_endpoint: samlConfig.assert_endpoint,
                };
                AuthManager._samlServiceProviderMap.set(samlMethod.id, new saml.ServiceProvider(sp_options));
                AuthManager._samlIdentityProviderMap.set(samlMethod.id, new saml.IdentityProvider(ip_options));
            }
        }
    }
    async initOauthMethods(trx) {
        let allOrganizationAuthMethods = await dal_manager_1.dbManager.accessOrganization.listAllOrganizationAuthMethods(trx);
        for (const organization of allOrganizationAuthMethods) {
            let oauthMethods = organization.authentications.filter((a) => a.authenticationMethod == app_enums_1.enums.AuthMethod.OAUTH2);
            for (const oauthMethod of oauthMethods) {
                let oauthStrategy = new passport_oauth2_1.default({
                    authorizationURL: oauthMethod.config.authorizationURL,
                    tokenURL: oauthMethod.config.tokenURL,
                    clientID: oauthMethod.config.clientID,
                    clientSecret: oauthMethod.config.clientSecret,
                    callbackURL: oauthMethod.config.callbackURL,
                    passReqToCallback: true,
                    scope: oauthMethod.config.scope,
                }, async (req, accessToken, refreshToken, profile, cb) => {
                    let user = await app_auth_1.authHelper.authenticateWithUsername(oauthMethod.id, null, profile[oauthMethod.config.usernameField], trx);
                    let scopes = req.query.scope ? req.query.scope.split(" ") : undefined;
                    let token = await app_auth_1.authHelper.generateToken(req, { user, clientId: undefined, scopes, orgIds: [organization.id], custom: undefined }, trx);
                    cb(null, {
                        access_token: token.access_token,
                        token_type: token.token_type,
                        expiresIn: token.expires_in,
                        refresh_token: token.refresh_token,
                        profile: user.userProfile,
                        organizations: user.organizations.find((o) => o.organization.id == organization.id),
                    });
                });
                passport_1.default.use("oauth_" + organization.code, oauthStrategy);
            }
        }
        this._router.post("/:code/oauth2/callback", async function (req, res, next) {
            if (!req.params.code) {
                throw (0, api_error_1.generateBadRequestApiError)({ message: "null organization code" });
            }
            await (0, app_middleware_1.invokeMiddleware)([passport_1.default.authorize("oauth_" + req.params.code)], req, res);
            res.json(req.account);
        });
    }
    async initLdapMethods(trx) {
        let allOrganizationAuthMethods = await dal_manager_1.dbManager.accessOrganization.listAllOrganizationAuthMethods(trx);
        for (const organization of allOrganizationAuthMethods) {
            let tokenMethods = organization.authentications.filter((a) => a.authenticationMethod == app_enums_1.enums.AuthMethod.LDAP);
            for (const ldapMethod of tokenMethods) {
                let ldapOptions = ldapMethod.config ? ldapMethod.config : null;
                if (!ldapOptions)
                    continue;
                let ldapStrategyOptions = {
                    server: {
                        url: ldapOptions.url,
                        bindDN: ldapOptions.bindDN,
                        bindCredentials: ldapOptions.bindCredentials,
                        searchBase: ldapOptions.searchBase,
                        searchFilter: ldapOptions.searchFilter,
                        tlsOptions: null,
                    },
                };
                if (ldapOptions.tlsOptions) {
                    ldapStrategyOptions.server.tlsOptions = {
                        ca: [fs_1.default.readFileSync(ldapOptions.tlsOptions.caFilePath)],
                        rejectUnauthorized: ldapOptions.tlsOptions.rejectUnauthorized,
                    };
                }
                else {
                    ldapStrategyOptions.server.tlsOptions = {
                        rejectUnauthorized: false,
                    };
                }
                let ldapStrategy = new LdapStrategy(ldapStrategyOptions);
                passport_1.default.use("ldap_" + organization.id + ldapMethod.id, ldapStrategy);
            }
        }
    }
    static async authUser(grantType, fromMobile) {
        return async function (req, res, next) {
            return await dal_manager_1.dbManager.systemTransaction(async (trx) => {
                try {
                    let grantTypeId = (0, api_validatorhelper_1.validateUUID)(req.params.grantTypeId, "grantTypeId");
                    let authMethod = await dal_manager_1.dbManager.accessOrganization.getAuthenticationMethodById(grantTypeId, trx);
                    if (!authMethod) {
                        res.status(app_enums_1.enums.HttpStatusCode.NOT_FOUND).send();
                        return;
                    }
                    req.body.organizationId = authMethod.organizationId;
                    switch (authMethod.authenticationMethod) {
                        case app_enums_1.enums.AuthMethod.Native: {
                            if (grantType != app_enums_1.enums.AuthGrantType.UsernamePassword) {
                                return res.json({
                                    message: "Invalid grant type for authentication method",
                                });
                            }
                            let user = await app_auth_1.authHelper.authenticateWithPasswordCredentialsByUrlEncodedRequest(req, res, trx);
                            if (user?.organizations) {
                                const organizationMultifactorAuthSettings = user.organizations.find((elem) => elem.organization.id === req.body.organizationId).organization
                                    .multiFactorAuthenticationSettings;
                                if (organizationMultifactorAuthSettings.option !== app_enums_1.enums.MultifactorAuthenticationOption.DISABLE) {
                                    return AuthManager.checkMultiFactorAuthSettings(req, res, next, app_enums_1.enums.AuthMethod.Native);
                                }
                                const rateLimitCheck = await (0, app_rate_limiter_1.authRateLimiter)(req.body.username, req.socket.remoteAddress);
                                if (!rateLimitCheck.success) {
                                    res.set("Retry-After", rateLimitCheck.retryInSecs?.toString());
                                    let message = i18n_1.default.__({ phrase: "ERRORS.GENERAL.TOO_MANY_REQUESTS", locale: req.locale ?? "tr" }, { waittime: Math.ceil(rateLimitCheck.retryInSecs / 60).toString() });
                                    res.status(app_enums_1.enums.HttpStatusCode.TOO_MANY_REQUEST).json({ message: message, noReport: true, showAsModal: true });
                                    return;
                                }
                                else {
                                    return AuthManager.authenticateToken(req, res, next, trx, user);
                                }
                            }
                        }
                        case app_enums_1.enums.AuthMethod.CAS: {
                            req.casOptions = authMethod.config;
                            if (!authMethod.config) {
                                return res.json({ message: "no configuration for CAS " });
                            }
                            if (!fromMobile) {
                                return AuthManager.authenticateCAS(req, res, next, trx);
                            }
                            return AuthManager.authenticateCASMobile(req, res, next, trx);
                        }
                        case app_enums_1.enums.AuthMethod.NTLM: {
                            if (grantType == app_enums_1.enums.AuthGrantType.SSO) {
                                let ntlmAuth = authMethod;
                                if (!ntlmAuth || !ntlmAuth.config) {
                                    throw (0, api_error_1.generateBadRequestApiError)({
                                        message: "no ntlm method is defined",
                                    });
                                }
                                try {
                                    let ntlmOptions = ntlmAuth.config;
                                    if (ntlmOptions.enabled) {
                                        if (!ntlmOptions.domain || !ntlmOptions.domaincontroller) {
                                            throw new Error("ntlm domain and controller have to be defined!");
                                        }
                                        let ntlmTls = undefined;
                                        if (ntlmOptions.tlsOptions) {
                                            let ca = fs_1.default.readFileSync(ntlmOptions.tlsOptions.caFilePath);
                                            ntlmTls = {
                                                ca: ca,
                                                rejectUnauthorized: ntlmOptions.tlsOptions.rejectUnauthorized,
                                            };
                                        }
                                        let _ntlmMiddleWare = (0, express_ntlm_1.default)({
                                            debug: ntlmOptions.debug
                                                ? function () {
                                                    var args = Array.prototype.slice.apply(arguments);
                                                    console.log.apply(null, args);
                                                }
                                                : undefined,
                                            domain: ntlmOptions.domain,
                                            tlsOptions: ntlmTls,
                                            domaincontroller: ntlmOptions.domaincontroller,
                                        });
                                        const ntlmPromise = new Promise((resolve, reject) => {
                                            let finished = false;
                                            res.on("finished", () => {
                                                if (!finished) {
                                                    finished = true;
                                                    resolve();
                                                }
                                            });
                                            _ntlmMiddleWare(req, res, () => {
                                                if (!finished) {
                                                    finished = true;
                                                    resolve();
                                                }
                                            });
                                        });
                                        await ntlmPromise;
                                        if (res.finished || res.headersSent) {
                                            if (res.statusCode !== app_enums_1.enums.HttpStatusCode.SUCCESS) {
                                                app_logs_1.logger.debug(`[app.auth.manager.ts] ntlm flow contiunes, status code: ${res.statusCode}`);
                                                return;
                                            }
                                            else {
                                                app_logs_1.logger.debug(`[app.auth.manager.ts] ntlm returned wih 200, should not happend as far as i can see`);
                                            }
                                        }
                                        else {
                                            if (req.ntlm) {
                                                app_logs_1.logger.debug(`[app.auth.manager.ts] ntlm flow success, continue for armon token`);
                                            }
                                            else {
                                                app_logs_1.logger.debug(`[app.auth.manager.ts] middleware called next even though no response is sent, and ntlm failed!!!!`);
                                            }
                                        }
                                    }
                                }
                                catch (err) {
                                    app_logs_1.logger.error("Error while parsing ntlm options file! %j", err);
                                    process.exit(1);
                                }
                                return AuthManager.authenticateNTLM(req, res, next, trx);
                            }
                        }
                        case app_enums_1.enums.AuthMethod.LDAP: {
                            if (grantType == app_enums_1.enums.AuthGrantType.UsernamePassword) {
                                const rateLimitCheck = await (0, app_rate_limiter_1.authRateLimiter)(req.body.username, req.socket.remoteAddress);
                                if (!rateLimitCheck.success) {
                                    res.set("Retry-After", rateLimitCheck.retryInSecs?.toString());
                                    let message = i18n_1.default.__({ phrase: "ERRORS.GENERAL.TOO_MANY_REQUESTS", locale: req.locale ?? "tr" }, { waittime: Math.ceil(rateLimitCheck.retryInSecs / 60).toString() });
                                    res.status(app_enums_1.enums.HttpStatusCode.TOO_MANY_REQUEST).json({ message: message, noReport: true, showAsModal: true });
                                    return;
                                }
                                else {
                                    let ldapPromise = new Promise((resolve, reject) => {
                                        passport_1.default.authenticate("ldap_" + authMethod.organizationId + grantTypeId, function (err, user, info) {
                                            if (err) {
                                                reject(err);
                                            }
                                            if (user) {
                                                return resolve(null);
                                            }
                                            reject(info);
                                        })(req, res, next);
                                    });
                                    try {
                                        await ldapPromise;
                                    }
                                    catch (error) {
                                        if (error.message === "Invalid username/password") {
                                            (0, dal_access_error_1.throwDbAccessBadRequestErrorTr)("ERRORS.IDENTITY.USERNAMEORPASSWORDWRONG");
                                        }
                                        else {
                                            throw error;
                                        }
                                    }
                                    return AuthManager.authenticateTokenWithLDAP(req, res, next, trx);
                                }
                            }
                            else {
                                return res.status(app_enums_1.enums.HttpStatusCode.INTERNAL_ERROR).json({
                                    message: "Invalid grant type for authentication method",
                                });
                            }
                        }
                        case app_enums_1.enums.AuthMethod.OAUTH2:
                            return AuthManager.authenticateOauth2(req, res, next);
                        case app_enums_1.enums.AuthMethod.OPENID: {
                            const openIdMethod = authMethod;
                            if (!openIdMethod || !openIdMethod.config) {
                                throw (0, api_error_1.generateBadRequestApiError)({
                                    message: "no openid method is defined",
                                });
                            }
                            const openIdConfig = openIdMethod.config;
                            if (!openIdConfig.clientId || !openIdConfig.clientSecret) {
                                throw (0, api_error_1.generateBadRequestApiError)({
                                    message: "openid client id and secret have to be defined!",
                                });
                            }
                            if (!openIdConfig.wellKnownEndpoint || !openIdConfig.redirectUris) {
                                throw (0, api_error_1.generateBadRequestApiError)({
                                    message: "openid wellKnown url and redirectUris have to be defined!",
                                });
                            }
                            const result = await AuthManager.authenticateOpenID(req.body.organizationId, req, res, openIdConfig, trx);
                            if (result?.body?.tokenId) {
                                res.json(result.body);
                            }
                            return;
                        }
                        case app_enums_1.enums.AuthMethod.OPENID_CLIENT: {
                            const openIdClientMethod = authMethod;
                            if (!openIdClientMethod || !openIdClientMethod.config) {
                                throw (0, api_error_1.generateBadRequestApiError)({
                                    message: "no openid method is defined",
                                });
                            }
                            const openIdClientConfig = openIdClientMethod.config;
                            if (!req.body.accessToken || !req.body.refreshToken || !req.body.idToken) {
                                throw (0, api_error_1.generateBadRequestApiError)({
                                    message: "some / all of openid method params are missing",
                                });
                            }
                            const result = await AuthManager.authenticateOpenIdByClientSide(req, res, openIdClientConfig, trx);
                            if (result?.body?.tokenId) {
                                res.json(result.body);
                            }
                            return;
                        }
                        case app_enums_1.enums.AuthMethod.SAML: {
                            const samlMethod = authMethod;
                            if (!authMethod || !authMethod.config) {
                                throw (0, api_error_1.generateBadRequestApiError)({
                                    message: "no saml method is defined",
                                });
                            }
                            const samlClientConfig = samlMethod.config;
                            const result = await AuthManager.authenticateSAML(req.body.organizationId, req, res, next, samlClientConfig, trx);
                            if (result?.body?.tokenId) {
                                res.json(result.body);
                            }
                            return;
                        }
                    }
                }
                catch (error) {
                    app_logs_1.logger.error(`Error while using authentication with grant type id ${req.params.grantTypeId}.`);
                    app_logs_1.logger.error(error);
                    if (error instanceof dal_access_error_1.DbAccessErrorTr) {
                        let translatedError = (0, api_error_1.generateTranslatedError)(error.errorCode, error.phrase, error.replacements);
                        let message = i18n_1.default.__({ phrase: translatedError.message, locale: "tr" });
                        res.status(translatedError.statusCode).json({ message: message, noReport: true });
                    }
                    else {
                        if (!error.statusCode || error.statusCode === 500) {
                            app_logs_1.logger.error("Invalid grant ", error || "something blew up and the err object was undefined");
                        }
                        res.status(error.statusCode || 500).json(error.body || error.message || error || "something blew up and the err object was undefined");
                    }
                }
            });
        };
    }
    static async authenticateNTLM(req, res, next, trx) {
        try {
            let ntlm = req.ntlm;
            if (ntlm && ntlm.Authenticated) {
                let user = await app_auth_1.authHelper.authenticateWithUsername(req.body.organizationId, res, ntlm.UserName, trx);
                let scopes = req.query.scope ? req.query.scope.split(" ") : undefined;
                let token = await app_auth_1.authHelper.generateToken(req, { user, clientId: undefined, scopes, orgIds: [req.body.organizationId], custom: undefined }, trx);
                res.json({
                    tokenId: token.tokenId,
                    token: token.access_token,
                    expiresIn: token.expires_in,
                    refreshToken: token.refresh_token,
                    organization: user.organizations.find((o) => o.organization.id == req.body.organizationId),
                });
            }
            else {
                throw new Error("Invalid ntlm authentication");
            }
        }
        catch (error) {
            app_logs_1.logger.error(`Error while authenticating NTLM.`);
            app_logs_1.logger.error(error?.message ?? error?.body?.message ?? error);
            res.status(app_enums_1.enums.HttpStatusCode.BAD_REQUEST).json({
                error: AccessTokenErrorType[AccessTokenErrorType.invalid_grant],
                error_description: error?.message ?? error?.body?.message ?? error,
            });
        }
    }
    static async authenticateCAS(req, res, next, trx) {
        let casOptions = req.casOptions;
        try {
            if (req.cookies && req.cookies.xcasusername) {
                try {
                    let user = await app_auth_1.authHelper.authenticateWithUsername(req.body.organizationId, res, req.cookies.xcasusername, trx);
                    const mobileInstallIdCheckResult = await app_auth_1.authHelper.checkMobileDeviceMismatch(req.body.organizationId, {
                        userAgent: req.header(app_constants_1.ArmonHeaders.UserAgent),
                        mobileInstallId: req.header(app_constants_1.ArmonHeaders.MobileInstallId),
                        userId: user.id,
                    }, trx);
                    if (!mobileInstallIdCheckResult) {
                        return res.status(app_enums_1.enums.HttpStatusCode.BAD_REQUEST).json({
                            error: AccessTokenErrorType[AccessTokenErrorType.mobile_device_mismatch],
                            error_description: req.header(app_constants_1.ArmonHeaders.MobileInstallId),
                        });
                    }
                    let scopes = req.query.scope ? req.query.scope.split(" ") : undefined;
                    let token = await app_auth_1.authHelper.generateToken(req, { user, clientId: undefined, scopes, orgIds: [req.body.organizationId], custom: undefined }, trx);
                    res.json({
                        tokenId: token.tokenId,
                        token: token.access_token,
                        expiresIn: token.expires_in,
                        refreshToken: token.refresh_token,
                        organization: user.organizations.find((o) => o.organization.id == req.body.organizationId),
                    });
                    res.end();
                }
                catch (error) {
                    app_logs_1.logger.error("error 1" + JSON.stringify(error));
                }
            }
            else {
                let ticket = req.query.ticket;
                let originalPath = req.originalUrl.split("?")[0];
                if (!ticket) {
                    var loginServerURL = url_1.default.parse(casOptions.casUrl + "/login", true);
                    loginServerURL.query.service = casOptions.serviceUrl + "/oauth/" + req.params.code + "/cas";
                    app_logs_1.logger.error(loginServerURL);
                    return res.status(app_enums_1.enums.HttpStatusCode.CONFLICT).json({
                        url: casOptions.casUrl +
                            url_1.default.format({
                                pathname: "/login",
                                query: {
                                    service: casOptions.serviceUrl + originalPath,
                                },
                            }),
                    });
                }
                let parsed = url_1.default.parse(casOptions.casUrl);
                var requestOptions = {
                    host: parsed.hostname,
                    port: 443,
                    method: "GET",
                    path: url_1.default.format({
                        pathname: "/cas/validate",
                        query: {
                            service: casOptions.serviceUrl + originalPath,
                            ticket: req.query.ticket,
                        },
                    }),
                };
                let casUser = null;
                let requestPromise = new Promise((resolve, reject) => {
                    var request = https_1.default.request(requestOptions, function (response) {
                        response.setEncoding("utf8");
                        var body = "";
                        response.on("data", function (chunk) {
                            return (body += chunk);
                        });
                        response.on("end", function () {
                            if (body.indexOf("yes") > -1) {
                                casUser = body.replace("yes", "").trim();
                            }
                            resolve(null);
                        });
                        response.on("error", function (err) {
                            app_logs_1.logger.error("Response error from CAS: ", err);
                            res.sendStatus(app_enums_1.enums.HttpStatusCode.UNAUTHORIZED_ACCESS);
                            reject(err);
                        });
                    });
                    request.on("error", function (err) {
                        app_logs_1.logger.error("Request error with CAS: ", err);
                        res.sendStatus(app_enums_1.enums.HttpStatusCode.UNAUTHORIZED_ACCESS);
                        reject(err);
                    });
                    request.end();
                });
                await requestPromise;
                if (casUser) {
                    res.cookie("xcasusername", casUser, {
                        maxAge: 15000,
                        domain: undefined,
                        path: undefined,
                        httpOnly: true,
                        secure: true,
                    });
                    let user = await dal_manager_1.dbManager.accessOAuth.getArmonApplicationUserByUsername(req.body.organizationId, casUser, trx);
                    if (user) {
                        res.redirect("/app");
                    }
                    else {
                        throw { user: user };
                    }
                    res.end();
                }
            }
        }
        catch (error) {
            app_logs_1.logger.error(`Error while authenticating CAS`);
            res.send(casOptions.errorText.replace(/{{user}}/g, error.message));
            app_logs_1.logger.error("cas error " + JSON.stringify(error));
        }
    }
    static async authenticateOpenID(organizationId, req, res, openIdConfig, trx) {
        try {
            const isMobileDevice = req.header(app_constants_1.ArmonHeaders.UserAgent).includes("mobilemanager");
            const openIdIssuer = await openid_client_1.Issuer.discover(openIdConfig.wellKnownEndpoint);
            const client = new openIdIssuer.Client({
                client_id: openIdConfig.clientId,
                client_secret: openIdConfig.clientSecret,
                redirect_uris: openIdConfig.redirectUris,
                response_types: openIdConfig.responseTypes,
            });
            if (req.cookies && req.cookies.xopenidrefreshtoken && req.cookies.xopenidaccesstoken && req.cookies.xopenidusername && req.cookies.xopenidgrantid) {
                const accessTokenOpen = jsonwebtoken_1.default.decode(req.cookies.xopenidaccesstoken, { complete: true });
                const idTokenOpen = jsonwebtoken_1.default.decode(req.cookies.xopenididtoken, { complete: true });
                if (accessTokenOpen.payload["exp"] < Math.floor(new Date().getTime() / 1000)) {
                    let refreshedToken = null;
                    try {
                        refreshedToken = await client.refresh(req.cookies.xopenidrefreshtoken);
                    }
                    catch (error) {
                        app_logs_1.logger.debug("refresh token expired. Redirected to " + app_config_1.appConfig.hostBaseRedirectUrl);
                        app_logs_1.logger.warn(error.message);
                        invalidateOpenIdCookies(res);
                        res.status(app_enums_1.enums.HttpStatusCode.CONFLICT).send({ url: app_config_1.appConfig.hostBaseRedirectUrl });
                        return;
                    }
                    const userinfo = await client.userinfo(refreshedToken.access_token);
                    const accessTokenDecoded = jsonwebtoken_1.default.decode(refreshedToken.access_token, { complete: true }).payload;
                    const idTokenDecoded = jsonwebtoken_1.default.decode(refreshedToken.access_token, { complete: true }).payload;
                    if (userinfo.sub !== idTokenDecoded["sub"]) {
                        throw new Error("Malformed user info.");
                    }
                    setOpenIdCookies(req.params.grantTypeId, res, refreshedToken, accessTokenDecoded, openIdConfig.binding.externalUserInfoKey);
                }
                const user = await app_auth_1.authHelper.authenticateWithCustomMapping(organizationId, { ...openIdConfig.binding, externalValueToMatch: req.cookies.xopenidusername }, trx);
                const mobileInstallIdCheckResult = await app_auth_1.authHelper.checkMobileDeviceMismatch(organizationId, {
                    userAgent: req.header(app_constants_1.ArmonHeaders.UserAgent),
                    mobileInstallId: req.header(app_constants_1.ArmonHeaders.MobileInstallId),
                    userId: user.id,
                }, trx);
                if (!mobileInstallIdCheckResult) {
                    res.status(app_enums_1.enums.HttpStatusCode.BAD_REQUEST).json({
                        error: AccessTokenErrorType[AccessTokenErrorType.mobile_device_mismatch],
                        error_description: req.header(app_constants_1.ArmonHeaders.MobileInstallId),
                    });
                    return;
                }
                let customData = app_auth_1.authHelper.calculateCustomTokenExpiration(req.cookies.xopenidaccesstoken, req.cookies.xopenidrefreshtoken);
                const token = await app_auth_1.authHelper.generateToken(req, { user, clientId: undefined, scopes: undefined, orgIds: [organizationId], custom: customData }, trx);
                return {
                    responseStatus: app_enums_1.enums.HttpStatusCode.SUCCESS,
                    body: {
                        tokenId: token.tokenId,
                        token: token.access_token,
                        expiresIn: token.expires_in,
                        refreshToken: token.refresh_token,
                        organization: user.organizations.find((o) => o.organization.id == organizationId),
                    },
                };
            }
            else {
                if (!req.query.code) {
                    app_logs_1.logger.debug("cookies and code not found");
                    const state = openid_client_1.generators.state();
                    const url = client.authorizationUrl({
                        state,
                        scope: openIdConfig.scope ?? "openid",
                    });
                    await dal_manager_1.dbManager.accessRedisCache.setExpireValue(state, state, 5 * 60 * 1000);
                    res.status(app_enums_1.enums.HttpStatusCode.CONFLICT).json({ url });
                    return;
                }
                const storedState = await dal_manager_1.dbManager.accessRedisCache.getValue(req.query.state);
                if (!storedState) {
                    throw new Error("Invalid state");
                }
                const tokenSet = await client.callback(openIdConfig.redirectUris[0], client.callbackParams(req), { state: storedState });
                app_logs_1.logger.debug("TokenSet:" + JSON.stringify(tokenSet, null, 4));
                const userinfo = await client.userinfo(tokenSet.access_token);
                const accessTokenDecoded = jsonwebtoken_1.default.decode(tokenSet.access_token, { complete: true }).payload;
                const idTokenDecoded = jsonwebtoken_1.default.decode(tokenSet.id_token, { complete: true }).payload;
                if (userinfo.sub !== idTokenDecoded["sub"]) {
                    throw new Error("Malformed user info.");
                }
                setOpenIdCookies(req.params.grantTypeId, res, tokenSet, accessTokenDecoded, openIdConfig.binding.externalUserInfoKey);
                const user = await app_auth_1.authHelper.authenticateWithCustomMapping(organizationId, { ...openIdConfig.binding, externalValueToMatch: accessTokenDecoded[openIdConfig.binding.externalUserInfoKey ?? "preferred_username"] }, trx);
                if (!isMobileDevice) {
                    res.redirect(app_config_1.appConfig.httpServer.hostBaseRedirectUrl);
                    return;
                }
                else {
                    let customData = app_auth_1.authHelper.calculateCustomTokenExpiration(tokenSet.access_token, tokenSet.refresh_token);
                    const mobileInstallIdCheckResult = await app_auth_1.authHelper.checkMobileDeviceMismatch(organizationId, {
                        userAgent: req.header(app_constants_1.ArmonHeaders.UserAgent),
                        mobileInstallId: req.header(app_constants_1.ArmonHeaders.MobileInstallId),
                        userId: user.id,
                    }, trx);
                    if (!mobileInstallIdCheckResult) {
                        res.status(app_enums_1.enums.HttpStatusCode.BAD_REQUEST).json({
                            error: AccessTokenErrorType[AccessTokenErrorType.mobile_device_mismatch],
                            error_description: req.header(app_constants_1.ArmonHeaders.MobileInstallId),
                        });
                        return;
                    }
                    const token = await app_auth_1.authHelper.generateToken(req, { user, clientId: undefined, scopes: undefined, orgIds: [organizationId], custom: customData }, trx);
                    return {
                        responseStatus: app_enums_1.enums.HttpStatusCode.SUCCESS,
                        body: {
                            tokenId: token.tokenId,
                            token: token.access_token,
                            expiresIn: token.expires_in,
                            refreshToken: token.refresh_token,
                            organization: user.organizations.find((o) => o.organization.id == organizationId),
                        },
                    };
                }
            }
        }
        catch (error) {
            app_logs_1.logger.error(error);
            if (req.cookies.xopenididtoken) {
                res.status(app_enums_1.enums.HttpStatusCode.CONFLICT).send({ url: new URL(`/static/${openIdConfig.noAuthStaticContentName}`, app_config_1.appConfig.httpServer.apiBasePath).toString() });
                return;
            }
            const response = handlebars_1.default.compile(fs_1.default.readFileSync(path_1.default.join(app_config_1.appConfig.httpServer.staticDirectory, `${openIdConfig.noAuthStaticContentName}.handlebars`), "utf8"))({});
            res.send(response);
            return;
        }
    }
    static async authenticateOpenIdByClientSide(req, res, openIdClientConfig, trx) {
        try {
            const request = req.body;
            const accessTokenOpen = jsonwebtoken_1.default.decode(request.accessToken, { complete: true });
            if (accessTokenOpen.payload["exp"] > Math.floor(new Date().getTime() / 1000)) {
                let customData = app_auth_1.authHelper.calculateCustomTokenExpiration(request.accessToken, request.refreshToken, -30, -30);
                const user = await app_auth_1.authHelper.authenticateWithCustomMapping(req.body.organizationId, { ...openIdClientConfig.binding, externalValueToMatch: accessTokenOpen.payload[openIdClientConfig.binding.externalUserInfoKey] }, trx);
                const mobileInstallIdCheckResult = await app_auth_1.authHelper.checkMobileDeviceMismatch(req.body.organizationId, {
                    userAgent: req.header(app_constants_1.ArmonHeaders.UserAgent),
                    mobileInstallId: req.header(app_constants_1.ArmonHeaders.MobileInstallId),
                    userId: user.id,
                }, trx);
                if (!mobileInstallIdCheckResult) {
                    res.status(app_enums_1.enums.HttpStatusCode.BAD_REQUEST).json({
                        error: AccessTokenErrorType[AccessTokenErrorType.mobile_device_mismatch],
                        error_description: req.header(app_constants_1.ArmonHeaders.MobileInstallId),
                    });
                    return;
                }
                const token = await app_auth_1.authHelper.generateToken(req, { user, clientId: undefined, scopes: undefined, orgIds: [req.body.organizationId], custom: customData }, trx);
                return {
                    responseStatus: app_enums_1.enums.HttpStatusCode.SUCCESS,
                    body: {
                        tokenId: token.tokenId,
                        token: token.access_token,
                        expiresIn: token.expires_in,
                        refreshToken: token.refresh_token,
                        organization: user.organizations.find((o) => o.organization.id == req.body.organizationId),
                    },
                };
            }
            else {
                app_logs_1.logger.error(`Open Id Access Token is expired! accessToken: ${request.accessToken}`);
                res.status(app_enums_1.enums.HttpStatusCode.UNAUTHORIZED_ACCESS).send({ message: "The token is expired! Login again!" });
            }
        }
        catch (error) {
            app_logs_1.logger.error(error);
            const message = i18n_1.default.__({ phrase: "ERRORS.GENERAL.OPENID_USER_NOT_MATCHED", locale: req.locale ?? "tr" });
            res.status(app_enums_1.enums.HttpStatusCode.NOT_FOUND).json({ message: message, noReport: true, showAsModal: true });
            return;
        }
    }
    static async authenticateCASMobile(req, res, next, trx) {
        let casOptions = req.casOptions;
        try {
            if (req.cookies && req.cookies.xcasusername) {
                try {
                    let user = await app_auth_1.authHelper.authenticateWithUsername(req.body.organizationId, res, req.cookies.xcasusername, trx);
                    const mobileInstallIdCheckResult = await app_auth_1.authHelper.checkMobileDeviceMismatch(req.body.organizationId, {
                        userAgent: req.header(app_constants_1.ArmonHeaders.UserAgent),
                        mobileInstallId: req.header(app_constants_1.ArmonHeaders.MobileInstallId),
                        userId: user.id,
                    }, trx);
                    if (!mobileInstallIdCheckResult) {
                        return res.status(app_enums_1.enums.HttpStatusCode.BAD_REQUEST).json({
                            error: AccessTokenErrorType[AccessTokenErrorType.mobile_device_mismatch],
                            error_description: req.header(app_constants_1.ArmonHeaders.MobileInstallId),
                        });
                    }
                    let scopes = req.query.scope ? req.query.scope.split(" ") : undefined;
                    let token = await app_auth_1.authHelper.generateToken(req, { user, clientId: undefined, scopes, orgIds: [req.body.organizationId], custom: undefined }, trx);
                    res.json({
                        tokenId: token.tokenId,
                        token: token.access_token,
                        expiresIn: token.expires_in,
                        refreshToken: token.refresh_token,
                        organization: user.organizations.find((o) => o.organization.id == req.body.organizationId),
                    });
                    res.end();
                }
                catch (error) {
                    app_logs_1.logger.error("error 1" + JSON.stringify(error));
                }
            }
            else {
                let ticket = req.query.ticket;
                let originalPath = req.originalUrl.split("?")[0];
                if (!ticket) {
                    return res.status(app_enums_1.enums.HttpStatusCode.CONFLICT).json({
                        url: casOptions.casUrl +
                            url_1.default.format({
                                pathname: "/login",
                                query: {
                                    service: casOptions.serviceUrl + originalPath,
                                },
                            }),
                    });
                }
                let parsed = url_1.default.parse(casOptions.casUrl);
                var requestOptions = {
                    host: parsed.hostname,
                    port: 443,
                    method: "GET",
                    path: url_1.default.format({
                        pathname: "/cas/validate",
                        query: {
                            service: casOptions.serviceUrl + originalPath,
                            ticket: req.query.ticket,
                        },
                    }),
                };
                let casUser = null;
                let requestPromise = new Promise((resolve, reject) => {
                    var request = https_1.default.request(requestOptions, function (response) {
                        response.setEncoding("utf8");
                        var body = "";
                        response.on("data", function (chunk) {
                            return (body += chunk);
                        });
                        response.on("end", function () {
                            if (body.indexOf("yes") > -1) {
                                casUser = body.replace("yes", "").trim();
                            }
                            resolve(null);
                        });
                        response.on("error", function (err) {
                            app_logs_1.logger.error("Response error from CAS: ", err);
                            res.sendStatus(app_enums_1.enums.HttpStatusCode.UNAUTHORIZED_ACCESS);
                            reject(err);
                        });
                    });
                    request.on("error", function (err) {
                        app_logs_1.logger.error("Request error with CAS: ", err);
                        res.sendStatus(app_enums_1.enums.HttpStatusCode.UNAUTHORIZED_ACCESS);
                        reject(err);
                    });
                    request.end();
                });
                await requestPromise;
                if (casUser) {
                    res.cookie("xcasusername", casUser, {
                        maxAge: 15000,
                        domain: undefined,
                        path: undefined,
                        httpOnly: true,
                        secure: true,
                    });
                    let user = await app_auth_1.authHelper.authenticateWithUsername(req.body.organizationId, res, casUser, trx);
                    let scopes = req.query.scope ? req.query.scope.split(" ") : undefined;
                    let token = await app_auth_1.authHelper.generateToken(req, { user, clientId: undefined, scopes, orgIds: [req.body.organizationId], custom: undefined }, trx);
                    res.json({
                        tokenId: token.tokenId,
                        token: token.access_token,
                        expiresIn: token.expires_in,
                        refreshToken: token.refresh_token,
                        organization: user.organizations.find((o) => o.organization.id == req.body.organizationId),
                    });
                    res.end();
                }
            }
        }
        catch (error) {
            app_logs_1.logger.error(`Error while authenting CAS mobile.`);
            app_logs_1.logger.error(error?.message ?? error?.body?.message ?? error);
            res.send(casOptions.errorText.replace(/{{user}}/g, error.message));
            app_logs_1.logger.error("cas error " + JSON.stringify(error));
        }
    }
    static async authenticateToken(req, res, next, trx, user) {
        if (!req.body) {
            return res.status(app_enums_1.enums.HttpStatusCode.BAD_REQUEST).json({
                error: AccessTokenErrorType[AccessTokenErrorType.unsupported_grant_type],
                error_description: req.body.grant_type,
            });
        }
        try {
            let clientSpecified = req.query.client_id !== undefined;
            let client = null;
            if (clientSpecified) {
                client = await AuthManager.validateClientForTokenRequest(req, res, trx);
                if (res.finished) {
                    return;
                }
            }
            const mobileInstallIdCheckResult = await app_auth_1.authHelper.checkMobileDeviceMismatch(req.body.organizationId, {
                userAgent: req.header(app_constants_1.ArmonHeaders.UserAgent),
                mobileInstallId: req.header(app_constants_1.ArmonHeaders.MobileInstallId),
                userId: user.id,
            }, trx);
            if (!mobileInstallIdCheckResult) {
                return res.status(app_enums_1.enums.HttpStatusCode.BAD_REQUEST).json({
                    error: AccessTokenErrorType[AccessTokenErrorType.mobile_device_mismatch],
                    error_description: req.header(app_constants_1.ArmonHeaders.MobileInstallId),
                });
            }
            let scopes = req.query.scope ? req.query.scope.split(" ") : undefined;
            let token = await app_auth_1.authHelper.generateToken(req, { user, clientId: client ? client.id : undefined, scopes, orgIds: [req.body.organizationId], custom: undefined }, trx);
            await (0, app_rate_limiter_1.authRateLimitReset)(req.body.username, req.socket.remoteAddress);
            if (client && req.query.redirect_uri) {
                res.redirect(req.query.redirect_uri + "?access_token=" + token.access_token + "&state=" + req.query.state);
            }
            else {
                res.json({
                    tokenId: token.tokenId,
                    token: token.access_token,
                    expiresIn: token.expires_in,
                    refreshToken: token.refresh_token,
                    organization: user.organizations.find((o) => o.organization.id == req.body.organizationId),
                });
            }
        }
        catch (error) {
            app_logs_1.logger.error(`Error while authenticating user with username and password`);
            app_logs_1.logger.error(error);
            if (error instanceof dal_access_error_1.DbAccessErrorTr) {
                let translatedError = (0, api_error_1.generateTranslatedError)(error.errorCode, error.phrase, error.replacements);
                let message = i18n_1.default.__({ phrase: translatedError.message, locale: "tr" });
                res.status(translatedError.statusCode).json({ message: message, noReport: true });
            }
            else {
                if (!error.statusCode || error.statusCode === 500) {
                    app_logs_1.logger.error("Invalid grant ", error || "something blew up and the err object was undefined");
                }
                res.status(error.statusCode || 500).json(error.body || error.message || error || "something blew up and the err object was undefined");
            }
        }
    }
    static async authenticateTokenWithLDAP(req, res, next, trx) {
        try {
            let user = await app_auth_1.authHelper.authenticateWithUsername(req.body.organizationId, res, req.body.username, trx);
            const organizationMultifactorAuthSettings = user.organizations.find((elem) => elem.organization.id === req.body.organizationId).organization.multiFactorAuthenticationSettings;
            if (organizationMultifactorAuthSettings.option !== app_enums_1.enums.MultifactorAuthenticationOption.DISABLE) {
                return AuthManager.checkMultiFactorAuthSettings(req, res, next, app_enums_1.enums.AuthMethod.LDAP);
            }
            const mobileInstallIdCheckResult = await app_auth_1.authHelper.checkMobileDeviceMismatch(req.body.organizationId, {
                userAgent: req.header(app_constants_1.ArmonHeaders.UserAgent),
                mobileInstallId: req.header(app_constants_1.ArmonHeaders.MobileInstallId),
                userId: user.id,
            }, trx);
            if (!mobileInstallIdCheckResult) {
                return res.status(app_enums_1.enums.HttpStatusCode.BAD_REQUEST).json({
                    error: AccessTokenErrorType[AccessTokenErrorType.mobile_device_mismatch],
                    error_description: req.header(app_constants_1.ArmonHeaders.MobileInstallId),
                });
            }
            let scopes = req.query.scope ? req.query.scope.split(" ") : undefined;
            let token = await app_auth_1.authHelper.generateToken(req, { user, clientId: undefined, scopes, orgIds: [req.body.organizationId], custom: undefined }, trx);
            await (0, app_rate_limiter_1.authRateLimitReset)(req.body.username, req.socket.remoteAddress);
            res.json({
                tokenId: token.tokenId,
                token: token.access_token,
                expiresIn: token.expires_in,
                refreshToken: token.refresh_token,
                organization: user.organizations.find((o) => o.organization.id == req.body.organizationId),
            });
            res.end();
        }
        catch (error) {
            if (error instanceof dal_access_error_1.DbAccessErrorTr) {
                let translatedError = (0, api_error_1.generateTranslatedError)(error.errorCode, error.phrase, error.replacements);
                let message = i18n_1.default.__({ phrase: translatedError.message, locale: "tr" });
                res.status(translatedError.statusCode).json({ message: message, noReport: true });
            }
            else {
                if (!error.statusCode || error.statusCode === 500) {
                    app_logs_1.logger.error("Invalid grant ", error || "something blew up and the err object was undefined");
                }
                res.status(error.statusCode || 500).json(error.body || error.message || error || "something blew up and the err object was undefined");
            }
        }
    }
    static async authenticateOauth2(req, res, next) {
        let authPromise = new Promise((resolve, reject) => {
            passport_1.default.authenticate("oauth_" + req.params.code, function (err, user, info) {
                if (err) {
                    reject();
                }
                if (user) {
                    return resolve(null);
                }
                reject(info);
            })(req, res, next);
        });
        await authPromise;
    }
    static async authenticateSAML(organizationId, req, res, next, samlConfig, trx) {
        try {
            const isMobileDevice = req.header(app_constants_1.ArmonHeaders.UserAgent).includes("mobilemanager");
            if (req.cookies && req.cookies.xsamluser && req.cookies.xsamlhash && req.cookies.xsamlgrantid) {
                let orgDef = await dal_manager_1.dbManager.accessSystem.getOrganizationDefinitionFileContentV3(organizationId);
                let hash = crypto_1.default.createHmac("sha512", orgDef.organizationPrivateKey);
                hash.update(req.cookies.xsamluser);
                let hashedEmail = hash.digest("hex");
                if (req.cookies.xsamlhash !== hashedEmail) {
                    res.status(app_enums_1.enums.HttpStatusCode.UNAUTHORIZED_ACCESS).json({
                        error: AccessTokenErrorType[AccessTokenErrorType.invalid_grant],
                        error_description: req.header(app_constants_1.ArmonHeaders.Authorization),
                    });
                    return;
                }
                try {
                    const user = await app_auth_1.authHelper.authenticateWithCustomMapping(organizationId, { ...samlConfig.binding, externalValueToMatch: req.cookies.xsamluser }, trx);
                    const mobileInstallIdCheckResult = await app_auth_1.authHelper.checkMobileDeviceMismatch(organizationId, {
                        userAgent: req.header(app_constants_1.ArmonHeaders.UserAgent),
                        mobileInstallId: req.header(app_constants_1.ArmonHeaders.MobileInstallId),
                        userId: user.id,
                    }, trx);
                    if (!mobileInstallIdCheckResult) {
                        res.status(app_enums_1.enums.HttpStatusCode.BAD_REQUEST).json({
                            error: AccessTokenErrorType[AccessTokenErrorType.mobile_device_mismatch],
                            error_description: req.header(app_constants_1.ArmonHeaders.MobileInstallId),
                        });
                        return;
                    }
                    const token = await app_auth_1.authHelper.generateToken(req, { user, clientId: undefined, scopes: undefined, orgIds: [organizationId] }, trx);
                    return {
                        responseStatus: app_enums_1.enums.HttpStatusCode.SUCCESS,
                        body: {
                            tokenId: token.tokenId,
                            token: token.access_token,
                            expiresIn: token.expires_in,
                            refreshToken: token.refresh_token,
                            organization: user.organizations.find((o) => o.organization.id == organizationId),
                        },
                    };
                }
                catch (error) {
                    res.status(app_enums_1.enums.HttpStatusCode.CONFLICT).send({ url: new URL(`/static/${samlConfig.noAuthStaticContentName}`, app_config_1.appConfig.httpServer.apiBasePath).toString() });
                    return;
                }
            }
            else {
                if (req.cookies && !req.cookies.xsamlstarted) {
                    res.cookie("xsamlstarted", true, {
                        maxAge: 1704085200,
                        httpOnly: true,
                        secure: process.env.HTTP_PROTOCOL === "https",
                        path: "/auth",
                    });
                    let resUrl = "";
                    let samlPromise = new Promise((resolve, reject) => {
                        AuthManager._samlServiceProviderMap
                            .get(req.params.grantTypeId)
                            .create_login_request_url(AuthManager._samlIdentityProviderMap.get(req.params.grantTypeId), {}, (err, url) => {
                            if (err) {
                                res.status(app_enums_1.enums.HttpStatusCode.INTERNAL_ERROR).send("Error generating login URL");
                            }
                            resUrl = url;
                            resolve(url);
                        });
                    });
                    await samlPromise;
                    res.status(app_enums_1.enums.HttpStatusCode.CONFLICT).json({
                        url: resUrl,
                    });
                    return;
                }
                let SAMLResponse = req.body.SAMLResponse;
                if (!SAMLResponse) {
                    invalidateSamlCookies(res);
                }
                const decodedResponse = Buffer.from(SAMLResponse, "base64").toString("utf-8");
                let samlParsed;
                xml2js.parseString(decodedResponse, { explicitArray: false }, (err, result) => {
                    if (err) {
                        app_logs_1.logger.error(`Error while parsing SAML response: ${err}`);
                        app_logs_1.logger.error(err);
                    }
                    else {
                        samlParsed = result;
                    }
                });
                if (!samlParsed) {
                    res.status(app_enums_1.enums.HttpStatusCode.INTERNAL_ERROR).send("Error parsing SAML response.");
                    return;
                }
                const assertion = samlParsed["samlp:Response"]["Assertion"];
                const userId = assertion["Subject"]["NameID"]["_"];
                const externalKey = this.getSAMLAttributeValue(samlParsed, samlConfig.binding.externalPathToInfoKey, samlConfig.binding.externalUserInfoKey);
                let orgDef = await dal_manager_1.dbManager.accessSystem.getOrganizationDefinitionFileContentV3(organizationId);
                let hash = crypto_1.default.createHmac("sha512", orgDef.organizationPrivateKey);
                hash.update(externalKey);
                let hashedEmail = hash.digest("hex");
                setSAMLCookies(req.params.grantTypeId, res, externalKey, hashedEmail, userId);
                if (!isMobileDevice) {
                    res.redirect(app_config_1.appConfig.httpServer.hostBaseRedirectUrl);
                    return;
                }
                else {
                    const user = await app_auth_1.authHelper.authenticateWithCustomMapping(organizationId, { ...samlConfig.binding, externalValueToMatch: externalKey }, trx);
                    const mobileInstallIdCheckResult = await app_auth_1.authHelper.checkMobileDeviceMismatch(organizationId, {
                        userAgent: req.header(app_constants_1.ArmonHeaders.UserAgent),
                        mobileInstallId: req.header(app_constants_1.ArmonHeaders.MobileInstallId),
                        userId: user.id,
                    }, trx);
                    if (!mobileInstallIdCheckResult) {
                        res.status(app_enums_1.enums.HttpStatusCode.BAD_REQUEST).json({
                            error: AccessTokenErrorType[AccessTokenErrorType.mobile_device_mismatch],
                            error_description: req.header(app_constants_1.ArmonHeaders.MobileInstallId),
                        });
                        return;
                    }
                    const token = await app_auth_1.authHelper.generateToken(req, { user, clientId: undefined, scopes: undefined, orgIds: [organizationId] }, trx);
                    return {
                        responseStatus: app_enums_1.enums.HttpStatusCode.SUCCESS,
                        body: {
                            tokenId: token.tokenId,
                            token: token.access_token,
                            expiresIn: token.expires_in,
                            refreshToken: token.refresh_token,
                            organization: user.organizations.find((o) => o.organization.id == organizationId),
                        },
                    };
                }
            }
        }
        catch (error) {
            app_logs_1.logger.error(error);
            invalidateSamlCookies(res);
            res.status(app_enums_1.enums.HttpStatusCode.INTERNAL_ERROR).send("An error occured during SAML login.");
            return;
        }
    }
    static getSAMLAttributeValue(samlParsed, path, attributeName) {
        let current = samlParsed;
        for (let key of path) {
            if (current[key]) {
                current = current[key];
            }
            else {
                return null;
            }
        }
        const attribute = current.find((attribute) => attribute.$.Name === attributeName);
        return attribute ? attribute.AttributeValue : null;
    }
    static async validateClientForTokenRequest(req, res, trx) {
        let clientId = null;
        if (!req.query.client_id || req.query.client_id.length !== 36) {
            res.writeHead(app_enums_1.enums.HttpStatusCode.BAD_REQUEST, "invalid client_id");
            res.end();
            return;
        }
        else {
            clientId = req.query.client_id;
        }
        let client = await dal_manager_1.dbManager.accessOAuth.getClientById(req.body.organizationId, clientId, trx);
        if (!client) {
            res.writeHead(app_enums_1.enums.HttpStatusCode.BAD_REQUEST, "unknown client");
            res.end();
            return;
        }
        if (client.redirectUris && JSON.parse(client.redirectUris).indexOf(req.query.redirect_uri) === -1) {
            res.writeHead(app_enums_1.enums.HttpStatusCode.BAD_REQUEST, "unmatched redirect_uri");
            res.end();
            return;
        }
        return client;
    }
    static async validateMobileClientForTokenRequest(req, res, trx) {
        let clientId = null;
        if (!req.query.client_id || req.query.client_id.length !== 36) {
            res.writeHead(app_enums_1.enums.HttpStatusCode.BAD_REQUEST, "invalid client_id");
            return res.end();
        }
        else {
            clientId = req.query.client_id;
        }
        let client = await dal_manager_1.dbManager.accessOAuth.getClientById(req.body.organizationId, clientId, trx);
        if (!client) {
            res.writeHead(app_enums_1.enums.HttpStatusCode.BAD_REQUEST, "unknown client");
            return res.end();
        }
        if (client.redirectUris && JSON.parse(client.redirectUris).indexOf(req.query.redirect_uri) === -1) {
            res.writeHead(app_enums_1.enums.HttpStatusCode.BAD_REQUEST, "unmatched redirect_uri");
            return res.end();
        }
        if (client.id != predefined_oAuthClients_1.PredefinedOAuthClients.ArmonMobileIOSClient.id && client.id != predefined_oAuthClients_1.PredefinedOAuthClients.ArmonMobileAndroidClient.id) {
            res.writeHead(app_enums_1.enums.HttpStatusCode.BAD_REQUEST, "not mobile client");
            return res.end();
        }
        return client;
    }
    static async checkMultiFactorAuthSettings(req, res, next, authMethod) {
        const organizationId = req.body.organizationId;
        const isMobileDevice = req.header(app_constants_1.ArmonHeaders.UserAgent).includes("mobilemanager");
        let user = null;
        switch (authMethod) {
            case app_enums_1.enums.AuthMethod.LDAP:
                await dal_manager_1.dbManager.systemTransaction(async (trx) => {
                    user = await app_auth_1.authHelper.authenticateWithUsername(req.body.organizationId, res, req.body.username, trx);
                });
                break;
            case app_enums_1.enums.AuthMethod.Native:
                await dal_manager_1.dbManager.systemTransaction(async (trx) => {
                    user = await app_auth_1.authHelper.authenticateWithPasswordCredentialsByUrlEncodedRequest(req, res, trx);
                });
                break;
            default:
                res.status(app_enums_1.enums.HttpStatusCode.FORBIDDEN).json({
                    errorMessage: "MFA is not available for SSO methods.",
                });
                break;
        }
        if (user?.id) {
            const userEmailAndPhoneNumber = await dal_manager_1.dbManager.accessUser.getUserEmailAndPhoneNumberInfo(organizationId, user.id);
            const organizationMFASettings = user.organizations.find((elem) => elem.organization.id === organizationId);
            if (organizationMFASettings.organization.multiFactorAuthenticationSettings &&
                organizationMFASettings.organization.multiFactorAuthenticationSettings.option !== app_enums_1.enums.MultifactorAuthenticationOption.DISABLE) {
                if (isMobileDevice) {
                    const viperVersion = req.header(app_constants_1.ArmonHeaders.ViperSoftwareVersion);
                    if (!viperVersion || +viperVersion < 5007007) {
                        return res.status(app_enums_1.enums.MultiFactorHttpsStatusCodes.OUTDATED_APP).json({
                            errorMessage: i18n_1.default.__({ phrase: "ERRORS.AUTH.OUTDATED_APP", locale: req.locale ?? "tr" }),
                        });
                    }
                }
                const token = crypto_1.default.randomInt(100000, 999999);
                const responseModel = {};
                if (organizationMFASettings.organization.multiFactorAuthenticationSettings.method === app_enums_1.enums.MultifactorAuthenticationVerificationMethod.EMAIL) {
                    if (!userEmailAndPhoneNumber.email) {
                        return res.status(app_enums_1.enums.HttpStatusCode.USER_PROFILE_MISSING_EMAIL_FIELDS).json({
                            errorMessage: i18n_1.default.__({ phrase: "ERRORS.USER.PROFILE_MISSING_EMAIL_FIELDS", locale: req.locale ?? "tr" }),
                        });
                    }
                    responseModel.email = userEmailAndPhoneNumber.email;
                }
                else {
                    if (!userEmailAndPhoneNumber.phoneNumber) {
                        return res.status(app_enums_1.enums.HttpStatusCode.USER_PROFILE_MISSING_PHONENUMBER_FIELDS).json({
                            errorMessage: i18n_1.default.__({ phrase: "ERRORS.USER.PROFILE_MISSING_PHONENUMBER_FIELDS", locale: req.locale ?? "tr" }),
                        });
                    }
                    responseModel.phoneNumber = userEmailAndPhoneNumber.phoneNumber;
                }
                const verificationToken = await dal_manager_1.dbManager.accessRedisCache.getVerificationToken(req.body.username);
                if (verificationToken && verificationToken.b) {
                    let userSettings = await dal_manager_1.dbManager.organizationTransaction(async (trx) => {
                        const userSettings = await dal_manager_1.dbManager.accessUser.getUserSettings({ userId: user.id, organizationId, trx });
                        return userSettings;
                    }, user.id, organizationId);
                    return res.status(app_enums_1.enums.MultiFactorHttpsStatusCodes.BLOCKED_USER).json({
                        errorMessage: i18n_1.default.__({ phrase: "ERRORS.AUTH.BLOCKED_USER", locale: userSettings.locale }),
                    });
                }
                await dal_manager_1.dbManager.accessRedisCache.setVerificationToken(req.body.username, { t: token, ets: 120, nfa: 0, b: false });
                const instanceData = { t: token, receiverUserIds: [user.id] };
                const sendVerification = async function sendVerification(trx) {
                    const notificationId = await (0, dal_memcache_1.getCacheUniqueNotificationIdOfOrganization)(req.body.organizationId, app_enums_1.enums.NotificationType.SendingVerificationToken, trx);
                    const notificationEventId = await dal_manager_1.dbManager.accessNotifications.addNotificationEvent({
                        createdT: new Date(),
                        notificationId,
                        organizationId: organizationId,
                        instanceData,
                        trx,
                    });
                    await (0, messageBroker_notification_pub_1.publishToNotificationService)({
                        a: 0,
                        i: notificationEventId,
                        n: app_enums_1.enums.AmqpMessageCode.Empty,
                        o: organizationId,
                        v: "1",
                        d: instanceData,
                    });
                };
                switch (organizationMFASettings.organization.multiFactorAuthenticationSettings.option) {
                    case app_enums_1.enums.MultifactorAuthenticationOption.REQUIRED:
                        await dal_manager_1.dbManager.systemTransaction(sendVerification);
                        break;
                    case app_enums_1.enums.MultifactorAuthenticationOption.OPTIONAL:
                        if (user.userProfile.mfaSettings.enabledStatus) {
                            await dal_manager_1.dbManager.systemTransaction(sendVerification);
                        }
                        else {
                            return await dal_manager_1.dbManager.systemTransaction(async (trx) => {
                                return await AuthManager.authenticateToken(req, res, next, trx, user);
                            });
                        }
                        break;
                }
                res.status(app_enums_1.enums.MultiFactorHttpsStatusCodes.AUTH_REQ).send(responseModel);
            }
        }
    }
}
exports.AuthManager = AuthManager;
AuthManager._samlServiceProviderMap = new Map();
AuthManager._samlIdentityProviderMap = new Map();
const setOpenIdCookies = (grantTypeId, res, tokenSet, decodedAccessToken, externalKey) => {
    res.cookie("xopenididtoken", tokenSet.id_token, {
        maxAge: 1704085200,
        httpOnly: true,
        secure: process.env.HTTP_PROTOCOL === "https",
        path: "/auth",
    });
    res.cookie("xopenidaccesstoken", tokenSet.access_token, {
        maxAge: 1704085200,
        httpOnly: true,
        secure: process.env.HTTP_PROTOCOL === "https",
        path: "/auth",
    });
    res.cookie("xopenidrefreshtoken", tokenSet.refresh_token, {
        maxAge: 1704085200,
        httpOnly: true,
        secure: process.env.HTTP_PROTOCOL === "https",
        path: "/auth",
    });
    res.cookie("xopenidusername", decodedAccessToken[externalKey ?? "preferred_username"], {
        maxAge: 1704085200,
        httpOnly: true,
        secure: process.env.HTTP_PROTOCOL === "https",
        path: "/auth",
    });
    res.cookie("xopenidgrantid", grantTypeId, {
        maxAge: 1704085200,
        httpOnly: true,
        secure: process.env.HTTP_PROTOCOL === "https",
        path: "/auth",
    });
};
const setSAMLCookies = (grantTypeId, res, email, hashedEmail, userId) => {
    res.cookie("xsamluser", email, {
        maxAge: 1704085200,
        httpOnly: true,
        secure: process.env.HTTP_PROTOCOL === "https",
        path: "/auth",
    });
    res.cookie("xsamluserId", userId, {
        maxAge: 1704085200,
        httpOnly: true,
        secure: process.env.HTTP_PROTOCOL === "https",
        path: "/auth",
    });
    res.cookie("xsamlhash", hashedEmail, {
        maxAge: 1704085200,
        httpOnly: true,
        secure: process.env.HTTP_PROTOCOL === "https",
        path: "/auth",
    });
    res.cookie("xsamlgrantid", grantTypeId, {
        maxAge: 1704085200,
        httpOnly: true,
        secure: process.env.HTTP_PROTOCOL === "https",
        path: "/auth",
    });
};
const invalidateOpenIdCookies = (res) => {
    res.clearCookie("xopenididtoken", { path: "/auth" });
    res.clearCookie("xopenidaccesstoken", { path: "/auth" });
    res.clearCookie("xopenidrefreshtoken", { path: "/auth" });
    res.clearCookie("xopenidusername", { path: "/auth" });
    res.clearCookie("xopenidgrantid", { path: "/auth" });
};
const invalidateSamlCookies = (res) => {
    res.clearCookie("xsamluser", { path: "/auth" });
    res.clearCookie("xsamluserId", { path: "/auth" });
    res.clearCookie("xsamlhash", { path: "/auth" });
    res.clearCookie("xsamlgrantid", { path: "/auth" });
    res.clearCookie("xsamlstarted", { path: "/auth" });
};
exports.authManager = new AuthManager();
