"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.authHelper = exports.UserJwtPayload = exports.ControlPanelJwtPayload = void 0;
const api_securityhelper_1 = require("./api/api.securityhelper");
const app_config_1 = require("./app.config");
const app_logs_1 = require("./app.logs");
const app_permission_1 = require("./app.permission");
const app_util_1 = require("./app.util");
const dal_manager_1 = require("./dal/dal.manager");
const dal_db_armon_schema_1 = require("./dal/db/armon/dal.db.armon.schema");
const app_constants_1 = require("./app.constants");
var jwt = require("jsonwebtoken");
class ControlPanelJwtPayload {
    constructor(decodedPayload) {
        this._id = decodedPayload.id;
        this._organizationId = decodedPayload.organizationId;
        this._caption = decodedPayload.caption;
        this._name = decodedPayload.name;
    }
    get id() {
        return this._id;
    }
    get organizationId() {
        return this._organizationId;
    }
    get caption() {
        return this._caption;
    }
    get name() {
        return this._name;
    }
}
exports.ControlPanelJwtPayload = ControlPanelJwtPayload;
class UserJwtPayload {
    constructor(decodedPayload) {
        this._tokenId = decodedPayload.i;
        this._userId = decodedPayload.u;
        this._organizations = [];
        for (let indexO = 0; indexO < decodedPayload.o.length; indexO++) {
            let o = decodedPayload.o[indexO];
            let organization = {
                organizationId: o.i,
                isDisabled: o.d,
                roleId: o.r,
                unitIds: o.u.map((u) => u.i),
                unitClaim: app_permission_1.permissions.considerPermissions((0, app_util_1.flatten)(o.u.map((m) => m.p.map((mm) => mm + ":" + m.i)))),
                organizationClaim: app_permission_1.permissions.considerPermissions(o.p),
            };
            this._organizations.push(organization);
        }
    }
    get userId() {
        return this._userId;
    }
    get tokenId() {
        return this._tokenId;
    }
    organizationUnitIdsOfOrganization(organizationId) {
        return this._organizations.find((o) => o.organizationId === organizationId).unitIds;
    }
    getOrganizationId() {
        return this._organizations[0].organizationId;
    }
    isPermittedForOrganization(organizationId, permissions) {
        let org = this._organizations.find((o) => o.organizationId === organizationId);
        if (!org) {
            return false;
        }
        return org.organizationClaim.isPermitted(permissions);
    }
    getMissingPermissions(organizationId, permissions, controlForUnits) {
        let org = this._organizations.find((o) => o.organizationId === organizationId);
        if (!org) {
            return [];
        }
        let missingPermissionsForUnits = [];
        let missingPermissions = [];
        let missingPermissionsForOrganization = this.getMissingPermissionsForOrganization(organizationId, permissions);
        if (controlForUnits) {
            missingPermissionsForUnits = this.getMissingPermissionsForAnyUnit(organizationId, permissions);
            for (const mpo of missingPermissionsForOrganization) {
                if (missingPermissionsForUnits.indexOf(mpo) > -1)
                    missingPermissions.push(mpo);
            }
            for (const mpu of missingPermissionsForUnits) {
                if (missingPermissionsForOrganization.indexOf(mpu) > -1)
                    missingPermissions.push(mpu);
            }
            missingPermissions = missingPermissions.filter((elem, pos) => {
                return missingPermissions.indexOf(elem) == pos;
            });
        }
        else {
            missingPermissions = missingPermissionsForOrganization;
        }
        return missingPermissions;
    }
    getMissingPermissionsForOrganization(organizationId, permissions) {
        let org = this._organizations.find((o) => o.organizationId === organizationId);
        if (!org) {
            return [];
        }
        return org.organizationClaim.findMissingPermissions(permissions);
    }
    isPermittedForUnit(organizationId, permissions, unitId) {
        let org = this._organizations.find((o) => o.organizationId === organizationId);
        if (!org) {
            return false;
        }
        return org.unitClaim.isPermitted(typeof permissions === "string" ? permissions + ":" + unitId : permissions.map((m) => m + ":" + unitId));
    }
    getMissingPermissionsForAnyUnit(organizationId, permissions) {
        let org = this._organizations.find((o) => o.organizationId === organizationId);
        if (!org) {
            return [];
        }
        if (typeof permissions === "string") {
            return [permissions];
        }
        else if (permissions.length === 1) {
            return permissions;
        }
        else {
            let result = [];
            let allUnitIds = org.unitIds;
            let presentPermissions = [];
            for (let index = 0; index < allUnitIds.length; index++) {
                let unitId = allUnitIds[index];
                let pp = org.unitClaim.findPresentPermissions(typeof permissions === "string" ? permissions + ":" + unitId : permissions.map((m) => m + ":" + unitId));
                for (const upp of pp) {
                    const parts = upp.split(":");
                    presentPermissions.push(parts[0] + ":" + parts[1]);
                }
            }
            presentPermissions = presentPermissions.filter((elem, pos) => {
                return presentPermissions.indexOf(elem) == pos;
            });
            for (const perm of permissions) {
                if (presentPermissions.indexOf(perm) === -1)
                    result.push(perm);
            }
            return result;
        }
    }
    hasPermissionForOrganizationOrAnyUnit(organizationId, permission) {
        let org = this._organizations.find((o) => o.organizationId === organizationId);
        if (!org) {
            return false;
        }
        return org.organizationClaim.isPermitted(permission) || org.unitClaim.isPermitted(permission);
    }
    isPermittedForAnyUnit(organizationId, permissions) {
        return this.getPermittedUnitIdsFor(organizationId, permissions).length > 0;
    }
    getPermittedUnitIdsFor(organizationId, permissions) {
        let org = this._organizations.find((o) => o.organizationId === organizationId);
        if (!org || !org.unitIds || org.unitIds.length === 0) {
            return [];
        }
        let allUnitIds = org.unitIds;
        let permittedUnitIds = [];
        for (let index = 0; index < allUnitIds.length; index++) {
            let unitId = allUnitIds[index];
            if (org.unitClaim.isPermitted(typeof permissions === "string" ? permissions + ":" + unitId : permissions.map((m) => m + ":" + unitId))) {
                permittedUnitIds.push(unitId);
            }
        }
        return permittedUnitIds;
    }
    isPermittedForAtLeastOneOfUnits(organizationId, permissions, unitIds) {
        let org = this._organizations.find((o) => o.organizationId === organizationId);
        if (!org) {
            return false;
        }
        for (let index = 0; index < unitIds.length; index++) {
            let unitId = unitIds[index];
            if (org.unitClaim.isPermitted(typeof permissions === "string" ? permissions + ":" + unitId : permissions.map((m) => m + ":" + unitId))) {
                return true;
            }
        }
        return false;
    }
    isPermittedForAllOfUnits(organizationId, permissions, unitIds) {
        let org = this._organizations.find((o) => o.organizationId === organizationId);
        if (!org) {
            return false;
        }
        let units = unitIds.join(",");
        return org.unitClaim.isPermitted(typeof permissions === "string" ? permissions + ":" + units : permissions.map((m) => m + ":" + units));
    }
}
exports.UserJwtPayload = UserJwtPayload;
class AuthHelper {
    constructor() {
        this.calculateCustomTokenExpiration = (accessToken, refreshToken, accessTokenOffset, refreshTokenOffset) => {
            const accessTokenOpen = jwt.decode(accessToken, { complete: true });
            const accessTokenExpireAt = accessTokenOpen.payload["exp"] - (accessTokenOffset ?? 10);
            const refreshTokenOpen = jwt.decode(refreshToken, { complete: true });
            const refreshTokenExpireAt = refreshTokenOpen?.payload?.["exp"] ? refreshTokenOpen.payload["exp"] - (refreshTokenOffset ?? 10) : null;
            return {
                accessTokenExpireAt,
                refreshTokenExpireAt,
            };
        };
        this.checkMobileDeviceMismatch = async (organizationId, params, trx) => {
            const isCheckMobileDeviceMismatchOnLoginEnabled = await dal_manager_1.dbManager.accessOrganization.isCheckMobileDeviceMismatchOnLoginEnabled(organizationId, trx);
            if (isCheckMobileDeviceMismatchOnLoginEnabled) {
                const isMobileDevice = params.userAgent.includes("mobilemanager");
                if (isMobileDevice) {
                    const mobileInstallId = params.mobileInstallId;
                    if (mobileInstallId) {
                        const latestMobileInstallId = await dal_manager_1.dbManager.accessOAuth.getLatestMobileInstallIdentifier(organizationId, params.userId, trx);
                        if (latestMobileInstallId) {
                            if (mobileInstallId != latestMobileInstallId) {
                                return false;
                            }
                        }
                        else {
                            app_logs_1.logger.info("Mobile device check is enabled for organization, but user's existing tokens do not contain any mobile install id.");
                            app_logs_1.logger.info("Org: " + organizationId + " - User: " + params.userId + " - Agent: " + params.userAgent);
                        }
                    }
                    else {
                        app_logs_1.logger.info("Mobile device check is enabled for organization, but user did not send and id.");
                        app_logs_1.logger.info("Org: " + organizationId + " - User: " + params.userId + " - Agent: " + params.userAgent);
                    }
                }
            }
            return true;
        };
    }
    async generateToken(req, params, trx) {
        const token = await dal_manager_1.dbManager.accessOAuth.upsertClientToken(params.orgIds[0], {
            user: params.user,
            ip: req.ip,
            userAgent: req.header(app_constants_1.ArmonHeaders.UserAgent),
            mobileInstallId: req.header(app_constants_1.ArmonHeaders.MobileInstallId),
            userAgentUniqueId: req.header(app_constants_1.ArmonHeaders.UserAgentId),
            clientId: params.clientId,
            scopes: params.scopes,
            custom: params.custom,
        }, trx);
        let organizationAuths = params.user.organizationAuths;
        if (params.orgIds && params.orgIds.length > 0) {
            organizationAuths = organizationAuths.filter((o) => params.orgIds.some((x) => x == o.i));
        }
        const userJwtPayload = { i: token.id, u: params.user.id, o: organizationAuths };
        if (params.clientId) {
            userJwtPayload.c = params.clientId;
        }
        if (params.scopes) {
            userJwtPayload.s = params.scopes;
        }
        const accessTokenExpiresIn = req.header(app_constants_1.ArmonHeaders.UserAgent).startsWith("mobilemanager")
            ? 3600 * 24 * 7
            : params.custom?.accessTokenExpireAt
                ? params.custom.accessTokenExpireAt - Math.floor(new Date().getTime() / 1000)
                : 3600;
        const jwtToken = jwt.sign(userJwtPayload, app_config_1.appConfig.jwtSecret, { expiresIn: accessTokenExpiresIn });
        return Promise.resolve({
            tokenId: token.id,
            access_token: jwtToken,
            expires_in: accessTokenExpiresIn,
            refresh_token: token.refreshToken,
            token_type: "Bearer",
        });
    }
    async refreshToken(req, params, trx) {
        let decodedToken = jwt.decode(params.authToken);
        if (!decodedToken) {
            (0, api_securityhelper_1.throwAuthError)("Token is not provided");
        }
        const now = new Date();
        const accessTokenExpireAt = params.isMobileApplication
            ? Math.floor(now.getTime() / 1000) + 3600 * 24 * 7
            : params.custom?.accessTokenExpireAt ?? Math.floor(now.getTime() / 1000) + 3600;
        const refreshTokenExpireAt = params.isMobileApplication
            ? Math.floor(now.getTime() / 1000) + 3600 * 24 * 7 * 4
            : params.custom?.refreshTokenExpireAt ?? Math.floor(now.getTime() / 1000) + 3600 * 24 * 7;
        const accessTokenExpiresIn = accessTokenExpireAt - Math.floor(now.getTime() / 1000);
        let userToken = await dal_manager_1.dbManager.accessOAuth.refreshToken(params.orgIds[0], {
            authTokenId: decodedToken.i,
            expirations: {
                accessTokenExpireAt,
                refreshTokenExpireAt,
            },
            refreshToken: params.refreshToken,
            userAgent: req.header(app_constants_1.ArmonHeaders.UserAgent),
            mobileInstallId: req.header(app_constants_1.ArmonHeaders.MobileInstallId),
            userAgentUniqueId: req.header(app_constants_1.ArmonHeaders.UserAgentId),
        }, trx);
        let organizationAuths = userToken.user.organizationAuths;
        if (params.orgIds && params.orgIds.length > 0) {
            organizationAuths = organizationAuths.filter((o) => params.orgIds.some((x) => x == o.i));
        }
        let userJwtPayload = {
            i: userToken.token.id,
            u: userToken.user.id,
            o: organizationAuths,
        };
        if (params.clientId) {
            userJwtPayload.c = params.clientId;
        }
        if (userToken.token.scopes) {
            userJwtPayload.s = userToken.token.scopes;
        }
        const jwtToken = jwt.sign(userJwtPayload, app_config_1.appConfig.jwtSecret, {
            expiresIn: accessTokenExpiresIn,
        });
        return Promise.resolve({
            token: {
                tokenId: userToken.token.id,
                access_token: jwtToken,
                expires_in: accessTokenExpiresIn,
                refresh_token: userToken.token.refreshToken,
                token_type: "Bearer",
            },
            user: userToken.user,
        });
    }
    async authenticateWithPasswordCredentialsByUrlEncodedRequest(req, res, trx) {
        return new Promise((resolve, reject) => {
            if (!res.finished) {
                if (!req.body.username) {
                    reject(new Error("Username is required"));
                }
                else if (!req.body.password) {
                    reject(new Error("Password is required"));
                }
                else {
                    dal_manager_1.dbManager.accessOAuth
                        .getArmonApplicationUserByPassword(req.body.organizationId, req.body.username, req.body.password, trx)
                        .then((user) => resolve(user))
                        .catch((err) => reject(err));
                }
            }
        });
    }
    async authenticateWithUsername(organizationId, res, username, trx) {
        return new Promise((resolve, reject) => {
            if (!res.finished) {
                if (!username) {
                    reject(new Error("Username is required"));
                }
                else {
                    dal_manager_1.dbManager.accessOAuth
                        .getArmonApplicationUserByUsername(organizationId, username, trx)
                        .then((user) => resolve(user))
                        .catch((err) => reject(err));
                }
            }
        });
    }
    async authenticateWithCustomMapping(organizationId, mapping, trx) {
        let qCondition = null;
        const binding = [];
        if (mapping.isExtension) {
            qCondition = `unaccent(lower(uop."extensionFields"->>'${mapping.internalPropertyToMatch}')) = unaccent(lower($${binding.length + 1}))`;
            binding.push(mapping.externalValueToMatch);
        }
        else {
            if (mapping.internalPropertyToMatch === "username") {
                qCondition = `unaccent(lower(u."${mapping.internalPropertyToMatch}")) = unaccent(lower($${binding.length + 1}))`;
                binding.push(mapping.externalValueToMatch);
            }
            else {
                qCondition = `unaccent(lower(uop."${mapping.internalPropertyToMatch}")) = unaccent(lower($${binding.length + 1}))`;
                binding.push(mapping.externalValueToMatch);
            }
        }
        const { rows, rowCount } = await trx.query(`SELECT u.id FROM "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.users}" u
				INNER JOIN "${organizationId}"."${dal_db_armon_schema_1.ArmonSchema.tableNames.userOrganizationProfiles}" uop
			 ON uop."userId" = u.id
			 WHERE ${qCondition}`, binding);
        app_logs_1.logger.debug(`openid binding: ${qCondition} ${binding}`);
        if (rowCount === 0) {
            app_logs_1.logger.warn(`User mapping [${mapping.externalValueToMatch}] not found in the database!`);
            throw new Error(mapping.externalValueToMatch);
        }
        if (rowCount > 1) {
            app_logs_1.logger.warn(`Multiple users for mapping[${mapping.externalValueToMatch}] found!`);
            throw new Error(mapping.externalValueToMatch);
        }
        const user = rows[0];
        const armonApplicationUser = await dal_manager_1.dbManager.accessOAuth.getArmonApplicationUser(organizationId, user.id, trx);
        if (!armonApplicationUser) {
            app_logs_1.logger.warn(`User Profile mapping [${mapping.externalValueToMatch}] not found in the database!`);
            throw new Error(mapping.externalValueToMatch);
        }
        return dal_manager_1.dbManager.accessOAuth.mapToArmonApplicationUser(armonApplicationUser);
    }
}
exports.authHelper = new AuthHelper();
